#!/usr/bin/perl
# Sample vectors for "sha1sum".

# Copyright (C) 2000-2017 Free Software Foundation, Inc.

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

use strict;

(my $program_name = $0) =~ s|.*/||;

# Turn off localization of executable's output.
@ENV{qw(LANGUAGE LANG LC_ALL)} = ('C') x 3;

# The data from which these tests were derived came from here:
# http://web.archive.org/web/20060505234703/http://csrc.nist.gov/cryptval/shs/sha1-vectors.zip

my @Tests =
    (
     ['1', {IN=> {f=> '0 1 ^'}},
     {OUT=>'DA39A3EE5E6B4B0D3255BFEF95601890AFD80709'}],
     ['2', {IN=> {f=> '5 0 2 1 2 1 2 ^'}},
     {OUT=>'3CDF2936DA2FC556BFA533AB1EB59CE710AC80E5'}],
     ['3', {IN=> {f=> '5 0 1 3 4 4 4 ^'}},
     {OUT=>'19C1E2048FA7393CFBF2D310AD8209EC11D996E5'}],
     ['4', {IN=> {f=> '7 0 4 3 4 4 1 4 4 ^'}},
     {OUT=>'CA775D8C80FAA6F87FA62BECA6CA6089D63B56E5'}],
     ['5', {IN=> {f=> '10 0 4 1 5 3 4 4 3 1 3 4 ^'}},
     {OUT=>'71AC973D0E4B50AE9E5043FF4D615381120A25A0'}],
     ['6', {IN=> {f=> '10 0 3 1 6 5 5 1 3 6 6 4 ^'}},
     {OUT=>'A6B5B9F854CFB76701C3BDDBF374B3094EA49CBA'}],
     ['7', {IN=> {f=> '13 1 3 2 5 3 3 3 4 6 6 1 4 6 2 ^'}},
     {OUT=>'D87A0EE74E4B9AD72E6847C87BDEEB3D07844380'}],
     ['8', {IN=> {f=> '16 1 3 5 5 1 2 1 3 3 6 3 5 2 3 5 7 2 ^'}},
     {OUT=>'1976B8DD509FE66BF09C9A8D33534D4EF4F63BFD'}],
     ['9', {IN=> {f=> '15 1 8 1 5 3 2 7 4 5 6 7 3 3 1 6 3 ^'}},
     {OUT=>'5A78F439B6DB845BB8A558E4CEB106CD7B7FF783'}],
     ['10', {IN=> {f=> '15 1 4 6 8 2 1 4 2 5 1 6 8 8 6 4 7 ^'}},
     {OUT=>'F871BCE62436C1E280357416695EE2EF9B83695C'}],
     ['11', {IN=> {f=> '18 1 1 2 7 3 8 6 7 5 4 3 4 3 5 3 3 2 6 8 ^'}},
     {OUT=>'62B243D1B780E1D31CF1BA2DE3F01C72AEEA0E47'}],
     ['12', {IN=> {f=> '16 0 9 8 1 8 1 7 6 7 7 1 2 6 9 5 4 7 ^'}},
     {OUT=>'1698994A273404848E56E7FDA4457B5900DE1342'}],
     ['13', {IN=> {f=> '18 0 7 1 7 3 9 4 7 7 5 2 8 1 7 8 2 7 2 9 ^'}},
     {OUT=>'056F4CDC02791DA7ED1EB2303314F7667518DEEF'}],
     ['14', {IN=> {f=> '19 1 2 3 1 8 8 6 9 10 3 10 8 9 2 4 1 5 1 5 9 ^'}},
     {OUT=>'9FE2DA967BD8441EEA1C32DF68DDAA9DC1FC8E4B'}],
     ['15', {IN=> {f=> '19 1 8 5 4 8 1 3 9 5 7 7 2 7 2 7 8 7 4 8 10 ^'}},
     {OUT=>'73A31777B4ACE9384EFA8BBEAD45C51A71ABA6DD'}],
     ['16', {IN=> {f=> '20 1 1 9 7 4 1 4 5 1 10 8 6 4 4 9 9 9 8 2 9 10 ^'}},
     {OUT=>'3F9D7C4E2384EDDABFF5DD8A31E23DE3D03F42AC'}],
     ['17', {IN=> {f=> '19 1 11 6 7 7 2 6 2 6 10 6 9 10 5 11 1 6 8 11 4 ^'}},
     {OUT=>'4814908F72B93FFD011135BEE347DE9A08DA838F'}],
     ['18', {IN=> {f=> '22 0 10 5 10 3 7 8 9 9 1 1 1 10 2 1 5 10 2 9 9 9 7 8 ^'}},
     {OUT=>'0978374B67A412A3102C5AA0B10E1A6596FC68EB'}],
     ['19', {IN=> {f=> '21 0 1 10 1 6 9 4 2 5 2 11 8 12 12 9 8 1 3 10 7 11 12 ^'}},
     {OUT=>'44AD6CB618BD935460D46D3F921D87B99AB91C1E'}],
     ['20', {IN=> {f=> '24 1 3 9 5 12 3 4 2 9 12 11 6 6 1 1 9 5 9 1 4 9 4 10 8 9 ^'}},
     {OUT=>'02DC989AF265B09CF8485640842128DCF95E9F39'}],
     ['21', {IN=> {f=> '25 1 3 2 3 11 1 12 5 6 2 7 8 4 8 8 9 9 8 4 9 1 4 8 10 9 9 ^'}},
     {OUT=>'67507B8D497B35D6E99FC01976D73F54AECA75CF'}],
     ['22', {IN=> {f=> '23 0 11 10 7 10 10 6 10 9 4 5 10 5 8 4 1 10 12 4 6 1 8 11 6 ^'}},
     {OUT=>'1EAE0373C1317CB60C36A42A867B716039D441F5'}],
     ['23', {IN=> {f=> '22 0 12 8 10 4 3 8 5 5 7 11 13 11 12 11 4 12 3 6 5 11 10 5 ^'}},
     {OUT=>'9C3834589E5BFFAC9F50950E0199B3EC2620BEC8'}],
     ['24', {IN=> {f=> '26 1 10 9 6 9 7 2 10 4 4 5 5 2 12 13 5 3 1 10 1 4 7 8 13 13 12 9 ^'}},
     {OUT=>'209F7ABC7F3B878EE46CDF3A1FBB9C21C3474F32'}],
     ['25', {IN=> {f=> '31 0 2 6 5 4 7 3 10 6 13 6 3 9 6 2 10 5 3 8 4 1 11 3 5 3 7 11 1 12 9 12 5 ^'}},
     {OUT=>'05FC054B00D97753A9B3E2DA8FBBA3EE808CEF22'}],
     ['26', {IN=> {f=> '27 1 14 5 1 3 7 2 3 9 3 4 14 4 4 10 8 5 14 1 11 12 12 10 4 13 7 11 9 ^'}},
     {OUT=>'0C4980EA3A46C757DFBFC5BAA38AC6C8E72DDCE7'}],
     ['27', {IN=> {f=> '30 1 4 9 5 5 8 9 5 10 4 2 4 7 9 9 6 3 5 1 8 3 2 13 3 14 9 8 9 10 14 10 ^'}},
     {OUT=>'96A460D2972D276928B69864445BEA353BDCFFD2'}],
     ['28', {IN=> {f=> '27 0 12 9 5 8 7 2 14 12 3 8 14 6 6 4 7 5 7 10 7 11 10 1 9 6 7 12 14 ^'}},
     {OUT=>'F3EF04D8FA8C6FA9850F394A4554C080956FA64B'}],
     ['29', {IN=> {f=> '24 0 12 9 9 2 11 13 12 11 11 6 14 13 10 5 6 8 10 4 3 11 11 14 5 14 ^'}},
     {OUT=>'F2A31D875D1D7B30874D416C4D2EA6BAF0FFBAFE'}],
     ['30', {IN=> {f=> '24 0 15 4 5 3 8 12 15 8 14 15 9 12 12 3 10 13 6 11 10 4 13 14 8 8 ^'}},
     {OUT=>'F4942D3B9E9588DCFDC6312A84DF75D05F111C20'}],
     ['31', {IN=> {f=> '28 1 1 8 1 5 11 4 9 12 4 13 15 5 9 11 7 14 11 1 11 7 8 8 11 1 13 15 12 13 ^'}},
     {OUT=>'310207DF35B014E4676D30806FA34424813734DD'}],
     ['32', {IN=> {f=> '32 1 5 8 3 8 10 7 8 1 5 13 12 14 5 3 6 4 12 15 6 6 10 11 13 9 1 11 6 10 3 7 14 2 ^'}},
     {OUT=>'4DA1955B2FA7C7E74E3F47D7360CE530BBF57CA3'}],
     ['33', {IN=> {f=> '31 0 10 3 5 1 14 11 11 16 1 2 2 11 6 13 15 12 6 5 16 2 14 2 10 12 2 5 5 6 10 13 15 ^'}},
     {OUT=>'74C4BC5B26FB4A08602D40CCEC6C6161B6C11478'}],
     ['34', {IN=> {f=> '34 0 3 10 8 16 9 5 12 15 4 11 13 3 6 5 10 8 1 3 9 3 11 1 2 16 12 10 6 1 9 1 16 5 6 14 ^'}},
     {OUT=>'0B103CE297338DFC7395F7715EE47539B556DDB6'}],
     ['35', {IN=> {f=> '30 1 1 12 4 4 2 15 13 15 11 15 5 11 9 7 15 16 6 16 12 3 2 10 16 5 5 7 1 7 11 16 ^'}},
     {OUT=>'EFC72D99E3D2311CE14190C0B726BDC68F4B0821'}],
     ['36', {IN=> {f=> '34 0 7 9 11 2 5 5 5 4 13 13 14 4 7 12 6 4 8 2 9 9 13 13 3 3 6 7 16 7 6 15 5 8 15 14 ^'}},
     {OUT=>'660EDAC0A8F4CE33DA0D8DBAE597650E97687250'}],
     ['37', {IN=> {f=> '36 1 4 6 16 15 11 14 14 4 7 10 3 4 10 3 6 7 14 4 6 6 5 2 7 8 16 2 12 16 10 14 3 2 3 7 14 3 ^'}},
     {OUT=>'FE0A55A988B3B93946A63EB36B23785A5E6EFC3E'}],
     ['38', {IN=> {f=> '32 0 15 10 9 1 14 10 14 6 6 16 3 2 3 8 3 12 8 11 17 3 9 7 16 14 4 11 15 5 13 9 5 17 ^'}},
     {OUT=>'0CBDF2A5781C59F907513147A0DE3CC774B54BF3'}],
     ['39', {IN=> {f=> '30 0 17 17 13 8 2 6 8 16 1 12 5 17 2 9 8 10 13 14 11 17 12 5 14 9 11 9 11 4 11 12 ^'}},
     {OUT=>'663E40FEE5A44BFCB1C99EA5935A6B5BC9F583B0'}],
     ['40', {IN=> {f=> '30 1 16 6 10 5 8 3 17 16 14 1 15 15 15 6 13 2 11 6 13 11 13 4 6 7 11 11 12 16 13 16 ^'}},
     {OUT=>'00162134256952DD9AE6B51EFB159B35C3C138C7'}],
     ['41', {IN=> {f=> '33 1 16 16 14 16 2 4 16 11 6 15 7 4 17 6 5 7 6 3 14 16 5 17 11 13 1 1 14 13 3 6 14 5 16 ^'}},
     {OUT=>'CEB88E4736E354416E2010FC1061B3B53B81664B'}],
     ['42', {IN=> {f=> '39 1 2 16 13 7 8 6 2 15 1 9 12 4 4 11 13 7 2 11 9 18 4 5 4 8 2 14 9 9 1 8 13 11 15 8 5 9 10 16 7 ^'}},
     {OUT=>'A6A2C4B6BCC41DDC67278F3DF4D8D0B9DD7784EF'}],
     ['43', {IN=> {f=> '34 0 2 7 1 1 17 13 6 11 10 8 5 12 15 6 15 10 12 4 18 1 2 8 11 12 16 10 12 18 11 16 12 11 17 6 ^'}},
     {OUT=>'C23D083CD8820B57800A869F5F261D45E02DC55D'}],
     ['44', {IN=> {f=> '34 1 4 7 13 7 10 7 10 6 1 12 7 18 11 18 2 10 15 10 14 8 18 9 9 12 12 3 13 12 6 4 9 17 13 17 ^'}},
     {OUT=>'E8AC31927B78DDEC41A31CA7A44EB7177165E7AB'}],
     ['45', {IN=> {f=> '40 0 5 7 3 2 1 17 14 4 16 6 13 1 13 6 6 10 1 3 18 3 11 7 9 5 7 11 17 1 9 16 5 15 10 17 3 8 15 17 8 12 ^'}},
     {OUT=>'E864EC5DBAB0F9FF6984AB6AD43A8C9B81CC9F9C'}],
     ['46', {IN=> {f=> '40 0 11 3 15 17 11 1 1 4 3 14 18 4 2 18 8 15 6 4 6 3 15 11 16 10 17 17 9 6 3 2 6 16 4 9 12 6 8 1 11 17 ^'}},
     {OUT=>'CFED6269069417A84D6DE2347220F4B858BCD530'}],
     ['47', {IN=> {f=> '37 1 2 19 12 8 16 14 2 9 16 2 6 6 7 9 10 9 11 9 14 11 15 5 16 9 2 17 2 8 15 8 4 3 14 14 16 16 12 ^'}},
     {OUT=>'D9217BFB46C96348722C3783D29D4B1A3FEDA38C'}],
     ['48', {IN=> {f=> '37 1 11 10 16 12 11 7 14 14 14 6 10 10 1 6 13 19 5 6 4 7 12 12 10 5 10 15 15 8 5 13 17 13 5 6 14 1 19 ^'}},
     {OUT=>'DEC24E5554F79697218D317315FA986229CE3350'}],
     ['49', {IN=> {f=> '38 1 2 6 5 17 9 11 18 18 8 6 13 15 3 3 15 5 13 18 3 2 5 5 14 7 13 4 17 7 2 17 3 18 15 7 15 16 18 11 ^'}},
     {OUT=>'83A099DF7071437BA5495A5B0BFBFEFE1C0EF7F3'}],
     ['50', {IN=> {f=> '38 1 12 8 6 3 17 12 13 19 15 9 7 17 16 15 3 11 11 5 2 13 19 16 2 4 16 7 8 1 2 9 17 12 3 5 18 19 11 9 ^'}},
     {OUT=>'AA3198E30891A83E33CE3BFA0587D86A197D4F80'}],
     ['51', {IN=> {f=> '39 1 14 16 14 8 9 16 5 1 6 3 17 18 16 9 1 15 9 10 9 19 1 3 3 20 11 13 17 1 19 8 3 4 3 7 1 14 19 19 19 ^'}},
     {OUT=>'9B6ACBEB4989CBEE7015C7D515A75672FFDE3442'}],
     ['52', {IN=> {f=> '37 1 18 13 11 5 18 4 19 10 6 19 11 17 10 10 7 9 13 16 9 10 18 4 12 5 16 5 20 12 3 8 10 1 18 1 6 20 14 ^'}},
     {OUT=>'B021EB08A436B02658EAA7BA3C88D49F1219C035'}],
     ['53', {IN=> {f=> '36 0 8 9 6 12 11 7 7 3 17 13 6 20 17 9 20 16 10 12 17 8 11 8 11 10 5 10 14 18 8 19 9 12 12 2 20 19 ^'}},
     {OUT=>'CAE36DAB8AEA29F62E0855D9CB3CD8E7D39094B1'}],
     ['54', {IN=> {f=> '39 0 12 16 20 3 9 9 19 17 13 13 4 17 2 11 7 14 3 6 16 13 10 13 5 16 10 2 8 2 17 19 4 17 7 19 6 9 15 15 6 ^'}},
     {OUT=>'02DE8BA699F3C1B0CB5AD89A01F2346E630459D7'}],
     ['55', {IN=> {f=> '43 0 7 2 18 5 7 18 5 2 15 7 11 10 9 3 2 14 19 3 11 8 18 15 5 3 5 12 15 16 10 17 7 19 16 2 1 16 6 3 19 12 5 18 16 ^'}},
     {OUT=>'88021458847DD39B4495368F7254941859FAD44B'}],
     ['56', {IN=> {f=> '49 1 9 11 2 1 12 11 14 12 14 10 4 11 6 8 16 7 5 11 20 8 17 4 14 4 15 3 2 2 4 3 2 3 14 15 10 2 12 7 3 7 20 20 19 10 2 3 1 10 20 ^'}},
     {OUT=>'91A165295C666FE85C2ADBC5A10329DAF0CB81A0'}],
     ['57', {IN=> {f=> '36 0 19 20 12 5 19 21 5 21 11 14 19 1 17 8 9 4 19 3 17 1 14 21 14 7 6 5 20 14 21 20 4 6 21 7 11 12 ^'}},
     {OUT=>'4B31312EAF8B506811151A9DBD162961F7548C4B'}],
     ['58', {IN=> {f=> '41 0 12 9 11 6 16 18 18 10 11 20 6 12 11 5 7 21 19 18 6 15 21 10 4 14 9 19 10 3 3 5 13 1 8 12 3 13 9 7 10 17 14 ^'}},
     {OUT=>'3FE70971B20558F7E9BAC303ED2BC14BDE659A62'}],
     ['59', {IN=> {f=> '45 0 10 6 8 3 17 18 3 21 19 6 17 15 4 9 15 9 15 14 4 7 14 8 10 13 4 11 10 7 6 21 1 14 5 11 7 7 2 13 13 3 9 13 8 14 20 ^'}},
     {OUT=>'93FB769D5BF49D6C563685954E2AECC024DC02D6'}],
     ['60', {IN=> {f=> '39 1 3 7 18 4 9 9 5 15 13 17 10 15 16 20 8 19 9 10 9 1 19 14 21 2 18 13 10 4 18 16 4 21 15 10 18 19 3 12 18 ^'}},
     {OUT=>'BC8827C3E614D515E83DEA503989DEA4FDA6EA13'}],
     ['61', {IN=> {f=> '41 0 14 4 13 11 1 11 1 10 2 12 4 21 10 21 18 9 2 16 7 20 6 7 12 19 20 1 13 12 10 8 21 15 7 19 13 6 8 19 20 18 19 ^'}},
     {OUT=>'E83868DBE4A389AB48E61CFC4ED894F32AE112AC'}],
     ['62', {IN=> {f=> '37 0 11 18 1 17 14 15 20 16 20 8 2 17 10 4 21 5 19 19 14 22 21 18 13 14 1 3 12 11 11 4 22 13 5 18 7 21 21 ^'}},
     {OUT=>'55C95459CDE4B33791B4B2BCAAF840930AF3F3BD'}],
     ['63', {IN=> {f=> '48 0 9 22 19 12 8 16 5 17 5 9 1 2 9 6 12 6 1 7 4 3 15 1 14 1 12 3 10 2 10 14 21 13 17 6 6 17 1 21 2 14 16 17 9 11 20 21 11 18 ^'}},
     {OUT=>'36BB0E2BA438A3E03214D9ED2B28A4D5C578FCAA'}],
     ['64', {IN=> {f=> '50 1 12 8 20 13 2 9 20 9 14 10 1 16 2 22 6 4 16 14 15 1 12 4 14 9 21 3 3 9 8 21 15 14 8 4 14 4 2 3 8 12 8 6 1 2 18 20 15 3 19 10 ^'}},
     {OUT=>'3ACBF874199763EBA20F3789DFC59572ACA4CF33'}],
     ['65', {IN=> {f=> '44 0 10 20 14 6 3 4 21 1 12 4 18 2 6 7 6 9 20 14 10 10 19 17 21 12 15 17 7 10 11 8 10 12 1 19 19 9 18 21 4 18 11 9 22 5 ^'}},
     {OUT=>'86BE037C4D509C9202020767D860DAB039CADACE'}],
     ['66', {IN=> {f=> '47 0 15 8 15 3 5 6 2 19 12 17 4 20 8 11 20 2 18 4 16 20 12 9 9 6 16 21 16 3 16 18 3 19 5 16 2 4 2 12 11 15 11 14 17 2 10 18 8 ^'}},
     {OUT=>'51B57D7080A87394EEC3EB2E0B242E553F2827C9'}],
     ['67', {IN=> {f=> '48 1 5 13 3 21 5 3 6 18 18 10 1 21 21 7 1 13 12 19 1 14 6 8 21 19 21 11 19 13 2 13 4 1 10 22 16 4 9 4 10 16 3 7 15 11 9 13 17 12 ^'}},
     {OUT=>'1EFBFA78866315CE6A71E457F3A750A38FACAB41'}],
     ['68', {IN=> {f=> '45 0 14 7 6 2 20 3 6 19 19 10 2 22 12 17 12 1 20 7 7 15 20 6 18 8 3 14 23 18 15 4 7 5 23 15 7 14 10 10 19 17 2 4 15 17 21 ^'}},
     {OUT=>'57D6CB41AEEC20236F365B3A490C61D0CFA39611'}],
     ['69', {IN=> {f=> '45 1 15 11 8 9 17 5 12 18 14 6 20 17 21 12 16 9 22 9 20 15 2 22 11 2 6 11 9 8 2 4 14 19 3 21 21 23 8 2 11 4 8 4 20 22 11 ^'}},
     {OUT=>'C532CB64B4BA826372BCCF2B4B5793D5B88BB715'}],
     ['70', {IN=> {f=> '38 0 21 18 22 10 19 9 14 17 23 21 10 7 15 13 16 5 4 10 13 14 20 23 12 20 23 18 10 12 8 21 11 6 12 7 19 14 18 17 ^'}},
     {OUT=>'15833B5631032663E783686A209C6A2B47A1080E'}],
     ['71', {IN=> {f=> '40 0 18 22 6 9 22 5 23 13 6 8 23 20 22 5 22 15 19 20 9 9 1 13 13 10 14 13 5 22 14 21 9 21 19 14 14 4 18 13 12 14 ^'}},
     {OUT=>'D04F2043C96E10CD83B574B1E1C217052CD4A6B2'}],
     ['72', {IN=> {f=> '48 1 7 3 15 5 17 14 23 14 5 17 22 11 1 8 13 23 6 21 3 6 11 7 23 8 6 21 4 4 22 19 13 8 5 19 7 5 23 1 4 19 11 23 11 21 14 1 3 21 ^'}},
     {OUT=>'E8882627C64DB743F7DB8B4413DD033FC63BEB20'}],
     ['73', {IN=> {f=> '43 0 22 14 11 7 18 16 17 24 12 12 3 13 19 16 22 4 16 4 6 23 8 18 11 2 3 20 22 9 21 8 23 1 23 20 7 16 13 23 4 13 3 7 22 ^'}},
     {OUT=>'CD2D32286B8867BC124A0AF2236FC74BE3622199'}],
     ['74', {IN=> {f=> '47 1 23 6 13 19 2 3 7 2 9 9 15 6 13 4 22 6 19 20 1 9 7 14 1 15 3 23 24 22 18 12 12 17 19 10 8 11 22 12 10 2 20 15 18 17 18 7 19 ^'}},
     {OUT=>'019B70D745375091ED5C7B218445EC986D0F5A82'}],
     ['75', {IN=> {f=> '47 1 12 21 6 12 4 7 18 17 3 2 14 24 14 1 23 1 11 15 10 6 18 20 7 1 8 1 16 6 20 23 23 21 10 10 12 24 10 11 23 2 12 23 9 3 24 24 10 ^'}},
     {OUT=>'E5FF5FEC1DADBAED02BF2DAD4026BE6A96B3F2AF'}],
     ['76', {IN=> {f=> '52 0 14 10 18 15 14 5 16 11 22 2 15 24 8 22 1 4 24 9 10 15 3 9 5 4 17 15 9 12 19 19 1 3 10 6 8 3 17 8 18 24 19 3 4 15 4 9 2 24 5 20 13 13 ^'}},
     {OUT=>'6F4E23B3F2E2C068D13921FE4E5E053FFED4E146'}],
     ['77', {IN=> {f=> '42 0 20 17 19 22 13 8 10 19 15 11 1 14 17 20 22 10 7 11 16 9 21 22 17 23 12 15 4 24 7 21 18 2 21 16 1 19 18 20 11 3 15 17 ^'}},
     {OUT=>'25E179602A575C915067566FBA6DA930E97F8678'}],
     ['78', {IN=> {f=> '50 0 18 1 6 14 5 5 5 19 13 10 24 19 16 24 15 13 2 19 15 24 21 17 4 13 17 1 1 9 1 10 2 18 1 21 19 5 18 12 2 22 16 23 15 19 6 18 9 1 23 5 ^'}},
     {OUT=>'67DED0E68E235C8A523E051E86108EEB757EFBFD'}],
     ['79', {IN=> {f=> '51 0 21 13 14 11 18 12 13 3 19 9 20 22 20 2 11 12 6 1 12 16 18 2 9 8 4 3 11 17 11 5 4 19 16 11 23 13 18 1 20 8 2 16 16 21 4 19 5 5 20 24 16 ^'}},
     {OUT=>'AF78536EA83C822796745556D62A3EE82C7BE098'}],
     ['80', {IN=> {f=> '53 1 20 25 17 11 8 4 19 25 17 7 16 21 6 4 8 2 15 9 2 9 19 3 6 3 3 10 25 13 15 7 8 20 21 12 10 12 5 24 11 20 3 13 13 16 9 13 10 3 9 16 3 7 25 ^'}},
     {OUT=>'64D7AC52E47834BE72455F6C64325F9C358B610D'}],
     ['81', {IN=> {f=> '49 1 9 9 14 2 13 17 25 2 18 5 19 23 9 25 9 10 23 12 12 7 13 8 15 7 1 6 21 2 8 7 6 16 14 14 12 15 13 24 10 15 11 10 8 14 15 21 25 21 25 ^'}},
     {OUT=>'9D4866BAA3639C13E541F250FFA3D8BC157A491F'}],
     ['82', {IN=> {f=> '47 0 9 18 20 22 21 20 11 14 23 22 10 13 14 8 19 12 2 11 20 23 13 4 10 6 5 7 23 11 3 16 8 21 4 8 18 5 12 14 8 6 20 19 24 8 23 17 23 ^'}},
     {OUT=>'2E258811961D3EB876F30E7019241A01F9517BEC'}],
     ['83', {IN=> {f=> '48 1 7 19 1 18 1 14 22 13 14 5 8 22 18 14 25 17 11 12 22 2 12 12 16 12 13 18 17 12 17 14 18 8 25 9 23 5 3 8 14 24 17 7 3 3 23 17 22 19 ^'}},
     {OUT=>'8E0EBC487146F83BC9077A1630E0FB3AB3C89E63'}],
     ['84', {IN=> {f=> '51 1 19 17 16 22 24 14 16 20 23 20 9 19 16 7 12 16 5 8 9 7 10 21 24 10 11 19 1 21 14 14 19 3 22 8 12 20 1 18 5 6 5 12 14 1 1 11 9 22 3 24 4 ^'}},
     {OUT=>'CE8953741FFF3425D2311FBBF4AB481B669DEF70'}],
     ['85', {IN=> {f=> '52 1 6 1 11 16 1 12 8 11 11 17 10 22 7 3 10 2 6 4 24 16 24 19 4 5 18 11 12 9 20 21 25 2 21 18 10 20 25 21 3 17 17 5 8 22 25 19 8 10 19 7 11 18 ^'}},
     {OUT=>'789D1D2DAB52086BD90C0E137E2515ED9C6B59B5'}],
     ['86', {IN=> {f=> '44 0 26 14 21 25 25 4 9 13 5 8 9 21 8 12 26 24 9 24 15 1 23 22 16 14 8 22 15 19 24 20 7 8 15 24 12 4 4 23 21 13 19 15 21 12 ^'}},
     {OUT=>'B76CE7472700DD68D6328B7AA8437FB051D15745'}],
     ['87', {IN=> {f=> '59 1 15 7 3 21 20 8 22 14 23 26 19 2 10 18 3 5 3 1 9 15 15 3 7 13 23 9 7 1 13 17 14 25 9 16 2 2 6 13 7 19 25 17 1 5 21 2 7 22 5 6 25 3 12 19 6 2 4 24 17 ^'}},
     {OUT=>'F218669B596C5FFB0B1C14BD03C467FC873230A0'}],
     ['88', {IN=> {f=> '60 0 9 18 20 19 4 11 14 1 6 8 26 6 9 22 4 10 2 7 21 9 8 24 25 14 22 12 22 3 23 3 3 20 6 11 23 6 1 7 5 18 5 15 25 26 1 1 10 11 11 4 12 11 20 3 14 2 3 2 23 15 ^'}},
     {OUT=>'1FF3BDBE0D504CB0CDFAB17E6C37ABA6B3CFFDED'}],
     ['89', {IN=> {f=> '49 0 12 17 24 11 8 6 24 16 15 22 21 14 6 12 20 19 5 5 12 11 6 23 2 16 23 7 24 6 21 2 17 17 5 25 11 25 20 25 24 18 6 12 19 25 7 6 5 2 25 ^'}},
     {OUT=>'2F3CBACBB14405A4652ED52793C1814FD8C4FCE0'}],
     ['90', {IN=> {f=> '54 1 12 16 1 15 7 1 26 19 19 13 20 11 17 6 20 5 24 24 1 21 11 9 20 21 15 10 19 26 3 2 6 7 12 9 10 8 14 10 15 5 17 8 21 1 20 25 6 19 8 3 22 16 16 20 ^'}},
     {OUT=>'982C8AB6CE164F481915AF59AAED9FFF2A391752'}],
     ['91', {IN=> {f=> '63 0 17 13 11 10 17 15 12 6 13 14 17 4 12 10 24 5 13 24 3 5 2 5 11 14 8 5 10 17 16 8 4 14 21 15 3 6 17 25 8 2 3 3 19 10 13 22 22 8 2 13 25 17 2 1 19 1 14 20 2 5 4 15 24 ^'}},
     {OUT=>'5CD92012D488A07ECE0E47901D0E083B6BD93E3F'}],
     ['92', {IN=> {f=> '49 0 14 20 7 25 20 26 20 16 7 17 17 22 1 13 6 5 1 18 14 15 23 15 10 5 19 18 18 26 12 13 3 25 12 21 16 24 4 16 3 6 26 26 10 20 13 1 20 24 15 ^'}},
     {OUT=>'69603FEC02920851D4B3B8782E07B92BB2963009'}],
     ['93', {IN=> {f=> '56 0 3 8 14 5 5 7 11 13 11 26 11 4 26 17 20 19 11 10 3 10 14 9 6 9 7 16 10 4 4 19 19 2 26 13 19 17 15 24 15 4 21 22 13 13 12 22 2 14 20 5 18 7 17 24 20 20 ^'}},
     {OUT=>'3E90F76437B1EA44CF98A08D83EA24CECF6E6191'}],
     ['94', {IN=> {f=> '58 1 6 17 9 20 2 10 19 3 22 4 1 11 3 5 3 21 11 15 12 23 26 5 2 27 6 5 16 6 3 2 23 5 3 20 20 4 24 2 18 21 7 14 10 27 23 6 24 6 19 23 3 9 22 16 21 17 19 23 ^'}},
     {OUT=>'34C09F107C42D990EB4881D4BF2DDDCAB01563AE'}],
     ['95', {IN=> {f=> '58 1 17 7 21 19 6 16 15 15 20 14 2 25 19 14 18 19 7 9 1 14 11 10 16 3 23 14 26 10 11 1 18 1 12 24 19 19 1 7 2 3 24 7 12 9 2 8 16 20 24 5 26 26 4 9 2 7 25 17 ^'}},
     {OUT=>'474BE0E5892EB2382109BFC5E3C8249A9283B03D'}],
     ['96', {IN=> {f=> '54 1 8 12 18 14 26 7 17 18 4 20 1 16 14 21 26 4 6 8 24 11 25 15 24 16 23 4 10 23 21 24 15 10 9 26 7 14 24 21 6 20 5 17 16 17 1 3 12 1 4 13 3 9 21 26 ^'}},
     {OUT=>'A04B4F75051786682483252438F6A75BF4705EC6'}],
     ['97', {IN=> {f=> '56 1 7 18 11 1 19 20 23 12 12 27 13 13 15 16 13 1 16 15 12 26 3 16 16 8 17 13 21 4 6 5 19 14 16 4 16 11 14 18 18 27 9 13 21 3 26 22 3 7 6 4 26 3 15 8 25 21 ^'}},
     {OUT=>'BE88A6716083EB50ED9416719D6A247661299383'}],
     ['98', {IN=> {f=> '50 1 20 13 9 11 20 6 11 21 27 25 20 7 4 18 26 16 27 5 12 19 7 23 6 25 25 2 11 13 25 21 18 17 6 12 14 13 24 11 14 19 26 27 25 6 1 15 4 7 27 15 ^'}},
     {OUT=>'C67E38717FEE1A5F65EC6C7C7C42AFC00CD37F04'}],
     ['99', {IN=> {f=> '51 0 15 16 26 27 23 14 12 28 22 15 8 19 2 20 13 1 24 2 25 1 6 19 19 8 11 24 24 21 13 27 5 11 28 17 7 25 6 23 24 14 25 12 5 13 26 2 5 8 10 16 17 ^'}},
     {OUT=>'959AC4082388E19E9BE5DE571C047EF10C174A8D'}],
     ['100', {IN=> {f=> '58 1 5 26 18 19 21 3 12 11 13 4 14 22 22 14 16 13 3 22 16 23 5 19 6 13 10 26 17 27 26 4 3 25 6 14 2 3 5 7 23 11 22 8 25 2 9 25 18 17 8 2 14 4 19 1 5 27 13 24 ^'}},
     {OUT=>'BAA7AA7B7753FA0ABDC4A541842B5D238D949F0A'}],
     ['101', {IN=> {f=> '53 0 2 27 28 2 17 23 10 27 18 26 7 22 16 3 27 1 26 21 28 10 3 6 2 2 10 17 13 16 6 17 21 23 13 20 22 5 6 11 12 12 8 23 13 17 9 23 20 3 28 27 12 17 22 ^'}},
     {OUT=>'351394DCEBC08155D100FCD488578E6AE71D0E9C'}],
     ['102', {IN=> {f=> '59 0 28 19 5 21 4 27 8 1 19 14 20 6 7 9 1 6 22 3 19 26 14 8 6 7 19 15 23 1 17 16 6 26 14 5 22 25 4 7 10 16 21 10 18 19 24 16 23 8 3 17 28 18 10 2 5 3 21 21 15 ^'}},
     {OUT=>'AB8BE94C5AF60D9477EF1252D604E58E27B2A9EE'}],
     ['103', {IN=> {f=> '58 0 6 24 1 4 24 18 10 22 1 21 12 5 4 4 20 25 24 26 8 25 11 2 7 27 22 19 4 18 27 10 28 4 12 24 8 16 12 11 16 17 25 8 12 16 1 9 9 10 5 24 23 18 5 14 18 8 4 28 ^'}},
     {OUT=>'3429EC74A695FDD3228F152564952308AFE0680A'}],
     ['104', {IN=> {f=> '61 0 5 17 8 28 1 22 4 11 3 2 17 3 14 9 27 13 18 24 9 8 7 28 25 14 21 27 24 6 18 16 2 12 15 9 14 10 1 8 17 4 6 15 26 11 15 2 28 20 26 16 3 7 5 8 9 26 10 12 25 11 22 ^'}},
     {OUT=>'907FA46C029BC67EAA8E4F46E3C2A232F85BD122'}],
     ['105', {IN=> {f=> '53 0 9 13 24 15 20 2 4 8 2 22 20 19 4 15 14 28 13 25 10 10 12 28 24 22 26 28 15 9 11 26 19 22 27 2 21 8 20 23 26 12 10 21 9 15 13 25 7 26 1 13 5 9 20 ^'}},
     {OUT=>'2644C87D1FBBBC0FC8D65F64BCA2492DA15BAAE4'}],
     ['106', {IN=> {f=> '58 0 3 9 21 22 7 1 23 28 1 2 8 22 12 18 28 5 18 14 7 11 17 20 20 7 21 13 8 28 21 22 2 16 20 15 28 9 3 22 13 10 23 4 16 11 14 1 10 8 14 14 15 18 13 12 21 18 25 28 ^'}},
     {OUT=>'110A3EEB408756E2E81ABAF4C5DCD4D4C6AFCF6D'}],
     ['107', {IN=> {f=> '60 1 29 20 2 29 22 8 16 20 4 12 9 6 12 16 16 7 9 20 29 11 9 4 1 15 25 16 29 10 22 7 2 8 5 18 14 23 24 4 6 26 3 11 6 12 1 7 14 24 14 6 10 21 16 23 29 25 6 14 17 24 ^'}},
     {OUT=>'CD4FDC35FAC7E1ADB5DE40F47F256EF74D584959'}],
     ['108', {IN=> {f=> '64 0 12 10 5 10 15 25 8 15 3 7 13 25 16 14 1 29 22 26 15 27 9 1 8 8 28 6 13 5 13 3 15 5 23 8 23 2 5 5 4 17 13 14 7 17 12 27 3 18 5 7 5 26 18 15 22 28 16 13 7 2 23 19 25 15 ^'}},
     {OUT=>'8E6E273208AC256F9ECCF296F3F5A37BC8A0F9F7'}],
     ['109', {IN=> {f=> '56 1 17 7 16 25 23 11 11 15 2 13 9 26 2 24 26 7 28 11 2 29 7 22 23 5 28 19 1 27 29 1 24 11 18 20 3 13 11 7 3 15 17 24 1 18 13 6 3 25 27 16 28 18 24 8 23 22 ^'}},
     {OUT=>'FE0606100BDBC268DB39B503E0FDFE3766185828'}],
     ['110', {IN=> {f=> '51 1 29 28 6 28 14 12 28 27 22 4 14 25 1 3 9 7 11 14 15 16 10 19 12 19 11 20 13 28 4 27 28 7 27 12 4 28 21 17 22 20 17 15 15 23 22 13 12 21 22 21 29 ^'}},
     {OUT=>'6C63C3E58047BCDB35A17F74EEBA4E9B14420809'}],
     ['111', {IN=> {f=> '64 1 12 14 12 18 27 8 7 4 9 14 16 15 8 11 21 20 10 10 21 23 20 2 11 23 1 11 1 5 3 23 16 15 27 14 5 16 3 22 2 3 24 3 19 29 4 4 10 8 20 14 15 1 26 12 27 25 4 28 22 11 19 19 24 9 ^'}},
     {OUT=>'BCC2BD305F0BCDA8CF2D478EF9FE080486CB265F'}],
     ['112', {IN=> {f=> '60 1 20 8 9 5 25 19 17 19 15 7 24 24 21 3 20 16 8 3 17 28 18 29 9 23 9 10 29 4 12 24 15 5 8 22 17 29 12 3 8 29 15 21 21 4 7 20 7 10 7 26 10 16 24 6 7 12 8 12 15 17 ^'}},
     {OUT=>'CE5223FD3DD920A3B666481D5625B16457DCB5E8'}],
     ['113', {IN=> {f=> '60 0 9 17 11 28 12 26 26 6 29 13 10 20 6 23 10 4 3 26 26 14 20 20 25 14 13 15 24 14 11 4 23 27 24 20 9 16 17 24 13 12 6 1 14 26 25 7 8 21 1 19 3 2 2 17 21 13 5 9 21 11 ^'}},
     {OUT=>'948886776E42E4F5FAE1B2D0C906AC3759E3F8B0'}],
     ['114', {IN=> {f=> '54 0 25 1 27 24 6 23 16 5 1 20 29 22 25 9 25 10 3 28 28 25 19 18 16 24 14 15 5 28 12 28 26 29 2 15 15 9 5 18 19 22 12 15 4 6 15 24 16 9 4 26 25 18 27 12 ^'}},
     {OUT=>'4C12A51FCFE242F832E3D7329304B11B75161EFB'}],
     ['115', {IN=> {f=> '61 1 20 4 26 12 3 22 1 22 30 3 28 10 9 24 14 29 6 30 3 10 20 14 6 3 19 21 21 28 16 18 11 30 11 20 30 1 9 8 11 5 19 10 24 4 22 4 2 26 5 15 20 8 3 13 30 18 8 1 25 28 19 ^'}},
     {OUT=>'C54BDD2050504D92F551D378AD5FC72C9ED03932'}],
     ['116', {IN=> {f=> '56 1 20 15 21 18 18 12 16 13 24 9 21 2 28 6 1 23 9 18 27 27 4 9 13 10 8 14 16 15 12 11 14 21 14 10 11 25 17 17 30 21 13 27 26 26 22 14 13 17 21 19 9 9 20 23 13 28 ^'}},
     {OUT=>'8F53E8FA79EA09FD1B682AF5ED1515ECA965604C'}],
     ['117', {IN=> {f=> '59 1 10 28 24 10 22 27 23 27 8 17 14 6 4 21 26 15 1 8 29 27 6 28 15 3 27 25 25 14 19 13 29 8 24 2 8 2 4 12 19 11 10 6 26 14 22 24 30 10 11 12 2 12 17 23 8 8 12 28 12 ^'}},
     {OUT=>'2D7E17F6294524CE78B33EAB72CDD08E5FF6E313'}],
     ['118', {IN=> {f=> '56 0 14 28 2 17 4 8 3 26 9 23 21 30 30 20 4 13 28 29 9 3 17 7 19 30 28 1 2 20 9 12 24 15 30 20 27 3 23 11 6 29 25 23 26 17 20 10 22 15 23 6 25 5 4 30 2 29 ^'}},
     {OUT=>'64582B4B57F782C9302BFE7D07F74AA176627A3A'}],
     ['119', {IN=> {f=> '63 1 23 15 27 14 26 1 1 7 19 12 7 6 20 18 14 4 15 17 28 7 11 7 8 9 22 17 12 5 23 18 25 18 6 12 26 30 12 30 14 3 1 18 10 20 27 21 8 6 24 26 20 11 24 7 2 4 18 15 14 30 16 19 14 ^'}},
     {OUT=>'6D88795B71D3E386BBD1EB830FB9F161BA98869F'}],
     ['120', {IN=> {f=> '52 0 27 15 4 19 25 29 29 7 14 18 9 11 9 27 11 15 29 9 28 20 2 30 26 21 17 8 28 17 22 29 24 8 11 18 29 15 6 7 27 27 17 24 18 23 11 19 8 30 5 24 22 24 ^'}},
     {OUT=>'86AD34A6463F12CEE6DE9596ABA72F0DF1397FD1'}],
     ['121', {IN=> {f=> '66 1 25 15 28 23 5 10 21 5 8 7 3 10 19 17 6 9 15 29 10 7 4 1 16 21 16 29 13 18 5 3 8 15 8 21 29 20 5 27 2 13 27 7 7 30 2 18 26 10 2 5 29 21 15 25 26 24 8 12 20 3 9 10 30 7 12 29 ^'}},
     {OUT=>'7EB46685A57C0D466152DC339C8122548C757ED1'}],
     ['122', {IN=> {f=> '53 1 30 26 20 11 22 19 27 2 16 10 6 4 24 17 20 25 20 15 8 23 23 20 30 18 16 3 30 15 26 23 28 7 21 8 7 31 31 14 26 18 3 1 26 28 15 25 11 31 3 25 9 21 30 ^'}},
     {OUT=>'E7A98FB0692684054407CC221ABC60C199D6F52A'}],
     ['123', {IN=> {f=> '67 0 2 6 14 4 9 5 28 8 17 22 1 4 8 7 10 14 19 10 14 8 27 9 24 26 4 30 11 8 19 5 21 7 2 27 20 16 20 20 22 14 13 16 26 14 10 3 25 22 25 23 21 10 15 15 29 8 13 4 2 13 22 20 7 4 20 31 23 ^'}},
     {OUT=>'34DF1306662206FD0A5FC2969A4BEEC4EB0197F7'}],
     ['124', {IN=> {f=> '65 0 2 2 28 13 19 14 12 23 27 6 2 14 2 22 6 25 30 29 31 13 14 16 31 12 16 30 5 14 31 11 4 1 1 25 21 13 26 22 21 5 22 14 29 1 21 3 14 30 4 2 29 12 15 23 3 15 5 1 6 23 22 13 1 14 23 ^'}},
     {OUT=>'56CF7EBF08D10F0CB9FE7EE3B63A5C3A02BCB450'}],
     ['125', {IN=> {f=> '59 1 25 5 15 6 13 3 22 11 23 31 24 6 5 20 4 14 3 29 8 29 19 7 29 23 25 28 19 11 15 27 21 14 1 19 20 26 12 7 12 1 18 13 29 28 23 29 14 23 7 1 9 29 24 5 30 18 5 25 30 ^'}},
     {OUT=>'3BAE5CB8226642088DA760A6F78B0CF8EDDEA9F1'}],
     ['126', {IN=> {f=> '55 1 31 25 13 7 24 25 24 1 12 19 9 7 6 28 20 14 28 21 19 31 20 20 6 24 18 27 24 4 18 21 1 31 15 1 15 2 27 4 26 25 4 23 19 2 31 22 30 21 22 5 27 12 30 28 31 ^'}},
     {OUT=>'6475DF681E061FA506672C27CBABFA9AA6DDFF62'}],
     ['127', {IN=> {f=> '62 0 27 15 18 14 25 15 17 7 28 11 28 29 30 1 17 12 10 2 18 20 21 2 11 12 5 4 12 25 14 5 5 24 22 18 31 15 22 29 11 3 21 31 21 27 3 28 7 10 25 2 15 30 9 30 7 22 15 9 3 20 24 14 ^'}},
     {OUT=>'79D81991FA4E4957C8062753439DBFD47BBB277D'}],
     ['128', {IN=> {f=> '60 0 28 14 18 9 27 14 22 27 31 10 8 14 7 15 7 20 5 26 1 29 7 17 17 8 3 13 27 18 8 31 27 28 22 22 17 19 18 18 11 19 13 25 10 19 6 28 4 31 23 10 18 26 31 5 10 13 12 8 15 27 ^'}},
     {OUT=>'BAE224477B20302E881F5249F52EC6C34DA8ECEF'}],
     ['129', {IN=> {f=> '60 1 24 22 4 29 22 31 28 20 4 16 21 3 1 15 5 15 6 30 3 29 29 7 27 20 2 20 31 22 26 9 29 16 4 26 32 17 20 14 28 17 19 6 24 11 26 28 5 18 15 8 16 20 21 4 9 12 4 8 17 29 ^'}},
     {OUT=>'EDE4DEB4293CFE4138C2C056B7C46FF821CC0ACC'}],
     ['130', {IN=> {f=> '69 1 5 3 11 15 12 24 31 23 1 6 28 2 8 31 6 7 30 5 19 23 12 6 9 31 19 17 24 25 22 6 12 16 3 7 9 9 11 29 4 11 2 5 13 29 10 12 30 32 18 28 18 27 3 30 4 4 26 6 13 31 13 2 11 7 24 4 17 29 12 ^'}},
     {OUT=>'A771FA5C812BD0C9596D869EC99E4F4AC988B13F'}],
     ['131', {IN=> {f=> '95 0 21 19 21 23 11 42 36 2 13 4 1 33 22 16 27 9 4 33 16 3 30 15 11 32 13 17 38 32 9 38 4 36 15 32 27 19 42 18 6 36 22 10 29 12 25 40 15 29 23 28 30 4 8 11 24 9 10 31 28 43 23 16 29 33 5 40 26 3 19 12 36 43 5 35 37 5 14 11 45 35 16 10 8 32 4 15 35 26 2 39 22 37 22 30 29 ^'}},
     {OUT=>'E99D566212BBBCEEE903946F6100C9C96039A8F4'}],
     ['132', {IN=> {f=> '106 1 18 14 51 2 6 32 51 9 32 50 44 46 51 8 11 53 45 55 16 10 3 52 8 20 20 46 46 13 32 2 46 50 43 25 54 9 31 29 2 47 15 29 24 45 44 18 37 14 28 39 36 44 47 16 50 10 44 24 53 35 22 40 20 15 51 22 18 22 42 6 54 49 38 21 7 13 30 16 7 52 16 22 13 38 7 11 44 33 9 25 13 37 42 14 45 53 30 38 5 25 5 35 38 22 28 53 ^'}},
     {OUT=>'B48CE6B1D13903E3925AE0C88CB931388C013F9C'}],
     ['133', {IN=> {f=> '127 0 58 35 43 28 5 28 63 8 12 25 9 47 53 29 62 7 37 2 3 48 5 12 55 56 28 35 12 63 6 58 27 27 48 44 35 14 17 22 56 10 8 1 16 15 42 63 14 51 57 19 41 7 8 56 47 34 52 22 48 60 43 9 1 52 4 21 49 61 18 50 23 13 46 62 23 45 62 9 56 18 23 31 8 30 27 36 13 38 4 58 53 47 24 18 41 58 19 12 18 52 42 29 44 45 26 63 34 32 41 64 15 26 55 19 2 49 6 30 53 13 54 12 53 37 12 37 43 ^'}},
     {OUT=>'E647D5BAF670D4BF3AFC0A6B72A2424B0C64F194'}],
     ['134', {IN=> {f=> '148 0 60 4 51 47 58 38 17 63 33 23 28 43 12 69 70 33 17 12 50 18 18 36 45 2 67 4 45 20 4 33 38 29 45 8 22 58 39 71 38 32 53 35 19 53 31 29 51 35 4 63 18 33 26 47 70 9 64 62 63 30 15 1 35 28 16 40 20 14 50 33 19 38 30 27 55 10 16 46 47 7 55 12 53 26 56 33 29 55 25 17 48 43 21 43 18 24 63 27 68 46 38 33 35 10 18 11 27 5 9 58 35 70 36 36 39 47 2 10 66 47 5 18 21 44 71 51 57 3 22 7 56 55 28 25 14 40 16 24 48 37 66 50 24 45 18 39 53 55 ^'}},
     {OUT=>'65C1CD932A06B05CD0B43AFB3BC7891F6BCEF45C'}],
     ['135', {IN=> {f=> '165 1 15 62 35 29 15 40 19 76 67 4 5 71 46 61 26 8 77 48 1 23 12 60 40 24 44 33 29 42 73 66 49 61 20 30 1 54 52 42 39 64 23 65 37 24 20 11 26 66 22 77 22 57 7 38 57 33 61 73 7 64 1 49 35 76 14 27 21 45 68 38 58 73 13 72 47 73 33 8 66 23 38 4 56 77 47 10 71 13 20 31 41 6 51 3 18 17 61 47 14 48 76 46 28 34 43 1 56 4 25 7 65 41 1 34 37 23 59 59 27 26 13 15 14 75 60 14 1 28 59 26 65 61 16 23 17 28 6 19 2 35 49 30 29 48 2 63 73 59 1 3 76 41 11 19 18 43 54 63 67 51 4 9 78 60 66 ^'}},
     {OUT=>'70FFAE353A5CD0F8A65A8B2746D0F16281B25EC7'}],
     ['136', {IN=> {f=> '181 0 18 19 84 17 12 10 57 18 77 51 52 16 39 74 49 52 63 38 72 2 15 64 83 62 49 56 11 26 68 58 83 33 23 50 63 71 53 27 84 22 39 41 52 58 11 64 7 60 45 70 22 5 73 38 30 30 48 21 75 80 40 21 8 53 9 26 30 34 81 71 71 51 23 75 33 41 23 32 5 8 66 40 72 40 16 66 45 14 48 34 21 41 27 3 55 27 37 23 41 65 4 57 51 74 22 19 75 42 16 19 46 16 10 48 20 19 37 41 14 57 9 17 55 38 5 60 7 46 20 43 36 39 52 20 10 62 45 23 46 7 35 75 29 70 35 36 34 25 12 15 84 26 10 6 71 29 79 33 32 25 59 76 82 64 58 7 8 19 41 74 2 53 65 24 1 55 51 36 21 79 7 ^'}},
     {OUT=>'CC8221F2B829B8CF39646BF46888317C3EB378EA'}],
     ['137', {IN=> {f=> '184 1 60 66 66 6 3 9 73 12 7 40 70 18 71 70 65 51 14 14 27 50 9 87 81 50 22 19 40 37 16 79 12 34 37 76 82 10 61 7 81 49 67 26 45 82 50 81 63 45 69 31 31 76 51 9 59 34 51 54 34 83 10 33 51 86 81 82 69 18 8 22 64 19 86 62 58 33 37 17 34 5 29 83 42 76 50 54 66 39 9 1 36 43 17 65 6 35 56 72 71 83 88 10 1 8 87 22 6 21 78 25 89 43 62 40 55 85 31 89 74 63 46 28 24 26 31 17 7 8 27 19 12 85 17 20 27 77 10 2 54 80 17 52 74 76 69 78 11 20 80 4 29 24 85 75 18 39 23 70 83 29 57 67 72 70 33 4 15 46 42 2 69 13 53 33 69 64 33 64 14 40 69 59 78 54 ^'}},
     {OUT=>'26ACCC2D6D51FF7BF3E5895588907765111BB69B'}],
     ['138', {IN=> {f=> '193 1 68 43 95 53 38 58 55 28 20 16 67 48 17 86 32 44 68 67 28 16 14 79 25 15 72 67 50 80 18 30 10 75 1 60 45 87 78 28 95 49 63 70 59 26 6 51 73 60 65 18 26 8 87 5 58 31 25 57 40 46 78 57 34 78 61 36 66 57 38 80 22 32 68 71 30 74 37 81 66 77 66 55 2 51 24 93 61 40 68 45 61 12 63 24 89 59 52 72 43 20 20 69 36 40 88 46 9 62 55 77 84 20 18 6 77 15 52 39 75 3 26 4 85 17 62 29 11 92 46 58 29 59 28 42 80 71 96 2 49 85 37 63 4 61 14 2 53 87 25 86 6 75 76 93 41 39 93 92 42 56 41 63 26 28 18 77 11 50 78 79 1 12 12 91 29 13 58 5 56 92 66 59 4 39 47 95 5 5 62 33 13 80 27 ^'}},
     {OUT=>'01072915B8E868D9B28E759CF2BC1AEA4BB92165'}],
     ['139', {IN=> {f=> '203 1 35 28 11 7 20 7 17 3 3 30 89 13 65 56 66 63 22 82 16 31 55 56 77 91 91 71 101 13 10 85 101 95 17 99 98 91 33 14 20 48 32 7 64 29 38 35 25 4 95 23 34 1 85 81 23 31 96 71 84 50 15 79 47 25 51 45 35 66 19 61 60 9 31 93 64 70 30 42 86 53 1 71 46 42 22 38 96 10 99 34 76 26 55 73 63 63 97 23 92 81 64 46 1 30 31 35 86 91 88 64 87 16 37 69 84 94 60 100 3 47 52 8 71 87 57 29 76 43 18 45 46 15 65 12 44 42 66 60 15 68 19 58 39 62 76 9 92 101 57 32 4 34 15 41 62 32 89 71 43 35 31 41 21 17 82 33 96 27 62 29 82 57 46 62 15 24 99 37 83 40 52 46 56 80 98 3 91 74 6 27 7 58 94 10 41 79 97 84 77 74 26 99 35 ^'}},
     {OUT=>'3016115711D74236ADF0C371E47992F87A428598'}],
     ['140', {IN=> {f=> '212 1 26 101 17 91 45 97 80 59 102 30 68 4 85 9 4 39 16 18 85 70 11 87 62 72 78 38 3 41 53 82 82 35 18 13 94 64 52 39 77 59 26 9 65 46 64 98 32 29 86 79 16 63 54 76 56 98 16 98 78 22 72 33 103 104 52 84 12 65 15 85 101 97 84 31 51 26 100 100 38 80 13 2 78 7 24 44 84 103 27 7 28 16 33 99 25 103 54 14 42 62 87 92 27 22 42 5 52 100 84 73 72 63 24 48 56 52 23 5 17 76 31 1 95 58 43 60 50 62 30 23 35 79 20 35 3 72 32 45 51 87 41 84 27 79 77 70 102 15 54 15 100 8 52 69 105 3 30 84 42 93 66 89 69 74 24 33 42 97 4 38 99 106 13 93 6 106 74 100 54 45 21 59 56 37 9 50 32 75 79 31 77 9 61 1 8 68 6 60 81 7 100 99 14 61 48 25 73 26 70 72 94 34 ^'}},
     {OUT=>'BF30417999C1368F008C1F19FECA4D18A5E1C3C9'}],
     ['141', {IN=> {f=> '233 0 11 98 110 88 35 110 35 64 49 88 93 28 85 6 78 65 90 52 24 97 51 39 51 59 23 1 3 49 33 11 78 27 35 55 64 5 102 4 70 25 56 58 38 66 11 31 96 66 104 59 41 86 58 29 79 41 40 72 51 12 92 34 52 44 69 104 21 97 89 96 48 21 4 61 40 28 67 34 23 85 44 22 62 52 33 84 23 30 73 74 4 79 12 81 47 80 53 47 89 40 19 80 62 34 61 29 41 95 43 1 70 63 55 53 18 19 13 48 10 19 89 49 4 52 53 56 76 10 8 104 77 15 28 38 75 109 3 85 90 8 40 8 93 90 43 39 14 60 17 36 78 56 105 80 35 75 36 58 82 50 100 98 45 74 13 66 95 72 71 95 34 14 98 72 33 38 37 52 6 14 107 59 3 29 61 67 98 92 5 93 17 98 36 87 41 75 71 57 88 17 25 91 84 3 58 20 92 69 51 50 36 31 14 25 18 30 18 1 41 104 30 82 59 87 70 34 96 28 47 62 81 103 48 ^'}},
     {OUT=>'62BA49087185F2742C26E1C1F4844112178BF673'}],
     ['142', {IN=> {f=> '234 1 63 90 108 108 102 64 82 88 4 111 76 97 22 1 108 41 34 91 33 20 25 24 26 8 83 11 31 7 85 109 106 4 105 85 68 28 33 99 53 8 16 12 11 74 17 83 66 70 16 30 9 67 68 34 24 81 47 92 72 47 37 33 38 92 17 8 28 88 22 62 69 32 89 75 3 72 96 85 13 105 24 38 37 94 115 83 72 108 114 24 93 76 103 60 99 102 9 43 10 59 95 46 33 93 15 26 69 44 2 86 107 55 45 61 65 92 66 9 55 39 70 83 29 98 67 13 111 15 20 31 62 8 2 51 20 19 33 44 14 115 71 112 97 10 41 28 53 51 26 57 15 38 98 55 106 22 56 31 50 95 107 110 84 70 10 108 96 73 100 25 36 55 88 71 63 96 30 90 96 79 22 7 30 23 28 59 89 8 51 99 47 86 34 18 43 65 98 104 107 49 7 79 71 8 57 21 29 80 2 74 78 44 57 9 61 22 13 68 52 91 74 98 43 30 58 68 95 101 72 102 76 42 99 61 ^'}},
     {OUT=>'E1F6B9536F384DD3098285BBFD495A474140DC5A'}],
     ['143', {IN=> {f=> '249 0 27 117 45 119 80 2 59 52 8 76 20 94 102 69 96 42 46 106 67 9 110 89 71 69 34 31 15 85 16 29 100 82 37 62 68 95 108 44 23 114 34 36 56 93 11 30 96 12 31 67 14 114 14 66 70 30 81 46 53 119 85 6 104 47 92 72 70 5 70 15 115 68 105 33 97 13 85 106 14 61 29 22 86 45 57 69 91 38 38 28 66 13 60 95 103 3 15 5 113 38 23 62 5 65 94 107 73 104 37 47 102 117 3 78 35 7 95 56 78 45 52 28 46 43 37 32 53 19 55 29 47 97 76 115 83 71 11 45 62 73 99 116 2 24 116 7 28 41 2 29 37 52 23 5 118 79 31 57 89 61 24 101 78 50 93 73 41 7 33 45 47 24 1 48 73 36 3 25 87 46 28 108 54 68 53 67 119 28 36 118 104 42 88 27 112 4 74 85 1 63 39 97 71 74 75 76 10 49 12 79 11 50 103 118 94 117 118 37 27 12 94 60 28 51 47 82 110 17 15 105 23 52 43 12 21 22 81 41 12 74 90 42 108 117 98 67 4 69 85 ^'}},
     {OUT=>'B522DAE1D67726EBA7C4136D4E2F6D6D645AC43E'}],
     ['144', {IN=> {f=> '243 0 76 81 26 101 13 68 62 106 87 19 98 32 81 63 79 93 31 121 123 75 52 11 66 41 54 87 38 5 104 62 51 38 55 29 31 120 44 16 48 94 46 105 91 66 78 27 43 6 64 2 55 79 75 84 113 22 4 113 109 31 33 17 96 11 29 63 98 103 107 116 34 14 9 95 38 18 51 75 33 109 118 55 66 4 76 7 75 70 82 74 23 1 26 69 40 112 99 47 65 31 70 119 52 103 88 85 86 28 16 12 76 25 22 78 64 21 86 27 61 77 72 108 2 18 106 119 121 54 16 85 72 2 73 26 88 66 60 80 35 24 117 63 24 44 67 52 122 119 33 72 16 99 98 69 54 19 42 28 53 114 32 117 81 100 57 49 123 56 21 68 80 53 95 1 45 95 107 98 87 1 27 24 99 116 16 67 1 113 91 84 25 40 25 72 3 28 90 87 112 80 16 117 45 77 36 90 105 59 88 122 64 108 108 71 98 18 50 115 93 105 77 35 6 46 55 47 102 4 26 87 111 120 81 113 4 57 105 3 84 94 115 61 73 ^'}},
     {OUT=>'E9A021C3EB0B9F2C710554D4BF21B19F78E09478'}],
     ['145', {IN=> {f=> '255 1 91 47 51 9 57 9 55 94 61 61 68 46 107 6 35 81 114 78 96 74 14 89 73 67 67 69 113 107 11 98 113 109 20 92 17 67 70 88 57 10 124 9 60 122 93 91 45 7 15 24 51 5 98 115 24 49 90 104 117 66 128 94 64 80 12 43 91 46 111 59 58 77 30 14 88 60 123 68 41 44 68 40 104 118 41 43 93 90 105 92 16 127 26 54 125 114 79 71 24 48 21 25 118 40 103 49 91 44 67 65 25 119 109 18 48 23 69 112 38 61 64 87 84 104 119 110 122 92 22 1 8 83 34 100 32 62 41 46 112 34 102 76 56 39 4 127 30 13 19 110 124 7 16 128 95 4 124 11 104 116 126 49 95 3 55 96 70 90 101 4 122 96 75 118 39 128 99 92 18 42 20 87 83 35 75 111 61 67 71 28 101 9 56 34 105 95 71 23 73 71 26 57 15 23 76 55 99 89 128 98 117 68 43 88 62 38 62 39 2 83 36 15 26 60 128 96 73 74 10 1 12 42 22 2 77 33 33 32 57 13 14 82 57 12 39 3 58 80 14 87 85 44 69 109 119 ^'}},
     {OUT=>'DF13573188F3BF705E697A3E1F580145F2183377'}],
     ['146', {IN=> {f=> '283 0 102 55 53 41 60 88 25 67 58 76 44 22 68 118 108 40 95 96 81 90 85 28 77 18 11 37 72 93 60 110 124 119 95 131 91 37 109 126 8 73 69 72 80 17 83 5 76 20 32 15 10 1 103 18 22 116 98 9 51 104 102 44 33 15 12 24 31 89 1 6 28 101 8 64 72 106 30 5 52 89 111 39 108 64 85 17 57 124 22 105 78 115 3 40 108 66 108 77 128 103 44 35 38 13 95 10 111 63 98 117 61 51 126 69 96 70 70 59 39 13 97 33 112 2 77 7 123 70 83 29 66 67 49 79 19 104 115 14 60 2 55 40 71 33 28 114 51 91 17 46 45 128 57 87 62 25 115 38 50 55 90 74 8 51 102 79 43 94 36 122 94 12 41 36 25 104 91 24 7 99 80 30 126 32 63 122 107 114 27 28 79 41 12 35 51 115 122 70 22 79 65 2 88 27 17 59 15 23 44 57 5 65 6 26 78 80 125 93 84 100 45 22 129 68 36 111 74 118 11 50 42 120 47 21 8 86 112 26 67 60 99 45 93 47 8 38 59 52 56 124 20 82 18 117 24 18 46 106 19 117 26 41 47 45 130 7 15 1 4 5 100 10 85 50 44 11 48 92 119 108 42 118 125 ^'}},
     {OUT=>'188835CFE52ECFA0C4135C2825F245DC29973970'}],
     ['147', {IN=> {f=> '272 0 8 61 99 70 96 20 87 123 134 82 22 2 110 118 33 86 5 7 5 94 56 15 60 96 54 13 22 55 99 4 25 105 17 37 69 10 38 117 117 30 70 13 9 109 115 62 94 52 66 117 100 135 7 75 23 5 81 110 31 118 29 1 62 11 41 88 109 119 102 37 3 30 123 47 31 56 134 29 124 116 118 99 21 56 77 91 23 37 135 81 44 51 67 95 51 133 30 57 67 116 122 48 100 7 132 97 106 69 93 4 95 125 102 103 119 81 57 133 96 37 118 50 117 113 81 127 17 45 103 32 121 129 60 43 65 127 30 36 132 110 52 53 35 71 12 76 22 72 130 112 99 76 26 21 73 63 63 97 23 58 115 132 114 1 132 31 35 18 23 54 30 53 118 37 35 84 94 60 100 3 47 18 110 105 87 57 63 76 43 52 45 46 49 65 12 10 42 66 60 117 34 19 92 5 28 76 9 126 101 125 32 38 34 15 7 62 32 21 3 43 69 31 109 123 51 116 135 130 129 130 63 14 57 80 62 15 126 31 105 83 108 120 80 124 46 98 105 91 6 6 27 7 58 128 78 7 79 63 84 77 74 128 65 61 95 121 17 24 123 117 51 122 ^'}},
     {OUT=>'41B615A34EE2CEC9D84A91B141CFAB115821950B'}],
     ['148', {IN=> {f=> '284 0 44 71 43 20 126 58 53 47 98 18 19 119 93 29 70 39 94 112 44 115 135 98 82 10 67 29 102 113 68 80 19 75 1 91 114 87 80 7 40 37 86 120 16 104 136 117 82 138 32 65 114 119 137 121 8 12 46 126 26 119 73 130 60 76 113 100 14 133 26 116 34 120 80 95 84 53 15 24 44 51 4 10 23 77 24 99 66 37 54 63 42 136 21 34 76 5 17 128 101 1 59 40 113 112 32 97 31 93 105 79 91 18 39 1 103 132 51 68 124 111 13 97 43 128 69 84 85 72 15 12 26 87 16 16 92 101 13 77 4 118 89 103 56 42 16 60 44 39 126 46 18 83 93 41 105 3 82 106 115 91 6 4 54 115 15 120 109 113 48 41 9 95 20 62 67 105 111 25 132 7 116 46 138 44 83 61 124 131 35 107 6 109 81 114 67 41 137 77 56 74 73 34 12 14 69 52 11 98 47 54 83 81 6 1 15 88 35 139 80 83 49 89 27 47 130 92 133 87 51 112 76 49 109 49 57 93 73 22 117 50 64 58 97 139 36 131 111 133 58 33 8 88 55 38 90 46 30 118 57 29 82 74 41 117 38 46 94 92 5 105 15 117 70 103 68 60 120 48 21 110 85 40 81 66 ^'}},
     {OUT=>'AB3DD6221D2AFE6613B815DA1C389EEC74AA0337'}],
     ['149', {IN=> {f=> '291 0 46 113 52 134 79 74 64 57 18 23 9 52 8 16 103 57 138 59 59 65 92 2 7 130 92 8 34 40 86 131 140 100 112 4 42 1 110 108 43 37 15 67 19 35 94 61 130 98 35 88 34 65 104 56 126 118 50 87 10 81 109 90 86 118 32 6 114 88 39 38 39 62 3 12 134 72 137 35 75 81 115 106 140 112 11 123 41 103 45 95 84 71 107 13 26 110 96 62 16 109 84 59 53 38 27 8 28 13 32 137 17 138 41 122 36 99 65 99 83 36 112 29 49 70 96 126 136 131 116 3 18 17 126 142 14 37 141 141 123 42 13 20 83 42 139 83 54 49 58 42 7 137 29 48 16 121 127 34 52 140 106 128 58 36 124 83 24 69 54 61 112 17 6 95 97 24 57 86 124 59 71 119 67 1 109 54 68 49 57 132 32 5 71 113 40 80 104 75 106 133 31 126 130 104 62 9 39 44 66 116 141 135 96 132 19 41 121 126 124 77 8 4 60 82 6 101 124 89 51 123 48 40 85 77 21 112 10 69 66 115 87 16 108 30 84 65 80 103 32 131 134 73 47 10 63 39 50 93 37 135 114 69 48 34 58 23 27 133 37 9 40 98 41 115 99 70 83 29 42 67 133 55 79 80 91 122 12 2 115 112 47 ^'}},
     {OUT=>'0706D414B4AA7FB4A9051AA70D6856A7264054FB'}],
     ['150', {IN=> {f=> '293 1 33 13 99 138 1 42 89 118 87 113 99 12 134 142 100 38 5 55 75 14 110 108 42 64 130 79 138 62 64 69 57 11 123 25 59 16 111 94 24 65 30 51 119 48 107 92 84 69 28 136 143 54 20 6 70 47 142 64 4 65 59 73 99 134 146 102 125 116 57 137 137 72 48 128 78 5 80 63 54 85 30 22 129 68 21 21 74 28 128 107 27 60 2 93 95 71 37 11 37 15 39 102 3 104 65 80 59 52 113 34 20 67 60 27 81 135 46 106 106 102 68 128 17 15 100 124 15 43 136 122 100 67 142 35 14 53 120 2 89 93 99 73 9 122 39 77 15 96 90 43 79 134 60 92 105 55 96 31 119 77 97 72 23 140 38 30 43 83 136 88 107 117 72 109 118 58 91 119 73 95 100 59 138 123 54 49 143 50 133 66 106 45 80 88 42 93 5 59 77 101 74 110 104 40 92 19 77 76 86 102 129 3 144 101 139 134 56 90 18 91 94 85 55 10 137 11 58 1 107 113 70 22 7 56 29 143 111 8 46 45 116 122 129 89 7 121 53 95 14 49 118 62 125 91 37 97 15 35 100 63 140 63 50 51 58 26 127 6 45 59 102 121 114 85 141 135 10 72 19 106 66 66 41 53 13 38 1 21 103 50 108 46 119 ^'}},
     {OUT=>'3CBF8151F3A00B1D5A809CBB8C4F3135055A6BD1'}],
     ['151', {IN=> {f=> '297 1 46 31 132 112 28 63 124 97 129 43 40 72 99 107 132 137 96 139 99 145 121 144 118 37 81 39 94 60 55 109 47 109 110 75 42 12 139 137 43 128 106 107 19 126 12 101 148 127 15 117 125 125 62 96 13 76 70 96 101 110 138 8 95 76 143 17 32 97 79 149 39 31 94 123 21 41 135 55 84 70 33 135 118 50 62 121 81 1 45 144 93 60 5 64 137 8 105 91 82 67 27 113 119 53 18 98 79 48 84 32 135 128 5 1 20 76 17 85 108 72 36 141 140 49 150 105 104 3 149 14 54 18 148 64 49 125 37 28 28 101 22 104 91 32 82 117 12 114 69 58 2 58 115 9 108 47 59 65 14 92 7 4 86 98 16 82 92 95 38 94 10 10 48 97 104 66 115 97 142 115 122 119 40 97 16 32 47 34 88 89 26 50 12 76 80 51 40 9 133 24 44 40 122 84 108 22 142 140 99 44 15 54 8 42 125 150 130 21 79 124 62 46 119 15 29 91 57 150 42 138 71 61 68 80 114 6 1 70 121 18 35 113 56 87 86 10 73 14 29 41 72 89 1 133 87 101 123 59 90 142 77 133 52 78 48 34 138 134 27 17 60 131 147 61 93 148 39 132 49 62 71 36 91 4 139 49 100 120 43 113 144 30 94 73 127 40 125 ^'}},
     {OUT=>'DA5D6A0319272BBCCEA63ACFA6799756FFDA6840'}],
     ['152', {IN=> {f=> '313 1 35 97 95 76 105 88 32 138 30 69 61 40 47 21 107 6 39 81 114 53 125 53 147 14 4 73 146 96 98 13 136 11 98 117 138 153 67 146 71 99 88 7 139 24 13 35 47 97 145 74 36 119 3 51 84 48 119 53 49 15 79 17 120 103 148 64 30 41 97 120 75 111 63 58 131 134 18 13 10 48 18 16 48 43 15 54 18 41 47 122 144 80 92 145 77 1 33 89 54 46 78 48 21 54 43 40 53 24 16 73 42 94 29 44 34 151 152 23 123 12 142 140 43 37 88 29 19 35 72 96 151 130 62 112 34 36 91 120 50 112 138 2 105 60 68 137 131 5 17 19 139 74 11 120 78 149 58 128 15 104 16 126 78 20 57 134 71 49 90 76 108 126 100 54 68 39 132 153 42 147 146 124 62 87 35 75 61 65 46 100 82 105 113 31 63 5 95 54 71 77 127 150 80 36 144 2 130 59 74 39 3 152 121 122 18 117 12 117 141 118 135 62 36 69 5 39 53 150 52 153 143 30 66 96 126 131 56 137 8 7 86 142 14 7 111 141 93 136 137 134 43 12 89 23 44 9 152 146 121 97 19 38 110 91 67 14 32 110 66 68 8 130 84 73 118 59 24 41 72 121 150 55 37 138 27 104 66 124 9 51 109 47 125 109 148 8 29 47 72 146 149 61 93 10 20 54 15 76 133 125 106 110 67 ^'}},
     {OUT=>'FB4429C95F6277B346D3B389413758DFFFEEDC98'}],
     ['153', {IN=> {f=> '330 0 23 9 26 136 27 51 115 122 44 106 6 146 108 113 85 51 8 96 47 56 137 62 59 89 143 71 140 14 85 156 139 99 154 30 53 115 35 147 108 148 58 52 28 103 19 92 95 152 152 10 11 13 155 67 11 83 101 69 153 152 45 141 14 120 129 140 119 59 2 89 73 70 83 29 16 67 81 29 1 54 65 96 117 2 37 47 128 33 3 89 108 98 139 49 78 27 103 39 119 94 132 90 38 132 55 65 131 90 58 2 54 100 69 118 22 44 19 7 148 93 25 29 123 81 64 131 55 30 1 89 38 97 82 64 9 28 86 123 151 10 133 40 154 102 4 111 65 9 63 59 124 116 72 105 76 57 137 97 32 145 108 78 112 50 43 34 75 20 22 129 68 11 118 74 125 118 57 17 20 129 53 65 61 144 1 17 142 156 52 100 54 15 20 59 52 63 131 20 57 124 31 125 46 106 76 92 8 98 154 152 80 114 15 140 136 112 100 17 92 25 151 150 80 99 69 83 49 43 156 102 19 57 122 96 30 3 39 134 40 32 75 5 76 127 138 99 17 57 52 150 130 18 127 33 23 116 107 78 77 77 42 69 68 48 41 69 33 75 40 49 128 103 4 146 93 10 83 66 96 152 30 38 12 33 5 39 47 41 34 60 74 20 42 156 67 46 56 102 89 3 124 81 99 104 56 50 8 61 74 55 15 87 108 28 138 47 93 60 2 124 46 126 103 91 145 36 25 116 122 51 ^'}},
     {OUT=>'2C6E30D9C895B42DCCCFC84C906EC88C09B20DE1'}],
     ['154', {IN=> {f=> '322 0 75 7 107 158 81 105 154 90 20 125 77 114 69 92 7 58 21 98 154 50 128 149 117 127 153 45 3 18 121 86 29 71 79 101 2 5 22 143 10 27 53 146 157 148 112 33 22 80 123 24 147 1 112 82 159 63 74 97 109 33 151 32 89 87 132 117 46 129 59 115 91 114 118 37 21 9 94 60 25 89 47 79 110 55 12 143 99 87 43 88 56 57 160 76 12 71 128 77 146 117 95 105 42 66 3 76 20 76 101 100 118 149 45 26 143 148 32 57 39 129 19 31 84 123 1 152 135 5 54 30 13 125 68 30 62 101 51 142 5 94 83 20 116 24 107 109 105 91 42 17 27 93 69 3 139 68 79 38 84 2 85 128 126 122 131 46 17 35 98 42 26 111 100 29 120 55 84 114 109 145 14 18 138 14 9 85 7 18 129 91 2 94 51 133 82 87 123 64 39 8 103 38 75 110 78 7 9 45 115 42 138 135 86 78 16 62 52 75 159 54 151 121 149 77 74 16 85 47 102 105 82 119 10 67 137 153 148 135 28 49 26 151 153 36 80 11 130 113 24 44 30 102 24 58 133 122 140 99 24 156 54 119 42 115 140 90 132 19 94 2 157 99 136 19 71 7 130 153 108 51 21 58 70 74 137 1 40 111 149 5 103 6 27 76 141 23 125 140 1 72 29 152 103 87 51 93 29 80 132 77 123 153 68 159 14 98 114 158 121 158 81 131 ^'}},
     {OUT=>'3DE3189A5E19F225CDCE254DFF23DACD22C61363'}],
     ['155', {IN=> {f=> '322 0 35 93 109 125 119 10 10 19 135 26 4 74 135 35 120 129 113 92 17 29 47 88 14 159 149 87 45 36 75 68 22 138 20 59 61 144 151 11 107 6 153 81 114 43 85 157 97 148 118 73 126 56 58 137 96 11 98 67 98 103 57 146 21 59 88 151 139 148 127 25 17 47 115 34 160 109 107 51 64 28 69 13 49 149 69 141 90 93 118 64 10 1 67 80 35 111 13 58 101 124 132 147 154 18 162 6 162 33 5 34 142 41 161 82 114 70 92 145 57 155 137 114 79 44 36 48 48 21 14 13 40 33 14 150 33 32 54 143 14 4 101 142 23 93 136 132 120 147 17 38 163 143 5 52 46 151 130 32 72 34 124 150 51 100 112 128 126 65 10 28 87 81 159 131 19 99 54 125 110 58 119 28 78 129 104 140 126 38 154 27 114 61 153 90 66 98 76 50 158 48 39 82 123 22 147 136 114 52 37 35 75 41 15 150 60 52 55 103 21 23 129 95 24 71 47 97 130 50 140 144 106 100 9 64 19 117 122 71 92 8 77 156 97 121 98 85 2 36 39 109 143 23 120 156 133 93 154 36 66 116 131 160 127 162 161 46 142 14 141 81 141 63 86 117 104 3 146 39 127 34 133 102 106 91 57 9 28 60 61 7 158 12 80 26 8 122 80 44 63 68 49 158 21 32 81 150 15 141 108 161 64 46 124 123 31 99 27 105 109 98 112 144 ^'}},
     {OUT=>'93530A9BC9A817F6922518A73A1505C411D05DA2'}],
     ['156', {IN=> {f=> '336 1 34 161 107 149 48 67 138 109 156 104 37 133 60 80 84 81 160 9 16 96 164 1 95 112 4 86 163 116 98 103 55 31 8 56 37 36 127 32 9 89 103 31 100 161 85 106 119 89 154 43 115 162 137 108 128 38 42 155 103 9 62 65 102 122 10 138 160 125 47 158 43 91 69 123 132 35 121 4 110 89 130 69 29 139 69 53 70 83 29 163 67 41 9 108 34 45 76 87 2 144 164 98 33 160 79 78 48 89 9 38 134 93 146 79 54 122 80 38 112 55 55 101 70 8 129 44 70 59 98 149 24 136 124 138 63 25 166 83 51 34 91 45 30 118 59 28 87 72 44 116 28 36 103 101 113 10 114 62 111 71 65 126 53 19 114 86 42 85 36 57 137 57 159 95 88 78 72 20 23 14 65 10 22 129 68 1 68 74 75 108 7 7 147 109 13 35 51 104 158 164 122 126 2 50 4 132 127 59 52 13 81 20 47 107 74 148 115 46 106 46 82 115 68 144 142 60 104 15 90 136 102 100 134 42 15 141 100 40 49 49 73 166 13 156 82 166 37 82 96 137 130 166 134 20 139 45 122 56 107 98 79 124 17 32 130 120 165 77 23 130 96 67 68 47 37 12 29 18 38 158 19 160 55 147 39 118 83 121 96 43 137 33 66 86 112 147 155 149 140 5 19 17 148 161 10 44 159 146 57 16 26 102 49 3 104 61 59 74 56 10 165 31 54 25 142 157 37 58 165 128 154 73 50 149 94 137 ^'}},
     {OUT=>'E31354345F832D31E05C1B842D405D4BD4588EC8'}],
     ['157', {IN=> {f=> '330 1 61 51 65 132 23 169 116 122 14 66 7 98 131 72 69 127 72 163 125 68 69 51 47 159 31 164 71 118 50 83 113 81 127 153 45 137 134 121 68 163 26 43 65 127 166 138 98 144 18 53 137 139 148 76 158 4 62 78 167 102 144 94 55 141 63 29 97 91 24 115 166 80 69 132 99 1 120 23 88 64 87 118 37 137 152 94 60 168 71 47 52 110 37 155 125 63 42 43 52 11 12 151 31 12 44 110 32 128 117 68 87 24 39 164 76 145 58 101 91 100 140 151 143 130 32 21 3 111 1 31 75 123 153 116 135 130 27 164 165 116 23 12 62 83 24 133 139 49 74 154 80 158 80 64 105 91 6 142 27 75 24 128 112 41 79 29 84 145 40 128 99 95 95 19 17 160 89 15 17 84 64 11 93 10 66 78 73 127 148 18 129 139 143 49 150 9 84 82 154 85 15 88 82 60 87 19 12 133 58 20 39 65 51 141 134 27 70 167 120 117 86 60 16 44 16 57 132 18 142 85 104 59 47 141 58 2 66 96 46 119 153 40 110 126 103 90 144 13 26 106 144 80 145 134 103 95 24 44 21 84 140 13 97 104 140 99 6 147 54 83 42 106 131 54 96 135 67 118 121 81 109 10 53 132 112 117 81 33 155 49 61 38 119 1 13 102 131 148 94 131 143 67 123 148 89 104 135 72 145 152 76 87 6 66 2 71 123 77 114 108 59 123 166 62 96 140 94 149 116 169 ^'}},
     {OUT=>'3FF76957E80B60CF74D015AD431FCA147B3AF232'}],
     ['158', {IN=> {f=> '349 0 125 17 93 82 80 110 156 147 156 99 154 4 29 90 163 120 84 113 56 8 157 29 61 169 141 113 78 48 50 13 138 11 50 61 99 106 2 107 6 117 81 114 34 49 112 52 130 82 73 108 20 22 110 60 11 98 22 62 58 48 146 149 23 88 142 139 121 91 16 163 2 88 171 133 100 62 51 46 10 24 150 49 131 60 114 63 84 91 64 165 138 40 44 172 111 141 58 74 115 96 129 145 164 153 170 126 24 169 16 115 41 125 46 87 61 92 145 39 155 92 78 70 35 27 21 48 21 151 159 40 15 5 132 170 23 18 107 160 150 56 133 23 66 109 123 102 102 172 166 145 116 151 34 1 151 130 5 36 34 97 114 15 82 128 112 119 99 29 138 165 42 36 159 95 19 63 36 89 101 40 92 1 33 93 104 113 126 2 136 96 52 108 90 57 89 31 5 113 30 39 37 96 4 147 127 105 43 165 35 75 23 143 105 24 25 10 94 12 160 102 95 170 71 20 70 112 23 95 144 61 73 137 55 1 81 95 26 65 172 41 147 79 103 80 40 121 36 12 64 98 169 93 111 115 48 127 9 39 107 131 115 118 162 161 10 142 14 123 54 141 36 41 99 77 140 128 167 82 25 106 57 70 64 21 19 15 34 126 149 167 53 163 127 86 35 8 54 23 40 140 3 169 45 150 152 96 81 143 28 28 124 87 13 90 9 87 109 53 67 164 28 131 89 149 42 55 126 79 132 74 19 133 30 68 72 75 148 9 10 72 152 144 83 106 153 74 163 98 152 ^'}},
     {OUT=>'34AE3B806BE143A84DCE82E4B830EB7D3D2BAC69'}],
     ['159', {IN=> {f=> '375 1 94 28 13 8 20 28 18 118 5 140 89 67 171 64 152 85 61 101 80 154 149 34 115 135 128 108 110 20 33 128 103 35 38 57 95 10 111 151 98 29 149 7 82 69 96 114 26 103 171 101 53 121 24 2 121 51 35 70 83 29 154 67 5 167 63 16 27 58 60 2 99 128 71 33 160 70 51 3 44 149 2 89 84 101 43 18 113 71 38 94 55 46 74 52 139 102 35 43 50 80 122 6 100 88 129 36 25 148 47 24 7 55 36 30 82 32 19 78 63 26 71 28 167 85 56 167 95 159 78 26 66 35 65 90 44 159 105 59 15 67 57 137 21 132 50 70 78 36 169 5 172 56 1 22 129 68 168 23 74 30 99 138 174 120 91 153 8 42 68 158 155 104 99 133 5 135 96 82 59 52 144 36 20 38 62 29 112 106 46 106 19 73 70 41 135 133 42 95 15 45 136 93 100 98 173 6 132 55 4 4 31 64 130 162 156 64 157 19 46 96 92 103 139 134 2 94 18 86 38 89 62 61 79 157 14 112 111 156 32 14 85 78 31 59 20 1 161 169 149 29 122 150 133 37 102 30 109 65 85 51 174 110 164 66 77 76 111 119 131 95 5 1 166 103 134 141 17 158 123 137 48 165 175 102 13 3 86 43 23 47 56 150 165 4 36 174 115 157 168 13 147 119 109 55 41 140 67 27 31 27 53 126 17 163 116 122 160 60 7 92 113 66 45 109 60 151 125 62 39 39 17 153 13 152 53 94 50 53 89 57 127 153 45 119 104 121 56 145 172 19 41 103 166 108 68 126 12 53 131 127 148 52 134 168 50 48 155 72 132 82 37 129 63 175 160 ^'}},
     {OUT=>'D7447E53D66BB5E4C26E8B41F83EFD107BF4ADDA'}],
     ['160', {IN=> {f=> '366 1 73 15 79 139 71 51 132 81 135 111 166 61 37 60 118 37 92 134 94 60 150 53 47 25 110 19 137 107 27 176 43 16 145 146 142 165 12 17 92 166 110 117 41 69 6 12 164 76 109 40 101 82 82 131 134 115 143 112 32 164 146 93 162 31 66 123 144 80 135 94 137 156 107 157 173 62 65 176 124 112 4 65 127 44 131 53 19 105 91 149 106 27 57 158 92 85 14 79 20 84 127 174 128 72 68 59 171 17 124 80 167 8 57 28 172 66 144 48 42 37 109 121 18 120 103 116 13 132 39 73 145 76 158 43 82 33 51 153 164 97 13 2 3 20 24 114 98 9 25 131 102 99 86 42 16 26 159 39 105 161 133 49 59 41 20 105 31 136 30 87 10 119 135 13 83 99 58 45 99 156 26 61 135 143 80 118 107 76 77 24 44 12 66 95 147 61 86 140 99 167 138 54 47 42 97 122 18 60 90 40 73 85 63 82 1 35 96 94 81 54 15 128 40 52 2 101 1 165 93 113 130 85 95 98 58 105 112 53 68 108 72 100 152 49 87 140 39 154 62 114 77 105 63 50 87 157 26 78 122 67 140 71 170 119 5 93 64 50 104 144 129 138 75 130 4 178 60 139 120 54 113 32 2 133 17 43 163 129 89 72 149 155 30 38 7 138 5 44 61 69 76 175 107 6 93 81 114 28 25 82 22 118 58 73 96 175 177 92 36 11 98 171 38 28 42 146 125 178 88 136 139 103 67 10 151 151 70 153 115 94 32 51 34 177 173 132 49 119 54 96 45 78 73 64 159 120 22 20 154 111 117 58 56 109 72 143 ^'}},
     {OUT=>'77DD2A4482705BC2E9DC96EC0A13395771AC850C'}],
     ['161', {IN=> {f=> '372 1 136 146 144 170 90 15 169 180 88 41 89 10 60 52 92 145 21 155 47 42 61 26 18 176 48 21 124 141 40 179 178 114 143 14 164 71 142 132 11 124 23 39 82 114 84 57 163 130 127 89 133 16 138 151 130 160 34 70 78 161 64 92 112 110 72 175 102 138 179 173 159 59 19 27 18 53 92 22 65 156 170 57 104 86 126 148 118 155 78 43 63 90 48 80 168 142 68 12 39 174 69 168 147 118 96 34 129 35 75 5 107 60 170 180 147 85 3 133 75 95 152 71 175 43 94 178 50 144 16 46 101 46 165 45 68 163 38 172 5 138 61 85 62 177 76 36 167 19 53 151 66 66 97 3 100 164 12 98 131 70 109 162 161 156 142 14 105 27 141 9 178 81 50 113 110 131 37 16 79 12 34 37 167 173 10 152 7 81 140 158 26 136 82 50 172 154 45 160 31 122 167 142 9 150 125 51 54 125 174 10 124 51 177 81 173 69 109 8 22 155 19 86 62 149 33 37 108 34 96 29 174 133 167 50 54 66 130 9 1 36 134 108 65 97 126 56 163 71 83 88 10 1 8 178 22 6 112 169 116 89 43 153 40 146 85 31 89 74 154 137 28 115 117 122 108 98 8 27 110 103 176 17 20 27 77 10 93 145 80 17 143 165 76 69 78 102 20 91 171 95 29 115 176 166 109 39 23 70 83 29 148 67 163 161 33 4 15 46 42 2 69 104 53 33 160 64 33 155 14 131 160 59 78 71 19 176 107 65 38 82 55 40 56 40 115 84 29 25 44 68 104 176 76 64 123 18 25 136 23 6 171 31 30 30 58 14 13 72 57 14 41 177 ^'}},
     {OUT=>'EAA1465DB1F59DE3F25EB8629602B568E693BB57'}],
     ['162', {IN=> {f=> '363 0 135 69 16 167 79 143 46 178 26 3 65 58 36 135 97 35 175 51 152 57 137 173 108 10 54 78 4 153 173 164 48 177 22 129 68 168 167 74 174 91 106 174 96 75 129 168 34 36 158 147 88 75 101 149 103 64 42 59 52 112 180 20 30 22 173 80 98 46 106 179 65 30 17 127 125 26 87 15 5 136 85 100 66 141 182 124 15 156 148 15 56 98 146 156 48 149 3 14 96 52 79 115 134 170 54 178 54 22 73 30 45 39 133 182 96 103 148 176 6 45 62 183 51 180 153 145 145 117 21 90 118 109 21 62 22 101 49 53 11 142 86 132 66 69 44 79 87 115 55 5 169 150 63 110 109 177 150 91 129 40 149 159 102 165 3 70 27 175 23 56 126 165 164 20 158 91 157 136 157 131 111 69 39 33 132 43 19 175 179 37 118 9 155 116 122 128 52 7 84 89 58 13 85 44 135 125 54 183 23 161 145 173 136 29 62 50 13 57 25 127 153 45 95 64 121 40 121 140 171 9 71 166 68 28 102 4 53 123 111 148 20 102 160 34 8 139 32 116 66 13 113 63 143 97 63 10 59 124 66 41 132 71 115 106 151 46 22 45 118 37 67 124 94 60 140 43 47 10 110 9 127 97 7 156 43 180 125 126 137 145 12 2 82 146 100 117 26 59 180 181 164 76 89 30 101 77 72 126 114 95 143 102 32 149 131 83 157 31 61 123 139 60 135 74 169 122 151 102 137 168 62 55 166 119 97 163 60 112 24 116 38 178 105 91 134 86 27 47 138 72 70 183 79 15 84 117 154 128 57 53 39 161 88 ^'}},
     {OUT=>'9329D5B40E0DC43AA25FED69A0FA9C211A948411'}],
     ['163', {IN=> {f=> '393 1 92 72 151 33 183 164 42 112 32 10 5 93 97 18 112 71 92 168 116 179 186 65 137 68 134 3 82 9 19 121 148 65 160 173 158 167 90 66 180 172 99 86 83 86 26 16 10 135 23 81 137 125 17 19 25 183 73 7 104 185 79 165 119 119 176 59 75 18 5 59 132 26 21 127 119 80 94 83 52 61 24 44 4 50 55 115 29 70 140 99 159 130 54 15 42 89 114 173 28 50 16 33 53 47 58 180 19 64 78 49 30 186 104 32 44 157 85 1 149 85 97 114 77 63 58 50 89 80 21 36 84 72 60 152 25 87 108 15 138 54 106 77 97 23 42 55 149 181 62 106 43 132 31 138 111 176 93 40 10 96 128 105 114 43 98 4 146 20 107 120 14 113 181 101 1 19 155 113 57 64 117 131 6 22 186 138 184 36 61 29 36 175 107 6 61 81 114 20 180 42 169 102 26 73 80 151 153 68 4 11 98 139 6 175 34 146 93 154 88 128 139 79 35 2 135 119 46 129 91 86 179 51 18 169 141 108 49 103 46 72 21 70 49 64 151 96 185 175 130 111 85 58 32 101 40 101 131 136 139 170 70 10 169 175 73 41 69 177 45 47 92 145 11 155 22 22 56 21 13 166 48 21 109 131 40 174 178 104 128 9 149 51 132 122 173 119 23 24 67 109 74 32 158 110 117 74 123 6 118 151 130 150 167 34 55 58 146 54 72 112 105 57 160 82 123 159 153 159 39 19 7 8 33 87 12 50 146 150 37 104 71 126 133 108 145 68 38 38 90 43 75 148 122 43 2 39 154 54 163 147 113 91 29 109 35 75 182 87 35 155 170 127 80 185 118 60 95 142 71 165 28 84 168 25 144 178 31 81 41 160 25 53 143 ^'}},
     {OUT=>'E94C0B6AA62AA08C625FAF817DDF8F51EC645273'}],
     ['164', {IN=> {f=> '381 1 14 172 163 130 45 69 46 145 36 36 151 169 13 135 42 26 81 153 76 148 178 90 131 30 101 162 161 132 142 14 89 3 141 175 146 65 26 89 94 99 187 8 55 162 2 13 143 173 2 120 173 41 132 150 2 112 42 18 140 130 37 128 23 106 159 118 167 150 101 11 30 109 150 184 124 19 169 73 165 53 109 158 172 147 11 46 38 149 25 21 92 184 64 179 158 133 135 34 38 58 114 9 183 4 118 76 49 89 102 40 163 47 75 80 176 175 8 154 14 180 104 153 84 89 11 129 8 138 85 181 73 66 154 121 20 115 93 114 108 82 182 19 86 103 176 183 186 177 53 10 69 137 56 1 135 141 68 69 54 86 12 75 171 87 187 107 144 150 93 23 7 70 83 29 140 67 139 153 183 178 189 30 18 2 29 72 29 33 160 56 9 123 164 107 136 19 70 31 177 152 99 57 38 66 55 32 32 24 83 60 21 1 36 52 80 168 44 32 115 184 25 120 181 172 155 189 22 30 26 180 5 64 49 188 1 28 111 57 176 167 67 131 22 160 186 169 65 34 30 117 91 17 163 39 134 57 137 155 90 170 42 78 170 141 167 158 42 177 22 129 68 168 143 74 150 85 82 174 78 63 111 156 28 12 158 141 76 57 77 125 79 40 12 59 52 88 156 20 24 182 149 56 92 46 106 167 59 189 121 119 14 81 15 165 136 79 100 42 117 182 118 175 138 124 3 50 74 134 156 36 143 181 180 96 22 61 97 134 164 24 166 30 10 61 6 33 9 115 176 84 97 142 152 15 50 165 45 168 135 133 127 93 15 66 94 91 9 32 16 95 37 29 171 118 68 108 66 63 20 55 186 ^'}},
     {OUT=>'7FF02B909D82AD668E31E547E0FB66CB8E213771'}],
     ['165', {IN=> {f=> '396 1 97 10 5 160 132 18 83 73 159 141 55 120 31 131 141 102 138 3 52 9 148 189 56 99 165 146 2 140 64 157 100 121 113 102 24 21 24 123 16 10 139 152 19 109 146 116 122 92 43 7 75 62 49 170 58 26 117 125 45 147 5 125 136 155 118 2 26 50 161 21 182 127 153 45 68 19 121 22 94 104 144 166 35 166 23 176 75 188 53 114 93 148 177 66 151 16 156 121 180 98 48 179 95 63 107 97 45 1 23 97 57 23 132 53 79 97 124 19 188 18 118 37 22 106 94 60 122 25 47 176 110 184 109 79 164 120 43 153 89 90 128 109 12 168 64 110 82 117 192 41 171 163 164 76 53 12 101 68 54 117 78 59 143 84 32 122 104 65 148 31 52 123 130 24 135 38 151 95 142 93 101 159 62 37 148 110 70 127 51 85 181 89 11 142 105 91 107 50 27 29 102 36 43 165 79 6 84 99 118 128 30 26 3 143 17 68 66 139 187 15 165 158 24 88 20 179 174 81 79 18 106 47 74 150 104 179 162 59 131 62 116 166 82 184 188 97 136 41 136 167 140 143 175 72 42 174 148 75 74 71 86 14 16 191 117 11 63 119 119 186 182 13 171 49 182 80 167 73 147 119 107 164 41 57 181 168 29 114 26 184 121 101 80 76 65 34 49 24 44 191 38 25 91 5 58 140 99 153 124 54 184 42 83 108 155 4 20 191 3 29 35 40 180 7 40 66 25 12 180 86 26 38 139 73 1 137 79 85 102 71 39 28 44 77 56 190 12 66 72 30 152 7 87 84 190 126 48 100 77 91 186 36 31 143 163 50 94 25 126 1 114 105 170 93 22 173 90 116 87 96 19 74 4 122 183 83 120 177 113 169 181 77 182 1 149 186 ^'}},
     {OUT=>'5BB3570858FA1744123BAC2873B0BB9810F53FA1'}],
     ['166', {IN=> {f=> '384 1 25 56 85 107 177 6 186 138 184 28 61 184 191 175 107 6 29 81 114 12 156 2 137 86 189 73 64 127 129 44 167 11 98 107 169 143 26 146 61 130 88 120 139 55 3 189 119 87 22 105 67 78 147 51 2 161 109 84 49 87 38 48 192 62 25 64 143 72 169 151 106 111 53 58 8 93 8 85 123 120 131 170 38 2 169 167 49 41 37 153 21 39 92 145 190 155 177 185 48 13 5 150 48 21 85 115 40 166 178 88 104 1 125 19 116 106 141 111 23 43 101 58 187 150 78 101 50 107 185 86 151 130 134 143 34 31 26 122 38 40 112 97 33 136 50 99 127 121 159 7 19 170 187 1 79 191 26 130 118 5 104 47 126 109 92 129 52 30 193 90 35 67 116 90 3 181 39 122 30 155 147 105 83 21 77 35 75 174 55 190 131 154 95 72 185 94 36 95 126 71 149 4 68 152 180 144 146 7 49 33 152 188 29 111 194 172 148 125 35 59 36 125 11 36 141 149 183 125 27 1 71 133 61 138 168 85 131 5 96 162 161 117 142 14 79 183 141 165 126 55 11 74 84 79 167 3 40 142 177 193 128 173 192 100 163 16 127 145 182 97 17 193 120 115 32 108 18 96 154 103 152 150 86 181 15 99 135 179 124 194 164 68 160 43 109 138 152 142 6 21 23 149 20 11 82 164 44 159 148 133 115 24 28 53 104 9 183 179 108 56 39 84 87 30 163 32 70 75 166 170 8 139 9 175 99 143 64 89 186 114 183 133 85 161 63 61 154 111 15 115 78 109 108 72 177 14 71 103 176 173 176 157 38 10 54 132 41 186 130 126 63 69 39 76 7 65 171 82 172 102 124 140 83 113 ^'}},
     {OUT=>'905F43940B3591CE39D1145ACB1ECA80AB5E43CD'}],
     ['167', {IN=> {f=> '396 1 189 70 83 29 132 67 115 145 151 170 181 14 192 2 187 40 5 33 160 48 183 91 132 83 112 177 62 189 153 128 91 49 38 50 55 24 8 8 51 36 13 175 28 36 56 160 12 107 168 25 104 157 156 139 165 14 30 192 164 195 56 41 180 159 28 79 41 144 167 51 115 188 136 154 145 65 2 22 93 83 191 147 23 110 57 137 131 66 138 26 78 146 125 159 150 34 177 22 129 68 168 111 74 118 77 50 174 54 47 87 140 20 178 158 133 60 33 45 93 47 8 170 59 52 56 124 20 16 150 117 24 84 46 106 151 51 158 173 113 111 196 73 15 133 136 71 100 10 85 182 110 143 114 92 185 42 42 118 156 20 135 173 156 96 180 37 73 134 156 182 150 196 192 45 172 17 167 91 168 68 89 134 120 190 173 34 141 37 152 111 117 103 61 7 34 62 67 191 190 8 87 21 195 139 86 44 76 66 55 186 23 31 87 183 5 155 122 191 68 53 149 136 35 115 26 121 131 102 123 3 42 197 133 179 56 84 165 136 190 130 49 157 80 101 103 97 197 11 19 118 1 5 119 137 9 104 193 141 116 122 72 38 7 70 47 44 155 43 16 107 125 40 127 193 105 131 145 108 185 6 50 141 1 167 127 153 45 53 192 121 12 79 84 129 151 15 166 196 156 60 188 53 109 83 148 162 46 146 6 136 111 160 88 38 169 85 63 87 97 35 194 3 82 52 13 132 43 59 92 109 4 178 3 118 37 195 96 94 60 112 15 47 166 110 179 99 69 149 100 43 138 69 70 123 89 12 158 54 90 72 117 182 31 166 153 164 76 33 2 101 63 44 112 58 39 143 74 32 107 89 55 143 31 47 123 125 4 135 18 141 80 137 88 81 154 187 ^'}},
     {OUT=>'336C79FBD82F33E490C577E3F791C3CBFE842AFF'}],
     ['168', {IN=> {f=> '406 0 23 134 103 49 99 44 64 160 68 190 114 105 91 86 22 27 15 74 8 22 151 79 199 84 85 90 128 9 5 175 129 17 40 59 125 187 194 144 151 3 60 6 158 153 67 58 18 99 19 53 129 90 179 134 52 124 55 95 138 82 170 167 69 122 13 108 160 119 115 161 51 14 167 120 47 60 57 86 16 184 96 197 42 98 112 165 154 199 157 21 168 52 146 66 126 119 93 150 20 36 153 140 194 93 26 156 114 80 80 55 44 13 35 24 44 191 24 190 63 177 44 140 99 146 117 54 163 42 76 101 134 176 185 177 168 1 21 19 180 193 12 52 197 191 173 65 19 31 118 59 1 123 72 71 88 64 11 193 37 63 28 169 184 45 72 195 152 186 87 56 176 112 41 93 77 84 158 29 3 136 142 36 80 4 119 166 86 98 163 93 1 145 83 102 66 75 191 46 4 94 155 55 120 149 113 148 181 49 175 180 142 87 5 51 65 92 167 196 186 138 184 23 61 164 171 175 107 6 9 81 114 7 141 177 117 76 174 73 54 112 114 29 152 11 98 87 154 123 21 146 41 115 88 115 139 40 183 189 109 67 7 90 52 73 127 51 192 156 89 69 49 77 33 33 182 57 10 64 138 57 159 136 91 111 33 58 193 88 188 75 118 110 126 170 18 197 169 162 34 41 17 138 6 34 92 145 185 155 157 170 43 8 140 48 21 70 105 40 161 178 78 89 196 110 199 106 96 121 106 23 185 28 96 48 167 145 58 91 35 97 180 66 151 130 124 128 34 16 6 107 28 20 112 92 18 121 30 84 107 101 159 187 19 155 182 181 74 186 11 120 98 185 104 32 126 94 82 119 42 25 173 90 30 62 96 70 178 176 39 102 15 150 147 100 78 16 57 35 75 169 35 170 116 144 75 146 ^'}},
     {OUT=>'5C6D07A6B44F7A75A64F6CE592F3BAE91E022210'}],
     ['169', {IN=> {f=> '409 1 185 70 12 95 110 71 133 183 52 136 148 144 114 186 17 25 144 164 5 79 178 172 124 117 19 43 20 93 174 36 125 117 151 109 3 164 55 101 37 122 152 77 131 168 88 162 161 93 142 14 63 167 141 149 94 39 190 50 68 47 135 198 16 110 153 177 104 173 192 68 147 179 119 137 166 73 180 169 88 91 24 76 10 80 146 79 128 150 62 149 194 83 111 171 124 170 156 60 152 27 109 106 120 134 201 184 202 149 12 198 66 132 12 127 132 133 83 8 12 45 88 9 183 155 92 24 23 76 63 14 163 8 62 67 150 162 8 115 1 167 91 127 32 89 162 90 159 125 85 129 47 53 154 95 7 115 54 101 108 56 169 6 47 103 176 157 160 125 14 10 30 124 17 178 122 102 55 69 15 60 202 49 171 74 148 94 92 124 67 200 184 70 83 29 127 67 100 140 131 165 176 4 182 2 167 20 193 33 160 43 173 71 112 68 97 157 57 169 138 113 86 44 38 40 55 19 196 201 31 21 8 165 23 26 41 155 195 183 102 158 25 94 142 146 129 150 9 30 177 154 195 51 36 175 139 28 59 31 124 167 41 105 173 121 134 130 65 185 17 78 78 181 137 13 95 57 137 116 51 118 16 78 131 115 154 145 29 177 22 129 68 168 91 74 98 72 30 174 39 37 72 130 15 163 158 128 50 18 25 73 27 191 150 59 52 36 104 20 11 130 97 4 79 46 106 141 46 138 163 108 106 191 68 15 113 136 66 100 193 65 182 105 123 99 72 180 37 22 108 156 10 130 168 141 96 160 22 58 134 151 162 140 181 187 35 157 7 147 76 163 58 84 129 100 190 153 24 126 32 142 96 107 88 41 2 14 42 52 186 170 3 82 11 180 119 66 29 56 66 50 171 3 11 77 163 5 150 112 128 ^'}},
     {OUT=>'7E0D3E9D33127F4A30EB8D9C134A58409FA8695B'}],
     ['170', {IN=> {f=> '413 1 47 25 135 129 7 108 19 107 117 102 102 3 28 190 112 165 56 63 165 122 183 116 28 157 52 73 89 90 169 202 12 111 185 203 91 116 200 97 193 134 116 122 44 31 7 63 26 37 134 22 2 93 125 33 99 186 77 124 131 94 171 183 50 113 178 146 127 153 45 32 164 121 203 58 56 108 130 192 166 168 128 39 188 53 102 69 148 141 18 139 197 108 97 132 74 24 155 71 63 59 97 21 194 180 61 45 204 132 29 31 85 88 188 164 187 118 37 167 82 94 60 98 1 47 152 110 172 85 55 128 72 43 117 41 42 116 61 12 144 40 62 58 117 168 17 159 139 164 76 5 193 101 56 30 105 30 11 143 60 32 86 68 41 136 31 40 123 118 181 135 195 127 59 130 81 53 147 62 13 124 98 34 79 39 49 145 53 180 94 105 91 71 2 27 5 54 193 7 141 79 199 84 75 70 128 199 195 160 119 17 20 54 115 187 184 129 146 193 40 201 143 138 57 43 18 94 204 38 114 80 179 114 47 119 50 80 118 82 160 152 49 112 198 88 155 104 95 151 36 199 162 100 27 50 47 86 195 16 179 81 192 27 83 107 150 134 194 147 1 158 32 131 61 111 119 83 140 5 21 133 120 174 78 26 136 109 65 80 40 29 203 25 24 44 191 14 170 43 162 34 140 99 141 112 54 148 42 71 96 119 161 165 167 148 186 11 4 180 188 197 42 182 181 168 50 14 26 103 49 1 113 67 61 78 59 196 173 32 53 8 154 169 30 72 175 152 176 87 36 166 102 36 88 77 79 138 24 188 131 127 26 70 194 114 146 66 93 158 93 191 125 78 92 51 60 176 26 4 74 135 35 120 129 113 133 181 29 170 170 137 77 190 46 45 77 157 191 186 138 184 18 61 144 151 175 107 6 194 81 114 2 126 110 ^'}},
     {OUT=>'9A5F50DFCFB19286206C229019F0ABF25283028C'}],
     ['171', {IN=> {f=> '427 1 85 60 150 73 38 88 90 5 128 11 98 55 130 91 13 146 9 91 88 107 139 16 159 189 93 35 191 66 28 65 95 51 184 148 57 45 49 61 25 9 166 49 194 64 130 33 143 112 67 111 1 58 177 80 164 59 110 94 118 170 194 197 169 154 10 41 193 114 190 26 92 145 177 155 125 146 35 200 124 48 21 46 89 40 153 178 62 65 196 86 175 90 80 89 98 23 169 4 88 32 135 137 26 75 11 81 172 34 151 130 108 104 34 200 182 83 12 196 112 84 202 97 206 60 75 69 159 163 19 131 174 157 66 178 195 104 66 161 104 8 126 70 66 103 26 17 141 90 22 54 64 38 146 168 39 70 199 142 147 92 70 8 25 35 75 161 3 138 92 128 43 59 185 55 205 95 100 71 123 173 42 126 128 144 94 176 205 20 139 149 198 59 168 172 109 112 9 33 10 73 154 36 115 97 131 99 196 144 45 81 22 112 142 72 131 148 83 162 161 78 142 14 53 157 141 139 74 29 180 35 58 27 115 198 1 90 138 167 89 173 192 48 137 159 114 132 156 58 160 154 68 76 19 56 5 70 141 64 113 150 47 129 184 73 96 166 124 155 151 55 147 17 109 86 100 129 201 164 192 149 7 193 56 112 200 107 122 133 63 206 2 40 78 9 183 140 82 4 13 71 48 4 163 201 57 62 140 157 8 100 204 162 86 117 12 89 147 75 144 120 85 109 37 48 154 85 2 115 39 96 108 46 164 1 32 103 176 147 150 105 207 10 15 119 2 173 117 87 50 69 50 202 39 171 69 133 89 72 114 57 195 179 70 83 29 122 67 85 135 111 160 171 202 172 2 147 183 33 160 38 163 51 92 53 82 137 52 149 123 98 81 39 38 30 55 14 186 196 11 6 3 155 18 16 26 150 180 168 97 148 25 84 127 136 119 135 4 30 162 144 195 46 31 170 119 28 190 ^'}},
     {OUT=>'DCA737E269F9D8626D488988C996E06B352C0708'}],
     ['172', {IN=> {f=> '443 1 17 96 167 27 91 152 100 106 109 65 164 10 57 71 167 123 209 74 57 137 95 30 90 2 78 110 101 147 138 22 177 22 129 68 168 63 74 70 65 2 174 18 23 51 116 8 142 158 121 36 207 207 45 209 170 122 59 52 8 76 20 4 102 69 186 72 46 106 127 39 110 149 101 99 184 61 15 85 136 59 100 172 37 182 98 95 78 44 173 30 204 94 156 206 123 161 120 96 132 1 37 134 144 134 126 160 180 21 136 203 119 55 156 44 77 122 72 190 125 10 105 25 128 75 93 67 13 205 196 14 31 179 142 206 75 207 159 91 38 8 28 66 43 150 185 193 63 135 5 143 98 143 32 5 125 124 197 103 14 97 107 102 87 3 18 185 97 155 56 48 165 112 178 106 13 157 32 53 79 85 149 197 7 106 175 203 71 101 195 92 193 129 116 122 24 26 7 58 11 32 119 7 202 83 125 28 79 181 57 119 121 84 161 168 50 93 163 131 127 153 45 17 144 121 198 43 36 93 115 177 166 148 108 24 188 53 97 59 148 126 208 134 192 88 87 112 64 14 145 61 63 39 97 11 194 165 46 40 199 132 19 11 80 73 178 154 177 118 37 147 72 94 60 88 201 47 142 110 167 75 45 113 52 43 102 21 22 111 41 12 134 30 42 48 117 158 7 154 129 164 76 195 188 101 51 20 100 10 201 143 50 32 71 53 31 131 31 35 123 113 166 135 180 117 44 125 76 33 142 62 3 114 93 19 59 34 34 130 38 170 74 105 91 56 192 27 205 34 178 202 131 79 199 84 65 50 128 189 185 145 109 17 49 105 187 174 114 141 183 20 196 128 123 47 28 18 89 189 23 99 70 179 94 42 114 45 65 98 82 150 137 29 102 183 68 150 89 75 141 21 184 157 80 7 40 37 86 190 16 174 66 187 12 68 102 135 114 189 137 191 148 12 116 56 96 119 73 130 200 6 113 100 154 63 26 116 104 50 80 25 14 193 83 ^'}},
     {OUT=>'B8FFC1D4972FCE63241E0E77850AC46DDE75DBFA'}],
     ['173', {IN=> {f=> '436 1 24 44 191 211 138 11 138 18 140 99 133 104 54 124 42 63 88 95 137 133 151 116 162 208 193 180 180 173 26 158 165 160 26 6 18 79 33 1 97 59 45 62 51 172 141 24 37 189 130 145 6 72 143 152 160 87 4 150 86 28 80 77 71 106 16 164 123 103 10 54 178 106 114 34 85 150 93 175 93 70 76 27 36 152 207 4 42 103 3 120 97 113 109 181 210 162 154 129 61 166 38 13 53 141 183 186 138 184 10 61 112 119 175 107 6 170 81 114 207 102 125 65 50 135 73 28 73 75 203 113 11 98 35 115 71 8 146 202 76 88 102 139 1 144 189 83 15 181 51 13 60 75 51 179 143 37 30 49 51 20 207 156 44 184 64 125 18 133 97 52 111 194 58 167 75 149 49 105 84 113 170 179 197 169 149 208 41 178 99 180 21 92 145 172 155 105 131 30 208 200 114 48 21 31 79 40 148 178 52 50 196 71 160 80 70 69 93 23 159 202 83 22 115 132 6 65 209 71 167 14 151 130 98 89 34 190 167 68 2 181 112 79 192 82 191 45 55 49 159 148 19 116 169 142 61 173 185 94 46 146 104 206 126 55 56 93 16 12 121 90 17 49 44 18 126 163 39 50 189 137 147 87 65 3 5 35 75 156 196 118 77 118 23 54 185 40 195 95 90 71 113 163 32 116 108 144 74 166 190 15 134 134 188 39 158 172 94 107 212 23 53 134 36 105 77 111 89 186 124 35 61 7 102 132 67 131 128 78 162 161 63 142 14 43 147 141 129 54 19 170 20 48 7 95 198 199 70 123 157 74 173 192 28 127 139 109 127 146 43 140 139 48 61 14 36 60 136 49 98 150 32 109 174 63 81 161 124 140 146 50 142 7 109 66 80 124 201 144 182 149 2 188 46 92 185 87 112 133 43 201 205 35 68 9 183 125 72 197 3 66 33 207 163 191 52 57 130 152 8 85 204 157 81 107 205 187 ^'}},
     {OUT=>'E9C9BF41C8549354151B977003CE1D830BE667DB'}],
     ['174', {IN=> {f=> '462 1 126 54 123 113 85 81 23 41 154 71 210 115 18 89 108 32 157 209 11 103 176 133 136 77 193 10 209 112 196 166 110 66 43 69 194 36 202 25 171 62 112 82 44 100 43 188 172 70 83 29 115 67 64 128 83 153 164 195 158 2 119 187 169 33 160 31 149 23 64 32 61 109 45 121 102 77 74 32 38 16 55 7 172 189 198 200 211 141 11 2 5 143 159 147 90 134 25 70 106 122 105 114 212 30 141 130 195 39 24 163 91 28 11 7 76 167 17 81 137 85 86 94 65 149 5 42 66 157 113 204 59 57 137 80 15 70 207 78 95 91 142 133 17 177 22 129 68 168 43 74 50 60 197 174 3 13 36 106 3 127 158 116 26 197 192 25 194 155 102 59 52 203 56 20 214 82 49 171 67 46 106 117 34 90 139 96 94 179 56 15 65 136 54 100 157 17 182 93 75 63 24 168 25 189 84 156 201 118 156 105 96 112 201 22 134 139 114 116 145 175 11 121 198 99 40 151 34 72 117 52 190 105 90 20 118 60 83 52 208 205 181 209 16 174 122 206 70 202 144 71 18 208 8 66 38 135 170 178 53 115 5 138 88 123 17 200 115 119 182 98 9 87 97 102 72 3 8 180 82 145 56 33 165 102 173 96 213 157 12 33 69 80 129 192 2 101 165 203 51 86 190 87 193 124 116 122 4 21 7 53 211 27 104 207 197 73 125 23 59 176 37 114 111 74 151 153 50 73 148 116 127 153 45 2 124 121 193 28 16 78 100 162 166 128 88 9 188 53 92 49 148 111 193 129 187 68 77 92 54 4 135 51 63 19 97 1 194 150 31 35 194 132 9 206 75 58 168 144 167 118 37 127 62 94 60 78 196 47 132 110 162 65 35 98 32 43 87 1 2 106 21 12 124 20 22 38 117 148 212 149 119 164 76 180 183 101 46 10 95 205 186 143 40 32 56 38 21 126 31 30 123 108 151 135 165 107 29 120 71 13 137 62 208 104 88 4 39 29 19 115 23 160 54 105 91 41 177 27 200 14 163 124 ^'}},
     {OUT=>'0942908960B54F96CB43452E583F4F9CB66E398A'}],
     ['175', {IN=> {f=> '453 0 115 79 199 84 49 18 128 173 169 121 93 17 186 41 89 187 158 90 133 167 206 188 104 99 31 4 18 81 165 217 75 54 179 62 34 106 37 41 66 82 134 113 215 86 159 36 142 65 43 125 215 160 149 48 193 24 21 86 182 16 166 42 179 206 44 94 111 82 181 121 167 132 198 92 48 72 119 57 114 184 200 81 68 122 39 26 84 96 26 80 1 208 177 217 24 44 191 206 118 209 123 8 140 99 128 99 54 109 42 58 83 80 122 113 141 96 147 203 183 180 175 158 16 143 155 155 11 1 13 64 23 1 87 54 35 52 46 157 121 19 27 174 115 130 209 72 123 152 150 87 202 140 76 23 75 77 66 86 11 149 118 88 44 168 101 94 14 80 145 93 165 73 65 66 12 21 137 192 4 22 83 201 120 77 113 94 181 195 157 144 124 51 151 33 211 38 131 178 186 138 184 5 61 92 99 175 107 6 155 81 114 207 87 105 45 40 120 73 18 58 60 193 98 11 98 15 100 51 3 146 187 61 88 97 139 204 129 189 73 213 171 36 216 55 55 51 174 138 17 15 49 41 15 197 146 39 174 64 120 3 123 82 37 111 179 58 157 70 134 39 100 74 108 170 164 197 169 144 198 41 163 84 170 16 92 145 167 155 85 116 25 208 200 104 48 21 16 69 40 143 178 42 35 196 56 145 70 60 49 88 23 149 192 78 12 95 127 204 55 199 61 162 212 151 130 88 74 34 180 152 53 210 166 112 74 182 67 176 30 35 29 159 133 19 101 164 127 56 168 175 84 26 131 104 196 126 40 46 83 6 7 101 90 12 44 24 216 106 158 39 30 179 132 147 82 60 216 203 35 75 151 181 98 62 108 3 49 185 25 185 95 80 71 103 153 22 106 88 144 54 156 175 10 129 119 178 19 148 172 79 102 207 13 208 33 114 36 95 57 91 79 176 104 25 41 210 92 122 62 131 108 73 162 161 48 142 14 33 137 141 119 34 9 160 5 38 205 75 198 189 50 108 112 ^'}},
     {OUT=>'FCE34051C34D4B81B85DDC4B543CDE8007E284B3'}],
     ['176', {IN=> {f=> '454 1 53 173 192 113 111 102 120 132 22 112 118 20 40 7 8 213 46 129 28 77 150 11 81 160 49 60 154 124 119 139 43 135 213 109 38 52 117 201 116 168 149 215 181 32 64 164 59 98 133 15 194 198 28 54 9 183 104 58 176 209 59 12 200 163 177 45 50 116 145 8 64 204 150 74 93 184 89 111 39 108 108 85 61 13 36 154 61 210 115 3 84 108 22 152 209 216 103 176 123 126 57 183 10 199 107 186 161 105 51 38 69 184 26 202 15 171 57 97 77 24 90 33 183 167 70 83 29 110 67 49 123 63 148 159 190 148 2 99 172 159 33 160 26 139 3 44 17 46 89 40 101 87 62 69 27 38 6 55 2 162 184 183 190 211 131 6 212 210 138 144 132 85 124 25 60 91 112 95 99 212 30 126 120 195 34 19 158 71 28 211 217 56 167 7 71 122 70 66 79 65 134 27 61 147 103 199 44 57 137 65 50 202 78 80 81 137 128 12 177 22 129 68 168 23 74 30 55 182 174 208 3 21 96 218 112 158 111 16 187 177 5 179 140 82 59 52 188 36 20 214 62 29 156 62 46 106 107 29 70 129 91 89 174 51 15 45 136 49 100 142 217 182 88 55 48 4 163 20 174 74 156 196 113 151 90 96 92 191 7 134 134 94 106 130 170 1 106 193 79 25 146 24 67 112 32 190 85 210 75 15 108 45 73 37 193 205 166 194 1 169 102 206 65 197 129 51 218 198 208 66 33 120 155 163 43 95 5 133 78 103 2 185 105 114 167 93 4 77 87 102 57 3 218 175 67 135 56 18 165 92 168 86 203 157 212 13 59 75 109 187 217 96 155 203 31 71 185 82 193 119 116 122 204 16 7 48 201 22 89 197 192 63 125 18 39 171 17 109 101 64 141 138 50 53 133 101 127 153 45 207 104 121 188 13 216 63 85 147 166 108 68 214 188 53 87 39 148 96 178 124 182 48 67 72 44 214 125 41 63 219 97 211 194 135 16 30 189 132 219 191 70 43 158 181 ^'}},
     {OUT=>'61E8916532503627F4024D13884640A46F1D61D4'}],
     ['177', {IN=> {f=> '475 0 153 118 37 99 48 94 60 64 189 47 118 110 155 51 21 77 4 43 66 195 196 99 215 12 110 6 216 24 117 134 205 142 105 164 76 159 176 101 39 218 88 184 165 143 26 32 35 17 7 119 31 23 123 101 130 135 144 93 8 113 64 207 130 62 201 90 81 205 11 22 220 94 2 146 26 105 91 20 156 27 193 208 142 178 107 79 199 84 41 2 128 165 161 109 85 17 174 37 81 187 150 78 129 159 194 184 92 87 23 214 18 77 153 209 63 46 179 46 30 102 33 29 50 82 126 101 203 78 147 20 138 53 27 117 207 148 145 32 181 16 13 86 178 16 162 30 175 198 32 90 99 66 177 113 155 124 186 80 44 60 119 49 106 176 192 65 52 106 27 26 68 92 14 80 211 200 169 213 24 44 191 202 102 197 111 140 99 124 95 54 97 42 54 79 68 110 97 133 80 135 199 175 180 171 146 8 131 147 151 221 219 9 52 15 1 79 50 27 44 42 145 105 15 19 162 103 118 201 72 107 152 142 87 190 132 68 19 71 77 62 70 7 137 114 76 214 36 160 97 78 220 76 141 93 157 57 61 58 9 125 180 4 6 67 189 120 61 113 82 181 183 153 136 120 43 139 29 199 26 123 174 186 138 184 1 61 76 83 175 107 6 143 81 114 207 75 89 29 32 108 73 10 46 48 185 86 11 98 221 88 35 221 146 175 49 88 93 139 196 117 189 65 201 163 24 208 51 39 51 170 134 1 3 49 33 11 189 138 35 166 64 116 213 115 70 25 111 167 58 149 66 122 31 96 66 104 170 152 197 169 140 190 41 151 72 162 12 92 145 163 155 69 104 21 208 200 96 48 21 4 61 40 139 178 34 23 196 44 133 62 52 33 84 23 141 184 74 4 79 123 192 47 191 53 158 200 151 130 80 62 34 172 140 41 206 154 112 70 174 55 164 18 19 13 159 121 19 89 160 115 52 164 167 76 10 119 104 188 126 28 38 75 220 3 85 90 8 40 8 204 90 154 39 14 171 128 147 78 56 216 191 35 75 147 169 82 50 100 209 45 185 13 177 95 150 ^'}},
     {OUT=>'F008D5D7853B6A17B7466CD9E18BD135E520FAF4'}],
     ['178', {IN=> {f=> '471 0 71 89 139 8 92 60 144 26 142 154 3 122 98 164 216 134 172 58 95 200 224 201 5 86 36 81 29 63 65 162 76 11 13 196 78 108 55 131 80 66 162 161 27 142 14 19 123 141 105 6 220 146 209 24 184 47 198 175 22 87 133 38 173 192 205 103 91 97 115 122 7 92 103 25 2 213 213 36 124 13 62 150 221 61 150 39 45 149 124 104 134 38 130 208 109 18 32 112 201 96 158 149 215 176 22 44 149 39 88 133 220 189 193 23 44 9 183 89 48 161 204 54 222 195 163 167 40 45 106 140 8 49 204 145 69 83 169 89 96 24 93 103 85 41 3 31 154 51 210 115 213 79 108 12 147 209 206 103 176 113 116 37 173 10 189 102 176 156 100 36 33 69 174 16 202 5 171 52 82 72 4 80 23 178 162 70 83 29 105 67 34 118 43 143 154 185 138 2 79 157 149 33 160 21 129 208 24 2 31 69 35 81 72 47 64 22 38 221 55 222 152 179 168 180 211 121 1 207 200 133 129 117 80 114 25 50 76 102 85 84 212 30 111 110 195 29 14 153 51 28 196 212 36 167 222 61 107 55 46 64 65 119 220 12 56 137 93 194 29 57 137 50 210 30 197 78 65 71 132 123 7 177 22 129 68 168 3 74 10 50 167 174 198 218 6 86 218 97 158 106 6 177 162 210 164 125 62 59 52 173 16 20 214 42 9 141 57 46 106 97 24 50 119 86 84 169 46 15 25 136 44 100 127 202 182 83 35 33 209 158 15 159 64 156 191 108 146 75 96 72 181 217 134 129 74 96 115 165 216 91 188 59 10 141 14 62 107 12 190 65 205 60 10 98 30 63 22 178 205 151 179 211 164 82 206 60 192 114 31 203 188 193 66 28 105 140 148 33 75 5 128 68 83 212 170 95 109 152 88 224 67 77 102 42 3 213 170 52 125 56 3 165 82 163 76 193 157 197 218 49 70 89 182 217 91 145 203 11 56 180 77 193 114 116 122 189 11 7 43 191 17 74 187 187 53 125 13 19 166 222 104 91 54 131 123 50 33 118 86 127 167 ^'}},
     {OUT=>'BD8D2E873CF659B5C77AAC1616827EF8A3B1A3B3'}],
     ['179', {IN=> {f=> '480 1 45 193 76 121 181 219 195 42 64 126 166 80 40 200 188 53 80 25 148 75 157 117 175 20 53 44 30 207 111 27 63 198 97 204 194 114 222 23 182 132 212 170 63 22 144 120 143 118 37 79 38 94 60 54 184 47 108 110 150 41 11 62 211 43 51 180 181 94 200 12 100 223 201 14 117 124 200 137 95 164 76 144 171 101 34 213 83 169 150 143 16 32 20 2 224 114 31 18 123 96 115 135 129 83 220 108 59 192 125 62 196 80 76 195 218 17 210 79 214 136 6 105 91 5 141 27 188 193 127 168 97 79 199 84 31 209 128 155 151 94 75 17 159 32 71 187 140 63 124 149 179 179 77 72 13 204 18 72 138 199 48 36 179 26 25 97 28 14 30 82 116 86 188 68 132 133 38 7 107 197 133 140 12 166 6 3 86 173 16 157 15 170 188 17 85 84 46 172 103 140 114 171 65 39 45 119 39 96 166 182 45 32 86 12 26 48 87 226 80 201 190 159 208 24 44 191 197 82 182 96 217 140 99 119 90 54 82 42 49 74 53 95 77 123 60 120 194 165 180 166 131 225 116 137 146 211 219 4 37 5 1 69 45 17 34 37 130 85 10 9 147 88 103 191 72 87 152 132 87 175 122 58 14 66 77 57 50 2 122 109 61 209 26 150 92 58 205 71 136 93 147 37 56 48 212 221 110 165 4 213 47 174 120 41 113 67 181 168 148 126 115 33 124 24 184 11 113 169 186 138 184 223 61 56 63 175 107 6 128 81 114 207 60 69 9 22 93 73 31 33 175 71 11 98 206 73 15 221 146 160 34 88 88 139 186 102 189 55 186 153 9 198 46 19 51 165 129 208 215 49 23 6 179 128 30 156 64 111 203 105 55 10 111 152 58 139 61 107 21 91 56 99 170 137 197 169 135 180 41 136 57 152 7 92 145 158 155 49 89 16 208 200 86 48 21 216 51 40 134 178 24 8 196 29 118 52 42 13 79 23 131 174 69 221 59 118 177 37 181 43 153 185 151 130 70 47 34 162 125 26 201 139 112 65 164 40 149 3 226 220 159 106 19 74 155 100 47 159 157 193 ^'}},
     {OUT=>'B25A04DD425302ED211A1C2412D2410FA10C63B6'}],
     ['180', {IN=> {f=> '471 0 211 98 104 174 126 7 24 61 213 225 57 90 1 33 209 183 62 147 39 215 157 121 147 71 49 216 170 35 75 140 148 54 29 86 188 38 185 221 163 95 58 71 81 131 84 44 144 10 134 142 228 118 86 156 204 126 172 46 91 196 220 197 218 70 36 73 13 47 57 154 60 3 226 188 70 100 51 131 64 62 162 161 15 142 14 11 115 141 97 219 216 138 201 16 172 31 198 167 6 75 125 26 173 192 193 95 75 93 111 114 224 76 91 213 13 227 201 213 28 120 1 50 150 213 45 142 31 33 145 124 92 130 34 126 204 109 2 16 108 201 80 150 149 215 172 14 28 137 23 80 133 208 185 189 19 36 9 183 77 40 149 200 50 214 191 163 159 36 41 98 136 8 37 204 141 65 75 157 89 84 12 81 99 85 25 224 27 154 43 210 115 205 75 108 4 143 209 198 103 176 105 108 21 165 10 181 98 168 152 96 24 29 69 166 8 202 226 171 48 70 68 217 72 15 174 158 70 83 29 101 67 22 114 27 139 150 181 130 2 63 145 141 33 160 17 121 196 8 219 19 53 31 65 60 35 60 18 38 217 55 222 144 175 156 172 211 113 226 203 192 129 117 105 76 106 25 42 64 94 77 72 212 30 99 102 195 25 10 149 35 28 184 208 20 167 218 53 95 43 30 52 65 107 220 52 129 85 190 17 57 137 38 202 14 193 78 53 63 128 119 3 177 22 129 68 168 216 74 223 46 155 174 190 214 223 78 218 85 158 102 227 169 150 198 152 113 46 59 52 161 20 214 26 222 129 53 46 106 89 20 34 111 82 80 165 42 15 9 136 40 100 115 190 182 79 19 21 197 154 11 147 56 156 187 104 142 63 96 56 173 209 134 125 58 88 103 161 212 79 184 43 227 137 6 58 103 225 190 49 201 48 6 90 18 55 10 166 205 139 167 203 160 66 206 56 188 102 15 191 180 181 66 24 93 128 136 25 59 5 124 60 67 204 158 87 105 140 84 224 59 69 102 30 3 209 166 40 117 56 220 165 74 159 68 185 157 185 206 41 66 167 ^'}},
     {OUT=>'A404E21588123E0893718B4B44E91414A785B91F'}],
     ['181', {IN=> {f=> '490 1 176 217 85 133 203 218 38 174 71 193 108 116 122 171 5 7 37 179 11 56 175 181 41 125 7 226 160 204 98 79 42 119 105 50 9 100 68 127 153 45 185 60 121 177 211 183 30 52 114 166 64 24 192 188 53 76 17 148 63 145 113 171 4 45 28 22 203 103 19 63 186 97 200 194 102 214 19 178 132 208 158 59 10 136 112 135 118 37 63 30 94 60 46 180 47 100 110 146 33 3 50 199 43 39 168 169 90 188 12 92 219 189 6 117 116 196 133 87 164 76 132 167 101 30 209 79 157 138 143 8 32 8 221 220 110 31 14 123 92 103 135 117 75 212 104 55 180 121 62 192 72 72 187 206 13 202 67 206 128 221 105 91 224 129 27 184 181 115 160 89 79 199 84 23 197 128 147 143 82 67 17 147 28 63 187 132 51 120 141 167 175 65 60 5 196 18 68 126 191 36 28 179 10 21 93 24 2 14 82 108 74 176 60 120 215 129 26 222 99 189 121 136 227 154 229 226 86 169 16 153 3 166 180 5 81 72 30 168 95 128 106 159 53 35 33 119 31 88 158 174 29 16 70 26 32 83 218 80 193 182 151 204 24 44 191 193 66 170 84 213 140 99 115 86 54 70 42 45 70 41 83 61 115 44 108 190 157 180 162 119 221 104 129 142 203 219 25 228 1 61 41 9 26 33 118 69 6 1 135 76 91 183 72 71 152 124 87 163 114 50 10 62 77 53 34 229 110 105 49 205 18 142 88 42 193 67 132 93 139 21 52 40 204 213 98 153 4 201 31 162 120 25 113 55 181 156 144 118 111 25 112 20 172 230 105 165 186 138 184 223 61 40 47 175 107 6 116 81 114 207 48 53 224 14 81 73 223 19 21 167 59 11 98 194 61 230 221 146 148 22 88 84 139 178 90 189 47 174 145 228 190 42 3 51 161 125 196 207 49 15 2 171 120 26 148 64 107 195 97 43 229 111 140 58 131 57 95 13 87 48 95 170 125 197 169 131 172 41 124 45 144 3 92 145 154 155 33 77 12 208 200 78 48 21 208 43 40 130 178 16 227 196 17 106 44 34 228 75 23 123 166 65 217 43 114 165 29 173 35 200 ^'}},
     {OUT=>'A1E13BC55BF6DAD83CF3AABDA3287AD68681EA64'}],
     ['182', {IN=> {f=> '479 0 167 151 130 58 29 34 150 107 8 195 121 112 59 152 22 131 218 208 202 159 88 19 56 149 82 41 153 145 54 199 86 104 166 126 228 16 53 209 225 41 90 230 29 197 171 46 143 39 203 149 117 147 67 45 216 158 35 75 136 136 38 17 78 176 34 185 213 155 95 50 71 73 123 225 76 28 144 227 126 130 228 114 74 148 192 118 172 34 87 192 216 193 206 54 36 65 230 31 49 146 44 228 214 180 62 92 47 131 48 58 162 161 3 142 14 3 107 141 89 207 212 130 193 8 160 15 198 159 223 63 117 14 173 192 181 87 59 89 107 106 216 60 79 201 1 227 189 213 20 116 222 38 150 205 29 134 23 21 141 124 80 126 30 122 200 109 219 104 201 64 142 149 215 168 6 12 125 7 72 133 196 181 185 15 28 9 183 65 32 137 196 46 206 187 163 151 32 37 90 132 8 25 204 137 61 67 145 89 72 69 95 85 9 220 23 154 35 210 115 197 71 108 229 139 209 190 103 176 97 100 5 157 10 173 94 160 148 92 12 25 69 158 202 222 171 44 58 64 205 64 7 170 154 70 83 29 97 67 10 110 11 135 146 177 122 2 47 133 133 33 160 13 113 184 225 211 7 37 27 49 48 23 56 14 38 213 55 222 136 171 144 164 211 105 226 199 184 125 105 93 72 98 25 34 52 86 69 60 212 30 87 94 195 21 6 145 19 28 172 204 4 167 214 45 83 31 14 40 65 95 220 221 48 121 77 186 5 57 137 26 194 231 189 78 41 55 124 115 232 177 22 129 68 168 204 74 211 42 143 174 182 210 215 70 218 73 158 98 223 161 138 186 140 101 30 59 52 149 217 20 214 10 210 117 49 46 106 81 16 18 103 78 76 161 38 15 226 136 36 100 103 178 182 75 3 9 185 150 7 135 48 156 183 100 138 51 96 40 165 201 134 121 42 80 91 157 208 67 180 27 219 133 231 54 99 213 190 33 197 36 2 82 6 47 231 154 205 127 155 195 156 50 206 52 184 90 232 179 172 169 66 20 81 116 124 17 43 5 120 52 51 196 146 79 101 128 80 224 65 ^'}},
     {OUT=>'D5FD35FFABED6733C92365929DF0FB4CAE864D15'}],
     ['183', {IN=> {f=> '503 1 55 102 9 3 202 159 19 103 56 206 165 60 152 54 171 157 164 185 27 59 45 171 217 80 123 203 203 23 169 66 193 103 116 122 156 7 32 169 6 41 165 176 31 125 2 211 155 189 93 69 32 109 90 50 225 85 53 127 153 45 175 40 121 172 201 168 15 37 99 166 44 4 182 188 53 71 7 148 48 130 108 166 220 35 8 12 198 93 9 63 171 97 195 194 87 204 14 173 132 203 143 54 231 126 102 125 118 37 43 20 94 60 36 175 47 90 110 141 23 229 35 184 43 24 153 154 85 173 12 82 214 174 232 117 106 191 128 77 164 76 117 162 101 25 204 74 142 123 143 234 32 229 211 215 105 31 9 123 87 88 135 102 65 202 99 50 165 116 62 187 62 67 177 191 8 192 52 196 118 206 105 91 214 114 27 179 166 100 150 79 79 199 84 13 182 128 137 133 67 57 17 132 23 53 187 122 36 115 131 152 170 50 45 231 186 18 63 111 181 21 18 179 226 16 88 19 223 230 82 98 59 161 50 105 200 124 11 207 89 179 106 131 212 139 224 221 86 164 16 148 224 161 170 226 76 57 10 163 85 113 96 144 38 30 18 119 21 78 148 164 9 232 50 221 26 12 78 208 80 183 172 141 199 24 44 191 188 46 155 69 208 140 99 110 81 54 55 42 40 65 26 68 41 105 24 93 185 147 180 157 104 216 89 119 137 193 219 231 10 223 1 51 36 235 16 28 103 49 1 227 120 61 76 173 72 51 152 114 87 148 104 40 5 57 77 48 14 229 95 100 34 200 8 132 83 22 178 62 127 93 129 1 47 30 194 203 83 138 4 186 11 147 120 5 113 40 181 141 139 108 106 15 97 15 157 220 95 160 186 138 184 223 61 20 27 175 107 6 101 81 114 207 33 33 209 4 66 73 218 4 6 157 44 11 98 179 46 215 221 146 133 7 88 79 139 168 75 189 37 159 135 218 180 37 219 51 156 120 181 197 49 5 233 161 110 21 138 64 102 185 87 28 219 111 125 58 121 52 80 3 82 38 90 170 110 197 169 126 162 41 109 30 134 234 92 145 149 155 13 62 7 208 200 68 48 21 198 33 40 125 178 6 217 196 2 91 34 24 213 70 23 113 161 ^'}},
     {OUT=>'C12E9C280EE9C079E0506FF89F9B20536E0A83EF'}],
     ['184', {IN=> {f=> '470 0 58 210 15 107 144 15 159 21 142 152 151 130 48 14 34 140 92 231 190 106 112 54 142 7 116 208 193 187 159 73 19 41 144 67 36 148 135 44 184 71 104 156 126 218 6 43 204 225 21 90 230 24 182 156 26 138 39 188 139 112 147 62 40 216 143 35 75 131 121 18 2 68 161 29 185 203 145 95 40 71 63 113 220 66 8 144 212 116 115 228 109 59 138 177 108 172 19 82 187 211 188 191 34 36 55 215 11 39 136 24 223 199 170 52 82 42 131 28 53 162 161 226 142 14 231 97 141 79 192 207 120 183 236 145 233 198 149 208 48 107 237 173 192 166 77 39 84 102 96 206 40 64 186 224 227 174 213 10 111 212 23 150 195 9 124 13 6 136 124 65 121 25 117 195 109 204 218 99 201 44 132 149 215 163 234 230 110 225 62 133 181 176 180 10 18 9 183 50 22 122 191 41 196 182 163 141 27 32 80 127 8 10 204 132 56 57 130 89 57 223 54 90 85 227 215 18 154 25 210 115 187 66 108 224 134 209 180 103 176 87 90 223 147 10 163 89 150 143 87 235 20 69 148 228 202 217 171 39 43 59 190 54 235 165 149 70 83 29 92 67 233 105 229 130 141 172 112 2 27 118 123 33 160 8 103 169 210 201 230 17 22 29 33 8 51 9 38 208 55 222 126 166 129 154 211 95 226 194 174 120 90 78 67 88 25 24 37 76 59 45 212 30 72 84 195 16 1 140 237 28 157 199 222 167 209 35 68 16 232 25 65 80 220 211 43 111 67 181 228 57 137 11 184 216 184 78 26 45 119 110 232 177 22 129 68 168 189 74 196 37 128 174 172 205 205 60 218 58 158 93 218 151 123 171 125 86 10 59 52 134 202 20 214 228 195 102 44 46 106 71 11 236 93 73 71 156 33 15 211 136 31 100 88 163 182 70 221 232 170 145 2 120 38 156 178 95 133 36 96 20 155 191 134 116 22 70 76 152 203 52 175 7 209 128 226 49 94 198 190 13 192 21 235 72 229 37 221 139 205 112 140 185 151 30 206 47 179 75 217 101 ^'}},
     {OUT=>'E22769DC00748A9BBD6C05BBC8E81F2CD1DC4E2D'}],
     ['185', {IN=> {f=> '502 0 158 148 66 13 60 95 103 3 15 5 113 38 23 182 125 65 94 107 73 224 37 47 102 237 3 198 155 7 95 56 198 165 52 148 46 163 157 152 173 19 55 29 167 217 76 115 203 191 11 165 62 193 99 116 122 144 236 7 28 161 2 29 157 172 23 125 238 199 151 177 89 61 24 101 78 50 213 73 41 127 153 45 167 24 121 168 193 156 3 25 87 166 28 228 174 188 53 67 239 148 36 118 104 162 208 27 232 4 194 85 1 63 159 97 191 194 75 196 10 169 132 199 131 50 223 118 94 117 118 37 27 12 94 60 28 171 47 82 110 137 15 225 23 172 43 12 141 142 81 161 12 74 210 162 228 117 98 187 124 69 164 76 105 158 101 21 200 70 130 111 143 230 32 221 203 211 101 31 5 123 83 76 135 90 57 194 95 46 153 112 62 183 54 63 169 179 4 184 40 188 110 194 105 91 206 102 27 175 154 88 142 71 79 199 84 5 170 128 129 125 55 49 17 120 19 45 187 114 24 111 123 140 166 38 33 227 178 18 59 99 173 9 10 179 214 12 84 15 215 218 82 90 47 149 42 93 188 120 239 195 81 171 94 127 200 127 220 217 86 160 16 144 216 157 162 218 72 45 234 159 77 101 88 132 26 26 6 119 13 70 140 156 233 220 34 213 26 236 74 200 80 175 164 133 195 24 44 191 184 30 143 57 204 140 99 106 77 54 43 42 36 61 14 56 25 97 8 81 181 139 180 153 92 212 77 111 133 185 219 231 238 219 1 43 32 231 8 24 91 33 237 223 108 49 64 165 72 35 152 106 87 136 96 32 1 53 77 44 238 229 83 96 22 196 124 79 6 166 58 123 93 121 225 43 22 186 195 71 126 4 174 235 135 120 229 113 28 181 129 135 100 102 7 85 11 145 212 87 156 186 138 184 223 61 4 11 175 107 6 89 81 114 207 21 17 197 236 54 73 214 232 234 149 32 11 98 167 34 203 221 146 121 235 88 75 139 160 63 189 29 147 127 210 172 33 207 51 152 116 169 189 49 237 233 153 102 17 130 64 98 177 79 16 211 111 113 58 113 48 68 235 78 30 86 170 98 197 169 122 154 41 97 18 126 234 92 145 145 155 237 50 209 ^'}},
     {OUT=>'F29835A93475740E888E8C14318F3CA45A3C8606'}],
     ['186', {IN=> {f=> '481 1 208 200 56 48 21 186 21 40 119 178 236 205 196 226 73 22 12 195 64 23 101 144 54 206 241 103 132 7 151 13 138 140 151 130 40 2 34 132 80 223 186 94 112 50 134 237 104 200 181 175 159 61 19 29 140 55 32 144 127 36 172 59 104 148 126 210 240 35 200 225 5 90 230 20 170 144 10 134 39 176 131 108 147 58 36 216 131 35 75 127 109 2 232 60 149 25 185 195 137 95 32 71 55 105 216 58 234 144 200 108 103 228 105 47 130 165 100 172 7 78 183 207 184 179 18 36 47 203 237 31 128 8 219 187 162 44 74 38 131 12 49 162 161 218 142 14 227 89 141 71 180 203 112 175 232 133 221 198 141 196 36 99 229 173 192 154 69 23 80 98 88 198 24 52 174 216 227 162 213 2 107 204 11 150 187 235 116 5 236 132 124 53 117 21 113 191 109 192 206 95 201 28 124 149 215 159 230 218 98 213 54 133 169 172 176 6 10 9 183 38 14 110 187 37 188 178 163 133 23 28 72 123 8 240 204 128 52 49 118 89 45 215 42 86 85 215 211 14 154 17 210 115 179 62 108 220 130 209 172 103 176 79 82 211 139 10 155 85 142 139 83 227 16 69 140 224 202 213 171 35 31 55 178 46 231 161 145 70 83 29 88 67 225 101 217 126 137 168 104 2 11 106 115 33 160 4 95 157 198 193 222 1 18 13 21 238 47 5 38 204 55 222 118 162 117 146 211 87 226 190 166 116 78 66 63 80 25 16 25 68 51 33 212 30 60 76 195 12 239 136 225 28 145 195 210 167 205 27 56 4 220 13 65 68 220 203 39 103 59 177 220 57 137 241 176 204 180 78 14 37 115 106 232 177 22 129 68 168 177 74 184 33 116 174 164 201 197 52 218 46 158 89 214 143 111 159 113 74 236 59 52 122 190 20 214 216 183 90 40 46 106 63 7 224 85 69 67 152 29 15 199 136 27 100 76 151 182 66 209 224 158 141 240 108 30 156 174 91 129 24 96 4 147 183 134 112 6 62 64 148 199 40 171 233 201 124 222 45 90 186 190 239 188 9 235 64 221 29 213 127 178 ^'}},
     {OUT=>'1A1D77C6D0F97C4B620FAA90F3F8644408E4B13D'}],
     ['187', {IN=> {f=> '508 1 94 122 173 145 6 206 41 173 57 199 146 150 136 66 9 48 83 91 239 243 5 109 30 7 174 113 57 90 95 69 224 29 39 102 229 3 194 151 239 87 56 190 165 44 144 38 155 157 140 161 11 51 13 163 217 72 107 203 179 243 161 58 193 95 116 122 132 236 7 24 153 242 17 149 168 15 125 238 187 147 165 85 53 16 93 66 50 201 61 29 127 153 45 159 8 121 164 185 144 235 13 75 166 12 216 166 188 53 63 235 148 24 106 100 158 196 19 220 240 190 77 237 63 147 97 187 194 63 188 6 165 132 195 119 46 215 110 86 109 118 37 11 4 94 60 20 167 47 74 110 133 7 221 11 160 43 129 130 77 149 12 66 206 150 224 117 90 183 120 61 164 76 93 154 101 17 196 66 118 99 143 226 32 213 195 207 97 31 1 123 79 64 135 78 49 186 91 42 141 108 62 179 46 59 161 167 176 28 180 102 182 105 91 198 90 27 171 142 76 134 63 79 199 84 241 158 128 121 117 43 41 17 108 15 37 187 106 12 107 115 128 162 26 21 223 170 18 55 87 165 241 2 179 202 8 80 11 207 206 82 82 35 137 34 81 176 116 231 183 73 163 82 123 188 115 216 213 86 156 16 140 208 153 154 210 68 33 222 155 69 89 80 120 14 22 238 119 5 62 132 148 221 208 18 205 26 224 70 192 80 167 156 125 191 24 44 191 180 14 131 45 200 140 99 102 73 54 31 42 32 57 2 44 9 89 236 69 177 131 180 149 80 208 65 103 129 177 219 231 230 215 1 35 28 227 20 79 17 237 219 96 37 52 157 72 19 152 98 87 124 88 24 241 49 77 40 226 229 71 92 10 192 236 116 75 234 154 54 119 93 113 213 39 14 178 187 59 114 4 162 223 123 120 217 113 16 181 117 131 92 98 243 73 7 133 204 79 152 186 138 184 223 61 232 239 175 107 6 77 81 114 207 9 1 185 232 42 73 210 224 226 141 20 11 98 155 22 191 221 146 109 227 88 71 139 152 51 189 21 135 119 202 164 29 195 51 148 112 157 181 49 233 233 145 94 13 122 64 94 169 71 4 203 111 101 58 105 44 56 231 74 22 82 170 86 197 169 118 146 41 85 6 118 234 92 145 141 149 ^'}},
     {OUT=>'4EC84870E9BDD25F523C6DFB6EDD605052CA4EAA'}],
     ['188', {IN=> {f=> '484 1 219 32 243 208 200 48 48 21 178 13 40 115 178 232 197 196 218 61 14 4 183 60 23 93 136 50 202 229 99 120 245 143 5 134 128 151 130 32 236 34 124 68 215 182 82 112 46 126 229 92 192 169 163 159 49 19 17 136 43 28 140 119 28 160 47 104 140 126 202 236 27 196 225 235 90 230 16 158 132 240 130 39 164 123 104 147 54 32 216 119 35 75 123 97 232 224 52 137 21 185 187 129 95 24 71 47 97 212 50 222 144 188 100 91 228 101 35 122 153 92 172 241 74 179 203 180 167 2 36 39 191 225 23 120 238 215 175 154 36 66 34 131 242 45 162 161 210 142 14 223 81 141 63 168 199 104 167 228 121 209 198 133 184 24 91 221 173 192 142 61 7 76 94 80 190 8 40 162 208 227 150 213 240 103 196 245 150 179 223 108 243 228 128 124 41 113 17 109 187 109 180 194 91 201 12 116 149 215 155 226 206 86 201 46 133 157 168 172 2 2 9 183 26 6 98 183 33 180 174 163 125 19 24 64 119 8 232 204 124 48 41 106 89 33 207 30 82 85 203 207 10 154 9 210 115 171 58 108 216 126 209 164 103 176 71 74 199 131 10 147 81 134 135 79 219 12 69 132 220 202 209 171 31 19 51 166 38 227 157 141 70 83 29 84 67 217 97 205 122 133 164 96 2 241 94 107 33 160 87 145 186 185 214 231 14 243 9 230 43 1 38 200 55 222 110 158 105 138 211 79 226 186 158 112 66 54 59 72 25 8 13 60 43 21 212 30 48 68 195 8 239 132 213 28 133 191 198 167 201 19 44 238 208 1 65 56 220 195 35 95 51 173 212 57 137 233 168 192 176 78 2 29 111 102 232 177 22 129 68 168 165 74 172 29 104 174 156 197 189 44 218 34 158 85 210 135 99 147 101 62 224 59 52 110 178 20 214 204 171 78 36 46 106 55 3 212 77 65 63 148 25 15 187 136 23 100 64 139 182 62 197 216 146 137 240 96 22 156 170 87 125 12 96 234 139 175 134 108 236 54 52 144 195 28 167 221 193 120 218 41 86 174 190 227 184 243 235 56 213 21 205 115 205 71 ^'}},
     {OUT=>'D689513FED08B80C39B67371959BC4E3FECB0537'}],
     ['189', {IN=> {f=> '506 0 110 165 141 238 206 37 169 45 187 134 142 124 66 5 36 71 79 235 231 5 105 22 239 166 101 49 86 83 65 224 21 31 102 221 3 190 147 231 79 56 182 165 36 140 30 147 157 128 149 3 47 245 159 217 68 99 203 167 235 157 54 193 91 116 122 120 236 7 20 145 242 5 141 164 7 125 238 175 143 153 81 45 8 85 54 50 189 49 17 127 153 45 151 240 121 160 177 132 227 1 63 166 244 204 158 188 53 59 231 148 12 94 96 154 184 11 208 236 186 69 233 63 135 97 183 194 51 180 2 161 132 191 107 42 207 102 78 101 118 37 243 244 94 60 12 163 47 66 110 129 247 217 247 148 43 236 117 118 73 137 12 58 202 138 220 117 82 179 116 53 164 76 81 150 101 13 192 62 106 87 143 222 32 205 187 203 93 31 245 123 75 52 135 66 41 178 87 38 129 104 62 175 38 55 153 155 244 168 16 172 94 170 105 91 190 78 27 167 130 64 126 55 79 199 84 237 146 128 113 109 31 33 17 96 11 29 187 98 103 107 116 158 14 9 219 162 18 51 75 157 233 242 179 190 4 76 7 199 194 82 74 23 125 26 69 164 112 223 171 65 155 70 119 176 103 212 209 86 152 16 136 200 149 146 202 64 21 210 151 61 77 72 108 2 18 230 119 245 54 124 140 209 196 2 197 26 212 66 184 80 159 148 117 187 24 44 191 176 246 119 33 196 140 99 98 69 54 19 42 28 53 238 32 241 81 224 57 173 123 180 145 68 204 53 95 125 169 219 231 222 211 1 27 24 223 240 16 67 1 237 215 84 25 40 149 72 3 152 90 87 112 80 16 241 45 77 36 214 229 59 88 246 188 232 108 71 222 142 50 115 93 105 201 35 6 170 179 47 102 4 150 211 111 120 205 113 4 181 105 127 84 94 239 61 3 121 196 71 148 186 138 184 223 61 220 227 175 107 6 65 81 114 207 245 233 173 228 30 73 206 216 218 133 8 11 98 143 10 179 221 146 97 219 88 67 139 144 39 189 13 123 111 194 156 25 183 51 144 108 145 173 49 229 233 137 86 9 114 64 90 161 63 240 195 111 89 58 97 40 44 227 70 14 78 170 74 197 169 114 138 41 73 242 110 234 177 ^'}},
     {OUT=>'C4FED58F209FC3C34AD19F86A6DACADC86C04D33'}],
     ['190', {IN=> {f=> '491 0 145 135 155 207 20 243 208 200 40 48 21 170 5 40 111 178 228 189 196 210 49 6 246 171 56 23 85 128 46 198 217 95 108 241 135 247 130 116 151 130 24 228 34 116 56 207 178 70 112 42 118 221 80 184 157 151 159 37 19 5 132 31 24 136 111 20 148 35 104 132 126 194 232 19 192 225 223 90 230 12 146 120 228 126 39 152 115 100 147 50 28 216 107 35 75 119 85 220 216 44 125 17 185 179 121 95 16 71 39 89 208 42 210 144 176 92 79 228 97 23 114 141 84 172 233 70 175 199 176 155 236 36 31 179 213 15 112 226 211 163 146 28 58 30 131 230 41 162 161 202 142 14 219 73 141 55 156 195 96 159 224 109 197 198 125 172 12 83 213 173 192 130 53 241 72 90 72 182 242 28 150 200 227 138 213 236 99 188 237 150 171 211 100 239 220 124 124 29 109 13 105 183 109 168 182 87 201 246 108 149 215 151 222 194 74 189 38 133 145 164 168 248 244 9 183 14 248 86 179 29 172 170 163 117 15 20 56 115 8 224 204 120 44 33 94 89 21 199 18 78 85 191 203 6 154 1 210 115 163 54 108 212 122 209 156 103 176 63 66 187 123 10 139 77 126 131 75 211 8 69 124 216 202 205 171 27 7 47 154 30 223 153 137 70 83 29 80 67 209 93 193 118 129 160 88 2 229 82 99 33 160 246 79 133 174 177 206 219 10 231 247 222 39 247 38 196 55 222 102 154 93 130 211 71 226 182 150 108 54 42 55 64 25 1 52 35 9 212 30 36 60 195 4 239 128 201 28 121 187 186 167 197 11 32 230 196 239 65 44 220 187 31 87 43 169 204 57 137 225 160 180 172 78 240 21 107 98 232 177 22 129 68 168 153 74 160 25 92 174 148 193 181 36 218 22 158 81 206 127 87 135 89 50 212 59 52 98 166 20 214 192 159 66 32 46 106 47 249 200 69 61 59 144 21 15 175 136 19 100 52 127 182 58 185 208 134 133 240 84 14 156 166 83 121 96 222 131 167 134 104 224 46 40 140 191 16 163 209 185 116 214 37 82 162 190 215 180 235 235 48 205 13 197 103 205 76 104 161 139 232 146 ^'}},
     {OUT=>'051888C6D00029C176DE792B84DECE2DC1C74B00'}],
     ['191', {IN=> {f=> '516 0 33 165 33 175 122 134 112 66 1 24 59 67 231 219 5 101 14 227 158 89 41 82 71 61 224 13 23 102 213 3 186 143 223 71 56 174 165 28 136 22 139 157 116 137 247 43 233 155 217 64 91 203 155 227 153 50 193 87 116 122 108 236 7 16 137 242 245 133 160 251 125 238 163 139 141 77 37 77 42 50 177 37 5 127 153 45 143 228 121 156 169 120 219 241 51 166 232 192 150 188 53 55 227 148 82 92 150 172 3 196 232 182 61 229 63 123 97 179 194 39 172 250 157 132 187 95 38 199 94 70 93 118 37 231 240 94 60 4 159 47 58 110 125 243 213 239 136 43 228 105 106 69 125 12 50 198 126 216 117 74 175 112 45 164 76 69 146 101 9 188 58 94 75 143 218 32 197 179 199 89 31 245 123 71 40 135 54 33 170 83 34 117 100 62 171 30 51 145 143 244 160 4 164 86 158 105 91 182 66 27 163 118 52 118 47 79 199 84 233 134 128 105 101 19 25 17 84 7 21 187 90 240 99 99 104 154 2 249 215 154 18 47 63 149 225 238 179 178 72 3 191 182 82 66 11 113 18 57 152 108 215 159 57 147 58 115 164 91 208 205 86 148 16 132 192 145 138 194 60 9 198 147 53 65 64 96 242 14 222 119 241 46 116 132 197 184 238 189 26 200 62 176 80 151 140 109 183 24 44 191 172 234 107 21 192 140 99 94 65 54 7 42 24 49 230 20 229 73 212 45 169 115 180 141 56 200 41 87 121 161 219 231 214 207 1 19 20 219 236 12 55 237 237 211 72 13 28 141 72 239 152 82 87 100 72 8 241 41 77 32 202 229 47 84 238 184 228 100 67 210 130 46 111 93 97 189 31 250 162 171 35 90 4 138 199 99 120 193 113 244 181 93 123 76 90 235 49 251 109 188 63 144 186 138 184 223 61 208 215 175 107 6 53 81 114 207 237 221 161 224 18 73 202 208 210 125 248 11 98 131 250 167 221 146 85 211 88 63 139 136 27 189 5 111 103 186 148 21 171 51 140 104 133 165 49 225 233 129 78 5 106 64 86 153 55 232 187 111 77 58 89 36 32 223 66 6 74 170 62 197 169 110 130 41 61 234 102 234 92 145 133 155 201 14 243 208 200 36 48 21 166 1 40 109 178 147 ^'}},
     {OUT=>'1A3540BEE05518505827954F58B751C475AEECE0'}],
     ['192', {IN=> {f=> '522 0 179 196 200 34 251 241 156 51 23 75 118 41 193 202 90 93 236 125 242 125 101 151 130 14 218 34 106 41 197 173 55 112 37 108 211 65 174 142 136 159 22 19 245 127 16 19 131 101 10 133 20 104 122 126 184 227 9 187 225 208 90 230 7 131 105 213 121 39 137 105 95 147 45 23 216 92 35 75 114 70 205 206 34 110 12 185 169 111 95 6 71 29 79 203 32 195 144 161 82 64 228 92 8 104 126 74 172 223 65 170 194 171 140 221 36 21 164 198 5 102 211 206 148 136 18 48 25 131 215 36 162 161 192 142 14 214 63 141 45 141 190 86 149 219 94 182 198 115 157 252 73 203 173 192 115 43 226 67 85 62 172 227 13 135 190 227 123 213 231 94 178 227 150 161 196 90 234 210 119 124 14 104 8 100 178 109 153 167 82 201 231 98 149 215 146 217 179 59 174 28 133 130 159 163 248 239 9 183 254 243 71 174 24 162 165 163 107 10 15 46 110 8 214 204 115 39 23 79 89 6 189 3 73 85 176 198 1 154 246 210 115 153 49 108 207 117 209 146 103 176 53 56 172 113 10 129 72 116 126 70 201 3 69 114 211 202 200 171 22 247 42 139 20 218 148 132 70 83 29 75 67 199 88 178 113 124 155 78 2 214 67 89 33 160 246 69 118 159 167 196 204 5 216 237 212 34 247 38 191 55 222 92 149 78 120 211 61 226 177 140 103 39 27 50 54 25 245 241 42 25 249 212 30 21 50 195 254 239 123 186 28 106 182 171 167 192 1 17 220 181 229 65 29 220 177 26 77 33 164 194 57 137 215 150 165 167 78 230 11 102 93 232 177 22 129 68 168 138 74 145 20 77 174 138 188 171 26 218 7 158 76 201 117 72 120 74 35 197 59 52 83 151 20 214 177 144 51 27 46 106 37 249 185 59 56 54 139 16 15 160 136 14 100 37 112 182 53 170 198 119 128 240 69 4 156 161 78 116 240 96 207 121 157 134 99 209 36 25 135 186 1 158 194 175 111 209 32 77 147 190 200 175 225 235 38 195 3 187 88 205 61 89 151 134 217 206 30 162 24 166 113 128 103 66 253 15 50 58 228 210 5 98 8 218 152 80 35 79 62 58 224 7 17 102 207 3 183 140 217 65 56 168 165 22 133 16 133 157 107 128 244 12 ^'}},
     {OUT=>'DFA19180359D5A7A38E842F172359CAF4208FC05'}],
     ['193', {IN=> {f=> '517 1 218 150 217 59 81 203 140 217 148 45 193 82 116 122 93 236 7 11 127 242 235 123 155 246 125 238 148 134 126 72 27 247 67 27 50 162 22 247 127 153 45 133 213 121 151 159 105 209 231 36 166 217 177 140 188 53 50 222 148 242 67 87 145 157 250 181 227 177 51 224 63 108 97 174 194 24 162 250 152 132 182 80 33 189 84 60 83 118 37 216 235 94 60 251 154 47 48 110 120 238 208 229 121 43 218 90 91 64 110 12 40 193 111 211 117 64 170 107 35 164 76 54 141 101 4 183 53 79 60 143 213 32 187 169 194 84 31 245 123 66 25 135 39 23 160 78 29 102 95 62 166 20 46 135 128 244 150 246 154 76 143 105 91 172 51 27 158 103 37 108 37 79 199 84 228 119 128 95 91 4 15 17 69 2 11 187 80 230 94 89 89 149 244 239 210 144 18 42 48 139 215 233 179 163 252 67 255 181 167 82 56 253 98 8 42 137 103 205 144 47 137 43 110 149 76 203 200 86 143 16 127 182 140 128 184 55 251 183 142 43 50 54 81 232 9 212 119 236 36 106 122 182 169 223 179 26 185 57 166 80 141 130 99 178 24 44 191 167 219 92 6 187 140 99 89 60 54 249 42 19 44 220 5 214 63 197 30 164 105 180 136 41 195 26 77 116 151 219 231 204 202 1 9 15 214 231 7 40 222 237 206 57 255 13 131 72 224 152 72 87 85 62 255 241 36 77 27 187 229 32 79 228 179 223 90 62 195 115 41 106 93 87 174 26 245 152 161 20 75 4 123 184 84 120 178 113 234 181 78 118 66 85 230 34 251 94 178 53 139 186 138 184 223 61 193 200 175 107 6 38 81 114 207 227 206 146 219 3 73 197 198 200 115 238 11 98 116 240 152 221 146 70 201 88 58 139 126 12 189 252 96 93 176 138 16 156 51 135 99 118 155 49 220 233 119 68 96 64 81 143 45 222 177 111 62 58 79 31 17 218 61 253 69 170 47 197 169 105 120 41 46 224 92 234 92 145 128 155 186 256 243 208 200 26 48 21 156 248 40 104 178 221 175 196 196 28 249 239 150 49 23 71 114 39 191 196 88 87 234 121 240 123 95 151 130 10 214 34 102 35 193 171 49 112 35 104 207 59 170 136 130 159 16 19 241 125 213 ^'}},
     {OUT=>'7B0FA84EBBCFF7D7F4500F73D79660C4A3431B67'}],
     ['194', {IN=> {f=> '529 1 15 127 93 2 121 8 104 114 126 176 223 1 183 225 196 90 230 3 119 93 201 117 39 125 97 91 147 41 19 216 80 35 75 110 58 193 198 26 98 8 185 161 103 95 257 71 21 71 199 24 183 144 149 74 52 228 88 255 96 114 66 172 215 61 166 190 167 128 209 36 13 152 186 256 94 199 202 136 128 10 40 21 131 203 32 162 161 184 142 14 210 55 141 37 129 186 78 141 215 82 170 198 107 145 244 65 195 173 192 103 35 214 63 81 54 164 215 1 123 182 227 111 213 227 90 170 219 150 153 184 82 230 202 115 124 2 100 4 96 174 109 141 155 78 201 219 90 149 215 142 213 167 47 162 20 133 118 155 159 248 235 9 183 246 239 59 170 20 154 161 163 99 6 11 38 106 8 206 204 111 35 15 67 89 253 181 250 69 85 164 194 256 154 242 210 115 145 45 108 203 113 209 138 103 176 45 48 160 105 10 121 68 108 122 66 193 258 69 106 207 202 196 171 18 239 38 127 12 214 144 128 70 83 29 71 67 191 84 166 109 120 151 70 2 202 55 81 33 160 246 61 106 147 159 188 192 1 204 229 204 30 247 38 187 55 222 84 145 66 112 211 53 226 173 132 99 27 15 46 46 25 241 233 34 17 241 212 30 9 42 195 254 239 119 174 28 94 178 159 167 188 252 5 212 169 221 65 17 220 169 22 69 25 160 186 57 137 207 142 153 163 78 222 3 98 89 232 177 22 129 68 168 126 74 133 16 65 174 130 184 163 18 218 254 158 72 197 109 60 108 62 23 185 59 52 71 139 20 214 165 132 39 23 46 106 29 249 173 51 52 50 135 12 15 148 136 10 100 25 100 182 49 158 190 107 124 240 57 255 156 157 74 112 232 96 195 113 149 134 95 197 28 13 131 182 248 154 182 167 107 205 28 73 135 190 188 171 217 235 30 187 254 179 76 205 49 77 143 130 205 206 26 158 12 154 101 120 91 66 253 3 38 46 224 198 5 94 206 144 68 27 75 50 54 224 258 9 102 199 3 179 136 209 57 56 160 165 14 129 8 125 157 95 116 240 36 212 148 217 57 77 203 134 213 146 43 193 80 116 122 87 236 7 9 123 242 231 119 153 244 125 238 142 132 120 70 23 245 63 21 50 156 16 243 127 153 45 129 207 121 149 155 99 205 227 30 166 211 57 ^'}},
     {OUT=>'9E886081C9ACAAD0F97B10810D1DE6FCDCE6B5F4'}],
     ['195', {IN=> {f=> '548 0 134 188 53 47 219 148 236 58 84 142 148 247 172 224 174 45 221 63 99 97 171 194 15 156 250 149 132 179 71 30 183 78 54 77 118 37 207 232 94 60 248 151 47 42 110 117 235 205 223 112 43 212 81 82 61 101 12 34 190 102 208 117 58 167 104 29 164 76 45 138 101 1 180 50 70 51 143 210 32 181 163 191 81 31 245 123 63 16 135 30 17 154 75 26 93 92 62 163 14 43 129 119 244 144 240 148 70 134 105 91 166 42 27 155 94 28 102 31 79 199 84 225 110 128 89 85 255 9 17 60 259 5 187 74 224 91 83 80 146 238 233 207 138 18 39 39 133 209 230 179 154 252 64 255 175 158 82 50 247 89 2 33 128 100 199 135 41 131 34 107 140 67 200 197 86 140 16 124 176 137 122 178 52 245 174 139 37 41 48 72 226 6 206 119 233 30 100 116 173 160 214 173 26 176 54 160 80 135 124 93 175 24 44 191 164 210 83 257 184 140 99 86 57 54 243 42 16 41 214 256 205 57 188 21 161 99 180 133 32 192 17 71 113 145 219 231 198 199 1 3 12 211 228 4 31 213 237 203 48 249 4 125 72 215 152 66 87 76 56 252 241 33 77 24 178 229 23 76 222 176 220 84 59 186 106 38 103 93 81 165 23 242 146 155 11 66 4 114 175 75 120 169 113 228 181 69 115 60 82 227 25 251 85 172 47 136 186 138 184 223 61 184 191 175 107 6 29 81 114 207 221 197 137 216 254 73 194 192 194 109 232 11 98 107 234 143 221 146 61 195 88 55 139 120 3 189 249 87 87 170 132 13 147 51 132 96 109 149 49 217 233 113 62 257 90 64 78 137 39 216 171 111 53 58 73 28 8 215 58 250 66 170 38 197 169 102 114 41 37 218 86 234 92 145 125 155 177 250 243 208 200 20 48 21 150 245 40 101 178 218 169 196 190 19 246 236 141 46 23 65 108 36 188 187 85 78 231 115 237 120 86 151 130 4 208 34 96 26 187 168 40 112 32 98 201 50 164 127 121 159 7 19 235 122 1 14 126 91 118 5 104 112 126 174 222 259 182 225 193 90 230 2 116 90 198 116 39 122 95 90 147 40 18 216 77 35 75 109 55 190 196 24 95 7 185 159 101 95 256 71 19 69 198 22 180 144 146 72 49 228 87 253 94 111 64 172 213 60 165 189 166 125 206 36 11 149 183 255 92 196 201 133 126 8 38 20 131 213 ^'}},
     {OUT=>'A4D46E4BA0AE4B012F75B1B50D0534D578AE9CB6'}],
     ['196', {IN=> {f=> '547 0 29 162 161 178 142 14 207 49 141 31 120 183 72 135 212 73 161 198 101 136 238 59 189 173 192 94 29 205 60 78 48 158 206 254 114 176 227 102 213 224 87 164 213 150 147 175 76 227 196 112 124 255 97 1 93 171 109 132 146 75 201 210 84 149 215 139 210 158 38 153 14 133 109 152 156 248 232 9 183 240 236 50 167 17 148 158 163 93 3 8 32 103 8 200 204 108 32 9 58 89 247 175 244 66 85 155 191 256 154 239 210 115 139 42 108 200 110 209 132 103 176 39 42 151 99 10 115 65 102 119 63 187 258 69 100 204 202 193 171 15 233 35 118 6 211 141 125 70 83 29 68 67 185 81 157 106 117 148 64 2 193 46 75 33 160 246 55 97 138 153 182 183 260 195 223 198 27 247 38 184 55 222 78 142 57 106 211 47 226 170 126 96 18 6 43 40 25 238 227 28 11 235 212 30 36 195 254 239 116 165 28 85 175 150 167 185 249 258 206 160 215 65 8 220 163 19 63 19 157 180 57 137 201 136 144 160 78 216 259 95 86 232 177 22 129 68 168 117 74 124 13 56 174 124 181 157 12 218 248 158 69 194 103 51 99 53 14 176 59 52 62 130 20 214 156 123 30 20 46 106 23 249 164 45 49 47 132 9 15 139 136 7 100 16 91 182 46 149 184 98 121 240 48 252 156 154 71 109 226 96 186 107 143 134 92 188 22 4 128 179 242 151 173 161 104 202 25 70 126 190 179 168 211 235 24 181 251 173 67 205 40 68 137 127 196 206 23 155 3 145 92 114 82 66 253 256 29 37 221 189 5 91 256 197 138 59 21 72 41 51 224 255 3 102 193 3 176 133 203 51 56 154 165 8 126 2 119 157 86 107 237 33 203 145 217 54 71 203 125 207 143 40 193 77 116 122 78 236 7 6 117 242 225 113 150 241 125 238 133 129 111 67 17 242 57 12 50 147 7 237 127 153 45 123 198 121 146 149 90 199 221 21 166 202 162 130 188 53 45 217 148 232 52 82 140 142 245 166 222 172 41 219 63 93 97 169 194 9 152 250 147 132 177 65 28 179 74 50 73 118 37 201 230 94 60 246 149 47 38 110 115 233 203 219 106 43 208 75 76 59 95 12 30 188 96 206 117 54 165 102 25 164 76 39 136 101 261 178 48 64 45 143 208 32 177 159 189 79 31 245 123 61 10 135 24 13 150 73 24 87 90 62 161 10 41 125 209 ^'}},
     {OUT=>'6342B199EE64C7B2C9CBCD4F2DCB65ACEF51516F'}],
     ['197', {IN=> {f=> '542 1 244 136 232 140 62 122 105 91 158 30 27 151 82 16 94 23 79 199 84 221 98 128 81 77 247 1 17 48 259 261 187 66 216 87 75 68 142 230 225 203 130 18 35 27 125 201 226 179 142 252 60 255 167 146 82 42 239 77 258 21 116 96 191 123 33 123 22 103 128 55 196 193 86 136 16 120 168 133 114 170 48 237 162 135 29 29 40 60 218 2 198 119 229 22 92 108 161 148 202 165 26 164 50 152 80 127 116 85 171 24 44 191 160 198 71 249 180 140 99 82 53 54 235 42 12 37 206 248 193 49 176 9 157 91 180 129 20 188 5 63 109 137 219 231 190 195 1 259 8 207 224 19 201 237 199 36 241 256 117 72 203 152 58 87 64 48 248 241 29 77 20 166 229 11 72 214 172 216 76 55 174 94 34 99 93 73 153 19 238 138 147 263 54 4 102 163 63 120 157 113 220 181 57 111 52 78 223 13 251 73 164 39 132 186 138 184 223 61 172 179 175 107 6 17 81 114 207 213 185 125 212 246 73 190 184 186 101 224 11 98 95 226 131 221 146 49 187 88 51 139 112 255 189 245 75 79 162 124 9 135 51 128 92 97 141 49 213 233 105 54 257 82 64 74 129 31 208 163 111 41 58 65 24 260 211 54 246 62 170 26 197 169 98 106 41 25 210 78 234 92 145 121 155 165 242 243 208 200 12 48 21 142 241 40 97 178 214 161 196 182 7 242 232 129 42 23 57 100 32 184 175 81 66 227 107 233 116 74 151 130 260 200 34 88 14 179 164 28 112 28 90 193 38 156 115 109 159 259 19 227 118 253 10 122 83 256 106 257 104 104 126 166 218 255 178 225 181 90 230 262 104 78 186 112 39 110 87 86 147 36 14 216 65 35 75 105 43 178 188 16 83 3 185 151 93 95 252 71 11 61 194 14 168 144 134 64 37 228 83 245 86 99 56 172 205 56 161 185 162 113 194 36 3 137 171 251 84 184 197 121 118 30 16 131 188 27 162 161 174 142 14 205 45 141 27 114 181 68 131 210 67 155 198 97 130 234 55 185 173 192 88 25 199 58 76 44 154 200 250 108 172 227 96 213 222 85 160 209 150 143 169 72 225 192 110 124 251 95 263 91 169 109 126 140 73 201 204 80 149 215 137 208 152 32 147 10 133 103 150 154 248 230 9 183 236 234 44 165 15 144 156 163 89 1 6 28 101 8 196 204 106 164 ^'}},
     {OUT=>'AABFD63688EB678357869130083E1B52F6EA861D'}],
     ['198', {IN=> {f=> '567 0 1 46 89 239 167 236 62 85 143 187 256 154 235 210 115 131 38 108 196 106 209 124 103 176 31 34 139 91 10 107 61 94 115 59 179 258 69 92 200 202 189 171 11 225 31 106 264 207 137 121 70 83 29 64 67 177 77 145 102 113 144 56 2 181 34 67 33 160 246 47 85 126 145 174 171 260 183 215 190 23 247 38 180 55 222 70 138 45 98 211 39 226 166 118 92 6 260 39 32 25 234 219 20 3 227 212 30 254 28 195 254 239 112 153 28 73 171 138 167 181 245 250 198 148 207 65 262 220 155 15 55 11 153 172 57 137 193 128 132 156 78 208 255 91 82 232 177 22 129 68 168 105 74 112 9 44 174 116 177 149 4 218 240 158 65 190 95 39 87 41 2 164 59 52 50 118 20 214 144 111 18 16 46 106 15 249 152 37 45 43 128 5 15 127 136 3 100 4 79 182 42 137 176 86 117 240 36 248 156 150 67 105 218 96 174 99 135 134 88 176 14 258 124 175 234 147 161 153 100 198 21 66 114 190 167 164 203 235 16 173 247 165 55 205 28 56 129 123 184 206 19 151 257 133 80 106 70 66 253 248 17 25 217 177 5 87 252 185 130 47 13 68 29 47 224 251 261 102 185 3 172 129 195 43 56 146 165 122 260 111 157 74 95 233 29 191 141 217 50 63 203 113 199 139 36 193 73 116 122 66 236 7 2 109 242 217 105 146 237 125 238 121 125 99 63 9 238 49 50 135 261 229 127 153 45 115 186 121 142 141 78 191 213 9 166 190 150 122 188 53 41 213 148 224 40 78 136 130 241 154 218 168 33 215 63 81 97 165 194 263 144 250 143 132 173 53 24 171 66 42 65 118 37 189 226 94 60 242 145 47 30 110 111 229 199 211 94 43 200 63 64 55 83 12 22 184 84 202 117 46 161 98 17 164 76 27 132 101 261 174 44 52 33 143 204 32 169 151 185 75 31 245 123 57 264 135 12 5 142 69 20 75 86 62 157 2 37 117 101 244 132 228 136 58 116 105 91 154 24 27 149 76 10 90 19 79 199 84 219 92 128 77 73 243 263 17 42 259 259 187 62 212 85 71 62 140 226 221 201 126 18 33 21 121 197 224 179 136 252 58 255 163 140 82 38 235 71 256 15 110 94 187 117 29 119 16 101 122 49 194 191 86 134 16 118 164 131 110 166 46 233 156 133 25 23 36 54 214 194 119 227 18 88 104 155 142 196 161 26 158 48 148 80 123 112 81 169 24 44 191 158 192 65 245 178 140 223 ^'}},
     {OUT=>'F732B7372DAF44801F81EFFE3108726239837936'}],
     ['199', {IN=> {f=> '551 1 78 49 54 227 42 8 33 198 240 181 41 164 265 153 83 180 125 8 184 261 55 105 129 219 231 182 191 1 255 4 203 220 264 7 189 237 195 24 233 248 109 72 191 152 50 87 52 40 244 241 25 77 16 154 229 267 68 206 168 212 68 51 162 82 30 95 93 65 141 15 234 130 139 255 42 4 90 151 51 120 145 113 212 181 45 107 44 74 219 1 251 61 156 31 128 186 138 184 223 61 160 167 175 107 6 5 81 114 207 205 173 113 208 238 73 186 176 178 93 216 11 98 83 218 119 221 146 37 179 88 47 139 104 247 189 241 63 71 154 116 5 123 51 124 88 85 133 49 209 233 97 46 257 74 64 70 121 23 200 155 111 29 58 57 20 252 207 50 242 58 170 14 197 169 94 98 41 13 202 70 234 92 145 117 155 153 234 243 208 200 4 48 21 134 237 40 93 178 210 153 196 174 263 238 228 117 38 23 49 92 28 180 163 77 54 223 99 229 112 62 151 130 256 192 34 80 2 171 160 16 112 24 82 185 26 148 103 97 159 251 19 219 114 245 6 118 75 252 94 249 104 96 126 158 214 251 174 225 169 90 230 262 92 66 174 108 39 98 79 82 147 32 10 216 53 35 75 101 31 166 180 8 71 267 185 143 85 95 248 71 3 53 190 6 156 144 122 56 25 228 79 237 78 87 48 172 197 52 157 181 158 101 182 36 263 125 159 247 76 172 193 109 110 260 22 12 131 176 23 162 161 166 142 14 201 37 141 19 102 177 60 123 206 55 143 198 89 118 226 47 177 173 192 76 17 187 54 72 36 146 188 242 96 164 227 84 213 218 81 152 201 150 135 157 64 221 184 106 124 243 91 263 87 165 109 114 128 69 201 192 72 149 215 133 204 140 20 135 2 133 91 146 150 248 226 9 183 228 230 32 161 11 136 152 163 81 265 2 20 97 8 188 204 102 26 265 40 89 235 163 232 60 85 137 185 256 154 233 210 115 127 36 108 194 104 209 120 103 176 27 30 133 87 10 103 59 90 113 57 175 258 69 88 198 202 187 171 9 221 29 100 262 205 135 119 70 83 29 62 67 173 75 139 100 111 142 52 2 175 28 63 33 160 246 43 79 120 141 170 165 260 177 211 186 21 247 38 178 55 222 66 136 39 94 211 35 226 164 114 90 256 37 28 25 232 215 16 267 223 212 30 250 24 195 254 239 110 147 28 67 169 132 167 179 243 246 194 142 203 65 263 ^'}},
     {OUT=>'5E9347FE4574CDCB80281ED092191199BADD7B42'}],
     ['200', {IN=> {f=> '578 0 220 147 11 47 3 149 164 57 137 185 120 120 152 78 200 251 87 78 232 177 22 129 68 168 93 74 100 5 32 174 108 173 141 266 218 232 158 61 186 87 27 75 29 260 152 59 52 38 106 20 214 132 99 6 12 46 106 7 249 140 29 41 39 124 1 15 115 136 269 100 262 67 182 38 125 168 74 113 240 24 244 156 146 63 101 210 96 162 91 127 134 84 164 6 250 120 171 226 143 149 145 96 194 17 62 102 190 155 160 195 235 8 165 243 157 43 205 16 44 121 119 172 206 15 147 249 121 68 98 58 66 253 240 5 13 213 165 5 83 248 173 122 35 5 64 17 43 224 247 257 102 177 3 168 125 187 35 56 138 165 262 118 256 103 157 62 83 229 25 179 137 217 46 55 203 101 191 135 32 193 69 116 122 54 236 7 268 101 242 209 97 142 233 125 238 109 121 87 59 1 234 41 258 50 123 253 221 127 153 45 107 174 121 138 133 66 183 205 267 166 178 138 114 188 53 37 209 148 216 28 74 132 118 237 142 214 164 25 211 63 69 97 161 194 255 136 250 139 132 169 41 20 163 58 34 57 118 37 177 222 94 60 238 141 47 22 110 107 225 195 203 82 43 192 51 52 51 71 12 14 180 72 198 117 38 157 94 9 164 76 15 128 101 261 170 40 40 21 143 200 32 161 143 181 71 31 245 123 53 256 135 267 134 65 16 63 82 62 153 264 33 109 89 244 124 220 128 50 104 105 91 146 12 27 145 64 268 82 11 79 199 84 215 80 128 69 65 235 259 17 30 259 255 187 54 204 81 63 50 136 218 213 197 118 18 29 9 113 189 220 179 124 252 54 255 155 128 82 30 227 59 252 3 98 90 179 105 21 111 4 97 110 37 190 187 86 130 16 114 156 127 102 158 42 225 144 129 17 11 28 42 206 266 186 119 223 10 80 96 143 130 184 153 26 146 44 140 80 115 104 73 165 24 44 191 154 180 53 237 174 140 99 76 47 54 223 42 6 31 194 236 175 37 158 261 151 79 180 123 2 182 257 51 103 125 219 231 178 189 1 253 2 201 218 264 1 183 237 193 18 229 244 105 72 185 152 46 87 46 36 242 241 23 77 14 148 229 263 66 202 166 210 64 49 156 76 28 93 93 61 135 13 232 126 135 251 36 4 84 145 45 120 139 113 208 181 39 105 40 72 217 265 251 55 152 27 126 186 138 184 223 61 154 161 175 107 6 269 81 114 207 201 167 107 206 234 73 184 172 174 89 212 11 98 77 214 113 221 146 31 175 88 45 139 100 243 189 213 ^'}},
     {OUT=>'D5776B7DFFF75C1358ABDBBB3F27A20BB6CA7C55'}],
     ['201', {IN=> {f=> '578 1 51 63 146 108 1 111 51 120 84 73 125 49 205 233 89 38 257 66 64 66 113 15 192 147 111 17 58 49 16 244 203 46 238 54 170 2 197 169 90 90 41 1 194 62 234 92 145 113 155 141 226 243 208 200 268 48 21 126 233 40 89 178 206 145 196 166 255 234 224 105 34 23 41 84 24 176 151 73 42 219 91 225 108 50 151 130 252 184 34 72 262 163 156 4 112 20 74 177 14 140 91 85 159 243 19 211 110 237 2 114 67 248 82 241 104 88 126 150 210 247 170 225 157 90 230 262 80 54 162 104 39 86 71 78 147 28 6 216 41 35 75 97 19 154 172 59 267 185 135 77 95 244 71 267 45 186 270 144 144 110 48 13 228 75 229 70 75 40 172 189 48 153 177 154 89 170 36 259 113 147 243 68 160 189 97 102 256 14 8 131 164 19 162 161 158 142 14 197 29 141 11 90 173 52 115 202 43 131 198 81 106 218 39 169 173 192 64 9 175 50 68 28 138 176 234 84 156 227 72 213 214 77 144 193 150 127 145 56 217 176 102 124 235 87 263 83 161 109 102 116 65 201 180 64 149 215 129 200 128 8 123 266 133 79 142 146 248 222 9 183 220 226 20 157 7 128 148 163 73 265 270 12 93 8 180 204 98 22 261 28 89 227 155 224 56 85 125 181 256 154 229 210 115 119 32 108 190 100 209 112 103 176 19 22 121 79 10 95 55 82 109 53 167 258 69 80 194 202 183 171 5 213 25 88 258 201 131 115 70 83 29 58 67 165 71 127 96 107 138 44 2 163 16 55 33 160 246 35 67 108 133 162 153 260 165 203 178 17 247 38 174 55 222 58 132 27 86 211 27 226 160 106 86 260 248 33 20 25 228 207 8 263 215 212 30 242 16 195 254 239 106 135 28 55 165 120 167 175 239 238 186 130 195 65 250 220 143 9 43 271 147 160 57 137 181 116 114 150 78 196 249 85 76 232 177 22 129 68 168 87 74 94 3 26 174 104 171 137 264 218 228 158 59 184 83 21 69 23 256 146 59 52 32 100 20 214 126 93 10 46 106 3 249 134 25 39 37 122 271 15 109 136 269 100 258 61 182 36 119 164 68 111 240 18 242 156 144 61 99 206 96 156 87 123 134 82 158 2 246 118 169 222 141 143 141 94 192 15 60 96 190 149 158 191 235 4 161 241 153 37 205 10 38 117 117 166 206 13 145 245 115 62 94 52 66 253 236 271 7 211 159 5 81 246 167 118 29 1 62 11 41 224 245 255 102 173 3 166 123 183 31 56 134 165 260 116 254 81 ^'}},
     {OUT=>'022B7ADA472FB7A9DA9219621C9C5F563D3792F6'}],
     ['202', {IN=> {f=> '583 1 157 50 71 225 21 167 133 217 42 47 203 89 183 131 28 193 65 116 122 42 236 7 268 93 242 201 89 138 229 125 238 97 117 75 55 267 230 33 250 50 111 245 213 127 153 45 99 162 121 134 125 54 175 197 259 166 166 126 106 188 53 33 205 148 208 16 70 128 106 233 130 210 160 17 207 63 57 97 157 194 247 128 250 135 132 165 29 16 155 50 26 49 118 37 165 218 94 60 234 137 47 14 110 103 221 191 195 70 43 184 39 40 47 59 12 6 176 60 194 117 30 153 90 1 164 76 3 124 101 261 166 36 28 9 143 196 32 153 135 177 67 31 245 123 49 248 135 262 263 126 61 12 51 78 62 149 260 29 101 77 244 116 212 120 42 92 105 91 138 27 141 52 260 74 3 79 199 84 211 68 128 61 57 227 255 17 18 259 251 187 46 196 77 55 38 132 210 205 193 110 18 25 271 105 181 216 179 112 252 50 255 147 116 82 22 219 47 248 265 86 86 171 93 13 103 266 93 98 25 186 183 86 126 16 110 148 123 94 150 38 217 132 125 9 273 20 30 198 266 178 119 219 2 72 88 131 118 172 145 26 134 40 132 80 107 96 65 161 24 44 191 150 168 41 229 170 140 99 72 43 54 215 42 2 27 186 228 163 29 146 253 147 71 180 119 264 178 249 43 99 117 219 231 170 185 1 249 272 197 214 264 263 171 237 189 6 221 236 97 72 173 152 38 87 34 28 238 241 19 77 10 136 229 255 62 194 162 206 56 45 144 64 24 89 93 53 123 9 228 118 127 243 24 4 72 133 33 120 127 113 200 181 27 101 32 68 213 257 251 43 144 19 122 186 138 184 223 61 142 149 175 107 6 261 81 114 207 193 155 95 202 226 73 180 164 166 81 204 11 98 65 206 101 221 146 19 167 88 41 139 92 235 189 235 45 59 142 104 273 105 51 118 82 67 121 49 203 233 85 34 257 62 64 64 109 11 188 143 111 11 58 45 14 240 201 44 236 52 170 270 197 169 88 86 41 269 190 58 234 92 145 111 155 135 222 243 208 200 266 48 21 122 231 40 87 178 204 141 196 162 251 232 222 99 32 23 37 80 22 174 145 71 36 217 87 223 106 44 151 130 250 180 34 68 258 159 154 272 112 18 70 173 8 136 85 79 159 239 19 207 108 233 112 63 246 76 237 104 84 126 146 208 245 168 225 151 90 230 262 74 48 156 102 39 80 67 76 147 26 4 216 35 35 75 95 13 148 168 270 53 267 185 131 73 95 242 71 265 41 184 268 138 144 104 44 7 228 73 225 66 69 36 272 ^'}},
     {OUT=>'7F1DE4ECA20362DA624653D225A5B3F7964A9FF2'}],
     ['203', {IN=> {f=> '588 0 181 44 149 173 150 77 158 36 255 101 135 239 60 148 185 85 94 252 6 4 131 152 15 162 161 150 142 14 193 21 141 3 78 169 44 107 198 31 119 198 73 94 210 31 161 173 192 52 1 163 46 64 20 130 164 226 72 148 227 60 213 210 73 136 185 150 119 133 48 213 168 98 124 227 83 263 79 157 109 90 104 61 201 168 56 149 215 125 196 116 272 111 262 133 67 138 142 248 218 9 183 212 222 8 153 3 120 144 163 65 265 270 4 89 8 172 204 94 18 257 16 89 219 147 216 52 85 113 177 256 154 225 210 115 111 28 108 186 96 209 104 103 176 11 14 109 71 10 87 51 74 105 49 159 258 69 72 190 202 179 171 1 205 21 76 254 197 127 111 70 83 29 54 67 157 67 115 92 103 134 36 2 151 4 47 33 160 246 27 55 96 125 154 141 260 153 195 170 13 247 38 170 55 222 50 128 15 78 211 19 226 156 98 82 252 240 29 12 25 224 199 259 207 212 30 234 8 195 254 239 102 123 28 43 161 108 167 171 235 230 178 118 187 65 242 220 135 5 35 267 143 152 57 137 173 108 102 146 78 188 245 81 72 232 177 22 129 68 168 75 74 82 275 14 174 96 167 129 260 218 220 158 55 180 75 9 57 11 248 134 59 52 20 88 20 214 114 81 264 6 46 106 271 249 122 17 35 33 118 271 15 97 136 269 100 250 49 182 32 107 156 56 107 240 6 238 156 140 57 95 198 96 144 79 115 134 78 146 270 238 114 165 214 137 131 133 90 188 11 56 84 190 137 154 183 235 272 153 237 145 25 205 274 26 109 113 154 206 9 141 237 103 50 86 40 66 253 228 263 271 207 147 5 77 242 155 110 17 269 58 275 37 224 241 251 102 165 3 162 119 175 23 56 126 165 256 112 250 91 157 44 65 223 19 161 131 217 40 43 203 83 179 129 26 193 63 116 122 36 236 7 268 89 242 197 85 136 227 125 238 91 115 69 53 265 228 29 246 50 105 241 209 127 153 45 95 156 121 132 121 48 171 193 255 166 160 120 102 188 53 31 203 148 204 10 68 126 100 231 124 208 158 13 205 63 51 97 155 194 243 124 250 133 132 163 23 14 151 46 22 45 118 37 159 216 94 60 232 135 47 10 110 101 219 189 191 64 43 180 33 34 45 53 12 2 174 54 192 117 26 151 88 273 164 76 273 122 101 261 164 34 22 3 143 194 32 149 131 175 65 31 245 123 47 244 135 258 261 122 59 10 45 76 62 147 258 27 97 71 244 112 208 116 38 86 105 91 134 270 27 139 46 256 70 275 79 199 160 ^'}},
     {OUT=>'CA0F2B1BFB4469C11ED006A994734F0F2F5EFD17'}],
     ['204', {IN=> {f=> '594 0 207 56 128 53 49 219 251 17 6 259 247 187 38 188 73 47 26 128 202 197 189 102 18 21 263 97 173 212 179 100 252 46 255 139 104 82 14 211 35 244 257 74 82 163 81 5 95 258 89 86 13 182 179 86 122 16 106 140 119 86 142 34 209 120 121 1 265 12 18 190 266 170 119 215 272 64 80 119 106 160 137 26 122 36 124 80 99 88 57 157 24 44 191 146 156 29 221 166 140 99 68 39 54 207 42 276 23 178 220 151 21 134 245 143 63 180 115 256 174 241 35 95 109 219 231 162 181 1 245 272 193 210 264 255 159 237 185 272 213 228 89 72 161 152 30 87 22 20 234 241 15 77 6 124 229 247 58 186 158 202 48 41 132 52 20 85 93 45 111 5 224 110 119 235 12 4 60 121 21 120 115 113 192 181 15 97 24 64 209 249 251 31 136 11 118 186 138 184 223 61 130 137 175 107 6 253 81 114 207 185 143 83 198 218 73 176 156 158 73 196 11 98 53 198 89 221 146 7 159 88 37 139 84 227 189 231 33 51 134 96 273 93 51 114 78 55 113 49 199 233 77 26 257 54 64 60 101 3 180 135 111 277 58 37 10 232 197 40 232 48 170 262 197 169 84 78 41 261 182 50 234 92 145 107 155 123 214 243 208 200 262 48 21 114 227 40 83 178 200 133 196 154 243 228 218 87 28 23 29 72 18 170 133 67 24 213 79 219 102 32 151 130 246 172 34 60 250 151 150 264 112 14 62 165 274 128 73 67 159 231 19 199 104 225 274 108 55 242 64 229 104 76 126 138 204 241 164 225 139 90 230 262 62 36 144 98 39 68 59 72 147 22 216 23 35 75 91 1 136 160 266 41 267 185 123 65 95 238 71 261 33 180 264 126 144 92 36 273 228 69 217 58 57 28 172 177 42 147 171 148 71 152 36 253 95 129 237 56 142 183 79 90 250 2 2 131 146 13 162 161 146 142 14 191 17 141 277 72 167 40 103 196 25 113 198 69 88 206 27 157 173 192 46 275 157 44 62 16 126 158 222 66 144 227 54 213 208 71 132 181 150 115 127 44 211 164 96 124 223 81 263 77 155 109 84 98 59 201 162 52 149 215 123 194 110 268 105 260 133 61 136 140 248 216 9 183 208 220 2 151 1 116 142 163 61 265 270 87 8 168 204 92 16 255 10 89 215 143 212 50 85 107 175 256 154 223 210 115 107 26 108 184 94 209 100 103 176 7 10 103 67 10 83 49 70 103 47 155 258 69 68 188 202 177 171 277 201 19 70 252 195 125 109 70 83 29 52 67 153 65 109 90 101 132 32 2 145 276 43 33 160 246 23 259 ^'}},
     {OUT=>'833D63F5C2EA0CD43EC15F2B9DD97FF12B030479'}],
     ['205', {IN=> {f=> '600 1 87 119 148 132 260 144 189 164 10 247 38 167 55 222 44 125 6 72 211 13 226 153 92 79 246 234 26 6 25 221 193 273 256 201 212 30 228 2 195 254 239 99 114 28 34 158 99 167 168 232 224 172 109 181 65 236 220 129 2 29 264 140 146 57 137 167 102 93 143 78 182 242 78 69 232 177 22 129 68 168 66 74 73 275 5 174 90 164 123 257 218 214 158 52 177 69 48 2 242 125 59 52 11 79 20 214 105 72 258 3 46 106 268 249 113 11 32 30 115 271 15 88 136 269 100 244 40 182 29 98 150 47 104 240 276 235 156 137 54 92 192 96 135 73 109 134 75 137 267 232 111 162 208 134 122 127 87 185 8 53 75 190 128 151 177 235 269 147 234 139 16 205 268 17 103 110 145 206 6 138 231 94 41 80 31 66 253 222 257 265 204 138 5 74 239 146 104 8 266 55 269 34 224 238 248 102 159 3 159 116 169 17 56 120 165 253 109 247 85 157 35 56 220 16 152 128 217 37 37 203 74 173 126 23 193 60 116 122 27 236 7 268 83 242 191 79 133 224 125 238 82 112 60 50 262 225 23 240 50 96 235 203 127 153 45 89 147 121 129 115 39 165 187 249 166 151 111 96 188 53 28 200 148 198 1 65 123 91 228 115 205 155 7 202 63 42 97 152 194 237 118 250 130 132 160 14 11 145 40 16 39 118 37 150 213 94 60 229 132 47 4 110 98 216 186 185 55 43 174 24 25 42 44 12 275 171 45 189 117 20 148 85 270 164 76 267 119 101 261 161 31 13 273 143 191 32 143 125 172 62 31 245 123 44 238 135 252 258 116 56 7 36 73 62 144 255 24 91 62 244 106 202 110 32 77 105 91 128 264 27 136 37 250 64 272 79 199 84 206 53 128 51 47 217 250 17 3 259 246 187 36 186 72 45 23 127 200 195 188 100 18 20 261 95 171 211 179 97 252 45 255 137 101 82 12 209 32 243 255 71 81 161 78 3 93 256 88 83 10 181 178 86 121 16 105 138 118 84 140 33 207 117 120 278 263 10 15 188 266 168 119 214 271 62 78 116 103 157 135 26 119 35 122 80 97 86 55 156 24 44 191 145 153 26 219 165 140 99 67 38 54 205 42 276 22 176 218 148 19 131 243 142 61 180 114 254 173 239 33 94 107 219 231 160 180 1 244 272 192 209 264 253 156 237 184 270 211 226 87 72 158 152 28 87 19 18 233 241 14 77 5 121 229 245 57 184 157 201 46 40 129 49 19 84 93 43 108 4 223 108 117 233 9 4 57 118 18 120 112 113 190 181 12 96 22 63 208 247 251 28 134 9 117 186 138 184 223 61 127 216 ^'}},
     {OUT=>'14FD356190416C00592B86FF7CA50B622F85593A'}],
     ['206', {IN=> {f=> '590 1 175 107 6 247 81 114 207 179 134 74 195 212 73 173 150 152 67 190 11 98 44 192 80 221 146 279 153 88 34 139 78 221 189 228 24 45 128 90 273 84 51 111 75 46 107 49 196 233 71 20 257 48 64 57 95 278 174 129 111 271 58 31 7 226 194 37 229 45 170 256 197 169 81 72 41 255 176 44 234 92 145 104 155 114 208 243 208 200 259 48 21 108 224 40 80 178 197 127 196 148 237 225 215 78 25 23 23 66 15 167 124 64 15 210 73 216 99 23 151 130 243 166 34 54 244 145 147 258 112 11 56 159 268 122 64 58 159 225 19 193 101 219 274 105 49 239 55 223 104 70 126 132 201 238 161 225 130 90 230 262 53 27 135 95 39 59 53 69 147 19 278 216 14 35 75 88 273 127 154 263 32 267 185 117 59 95 235 71 258 27 177 261 117 144 83 30 267 228 66 211 52 48 22 172 171 39 144 168 145 62 143 36 250 86 120 234 50 133 180 70 84 247 277 280 131 137 10 162 161 140 142 14 188 11 141 274 63 164 34 97 193 16 104 198 63 79 200 21 151 173 192 37 272 148 41 59 10 120 149 216 57 138 227 45 213 205 68 126 175 150 109 118 38 208 158 93 124 217 78 263 74 152 109 75 89 56 201 153 46 149 215 120 191 101 262 96 257 133 52 133 137 248 213 9 183 202 217 274 148 279 110 139 163 55 265 270 275 84 8 162 204 89 13 252 1 89 209 137 206 47 85 98 172 256 154 220 210 115 101 23 108 181 91 209 94 103 176 1 4 94 61 10 77 46 64 100 44 149 258 69 62 185 202 174 171 277 195 16 61 249 192 122 106 70 83 29 49 67 147 62 100 87 98 129 26 2 136 270 37 33 160 246 17 40 81 115 144 126 260 138 185 160 8 247 38 165 55 222 40 123 68 211 9 226 151 88 77 242 230 24 2 25 219 189 271 254 197 212 30 224 279 195 254 239 97 108 28 28 156 93 167 166 230 220 168 103 177 65 232 220 125 25 262 138 142 57 137 163 98 87 141 78 178 240 76 67 232 177 22 129 68 168 60 74 67 275 280 174 86 162 119 255 218 210 158 50 175 65 275 42 277 238 119 59 52 5 73 20 214 99 66 254 1 46 106 266 249 107 7 30 28 113 271 15 82 136 269 100 240 34 182 27 92 146 41 102 240 272 233 156 135 52 90 188 96 129 69 105 134 73 131 265 228 109 160 204 132 116 123 85 183 6 51 69 190 122 149 173 235 267 143 232 135 10 205 264 11 99 108 139 206 4 136 227 88 35 76 25 66 253 218 253 261 202 132 5 72 237 140 203 ^'}},
     {OUT=>'4AB6B57EDDEF1CE935622F935C1619AE7C1667D6'}],
     ['207', {IN=> {f=> '620 0 279 262 51 261 30 224 234 244 102 151 3 155 112 161 9 56 112 165 249 105 243 77 157 23 44 216 12 140 124 217 33 29 203 62 165 122 19 193 56 116 122 15 236 7 268 75 242 183 71 129 220 125 238 70 108 48 46 258 221 15 232 50 84 227 195 127 153 45 81 135 121 125 107 27 157 179 241 166 139 99 88 188 53 24 196 148 190 272 61 119 79 224 103 201 151 282 198 63 30 97 148 194 229 110 250 126 132 156 2 7 137 32 8 31 118 37 138 209 94 60 225 128 47 279 110 94 212 182 177 43 43 166 12 13 38 32 12 271 167 33 185 117 12 144 81 266 164 76 259 115 101 261 157 27 1 265 143 187 32 135 117 168 58 31 245 123 40 230 135 244 254 108 52 3 24 69 62 140 251 20 83 50 244 98 194 102 24 65 105 91 120 256 27 132 25 242 56 268 79 199 84 202 41 128 43 39 209 246 17 274 259 242 187 28 178 68 37 11 123 192 187 184 92 18 16 253 87 163 207 179 85 252 41 255 129 89 82 4 201 20 239 247 59 77 153 66 278 85 248 84 71 281 177 174 86 117 16 101 130 114 76 132 29 199 105 116 274 255 2 3 180 266 160 119 210 267 54 70 104 91 145 127 26 107 31 114 80 89 78 47 152 24 44 191 141 141 14 211 161 140 99 63 34 54 197 42 276 18 168 210 136 11 119 235 138 53 180 110 246 169 231 25 90 99 219 231 152 176 1 240 272 188 205 264 245 144 237 180 262 203 218 79 72 146 152 20 87 7 10 229 241 10 77 1 109 229 237 53 176 153 197 38 36 117 37 15 80 93 35 96 219 100 109 225 280 4 45 106 6 120 100 113 182 181 92 14 59 204 239 251 16 126 1 113 186 138 184 223 61 115 122 175 107 6 243 81 114 207 175 128 68 193 208 73 171 146 148 63 186 11 98 38 188 74 221 146 275 149 88 32 139 74 217 189 226 18 41 124 86 273 78 51 109 73 40 103 49 194 233 67 16 257 44 64 55 91 276 170 125 111 267 58 27 5 222 192 35 227 43 170 252 197 169 79 68 41 251 172 40 234 92 145 102 155 108 204 243 208 200 257 48 21 104 222 40 78 178 195 123 196 144 233 223 213 72 23 23 19 62 13 165 118 62 9 208 69 214 97 17 151 130 241 162 34 50 240 141 145 254 112 9 52 155 264 118 58 52 159 221 19 189 99 215 274 103 45 237 49 219 104 66 126 128 199 236 159 225 124 90 230 262 47 21 129 93 39 53 49 67 147 17 278 216 8 35 75 86 269 121 150 261 26 267 185 113 55 95 233 71 256 23 175 259 111 144 77 26 263 228 64 207 48 42 18 172 167 37 142 166 143 56 137 36 248 80 114 232 46 127 257 ^'}},
     {OUT=>'B456A6A968ACD66CAA974F96A9A916E700AA3C5D'}],
     ['208', {IN=> {f=> '605 1 58 76 243 273 280 131 125 6 162 161 132 142 14 184 3 141 270 51 160 26 89 189 4 92 198 55 67 192 13 143 173 192 25 268 136 37 55 2 112 137 208 45 130 227 33 213 201 64 118 167 150 101 106 30 204 150 89 124 209 74 263 70 148 109 63 77 52 201 141 38 149 215 116 187 89 254 84 253 133 40 129 133 248 209 9 183 194 213 266 144 279 102 135 163 47 265 270 271 80 8 154 204 85 9 248 274 89 201 129 198 43 85 86 168 256 154 216 210 115 93 19 108 177 87 209 86 103 176 278 281 82 53 10 69 42 56 96 40 141 258 69 54 181 202 170 171 277 187 12 49 245 188 118 102 70 83 29 45 67 139 58 88 83 94 125 18 2 124 262 29 33 160 246 9 28 69 107 136 114 260 126 177 152 4 247 38 161 55 222 32 119 273 60 211 1 226 147 80 73 234 222 20 279 25 215 181 267 250 189 212 30 216 275 195 254 239 93 96 28 16 152 81 167 162 226 212 160 91 169 65 224 220 117 281 17 258 134 134 57 137 155 90 75 137 78 170 236 72 63 232 177 22 129 68 168 48 74 55 275 272 174 78 158 111 251 218 202 158 46 171 57 267 30 269 230 107 59 52 278 61 20 214 87 54 246 282 46 106 262 249 95 284 26 24 109 271 15 70 136 269 100 232 22 182 23 80 138 29 98 240 264 229 156 131 48 86 180 96 117 61 97 134 69 119 261 220 105 156 196 128 104 115 81 179 2 47 57 190 110 145 165 235 263 135 228 127 283 205 256 284 91 104 127 206 132 219 76 23 68 13 66 253 210 245 253 198 120 5 68 233 128 92 275 260 49 257 28 224 232 242 102 147 3 153 110 157 5 56 108 165 247 103 241 73 157 17 38 214 10 134 122 217 31 25 203 56 161 120 17 193 54 116 122 9 236 7 268 71 242 179 67 127 218 125 238 64 106 42 44 256 219 11 228 50 78 223 191 127 153 45 77 129 121 123 103 21 153 175 237 166 133 93 84 188 53 22 194 148 186 268 59 117 73 222 97 199 149 280 196 63 24 97 146 194 225 106 250 124 132 154 281 5 133 28 4 27 118 37 132 207 94 60 223 126 47 277 110 92 210 180 173 37 43 162 6 7 36 26 12 269 165 27 183 117 8 142 79 264 164 76 255 113 101 261 155 25 280 261 143 185 32 131 113 166 56 31 245 123 38 226 135 240 252 104 50 1 18 67 62 138 249 18 79 44 244 94 190 98 20 59 105 91 116 252 27 130 19 238 52 266 79 199 84 200 35 128 39 35 205 244 17 270 259 240 187 24 174 66 33 5 121 188 183 182 88 18 14 249 83 159 205 179 79 252 186 ^'}},
     {OUT=>'FD1C257FE046B2A27E2F0CD55ED2DECA845F01D7'}],
     ['209', {IN=> {f=> '615 0 255 121 77 82 283 193 8 235 239 47 73 145 54 274 77 240 80 59 273 173 170 86 113 16 97 122 110 68 124 25 191 93 112 270 247 281 278 172 266 152 119 206 263 46 62 92 79 133 119 26 95 27 106 80 81 70 39 148 24 44 191 137 129 2 203 157 140 99 59 30 54 189 42 276 14 160 202 124 3 107 227 134 45 180 106 238 165 223 17 86 91 219 231 144 172 1 236 272 184 201 264 237 132 237 176 254 195 210 71 72 134 152 12 87 282 2 225 241 6 77 284 97 229 229 49 168 149 193 30 32 105 25 11 76 93 27 84 283 215 92 101 217 272 4 33 94 281 120 88 113 174 181 275 88 6 55 200 231 251 4 118 280 109 186 138 184 223 61 103 110 175 107 6 235 81 114 207 167 116 56 189 200 73 167 138 140 55 178 11 98 26 180 62 221 146 267 141 88 28 139 66 209 189 222 6 33 116 78 273 66 51 105 69 28 95 49 190 233 59 8 257 36 64 51 83 272 162 117 111 259 58 19 1 214 188 31 223 39 170 244 197 169 75 60 41 243 164 32 234 92 145 98 155 96 196 243 208 200 253 48 21 96 218 40 74 178 191 115 196 136 225 219 209 60 19 23 11 54 9 161 106 58 284 204 61 210 93 5 151 130 237 154 34 42 232 133 141 246 112 5 44 147 256 110 46 40 159 213 19 181 95 207 274 99 37 233 37 211 104 58 126 120 195 232 155 225 112 90 230 262 35 9 117 89 39 41 41 63 147 13 278 216 283 35 75 82 261 109 142 257 14 267 185 105 47 95 229 71 252 15 171 255 99 144 65 18 255 228 60 199 40 30 10 172 159 33 138 162 139 44 125 36 244 68 102 228 38 115 174 52 72 241 271 280 131 119 4 162 161 128 142 14 182 286 141 268 45 158 22 85 187 285 86 198 51 61 188 9 139 173 192 19 266 130 35 53 285 108 131 204 39 126 227 27 213 199 62 114 163 150 97 100 26 202 146 87 124 205 72 263 68 146 109 57 71 50 201 135 34 149 215 114 185 83 250 78 251 133 34 127 131 248 207 9 183 190 211 262 142 279 98 133 163 43 265 270 269 78 8 150 204 83 7 246 270 89 197 125 194 41 85 80 166 256 154 214 210 115 89 17 108 175 85 209 82 103 176 276 279 76 49 10 65 40 52 94 38 137 258 69 50 179 202 168 171 277 183 10 43 243 186 116 100 70 83 29 43 67 135 56 82 81 92 123 14 2 118 258 25 33 160 246 5 22 63 103 132 108 260 120 173 148 2 247 38 159 55 222 28 117 269 56 211 284 226 145 76 71 230 218 18 277 25 213 177 265 248 185 212 30 212 273 195 254 239 91 90 28 10 150 75 167 160 224 175 ^'}},
     {OUT=>'66E0D01780F1063E2929EAAD74826BC64060E38C'}],
     ['210', {IN=> {f=> '613 0 152 79 161 65 216 220 109 281 9 254 130 126 57 137 147 82 63 133 78 162 232 68 59 232 177 22 129 68 168 36 74 43 275 264 174 70 154 103 247 218 194 158 42 167 49 259 18 261 222 95 59 52 270 49 20 214 75 42 238 282 46 106 258 249 83 280 22 20 105 271 15 58 136 269 100 224 10 182 19 68 130 17 94 240 256 225 156 127 44 82 172 96 105 53 89 134 65 107 257 212 101 152 188 124 92 107 77 175 287 43 45 190 98 141 157 235 259 127 224 119 275 205 248 276 83 100 115 206 285 128 211 64 11 60 1 66 253 202 237 245 194 108 5 64 229 116 84 267 256 45 249 24 224 228 238 102 139 3 149 106 149 286 56 100 165 243 99 237 65 157 5 26 210 6 122 118 217 27 17 203 44 153 116 13 193 50 116 122 286 236 7 268 63 242 171 59 123 214 125 238 52 102 30 40 252 215 3 220 50 66 215 183 127 153 45 69 117 121 119 95 9 145 167 229 166 121 81 76 188 53 18 190 148 178 260 55 113 61 218 85 195 145 276 192 63 12 97 142 194 217 98 250 120 132 150 273 1 125 20 285 19 118 37 120 203 94 60 219 122 47 273 110 88 206 176 165 25 43 154 283 284 32 14 12 265 161 15 179 117 138 75 260 164 76 247 109 101 261 151 21 272 253 143 181 32 123 105 162 52 31 245 123 34 218 135 232 248 96 46 286 6 63 62 134 245 14 71 32 244 86 182 90 12 47 105 91 108 244 27 126 7 230 44 262 79 199 84 196 23 128 31 27 197 240 17 262 259 236 187 16 166 62 25 282 117 180 175 178 80 18 10 241 75 151 201 179 67 252 35 255 117 71 82 281 189 2 233 235 41 71 141 48 272 73 236 78 53 269 171 168 86 111 16 95 118 108 64 120 23 187 87 110 268 243 279 274 168 266 148 119 204 261 42 58 86 73 127 115 26 89 25 102 80 77 66 35 146 24 44 191 135 123 285 199 155 140 99 57 28 54 185 42 276 12 156 198 118 288 101 223 132 41 180 104 234 163 219 13 84 87 219 231 140 170 1 234 272 182 199 264 233 126 237 174 250 191 206 67 72 128 152 8 87 278 287 223 241 4 77 284 91 229 225 47 164 147 191 26 30 99 19 9 74 93 23 78 283 213 88 97 213 268 4 27 88 277 120 82 113 170 181 271 86 2 53 198 227 251 287 114 278 107 186 138 184 223 61 97 104 175 107 6 231 81 114 207 163 110 50 187 196 73 165 134 136 51 174 11 98 20 176 56 221 146 263 137 88 26 139 62 205 189 220 29 112 74 273 60 51 103 67 22 91 49 188 233 55 4 257 32 64 49 79 270 158 113 111 255 58 15 288 210 213 ^'}},
     {OUT=>'A8478DF406F179FD4EF97F4574D7F99EA1CE9EB8'}],
     ['211', {IN=> {f=> '624 1 28 220 36 170 238 197 169 72 54 41 237 158 26 234 92 145 95 155 87 190 243 208 200 250 48 21 90 215 40 71 178 188 109 196 130 219 216 206 51 16 23 5 48 6 158 97 55 278 201 55 207 90 286 151 130 234 148 34 36 226 127 138 240 112 2 38 141 250 104 37 31 159 207 19 175 92 201 274 96 31 230 28 205 104 52 126 114 192 229 152 225 103 90 230 262 26 108 86 39 32 35 60 147 10 278 216 277 35 75 79 255 100 136 254 5 267 185 99 41 95 226 71 249 9 168 252 90 144 56 12 249 228 57 193 34 21 4 172 153 30 135 159 136 35 116 36 241 59 93 225 32 106 171 43 66 238 268 280 131 110 1 162 161 122 142 14 179 283 141 265 36 155 16 79 184 279 77 198 45 52 182 3 133 173 192 10 263 121 32 50 282 102 122 198 30 120 227 18 213 196 59 108 157 150 91 91 20 199 140 84 124 199 69 263 65 143 109 48 62 47 201 126 28 149 215 111 182 74 244 69 248 133 25 124 128 248 204 9 183 184 208 256 139 279 92 130 163 37 265 270 266 75 8 144 204 80 4 243 264 89 191 119 188 38 85 71 163 256 154 211 210 115 83 14 108 172 82 209 76 103 176 273 276 67 43 10 59 37 46 91 35 131 258 69 44 176 202 165 171 277 177 7 34 240 183 113 97 70 83 29 40 67 129 53 73 78 89 120 8 2 109 252 19 33 160 246 289 13 54 97 126 99 260 111 167 142 289 247 38 156 55 222 22 114 263 50 211 281 226 142 70 68 224 212 15 274 25 210 171 262 245 179 212 30 206 270 195 254 239 88 81 28 1 147 66 167 157 221 202 150 76 159 65 214 220 107 281 7 253 129 124 57 137 145 80 60 132 78 160 231 67 58 232 177 22 129 68 168 33 74 40 275 262 174 68 153 101 246 218 192 158 41 166 47 257 15 259 220 92 59 52 268 46 20 214 72 39 236 282 46 106 257 249 80 279 21 19 104 271 15 55 136 269 100 222 7 182 18 65 128 14 93 240 254 224 156 126 43 81 170 96 102 51 87 134 64 104 256 210 100 151 186 123 89 105 76 174 287 42 42 190 95 140 155 235 258 125 223 117 273 205 246 274 81 99 112 206 285 127 209 61 8 58 288 66 253 200 235 243 193 105 5 63 228 113 82 265 255 44 247 23 224 227 237 102 137 3 148 105 147 285 56 98 165 242 98 236 63 157 2 23 209 5 119 117 217 26 15 203 41 151 115 12 193 49 116 122 284 236 7 268 61 242 169 57 122 213 125 238 49 101 27 39 251 214 1 218 50 63 213 181 127 153 45 67 114 121 118 93 6 143 165 227 166 118 78 74 188 53 17 189 148 176 258 54 112 58 217 82 194 144 275 191 141 ^'}},
     {OUT=>'248E58CF09A372114FC2F93B09C5FC14F3D0059E'}],
     ['212', {IN=> {f=> '628 1 3 97 139 194 211 92 250 117 132 147 267 290 119 14 282 13 118 37 111 200 94 60 216 119 47 270 110 85 203 173 159 16 43 148 277 278 29 5 12 262 158 6 176 117 286 135 72 257 164 76 241 106 101 261 148 18 266 247 143 178 32 117 99 159 49 31 245 123 31 212 135 226 245 90 43 286 289 60 62 131 242 11 65 23 244 80 176 84 6 38 105 91 102 238 27 123 290 224 38 259 79 199 84 193 14 128 25 21 191 237 17 256 259 233 187 10 160 59 19 276 114 174 169 175 74 18 7 235 69 145 198 179 58 252 32 255 111 62 82 278 183 285 230 229 32 68 135 39 269 67 230 75 44 263 168 165 86 108 16 92 112 105 58 114 20 181 78 107 265 237 276 268 162 266 142 119 201 258 36 52 77 64 118 109 26 80 22 96 80 71 60 29 143 24 44 191 132 114 279 193 152 140 99 54 25 54 179 42 276 9 150 192 109 285 92 217 129 35 180 101 228 160 213 7 81 81 219 231 134 167 1 231 272 179 196 264 227 117 237 171 244 185 200 61 72 119 152 2 87 272 284 220 241 1 77 284 82 229 219 44 158 144 188 20 27 90 10 6 71 93 17 69 283 210 82 91 207 262 4 18 79 271 120 73 113 164 181 265 83 288 50 195 221 251 281 108 275 104 186 138 184 223 61 88 95 175 107 6 225 81 114 207 157 101 41 184 190 73 162 128 130 45 168 11 98 11 170 47 221 146 257 131 88 23 139 56 199 189 217 283 23 106 68 273 51 51 100 64 13 85 49 185 233 49 290 257 26 64 46 73 267 152 107 111 249 58 9 288 204 183 26 218 34 170 234 197 169 70 50 41 233 154 22 234 92 145 93 155 81 186 243 208 200 248 48 21 86 213 40 69 178 186 105 196 126 215 214 204 45 14 23 1 44 4 156 91 53 274 199 51 205 88 282 151 130 232 144 34 32 222 123 136 236 112 34 137 246 100 31 25 159 203 19 171 90 197 274 94 27 228 22 201 104 48 126 110 190 227 150 225 97 90 230 262 20 286 102 84 39 26 31 58 147 8 278 216 273 35 75 77 251 94 132 252 291 267 185 95 37 95 224 71 247 5 166 250 84 144 50 8 245 228 55 189 30 15 172 149 28 133 157 134 29 110 36 239 53 87 223 28 100 169 37 62 236 266 280 131 104 291 162 161 118 142 14 177 281 141 263 30 153 12 75 182 275 71 198 41 46 178 291 129 173 192 4 261 115 30 48 280 98 116 194 24 116 227 12 213 194 57 104 153 150 87 85 16 197 136 82 124 195 67 263 63 141 109 42 56 45 201 120 24 149 215 109 180 68 240 63 246 133 19 122 126 248 202 9 183 180 206 252 137 279 88 128 163 33 265 270 264 73 8 140 204 78 2 241 76 ^'}},
     {OUT=>'F15767DE91796A6816977EFA4FCED4B7FD9B8A57'}],
     ['213', {IN=> {f=> '622 0 89 183 111 180 34 85 59 159 256 154 207 210 115 75 10 108 168 78 209 68 103 176 269 272 55 35 10 51 33 38 87 31 123 258 69 36 172 202 161 171 277 169 3 22 236 179 109 93 70 83 29 36 67 121 49 61 74 85 116 2 97 244 11 33 160 246 285 1 42 89 118 87 260 99 159 134 289 247 38 152 55 222 14 110 255 42 211 277 226 138 62 64 216 204 11 270 25 206 163 258 241 171 212 30 198 266 195 254 239 84 69 28 283 143 54 167 153 217 194 142 64 151 65 206 220 99 281 293 249 125 116 57 137 137 72 48 128 78 152 227 63 54 232 177 22 129 68 168 21 74 28 275 254 174 60 149 93 242 218 184 158 37 162 39 249 3 251 212 80 59 52 260 34 20 214 60 27 228 282 46 106 253 249 68 275 17 15 100 271 15 43 136 269 100 214 289 182 14 53 120 2 89 240 246 220 156 122 39 77 162 96 90 43 79 134 60 92 252 202 96 147 178 119 77 97 72 170 287 38 30 190 83 136 147 235 254 117 219 109 265 205 238 266 73 95 100 206 285 123 201 49 290 50 280 66 253 192 227 235 189 93 5 59 224 101 74 257 251 40 239 19 224 223 233 102 129 3 144 101 139 281 56 90 165 238 94 232 55 157 284 11 205 1 107 113 217 22 7 203 29 143 111 8 193 45 116 122 276 236 7 268 53 242 161 49 118 209 125 238 37 97 15 35 247 210 287 210 50 51 205 173 127 153 45 59 102 121 114 85 288 135 157 219 166 106 66 66 188 53 13 185 148 168 250 50 108 46 213 70 190 140 271 187 63 291 97 137 194 207 88 250 115 132 145 263 290 115 10 280 9 118 37 105 198 94 60 214 117 47 268 110 83 201 171 155 10 43 144 273 274 27 293 12 260 156 174 117 284 133 70 255 164 76 237 104 101 261 146 16 262 243 143 176 32 113 95 157 47 31 245 123 29 208 135 222 243 86 41 286 285 58 62 129 240 9 61 17 244 76 172 80 2 32 105 91 98 234 27 121 286 220 34 257 79 199 84 191 8 128 21 17 187 235 17 252 259 231 187 6 156 57 15 272 112 170 165 173 70 18 5 231 65 141 196 179 52 252 30 255 107 56 82 276 179 281 228 225 26 66 131 33 267 63 226 73 38 259 166 163 86 106 16 90 108 103 54 110 18 177 72 105 263 233 274 264 158 266 138 119 199 256 32 48 71 58 112 105 26 74 20 92 80 67 56 25 141 24 44 191 130 108 275 189 150 140 99 52 23 54 175 42 276 7 146 188 103 283 86 213 127 31 180 99 224 158 209 3 79 77 219 231 130 165 1 229 272 177 194 264 223 111 237 169 240 181 196 57 72 113 152 292 87 268 282 218 241 293 77 284 76 229 80 ^'}},
     {OUT=>'36A6BC5E680E15675D9696338C88B36248BBBAF4'}],
     ['214', {IN=> {f=> '635 1 40 150 140 184 12 23 78 294 2 67 93 9 57 283 206 74 83 199 254 4 6 67 263 120 61 113 156 181 257 79 284 46 191 213 251 273 100 271 100 186 138 184 223 61 76 83 175 107 6 217 81 114 207 149 89 29 180 182 73 158 120 122 37 160 11 98 295 162 35 221 146 249 123 88 19 139 48 191 189 213 275 15 98 60 273 39 51 96 60 1 77 49 181 233 41 286 257 18 64 42 65 263 144 99 111 241 58 1 288 196 179 22 214 30 170 226 197 169 66 42 41 225 146 14 234 92 145 89 155 69 178 243 208 200 244 48 21 78 209 40 65 178 182 97 196 118 207 210 200 33 10 23 289 36 152 79 49 266 195 43 201 84 274 151 130 228 136 34 24 214 115 132 228 112 292 26 129 238 92 19 13 159 195 19 163 86 189 274 90 19 224 10 193 104 40 126 102 186 223 146 225 85 90 230 262 8 278 90 80 39 14 23 54 147 4 278 216 265 35 75 73 243 82 124 248 283 267 185 87 29 95 220 71 243 293 162 246 72 144 38 237 228 51 181 22 3 288 172 141 24 129 153 130 17 98 36 235 41 75 219 20 88 165 25 54 232 262 280 131 92 291 162 161 110 142 14 173 277 141 259 18 149 4 67 178 267 59 198 33 34 170 287 121 173 192 288 257 103 26 44 276 90 104 186 12 108 227 213 190 53 96 145 150 79 73 8 193 128 78 124 187 63 263 59 137 109 30 44 41 201 108 16 149 215 105 176 56 232 51 242 133 7 118 122 248 198 9 183 172 202 244 133 279 80 124 163 25 265 270 260 69 8 132 204 74 294 237 252 89 179 107 176 32 85 53 157 256 154 205 210 115 71 8 108 166 76 209 64 103 176 267 270 49 31 10 47 31 34 85 29 119 258 69 32 170 202 159 171 277 165 1 16 234 177 107 91 70 83 29 34 67 117 47 55 72 83 114 292 2 91 240 7 33 160 246 283 291 36 85 114 81 260 93 155 130 289 247 38 150 55 222 10 108 251 38 211 275 226 136 58 62 212 200 9 268 25 204 159 256 239 167 212 30 194 264 195 254 239 82 63 28 279 141 48 167 151 215 190 138 58 147 65 202 220 95 281 291 247 123 112 57 137 133 68 42 126 78 148 225 61 52 232 177 22 129 68 168 15 74 22 275 250 174 56 147 89 240 218 180 158 35 160 35 245 293 247 208 74 59 52 256 28 20 214 54 21 224 282 46 106 251 249 62 273 15 13 98 271 15 37 136 269 100 210 285 182 12 47 116 292 87 240 242 218 156 120 37 75 158 96 84 39 75 134 58 86 250 198 94 145 174 117 71 93 70 168 287 36 24 190 77 134 143 235 252 113 217 105 261 205 234 262 69 93 94 206 285 121 197 43 286 46 276 66 253 188 223 231 187 87 5 57 222 95 285 ^'}},
     {OUT=>'4DEA6251B2A6DF017A8093AB066EE3863A4EC369'}],
     ['215', {IN=> {f=> '636 0 251 248 37 233 16 224 220 230 102 123 3 141 98 133 278 56 84 165 235 91 229 49 157 278 2 202 295 98 110 217 19 1 203 20 137 108 5 193 42 116 122 270 236 7 268 47 242 155 43 115 206 125 238 28 94 6 32 244 207 284 204 50 42 199 167 127 153 45 53 93 121 111 79 282 129 151 213 166 97 57 60 188 53 10 182 148 162 244 47 105 37 210 61 187 137 268 184 63 285 97 134 194 201 82 250 112 132 142 257 290 109 4 277 3 118 37 96 195 94 60 211 114 47 265 110 80 198 168 149 1 43 138 267 268 24 287 12 257 153 288 171 117 281 130 67 252 164 76 231 101 101 261 143 13 256 237 143 173 32 107 89 154 44 31 245 123 26 202 135 216 240 80 38 286 279 55 62 126 237 6 55 8 244 70 166 74 293 23 105 91 92 228 27 118 280 214 28 254 79 199 84 188 296 128 15 11 181 232 17 246 259 228 187 150 54 9 266 109 164 159 170 64 18 2 225 59 135 193 179 43 252 27 255 101 47 82 273 173 275 225 219 17 63 125 24 264 57 220 70 29 253 163 160 86 103 16 87 102 100 48 104 15 171 63 102 260 227 271 258 152 266 132 119 196 253 26 42 62 49 103 99 26 65 17 86 80 61 50 19 138 24 44 191 127 99 269 183 147 140 99 49 20 54 169 42 276 4 140 182 94 280 77 207 124 25 180 96 218 155 203 294 76 71 219 231 124 162 1 226 272 174 191 264 217 102 237 166 234 175 190 51 72 104 152 289 87 262 279 215 241 293 77 284 67 229 209 39 148 139 183 10 22 75 292 1 66 93 7 54 283 205 72 81 197 252 4 3 64 261 120 58 113 154 181 255 78 283 45 190 211 251 271 98 270 99 186 138 184 223 61 73 80 175 107 6 215 81 114 207 147 86 26 179 180 73 157 118 120 35 158 11 98 293 160 32 221 146 247 121 88 18 139 46 189 189 212 273 13 96 58 273 36 51 95 59 295 75 49 180 233 39 285 257 16 64 41 63 262 142 97 111 239 58 296 288 194 178 21 213 29 170 224 197 169 65 40 41 223 144 12 234 92 145 88 155 66 176 243 208 200 243 48 21 76 208 40 64 178 181 95 196 116 205 209 199 30 9 23 288 34 296 151 76 48 264 194 41 200 83 272 151 130 227 134 34 22 212 113 131 226 112 292 24 127 236 90 16 10 159 193 19 161 85 187 274 89 17 223 7 191 104 38 126 100 185 222 145 225 82 90 230 262 5 276 87 79 39 11 21 53 147 3 278 216 263 35 75 72 241 79 122 247 281 267 185 85 27 95 219 71 242 292 161 245 69 144 35 295 235 228 50 179 20 287 172 139 23 128 152 129 14 95 36 234 38 72 218 18 85 164 22 52 231 261 280 131 89 291 162 161 108 142 14 172 276 91 ^'}},
     {OUT=>'D30E70E357D57E3D82CA554B8A3D58DFF528FA94'}],
     ['216', {IN=> {f=> '635 1 256 9 146 297 61 175 261 50 198 27 25 164 284 115 173 192 282 254 94 23 41 273 84 95 180 3 102 227 290 213 187 50 90 139 150 73 64 2 190 122 75 124 181 60 263 56 134 109 21 35 38 201 99 10 149 215 102 173 47 226 42 239 133 297 115 119 248 195 9 183 166 199 238 130 279 74 121 163 19 265 270 257 66 8 126 204 71 294 234 246 89 173 101 170 29 85 44 154 256 154 202 210 115 65 5 108 163 73 209 58 103 176 264 267 40 25 10 41 28 28 82 26 113 258 69 26 167 202 156 171 277 159 297 7 231 174 104 88 70 83 29 31 67 111 44 46 69 80 111 289 2 82 234 1 33 160 246 280 285 27 79 108 72 260 84 149 124 289 247 38 147 55 222 4 105 245 32 211 272 226 133 52 59 206 194 6 265 25 201 153 253 236 161 212 30 188 261 195 254 239 79 54 28 273 138 39 167 148 212 184 132 49 141 65 196 220 89 281 288 244 120 106 57 137 127 62 33 123 78 142 222 58 49 232 177 22 129 68 168 6 74 13 275 244 174 50 144 83 237 218 174 158 32 157 29 239 287 241 202 65 59 52 250 19 20 214 45 12 218 282 46 106 248 249 53 270 12 10 95 271 15 28 136 269 100 204 279 182 9 38 110 286 84 240 236 215 156 117 34 72 152 96 75 33 69 134 55 77 247 192 91 142 168 114 62 87 67 165 287 33 15 190 68 131 137 235 249 107 214 99 255 205 228 256 63 90 85 206 285 118 191 34 280 40 270 66 253 182 217 225 184 78 5 54 219 86 64 247 246 35 229 14 224 218 228 102 119 3 139 96 129 276 56 80 165 233 89 227 45 157 274 295 200 295 92 108 217 17 296 203 14 133 106 3 193 40 116 122 266 236 7 268 43 242 151 39 113 204 125 238 22 92 30 242 205 282 200 50 36 195 163 127 153 45 49 87 121 109 75 278 125 147 209 166 91 51 56 188 53 8 180 148 158 240 45 103 31 208 55 185 135 266 182 63 281 97 132 194 197 78 250 110 132 140 253 290 105 275 298 118 37 90 193 94 60 209 112 47 263 110 78 196 166 145 294 43 134 263 264 22 283 12 255 151 284 169 117 279 128 65 250 164 76 227 99 101 261 141 11 252 233 143 171 32 103 85 152 42 31 245 123 24 198 135 212 238 76 36 286 275 53 62 124 235 4 51 2 244 66 162 70 291 17 105 91 88 224 27 116 276 210 24 252 79 199 84 186 292 128 11 7 177 230 17 242 259 226 187 295 146 52 5 262 107 160 155 168 60 18 221 55 131 191 179 37 252 25 255 97 41 82 271 169 271 223 215 11 61 121 18 262 53 216 68 23 249 161 158 86 101 16 85 98 98 44 100 13 167 57 100 258 223 269 254 148 266 128 119 194 251 22 38 56 43 275 ^'}},
     {OUT=>'70CA84D827F7FD61446233F88CF2F990B0F3E2AA'}],
     ['217', {IN=> {f=> '642 0 91 26 53 13 78 80 53 42 11 134 24 44 191 123 87 261 175 143 140 99 45 16 54 161 42 276 132 174 82 276 65 199 120 17 180 92 210 151 195 290 72 63 219 231 116 158 1 222 272 170 187 264 209 90 237 162 226 167 182 43 72 92 152 285 87 254 275 211 241 293 77 284 55 229 201 35 140 135 179 2 18 63 284 298 62 93 300 42 283 201 64 73 189 244 4 292 52 253 120 46 113 146 181 247 74 279 41 186 203 251 263 90 266 95 186 138 184 223 61 61 68 175 107 6 207 81 114 207 139 74 14 175 172 73 153 110 112 27 150 11 98 285 152 20 221 146 239 113 88 14 139 38 181 189 208 265 5 88 50 273 24 51 91 55 287 67 49 176 233 31 281 257 8 64 37 55 258 134 89 111 231 58 292 288 186 174 17 209 25 170 216 197 169 61 32 41 215 136 4 234 92 145 84 155 54 168 243 208 200 239 48 21 68 204 40 60 178 177 87 196 108 197 205 195 18 5 23 284 26 296 147 64 44 256 190 33 196 79 264 151 130 223 126 34 14 204 105 127 218 112 292 16 119 228 82 4 299 159 185 19 153 81 179 274 85 9 219 296 183 104 30 126 92 181 218 141 225 70 90 230 262 294 268 75 75 39 300 13 49 147 300 278 216 255 35 75 68 233 67 114 243 273 267 185 77 19 95 215 71 238 288 157 241 57 144 23 291 227 228 46 171 12 289 283 172 131 19 124 148 125 2 83 36 230 26 60 214 10 73 160 10 44 227 257 280 131 77 291 162 161 100 142 14 168 272 141 254 3 144 295 57 173 257 44 198 23 19 160 282 111 173 192 278 252 88 21 39 271 80 89 176 298 98 227 286 213 185 48 86 135 150 69 58 299 188 118 73 124 177 58 263 54 132 109 15 29 36 201 93 6 149 215 100 171 41 222 36 237 133 293 113 117 248 193 9 183 162 197 234 128 279 70 119 163 15 265 270 255 64 8 122 204 69 294 232 242 89 169 97 166 27 85 38 152 256 154 200 210 115 61 3 108 161 71 209 54 103 176 262 265 34 21 10 37 26 24 80 24 109 258 69 22 165 202 154 171 277 155 297 1 229 172 102 86 70 83 29 29 67 107 42 40 67 78 109 287 2 76 230 298 33 160 246 278 281 21 75 104 66 260 78 145 120 289 247 38 145 55 222 103 241 28 211 270 226 131 48 57 202 190 4 263 25 199 149 251 234 157 212 30 184 259 195 254 239 77 48 28 269 136 33 167 146 210 180 128 43 137 65 192 220 85 281 286 242 118 102 57 137 123 58 27 121 78 138 220 56 47 232 177 22 129 68 168 74 7 275 240 174 46 142 79 235 218 170 158 30 155 25 235 283 237 198 59 59 52 246 13 20 214 39 6 214 282 46 106 246 249 47 268 10 8 93 271 15 22 136 269 100 200 275 42 ^'}},
     {OUT=>'8D500C9CFDE0288530A2106B70BED39326C52C3C'}],
     ['218', {IN=> {f=> '644 0 6 29 104 280 81 240 230 212 156 114 31 69 146 96 66 27 63 134 52 68 244 186 88 139 162 111 53 81 64 162 287 30 6 190 59 128 131 235 246 101 211 93 249 205 222 250 57 87 76 206 285 115 185 25 274 34 264 66 253 176 211 219 181 69 5 51 216 77 58 241 243 32 223 11 224 215 225 102 113 3 136 93 123 273 56 74 165 230 86 224 39 157 268 289 197 295 83 105 217 14 293 203 5 127 103 193 37 116 122 260 236 7 268 37 242 145 33 110 201 125 238 13 89 293 27 239 202 279 194 50 27 189 157 127 153 45 43 78 121 106 69 272 119 141 203 166 82 42 50 188 53 5 177 148 152 234 42 100 22 205 46 182 132 263 179 63 275 97 129 194 191 72 250 107 132 137 247 290 99 296 272 295 118 37 81 190 94 60 206 109 47 260 110 75 193 163 139 288 43 128 257 258 19 277 12 252 148 278 166 117 276 125 62 247 164 76 221 96 101 261 138 8 246 227 143 168 32 97 79 149 39 31 245 123 21 192 135 206 235 70 33 286 269 50 62 121 232 1 45 295 244 60 156 64 288 8 105 91 82 218 27 113 270 204 18 249 79 199 84 183 286 128 5 1 171 227 17 236 259 223 187 292 140 49 301 256 104 154 149 165 54 18 299 215 49 125 188 179 28 252 22 255 91 32 82 268 163 265 220 209 2 58 115 9 259 47 210 65 14 243 158 155 86 98 16 82 92 95 38 94 10 161 48 97 255 217 266 248 142 266 122 119 191 248 16 32 47 34 88 89 26 50 12 76 80 51 40 9 133 24 44 191 122 84 259 173 142 140 99 44 15 54 159 42 276 301 130 172 79 275 62 197 119 15 180 91 208 150 193 289 71 61 219 231 114 157 1 221 272 169 186 264 207 87 237 161 224 165 180 41 72 89 152 284 87 252 274 210 241 293 77 284 52 229 199 34 138 134 178 17 60 282 298 61 93 299 39 283 200 62 71 187 242 4 290 49 251 120 43 113 144 181 245 73 278 40 185 201 251 261 88 265 94 186 138 184 223 61 58 65 175 107 6 205 81 114 207 137 71 11 174 170 73 152 108 110 25 148 11 98 283 150 17 221 146 237 111 88 13 139 36 179 189 207 263 3 86 48 273 21 51 90 54 285 65 49 175 233 29 280 257 6 64 36 53 257 132 87 111 229 58 291 288 184 173 16 208 24 170 214 197 169 60 30 41 213 134 2 234 92 145 83 155 51 166 243 208 200 238 48 21 66 203 40 59 178 176 85 196 106 195 204 194 15 4 23 283 24 296 146 61 43 254 189 31 195 78 262 151 130 222 124 34 12 202 103 126 216 112 292 14 117 226 80 1 297 159 183 19 151 80 177 274 84 7 218 294 181 104 28 126 90 180 217 140 225 67 90 230 262 292 266 72 74 39 298 11 48 147 300 278 216 253 265 ^'}},
     {OUT=>'F3D4D139EDFC24596377BC97A96FB7621F27FFC7'}],
     ['219', {IN=> {f=> '638 1 75 65 227 58 108 240 267 267 185 71 13 95 212 71 235 285 154 238 48 144 14 288 221 228 43 165 6 283 280 172 125 16 121 145 122 297 74 36 227 17 51 211 4 64 157 1 38 224 254 280 131 68 291 162 161 94 142 14 165 269 141 251 298 141 292 51 170 251 35 198 17 10 154 279 105 173 192 272 249 79 18 36 268 74 80 170 292 92 227 280 213 182 45 80 129 150 63 49 296 185 112 70 124 171 55 263 51 129 109 6 20 33 201 84 149 215 97 168 32 216 27 234 133 287 110 114 248 190 9 183 156 194 228 125 279 64 116 163 9 265 270 252 61 8 116 204 66 294 229 236 89 163 91 160 24 85 29 149 256 154 197 210 115 55 108 158 68 209 48 103 176 259 262 25 15 10 31 23 18 77 21 103 258 69 16 162 202 151 171 277 149 297 296 226 169 99 83 70 83 29 26 67 101 39 31 64 75 106 284 2 67 224 295 33 160 246 275 275 12 69 98 57 260 69 139 114 289 247 38 142 55 222 298 100 235 22 211 267 226 128 42 54 196 184 1 260 25 196 143 248 231 151 212 30 178 256 195 254 239 74 39 28 263 133 24 167 143 207 174 122 34 131 65 186 220 79 281 283 239 115 96 57 137 117 52 18 118 78 132 217 53 44 232 177 22 129 68 168 295 74 302 275 234 174 40 139 73 232 218 164 158 27 152 19 229 277 231 192 50 59 52 240 4 20 214 30 301 208 282 46 106 243 249 38 265 7 5 90 271 15 13 136 269 100 194 269 182 4 23 100 276 79 240 226 210 156 112 29 67 142 96 60 23 59 134 50 62 242 182 86 137 158 109 47 77 62 160 287 28 190 53 126 127 235 244 97 209 89 245 205 218 246 53 85 70 206 285 113 181 19 270 30 260 66 253 172 207 215 179 63 5 49 214 71 54 237 241 30 219 9 224 213 223 102 109 3 134 91 119 271 56 70 165 228 84 222 35 157 264 285 195 295 77 103 217 12 291 203 303 123 101 302 193 35 116 122 256 236 7 268 33 242 141 29 108 199 125 238 7 87 289 25 237 200 277 190 50 21 185 153 127 153 45 39 72 121 104 65 268 115 137 199 166 76 36 46 188 53 3 175 148 148 230 40 98 16 203 40 180 130 261 177 63 271 97 127 194 187 68 250 105 132 135 243 290 95 294 270 293 118 37 75 188 94 60 204 107 47 258 110 73 191 161 135 284 43 124 253 254 17 273 12 250 146 274 164 117 274 123 60 245 164 76 217 94 101 261 136 6 242 223 143 166 32 93 75 147 37 31 245 123 19 188 135 202 233 66 31 286 265 48 62 119 230 303 41 291 244 56 152 60 286 2 105 91 78 214 27 111 266 200 14 247 79 199 84 181 282 128 1 301 167 225 17 232 259 221 187 290 136 47 299 252 102 150 145 163 157 ^'}},
     {OUT=>'5509BAFFAC6D507860CEFC5AB5832CB63CD4B687'}],
     ['220', {IN=> {f=> '653 0 18 299 207 41 117 184 179 16 252 18 255 83 20 82 264 155 257 216 201 296 54 107 303 255 39 202 61 2 235 154 151 86 94 16 78 84 91 30 86 6 153 36 93 251 209 262 240 134 266 114 119 187 244 8 24 35 22 76 81 26 38 8 68 80 43 32 1 129 24 44 191 118 72 251 165 138 140 99 40 11 54 151 42 276 301 122 164 67 271 50 189 115 7 180 87 200 146 185 285 67 53 219 231 106 153 1 217 272 165 182 264 199 75 237 157 216 157 172 33 72 77 152 280 87 244 270 206 241 293 77 284 40 229 191 30 130 130 174 298 13 48 274 298 57 93 295 27 283 196 54 63 179 234 4 282 37 243 120 31 113 136 181 237 69 274 36 181 193 251 253 80 261 90 186 138 184 223 61 46 53 175 107 6 197 81 114 207 129 59 305 170 162 73 148 100 102 17 140 11 98 275 142 5 221 146 229 103 88 9 139 28 171 189 203 255 301 78 40 273 9 51 86 50 277 57 49 171 233 21 276 257 304 64 32 45 253 124 79 111 221 58 287 288 176 169 12 204 20 170 206 197 169 56 22 41 205 126 300 234 92 145 79 155 39 158 243 208 200 234 48 21 58 199 40 55 178 172 77 196 98 187 200 190 3 23 279 16 296 142 49 39 246 185 23 191 74 254 151 130 218 116 34 4 194 95 122 208 112 292 6 109 218 72 295 289 159 175 19 143 76 169 274 80 305 214 286 173 104 20 126 82 176 213 136 225 55 90 230 262 284 258 60 70 39 290 3 44 147 300 278 216 245 35 75 63 223 52 104 238 263 267 185 67 9 95 210 71 233 283 152 236 42 144 8 286 217 228 41 161 2 279 278 172 121 14 119 143 120 293 68 36 225 11 45 209 58 155 301 34 222 252 280 131 62 291 162 161 90 142 14 163 267 141 249 294 139 290 47 168 247 29 198 13 4 150 277 101 173 192 268 247 73 16 34 266 70 74 166 288 88 227 276 213 180 43 76 125 150 59 43 294 183 108 68 124 167 53 263 49 127 109 14 31 201 78 302 149 215 95 166 26 212 21 232 133 283 108 112 248 188 9 183 152 192 224 123 279 60 114 163 5 265 270 250 59 8 112 204 64 294 227 232 89 159 87 156 22 85 23 147 256 154 195 210 115 51 304 108 156 66 209 44 103 176 257 260 19 11 10 27 21 14 75 19 99 258 69 12 160 202 149 171 277 145 297 292 224 167 97 81 70 83 29 24 67 97 37 25 62 73 104 282 2 61 220 293 33 160 246 273 271 6 65 94 51 260 63 135 110 289 247 38 140 55 222 296 98 231 18 211 265 226 126 38 52 192 180 305 258 25 194 139 246 229 147 212 30 174 254 195 254 239 72 33 28 259 131 18 167 141 205 170 118 28 127 65 182 220 75 281 281 237 113 92 57 137 113 48 12 116 78 128 215 51 42 232 177 22 129 68 168 291 161 ^'}},
     {OUT=>'0C0AEA0C2FD7A620C77866B1A177481E26B4F592'}],
     ['221', {IN=> {f=> '653 0 296 275 228 174 34 136 67 229 218 158 158 24 149 13 223 271 225 186 41 59 52 234 302 20 214 21 295 202 282 46 106 240 249 29 262 4 2 87 271 15 4 136 269 100 188 263 182 1 14 94 270 76 240 220 207 156 109 26 64 136 96 51 17 53 134 47 53 239 176 83 134 152 106 38 71 59 157 287 25 298 190 44 123 121 235 241 91 206 83 239 205 212 240 47 82 61 206 285 110 175 10 264 24 254 66 253 166 201 209 176 54 5 46 211 62 48 231 238 27 213 6 224 210 220 102 103 3 131 88 113 268 56 64 165 225 81 219 29 157 258 279 192 295 68 100 217 9 288 203 297 117 98 302 193 32 116 122 250 236 7 268 27 242 135 23 105 196 125 238 305 84 283 22 234 197 274 184 50 12 179 147 127 153 45 33 63 121 101 59 262 109 131 193 166 67 27 40 188 53 172 148 142 224 37 95 7 200 31 177 127 258 174 63 265 97 124 194 181 62 250 102 132 132 237 290 89 291 267 290 118 37 66 185 94 60 201 104 47 255 110 70 188 158 129 278 43 118 247 248 14 267 12 247 143 268 161 117 271 120 57 242 164 76 211 91 101 261 133 3 236 217 143 163 32 87 69 144 34 31 245 123 16 182 135 196 230 60 28 286 259 45 62 116 227 303 35 285 244 50 146 54 283 300 105 91 72 208 27 108 260 194 8 244 79 199 84 178 276 128 302 298 161 222 17 226 259 218 187 287 130 44 296 246 99 144 139 160 44 18 299 205 39 115 183 179 13 252 17 255 81 17 82 263 153 255 215 199 294 53 105 301 254 37 200 60 306 233 153 150 86 93 16 77 82 90 28 84 5 151 33 92 250 207 261 238 132 266 112 119 186 243 6 22 32 19 73 79 26 35 7 66 80 41 30 306 128 24 44 191 117 69 249 163 137 140 99 39 10 54 149 42 276 301 120 162 64 270 47 187 114 5 180 86 198 145 183 284 66 51 219 231 104 152 1 216 272 164 181 264 197 72 237 156 214 155 170 31 72 74 152 279 87 242 269 205 241 293 77 284 37 229 189 29 128 129 173 297 12 45 272 298 56 93 294 24 283 195 52 61 177 232 4 280 34 241 120 28 113 134 181 235 68 273 35 180 191 251 251 78 260 89 186 138 184 223 61 43 50 175 107 6 195 81 114 207 127 56 303 169 160 73 147 98 100 15 138 11 98 273 140 2 221 146 227 101 88 8 139 26 169 189 202 253 300 76 38 273 6 51 85 49 275 55 49 170 233 19 275 257 303 64 31 43 252 122 77 111 219 58 286 288 174 168 11 203 19 170 204 197 169 55 20 41 203 124 299 234 92 145 78 155 36 156 243 208 200 233 48 21 56 198 40 54 178 171 75 196 96 185 199 189 306 23 278 14 296 141 46 38 244 184 21 190 73 252 151 130 217 114 34 2 192 93 121 206 112 292 4 107 216 70 293 112 ^'}},
     {OUT=>'149176007FEE58A591E3F00F8DB658B605F8390C'}],
     ['222', {IN=> {f=> '646 1 159 169 19 137 73 163 274 77 302 211 280 167 104 14 126 76 173 210 133 225 46 90 230 262 278 252 51 67 39 284 306 41 147 300 278 216 239 35 75 60 217 43 98 235 257 267 185 61 3 95 207 71 230 280 149 233 33 144 308 283 211 228 38 155 305 273 275 172 115 11 116 140 117 287 59 36 222 2 36 206 303 49 152 295 28 219 249 280 131 53 291 162 161 84 142 14 160 264 141 246 288 136 287 41 165 241 20 198 7 304 144 274 95 173 192 262 244 64 13 31 263 64 65 160 282 82 227 270 213 177 40 70 119 150 53 34 291 180 102 65 124 161 50 263 46 124 109 300 5 28 201 69 299 149 215 92 163 17 206 12 229 133 277 105 109 248 185 9 183 146 189 218 120 279 54 111 163 308 265 270 247 56 8 106 204 61 294 224 226 89 153 81 150 19 85 14 144 256 154 192 210 115 45 304 108 153 63 209 38 103 176 254 257 10 5 10 21 18 8 72 16 93 258 69 6 157 202 146 171 277 139 297 286 221 164 94 78 70 83 29 21 67 91 34 16 59 70 101 279 2 52 214 290 33 160 246 270 265 306 59 88 42 260 54 129 104 289 247 38 137 55 222 293 95 225 12 211 262 226 123 32 49 186 174 305 255 25 191 133 243 226 141 212 30 168 251 195 254 239 69 24 28 253 128 9 167 138 202 164 112 19 121 65 176 220 69 281 278 234 110 86 57 137 107 42 3 113 78 122 212 48 39 232 177 22 129 68 168 285 74 292 275 224 174 30 134 63 227 218 154 158 22 147 9 219 267 221 182 35 59 52 230 298 20 214 15 291 198 282 46 106 238 249 23 260 2 85 271 15 307 136 269 100 184 259 182 308 8 90 266 74 240 216 205 156 107 24 62 132 96 45 13 49 134 45 47 237 172 81 132 148 104 32 67 57 155 287 23 294 190 38 121 117 235 239 87 204 79 235 205 208 236 43 80 55 206 285 108 171 4 260 20 250 66 253 162 197 205 174 48 5 44 209 56 44 227 236 25 209 4 224 208 218 102 99 3 129 86 109 266 56 60 165 223 79 217 25 157 254 275 190 295 62 98 217 7 286 203 293 113 96 302 193 30 116 122 246 236 7 268 23 242 131 19 103 194 125 238 301 82 279 20 232 195 272 180 50 6 175 143 127 153 45 29 57 121 99 55 258 105 127 189 166 61 21 36 188 53 307 170 148 138 220 35 93 1 198 25 175 125 256 172 63 261 97 122 194 177 58 250 100 132 130 233 290 85 289 265 288 118 37 60 183 94 60 199 102 47 253 110 68 186 156 125 274 43 114 243 244 12 263 12 245 141 264 159 117 269 118 55 240 164 76 207 89 101 261 131 1 232 213 143 161 32 83 65 142 32 31 245 123 14 178 135 192 228 56 26 286 255 43 62 114 225 303 31 281 244 46 142 50 281 296 105 91 238 ^'}},
     {OUT=>'17C0D7B0256159F3626786FFDB20237AE154FA84'}],
     ['223', {IN=> {f=> '647 0 200 27 104 252 186 240 79 199 84 174 268 128 298 294 153 218 17 218 259 214 187 283 122 40 292 238 95 136 131 156 36 18 299 197 31 107 179 179 1 252 13 255 73 5 82 259 145 247 211 191 286 49 97 293 250 29 192 56 298 225 149 146 86 89 16 73 74 86 20 76 1 143 21 88 246 199 257 230 124 266 104 119 182 239 309 14 20 7 61 71 26 23 3 58 80 33 22 302 124 24 44 191 113 57 241 155 133 140 99 35 6 54 141 42 276 301 112 154 52 266 35 179 110 308 180 82 190 141 175 280 62 43 219 231 96 148 1 212 272 160 177 264 189 60 237 152 206 147 162 23 72 62 152 275 87 234 265 201 241 293 77 284 25 229 181 25 120 125 169 293 8 33 264 298 52 93 290 12 283 191 44 53 169 224 4 272 22 233 120 16 113 126 181 227 64 269 31 176 183 251 243 70 256 85 186 138 184 223 61 31 38 175 107 6 187 81 114 207 119 44 295 165 152 73 143 90 92 7 130 11 98 265 132 301 221 146 219 93 88 4 139 18 161 189 198 245 296 68 30 273 305 51 81 45 267 47 49 166 233 11 271 257 299 64 27 35 248 114 69 111 211 58 282 288 166 164 7 199 15 170 196 197 169 51 12 41 195 116 295 234 92 145 74 155 24 148 243 208 200 229 48 21 48 194 40 50 178 167 67 196 88 177 195 185 299 306 23 274 6 296 137 34 34 236 180 13 186 69 244 151 130 213 106 34 305 184 85 117 198 112 292 307 99 208 62 285 279 159 165 19 133 71 159 274 75 300 209 276 163 104 10 126 72 171 208 131 225 40 90 230 262 274 248 45 65 39 280 304 39 147 300 278 216 235 35 75 58 213 37 94 233 253 267 185 57 310 95 205 71 228 278 147 231 27 144 304 281 207 228 36 151 303 269 273 172 111 9 114 138 115 283 53 36 220 307 30 204 301 43 150 291 24 217 247 280 131 47 291 162 161 80 142 14 158 262 141 244 284 134 285 37 163 237 14 198 3 300 140 272 91 173 192 258 242 58 11 29 261 60 59 156 278 78 227 266 213 175 38 66 115 150 49 28 289 178 98 63 124 157 48 263 44 122 109 296 310 26 201 63 297 149 215 90 161 11 202 6 227 133 273 103 107 248 183 9 183 142 187 214 118 279 50 109 163 306 265 270 245 54 8 102 204 59 294 222 222 89 149 77 146 17 85 8 142 256 154 190 210 115 41 304 108 151 61 209 34 103 176 252 255 4 1 10 17 16 4 70 14 89 258 69 2 155 202 144 171 277 135 297 282 219 162 92 76 70 83 29 19 67 87 32 10 57 68 99 277 2 46 210 288 33 160 246 268 261 302 55 84 36 260 48 125 100 289 247 38 135 55 222 291 93 221 8 211 260 226 121 28 47 182 170 305 253 25 189 129 241 224 137 212 30 164 249 195 254 239 86 ^'}},
     {OUT=>'741A58618ABEB1D983D67AFDCBC49AA397A3B8E0'}],
     ['224', {IN=> {f=> '665 0 15 28 247 125 167 135 199 158 106 10 115 65 170 220 63 281 275 231 107 80 57 137 101 36 306 110 78 116 209 45 36 232 177 22 129 68 168 279 74 286 275 218 174 24 131 57 224 218 148 158 19 144 3 213 261 215 176 26 59 52 224 292 20 214 6 285 192 282 46 106 235 249 14 257 311 309 82 271 15 301 136 269 100 178 253 182 308 311 84 260 71 240 210 202 156 104 21 59 126 96 36 7 43 134 42 38 234 166 78 129 142 101 23 61 54 152 287 20 288 190 29 118 111 235 236 81 201 73 229 205 202 230 37 77 46 206 285 105 165 307 254 14 244 66 253 156 191 199 171 39 5 41 206 47 38 221 233 22 203 1 224 205 215 102 93 3 126 83 103 263 56 54 165 220 76 214 19 157 248 269 187 295 53 95 217 4 283 203 287 107 93 302 193 27 116 122 240 236 7 268 17 242 125 13 100 191 125 238 295 79 273 17 229 192 269 174 50 309 169 137 127 153 45 23 48 121 96 49 252 99 121 183 166 52 12 30 188 53 307 167 148 132 214 32 90 304 195 16 172 122 253 169 63 255 97 119 194 171 52 250 97 132 127 227 290 79 286 262 285 118 37 51 180 94 60 196 99 47 250 110 65 183 153 119 268 43 108 237 238 9 257 12 242 138 258 156 117 266 115 52 237 164 76 201 86 101 261 128 310 226 207 143 158 32 77 59 139 29 31 245 123 11 172 135 186 225 50 23 286 249 40 62 111 222 303 25 275 244 40 136 44 278 290 105 91 62 198 27 103 250 184 310 239 79 199 84 173 266 128 297 293 151 217 17 216 259 213 187 282 120 39 291 236 94 134 129 155 34 18 299 195 29 105 178 179 310 252 12 255 71 2 82 258 143 245 210 189 284 48 95 291 249 27 190 55 296 223 148 145 86 88 16 72 72 85 18 74 141 18 87 245 197 256 228 122 266 102 119 181 238 308 12 17 4 58 69 26 20 2 56 80 31 20 301 123 24 44 191 112 54 239 153 132 140 99 34 5 54 139 42 276 301 110 152 49 265 32 177 109 307 180 81 188 140 173 279 61 41 219 231 94 147 1 211 272 159 176 264 187 57 237 151 204 145 160 21 72 59 152 274 87 232 264 200 241 293 77 284 22 229 179 24 118 124 168 292 7 30 262 298 51 93 289 9 283 190 42 51 167 222 4 270 19 231 120 13 113 124 181 225 63 268 30 175 181 251 241 68 255 84 186 138 184 223 61 28 35 175 107 6 185 81 114 207 117 41 293 164 150 73 142 88 90 5 128 11 98 263 130 299 221 146 217 91 88 3 139 16 159 189 197 243 295 66 28 273 303 51 80 44 265 45 49 165 233 9 270 257 298 64 26 33 247 112 67 111 209 58 281 288 164 163 6 198 14 170 194 197 169 50 10 41 193 114 294 234 92 145 73 155 21 146 243 208 200 228 48 21 46 193 40 49 178 166 65 196 86 175 194 184 297 306 23 273 206 ^'}},
     {OUT=>'B738D6B3409EB9ED2F1719B84D13F7C36169CDEC'}],
     ['225', {IN=> {f=> '641 0 296 134 25 31 230 177 7 183 66 238 151 130 210 100 34 302 178 79 114 192 112 292 304 93 202 56 279 273 159 159 19 127 68 153 274 72 297 206 270 157 104 4 126 66 168 205 128 225 31 90 230 262 268 242 36 62 39 274 301 36 147 300 278 216 229 35 75 55 207 28 88 230 247 267 185 51 307 95 202 71 225 275 144 228 18 144 298 278 201 228 33 145 300 263 270 172 105 6 111 135 112 277 44 36 217 301 21 201 298 34 147 285 18 214 244 280 131 38 291 162 161 74 142 14 155 259 141 241 278 131 282 31 160 231 5 198 311 294 134 269 85 173 192 252 239 49 8 26 258 54 50 150 272 72 227 260 213 172 35 60 109 150 43 19 286 175 92 60 124 151 45 263 41 119 109 290 304 23 201 54 294 149 215 87 158 2 196 311 224 133 267 100 104 248 180 9 183 136 184 208 115 279 44 106 163 303 265 270 242 51 8 96 204 56 294 219 216 89 143 71 140 14 85 313 139 256 154 187 210 115 35 304 108 148 58 209 28 103 176 249 252 309 309 10 11 13 312 67 11 83 258 69 310 152 202 141 171 277 129 297 276 216 159 89 73 70 83 29 16 67 81 29 1 54 65 96 274 2 37 204 285 33 160 246 265 255 296 49 78 27 260 39 119 94 289 247 38 132 55 222 288 90 215 2 211 257 226 118 22 44 176 164 305 250 25 186 123 238 221 131 212 30 158 246 195 254 239 64 9 28 243 123 308 167 133 197 154 102 4 111 65 166 220 59 281 273 229 105 76 57 137 97 32 302 108 78 112 207 43 34 232 177 22 129 68 168 275 74 282 275 214 174 20 129 53 222 218 144 158 17 142 313 209 257 211 172 20 59 52 220 288 20 214 281 188 282 46 106 233 249 8 255 311 309 80 271 15 297 136 269 100 174 249 182 308 307 80 256 69 240 206 200 156 102 19 57 122 96 30 3 39 134 40 32 232 162 76 127 138 99 17 57 52 150 287 18 284 190 23 116 107 235 234 77 199 69 225 205 198 226 33 75 40 206 285 103 161 303 250 10 240 66 253 152 187 195 169 33 5 39 204 41 34 217 231 20 199 313 224 203 213 102 89 3 124 81 99 261 56 50 165 218 74 212 15 157 244 265 185 295 47 93 217 2 281 203 283 103 91 302 193 25 116 122 236 236 7 268 13 242 121 9 98 189 125 238 291 77 269 15 227 190 267 170 50 305 165 133 127 153 45 19 42 121 94 45 248 95 117 179 166 46 6 26 188 53 307 165 148 128 210 30 88 300 193 10 170 120 251 167 63 251 97 117 194 167 48 250 95 132 125 223 290 75 284 260 283 118 37 45 178 94 60 194 97 47 248 110 63 181 151 115 264 43 104 233 234 7 253 12 240 136 254 154 117 264 113 50 235 164 76 197 84 101 261 126 310 222 203 143 276 ^'}},
     {OUT=>'3D33DE31F64055D3B128AC9A6AA3F92DFD4F5330'}],
     ['226', {IN=> {f=> '656 0 32 69 51 135 25 31 245 123 7 164 135 178 221 42 19 286 241 36 62 107 218 303 17 267 244 32 128 36 274 282 105 91 54 190 27 99 242 176 306 235 79 199 84 169 258 128 293 289 143 213 17 208 259 209 187 278 112 35 287 228 90 126 121 151 26 18 299 187 21 97 174 179 302 252 8 255 63 306 82 254 135 237 206 181 276 44 87 283 245 19 182 51 288 215 144 141 86 84 16 68 64 81 10 66 312 133 6 83 241 189 252 220 114 266 94 119 177 234 304 4 5 308 46 61 26 8 314 48 80 23 12 297 119 24 44 191 108 42 231 145 128 140 99 30 1 54 131 42 276 301 102 144 37 261 20 169 105 303 180 77 180 136 165 275 57 33 219 231 86 143 1 207 272 155 172 264 179 45 237 147 196 137 152 13 72 47 152 270 87 224 260 196 241 293 77 284 10 229 171 20 110 120 164 288 3 18 254 298 47 93 285 313 283 186 34 43 159 214 4 262 7 223 120 1 113 116 181 217 59 264 26 171 173 251 233 60 251 80 186 138 184 223 61 16 23 175 107 6 177 81 114 207 109 29 285 160 142 73 138 80 82 313 120 11 98 255 122 291 221 146 209 83 88 315 139 8 151 189 193 235 291 58 20 273 295 51 76 40 257 37 49 161 233 1 266 257 294 64 22 25 243 104 59 111 201 58 277 288 156 159 2 194 10 170 186 197 169 46 2 41 185 106 290 234 92 145 69 155 9 138 243 208 200 224 48 21 38 189 40 45 178 162 57 196 78 167 190 180 289 306 23 269 312 296 132 19 29 226 175 3 181 64 234 151 130 208 96 34 300 174 75 112 188 112 292 302 89 198 52 275 269 159 155 19 123 66 149 274 70 295 204 266 153 104 126 62 166 203 126 225 25 90 230 262 264 238 30 60 39 270 299 34 147 300 278 216 225 35 75 53 203 22 84 228 243 267 185 47 305 95 200 71 223 273 142 226 12 144 294 276 197 228 31 141 298 259 268 172 101 4 109 133 110 273 38 36 215 297 15 199 296 28 145 281 14 212 242 280 131 32 291 162 161 70 142 14 153 257 141 239 274 129 280 27 158 227 315 198 309 290 130 267 81 173 192 248 237 43 6 24 256 50 44 146 268 68 227 256 213 170 33 56 105 150 39 13 284 173 88 58 124 147 43 263 39 117 109 286 300 21 201 48 292 149 215 85 156 312 192 307 222 133 263 98 102 248 178 9 183 132 182 204 113 279 40 104 163 301 265 270 240 49 8 92 204 54 294 217 212 89 139 67 136 12 85 309 137 256 154 185 210 115 31 304 108 146 56 209 24 103 176 247 250 305 307 10 7 11 310 65 9 79 258 69 308 150 202 139 171 277 125 297 272 214 157 87 71 70 83 29 14 67 77 27 311 52 63 94 272 2 31 200 283 33 160 246 263 251 292 45 74 21 260 33 115 90 289 247 38 130 55 222 286 88 211 227 ^'}},
     {OUT=>'B6925F4DF94949B8844C867428BA3DEDF4CF2B51'}],
     ['227', {IN=> {f=> '668 0 211 254 226 115 16 41 170 158 305 247 25 183 117 235 218 125 212 30 152 243 195 254 239 61 28 237 120 302 167 130 194 148 96 312 105 65 160 220 53 281 270 226 102 70 57 137 91 26 296 105 78 106 204 40 31 232 177 22 129 68 168 269 74 276 275 208 174 14 126 47 219 218 138 158 14 139 310 203 251 205 166 11 59 52 214 282 20 214 308 275 182 282 46 106 230 249 316 252 311 309 77 271 15 291 136 269 100 168 243 182 308 301 74 250 66 240 200 197 156 99 16 54 116 96 21 314 33 134 37 23 229 156 73 124 132 96 8 51 49 147 287 15 278 190 14 113 101 235 231 71 196 63 219 205 192 220 27 72 31 206 285 100 155 297 244 4 234 66 253 146 181 189 166 24 5 36 201 32 28 211 228 17 193 313 224 200 210 102 83 3 121 78 93 258 56 44 165 215 71 209 9 157 238 259 182 295 38 90 217 316 278 203 277 97 88 302 193 22 116 122 230 236 7 268 7 242 115 3 95 186 125 238 285 74 263 12 224 187 264 164 50 299 159 127 127 153 45 13 33 121 91 39 242 89 111 173 166 37 314 20 188 53 307 162 148 122 204 27 85 294 190 1 167 117 248 164 63 245 97 114 194 161 42 250 92 132 122 217 290 69 281 257 280 118 37 36 175 94 60 191 94 47 245 110 60 178 148 109 258 43 98 227 228 4 247 12 237 133 248 151 117 261 110 47 232 164 76 191 81 101 261 123 310 216 197 143 153 32 67 49 134 24 31 245 123 6 162 135 176 220 40 18 286 239 35 62 106 217 303 15 265 244 30 126 34 273 280 105 91 52 188 27 98 240 174 305 234 79 199 84 168 256 128 292 288 141 212 17 206 259 208 187 277 110 34 286 226 89 124 119 150 24 18 299 185 19 95 173 179 300 252 7 255 61 304 82 253 133 235 205 179 274 43 85 281 244 17 180 50 286 213 143 140 86 83 16 67 62 80 8 64 312 131 3 82 240 187 251 218 112 266 92 119 176 233 303 2 2 306 43 59 26 5 314 46 80 21 10 296 118 24 44 191 107 39 229 143 127 140 99 29 54 129 42 276 301 100 142 34 260 17 167 104 302 180 76 178 135 163 274 56 31 219 231 84 142 1 206 272 154 171 264 177 42 237 146 194 135 150 11 72 44 152 269 87 222 259 195 241 293 77 284 7 229 169 19 108 119 163 287 2 15 252 298 46 93 284 311 283 185 32 41 157 212 4 260 4 221 120 315 113 114 181 215 58 263 25 170 171 251 231 58 250 79 186 138 184 223 61 13 20 175 107 6 175 81 114 207 107 26 283 159 140 73 137 78 80 312 118 11 98 253 120 289 221 146 207 81 88 315 139 6 149 189 192 233 290 56 18 273 293 51 75 39 255 35 49 160 233 316 265 257 293 64 21 23 242 102 57 111 199 58 276 288 154 158 1 193 9 170 184 197 169 45 41 183 104 289 234 92 145 68 155 6 136 243 208 203 ^'}},
     {OUT=>'CF5E7256292ABEC431D8E8B9CBEAF22AF072377E'}],
     ['228', {IN=> {f=> '656 0 221 48 21 32 186 40 42 178 159 51 196 72 161 187 177 283 306 23 266 309 296 129 10 26 220 172 316 178 61 228 151 130 205 90 34 297 168 69 109 182 112 292 299 83 192 46 269 263 159 149 19 117 63 143 274 67 292 201 260 147 104 313 126 56 163 200 123 225 16 90 230 262 258 232 21 57 39 264 296 31 147 300 278 216 219 35 75 50 197 13 78 225 237 267 185 41 302 95 197 71 220 270 139 223 3 144 288 273 191 228 28 135 295 253 265 172 95 1 106 130 107 267 29 36 212 291 6 196 293 19 142 275 8 209 239 280 131 23 291 162 161 64 142 14 150 254 141 236 268 126 277 21 155 221 309 198 306 284 124 264 75 173 192 242 234 34 3 21 253 44 35 140 262 62 227 250 213 167 30 50 99 150 33 4 281 170 82 55 124 141 40 263 36 114 109 280 294 18 201 39 289 149 215 82 153 306 186 301 219 133 257 95 99 248 175 9 183 126 179 198 110 279 34 101 163 298 265 270 237 46 8 86 204 51 294 214 206 89 133 61 130 9 85 303 134 256 154 182 210 115 25 304 108 143 53 209 18 103 176 244 247 299 304 10 1 8 307 62 6 73 258 69 305 147 202 136 171 277 119 297 266 211 154 84 68 70 83 29 11 67 71 24 305 49 60 91 269 2 22 194 280 33 160 246 260 245 286 39 68 12 260 24 109 84 289 247 38 127 55 222 283 85 205 311 211 252 226 113 12 39 166 154 305 245 25 181 113 233 216 121 212 30 148 241 195 254 239 59 313 28 233 118 298 167 128 192 144 92 308 101 65 156 220 49 281 268 224 100 66 57 137 87 22 292 103 78 102 202 38 29 232 177 22 129 68 168 265 74 272 275 204 174 10 124 43 217 218 134 158 12 137 308 199 247 201 162 5 59 52 210 278 20 214 304 271 178 282 46 106 228 249 312 250 311 309 75 271 15 287 136 269 100 164 239 182 308 297 70 246 64 240 196 195 156 97 14 52 112 96 15 312 29 134 35 17 227 152 71 122 128 94 2 47 47 145 287 13 274 190 8 111 97 235 229 67 194 59 215 205 188 216 23 70 25 206 285 98 151 293 240 230 66 253 142 177 185 164 18 5 34 199 26 24 207 226 15 189 313 224 198 208 102 79 3 119 76 89 256 56 40 165 213 69 207 5 157 234 255 180 295 32 88 217 316 276 203 273 93 86 302 193 20 116 122 226 236 7 268 3 242 111 318 93 184 125 238 281 72 259 10 222 185 262 160 50 295 155 123 127 153 45 9 27 121 89 35 238 85 107 169 166 31 310 16 188 53 307 160 148 118 200 25 83 290 188 314 165 115 246 162 63 241 97 112 194 157 38 250 90 132 120 213 290 65 279 255 278 118 37 30 173 94 60 189 92 47 243 110 58 176 146 105 254 43 94 223 224 2 243 12 235 131 244 149 117 259 108 45 230 164 76 187 79 101 261 109 ^'}},
     {OUT=>'975DCE94902923977F129C0E4ACF40AD28DDB9AA'}],
     ['229', {IN=> {f=> '656 0 310 208 189 143 149 32 59 41 130 20 31 245 123 2 154 135 168 216 32 14 286 231 31 62 102 213 303 7 257 244 22 118 26 269 272 105 91 44 180 27 94 232 166 301 230 79 199 84 164 248 128 288 284 133 208 17 198 259 204 187 273 102 30 282 218 85 116 111 146 16 18 299 177 11 87 169 179 292 252 3 255 53 296 82 249 125 227 201 171 266 39 77 273 240 9 172 46 278 205 139 136 86 79 16 63 54 76 56 312 123 312 78 236 179 247 210 104 266 84 119 172 229 299 315 311 298 31 51 26 314 314 38 80 13 2 292 114 24 44 191 103 27 221 135 123 140 99 25 317 54 121 42 276 301 92 134 22 256 5 159 100 298 180 72 170 131 155 270 52 23 219 231 76 138 1 202 272 150 167 264 169 30 237 142 186 127 142 3 72 32 152 265 87 214 255 191 241 293 77 284 316 229 161 15 100 115 159 283 319 3 244 298 42 93 280 303 283 181 24 33 149 204 4 252 313 213 120 307 113 106 181 207 54 259 21 166 163 251 223 50 246 75 186 138 184 223 61 1 8 175 107 6 167 81 114 207 99 14 275 155 132 73 133 70 72 308 110 11 98 245 112 281 221 146 199 73 88 315 139 319 141 189 188 225 286 48 10 273 285 51 71 35 247 27 49 156 233 312 261 257 289 64 17 15 238 94 49 111 191 58 272 288 146 154 318 189 5 170 176 197 169 41 313 41 175 96 285 234 92 145 64 155 315 128 243 208 200 219 48 21 28 184 40 40 178 157 47 196 68 157 185 175 279 306 23 264 307 296 127 4 24 216 170 314 176 59 224 151 130 203 86 34 295 164 65 107 178 112 292 297 79 188 42 265 259 159 145 19 113 61 139 274 65 290 199 256 143 104 311 126 52 161 198 121 225 10 90 230 262 254 228 15 55 39 260 294 29 147 300 278 216 215 35 75 48 193 7 74 223 233 267 185 37 300 95 195 71 218 268 137 221 318 144 284 271 187 228 26 131 293 249 263 172 91 320 104 128 105 263 23 36 210 287 194 291 13 140 271 4 207 237 280 131 17 291 162 161 60 142 14 148 252 141 234 264 124 275 17 153 217 305 198 304 280 120 262 71 173 192 238 232 28 1 19 251 40 29 136 258 58 227 246 213 165 28 46 95 150 29 319 279 168 78 53 124 137 38 263 34 112 109 276 290 16 201 33 287 149 215 80 151 302 182 297 217 133 253 93 97 248 173 9 183 122 177 194 108 279 30 99 163 296 265 270 235 44 8 82 204 49 294 212 202 89 129 57 126 7 85 299 132 256 154 180 210 115 21 304 108 141 51 209 14 103 176 242 245 295 302 10 318 6 305 60 4 69 258 69 303 145 202 134 171 277 115 297 262 209 152 82 66 70 83 29 9 67 67 22 301 47 58 89 267 2 16 190 278 33 160 246 258 241 282 35 64 6 260 18 105 80 289 247 38 125 55 292 ^'}},
     {OUT=>'333B0259B18CE64D6B52CF563DD3041E5F63A516'}],
    );

sub binary_expand ($$)
{
  my ($test_name, $line) = @_;
  my ($n, $b, $rest) = split ' ', $line, 3;
  defined $n && defined $b or die "$test_name: too few args\n";
  my @a = split ' ', $rest, $n + 1;
  my $caret = pop @a;
  $caret eq '^' or die "test $test_name: @a missing '^'\n";
  $b eq '1' || $b eq '0' or die "test $test_name: bad 'b'=$b\n";
  my $n_bad = @a;
  @a == $n or
    die "test $test_name: wrong number of args (expected $n, found $n_bad)\n";
  my $bit_string = '';
  foreach my $a (@a)
    {
      $bit_string .= $b x $a;
      $b = 1 - $b;
    }
  my $t = pack ("B*", $bit_string);
  # print "$bit_string\n $t\n";
  return $t;
}

my $t;
foreach $t (@Tests)
  {
    # Expand each input.
    my $in = $t->[1]->{IN};
    $in->{f} = binary_expand $t->[0], $in->{f};

    # Convert each expected output string to lower case, and append "  f\n".
    my $h = $t->[2];
    $h->{OUT} = lc $h->{OUT} . "  f\n";

    # Insert the '--text' argument for each test.
    splice @$t, 1, 0, '--text';
  }

my $save_temps = $ENV{DEBUG};
my $verbose = $ENV{VERBOSE};

my $prog = 'sha1sum';
my $fail = run_tests ($program_name, $prog, \@Tests, $save_temps, $verbose);
exit $fail;
