use std::ops::Deref;
use std::str;

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct LanguageSubtag([u8; 3]);

impl LanguageSubtag {
    const fn new(subtag: [char; 3]) -> Self {
        LanguageSubtag([subtag[0] as u8, subtag[1] as u8, subtag[2] as u8])
    }
}

impl Deref for LanguageSubtag {
    type Target = str;

    fn deref(&self) -> &str {
        let mut end = 3;
        while self.0[end - 1] == b' ' {
            end -= 1;
        }
        unsafe { str::from_utf8_unchecked(&self.0[..end]) }
    }
}

impl str::FromStr for LanguageSubtag {
    type Err = ();

    fn from_str(input: &str) -> Result<Self, ()> {
        if 2 <= input.len() && input.len() <= 3 {
            let mut value = [b' '; 3];
            value[..input.len()].copy_from_slice(input.as_bytes());
            Ok(LanguageSubtag(value))
        } else {
            Err(())
        }
    }
}

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct ScriptSubtag([u8; 4]);

impl ScriptSubtag {
    const fn new(subtag: [char; 4]) -> Self {
        ScriptSubtag([
            subtag[0] as u8,
            subtag[1] as u8,
            subtag[2] as u8,
            subtag[3] as u8,
        ])
    }
}

impl Deref for ScriptSubtag {
    type Target = str;

    fn deref(&self) -> &str {
        unsafe { str::from_utf8_unchecked(&self.0) }
    }
}

impl str::FromStr for ScriptSubtag {
    type Err = ();

    fn from_str(input: &str) -> Result<Self, ()> {
        if input.len() == 4 {
            let mut value = [b' '; 4];
            value.copy_from_slice(input.as_bytes());
            Ok(ScriptSubtag(value))
        } else {
            Err(())
        }
    }
}

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct RegionSubtag([u8; 3]);

impl RegionSubtag {
    const fn new(subtag: [char; 3]) -> Self {
        RegionSubtag([subtag[0] as u8, subtag[1] as u8, subtag[2] as u8])
    }
}

impl Deref for RegionSubtag {
    type Target = str;

    fn deref(&self) -> &str {
        let mut end = 3;
        while self.0[end - 1] == b' ' {
            end -= 1;
        }
        unsafe { str::from_utf8_unchecked(&self.0[..end]) }
    }
}

impl str::FromStr for RegionSubtag {
    type Err = ();

    fn from_str(input: &str) -> Result<Self, ()> {
        if 2 <= input.len() && input.len() <= 3 {
            let mut value = [b' '; 3];
            value[..input.len()].copy_from_slice(input.as_bytes());
            Ok(RegionSubtag(value))
        } else {
            Err(())
        }
    }
}
pub const LANGUAGES: [LanguageSubtag; 8212] = [
    LanguageSubtag::new(['a', 'a', ' ']),
    LanguageSubtag::new(['a', 'a', 'a']),
    LanguageSubtag::new(['a', 'a', 'b']),
    LanguageSubtag::new(['a', 'a', 'c']),
    LanguageSubtag::new(['a', 'a', 'd']),
    LanguageSubtag::new(['a', 'a', 'e']),
    LanguageSubtag::new(['a', 'a', 'f']),
    LanguageSubtag::new(['a', 'a', 'g']),
    LanguageSubtag::new(['a', 'a', 'h']),
    LanguageSubtag::new(['a', 'a', 'i']),
    LanguageSubtag::new(['a', 'a', 'k']),
    LanguageSubtag::new(['a', 'a', 'l']),
    LanguageSubtag::new(['a', 'a', 'm']),
    LanguageSubtag::new(['a', 'a', 'n']),
    LanguageSubtag::new(['a', 'a', 'o']),
    LanguageSubtag::new(['a', 'a', 'p']),
    LanguageSubtag::new(['a', 'a', 'q']),
    LanguageSubtag::new(['a', 'a', 's']),
    LanguageSubtag::new(['a', 'a', 't']),
    LanguageSubtag::new(['a', 'a', 'u']),
    LanguageSubtag::new(['a', 'a', 'v']),
    LanguageSubtag::new(['a', 'a', 'w']),
    LanguageSubtag::new(['a', 'a', 'x']),
    LanguageSubtag::new(['a', 'a', 'z']),
    LanguageSubtag::new(['a', 'b', ' ']),
    LanguageSubtag::new(['a', 'b', 'a']),
    LanguageSubtag::new(['a', 'b', 'b']),
    LanguageSubtag::new(['a', 'b', 'c']),
    LanguageSubtag::new(['a', 'b', 'd']),
    LanguageSubtag::new(['a', 'b', 'e']),
    LanguageSubtag::new(['a', 'b', 'f']),
    LanguageSubtag::new(['a', 'b', 'g']),
    LanguageSubtag::new(['a', 'b', 'h']),
    LanguageSubtag::new(['a', 'b', 'i']),
    LanguageSubtag::new(['a', 'b', 'j']),
    LanguageSubtag::new(['a', 'b', 'l']),
    LanguageSubtag::new(['a', 'b', 'm']),
    LanguageSubtag::new(['a', 'b', 'n']),
    LanguageSubtag::new(['a', 'b', 'o']),
    LanguageSubtag::new(['a', 'b', 'p']),
    LanguageSubtag::new(['a', 'b', 'q']),
    LanguageSubtag::new(['a', 'b', 'r']),
    LanguageSubtag::new(['a', 'b', 's']),
    LanguageSubtag::new(['a', 'b', 't']),
    LanguageSubtag::new(['a', 'b', 'u']),
    LanguageSubtag::new(['a', 'b', 'v']),
    LanguageSubtag::new(['a', 'b', 'w']),
    LanguageSubtag::new(['a', 'b', 'x']),
    LanguageSubtag::new(['a', 'b', 'y']),
    LanguageSubtag::new(['a', 'b', 'z']),
    LanguageSubtag::new(['a', 'c', 'a']),
    LanguageSubtag::new(['a', 'c', 'b']),
    LanguageSubtag::new(['a', 'c', 'd']),
    LanguageSubtag::new(['a', 'c', 'e']),
    LanguageSubtag::new(['a', 'c', 'f']),
    LanguageSubtag::new(['a', 'c', 'h']),
    LanguageSubtag::new(['a', 'c', 'i']),
    LanguageSubtag::new(['a', 'c', 'k']),
    LanguageSubtag::new(['a', 'c', 'l']),
    LanguageSubtag::new(['a', 'c', 'm']),
    LanguageSubtag::new(['a', 'c', 'n']),
    LanguageSubtag::new(['a', 'c', 'p']),
    LanguageSubtag::new(['a', 'c', 'q']),
    LanguageSubtag::new(['a', 'c', 'r']),
    LanguageSubtag::new(['a', 'c', 's']),
    LanguageSubtag::new(['a', 'c', 't']),
    LanguageSubtag::new(['a', 'c', 'u']),
    LanguageSubtag::new(['a', 'c', 'v']),
    LanguageSubtag::new(['a', 'c', 'w']),
    LanguageSubtag::new(['a', 'c', 'x']),
    LanguageSubtag::new(['a', 'c', 'y']),
    LanguageSubtag::new(['a', 'c', 'z']),
    LanguageSubtag::new(['a', 'd', 'a']),
    LanguageSubtag::new(['a', 'd', 'b']),
    LanguageSubtag::new(['a', 'd', 'd']),
    LanguageSubtag::new(['a', 'd', 'e']),
    LanguageSubtag::new(['a', 'd', 'f']),
    LanguageSubtag::new(['a', 'd', 'g']),
    LanguageSubtag::new(['a', 'd', 'h']),
    LanguageSubtag::new(['a', 'd', 'i']),
    LanguageSubtag::new(['a', 'd', 'j']),
    LanguageSubtag::new(['a', 'd', 'l']),
    LanguageSubtag::new(['a', 'd', 'n']),
    LanguageSubtag::new(['a', 'd', 'o']),
    LanguageSubtag::new(['a', 'd', 'p']),
    LanguageSubtag::new(['a', 'd', 'q']),
    LanguageSubtag::new(['a', 'd', 'r']),
    LanguageSubtag::new(['a', 'd', 's']),
    LanguageSubtag::new(['a', 'd', 't']),
    LanguageSubtag::new(['a', 'd', 'u']),
    LanguageSubtag::new(['a', 'd', 'w']),
    LanguageSubtag::new(['a', 'd', 'x']),
    LanguageSubtag::new(['a', 'd', 'y']),
    LanguageSubtag::new(['a', 'd', 'z']),
    LanguageSubtag::new(['a', 'e', ' ']),
    LanguageSubtag::new(['a', 'e', 'a']),
    LanguageSubtag::new(['a', 'e', 'b']),
    LanguageSubtag::new(['a', 'e', 'c']),
    LanguageSubtag::new(['a', 'e', 'd']),
    LanguageSubtag::new(['a', 'e', 'e']),
    LanguageSubtag::new(['a', 'e', 'k']),
    LanguageSubtag::new(['a', 'e', 'l']),
    LanguageSubtag::new(['a', 'e', 'm']),
    LanguageSubtag::new(['a', 'e', 'n']),
    LanguageSubtag::new(['a', 'e', 'q']),
    LanguageSubtag::new(['a', 'e', 'r']),
    LanguageSubtag::new(['a', 'e', 's']),
    LanguageSubtag::new(['a', 'e', 'u']),
    LanguageSubtag::new(['a', 'e', 'w']),
    LanguageSubtag::new(['a', 'e', 'y']),
    LanguageSubtag::new(['a', 'e', 'z']),
    LanguageSubtag::new(['a', 'f', ' ']),
    LanguageSubtag::new(['a', 'f', 'a']),
    LanguageSubtag::new(['a', 'f', 'b']),
    LanguageSubtag::new(['a', 'f', 'd']),
    LanguageSubtag::new(['a', 'f', 'e']),
    LanguageSubtag::new(['a', 'f', 'g']),
    LanguageSubtag::new(['a', 'f', 'h']),
    LanguageSubtag::new(['a', 'f', 'i']),
    LanguageSubtag::new(['a', 'f', 'k']),
    LanguageSubtag::new(['a', 'f', 'n']),
    LanguageSubtag::new(['a', 'f', 'o']),
    LanguageSubtag::new(['a', 'f', 'p']),
    LanguageSubtag::new(['a', 'f', 's']),
    LanguageSubtag::new(['a', 'f', 't']),
    LanguageSubtag::new(['a', 'f', 'u']),
    LanguageSubtag::new(['a', 'f', 'z']),
    LanguageSubtag::new(['a', 'g', 'a']),
    LanguageSubtag::new(['a', 'g', 'b']),
    LanguageSubtag::new(['a', 'g', 'c']),
    LanguageSubtag::new(['a', 'g', 'd']),
    LanguageSubtag::new(['a', 'g', 'e']),
    LanguageSubtag::new(['a', 'g', 'f']),
    LanguageSubtag::new(['a', 'g', 'g']),
    LanguageSubtag::new(['a', 'g', 'h']),
    LanguageSubtag::new(['a', 'g', 'i']),
    LanguageSubtag::new(['a', 'g', 'j']),
    LanguageSubtag::new(['a', 'g', 'k']),
    LanguageSubtag::new(['a', 'g', 'l']),
    LanguageSubtag::new(['a', 'g', 'm']),
    LanguageSubtag::new(['a', 'g', 'n']),
    LanguageSubtag::new(['a', 'g', 'o']),
    LanguageSubtag::new(['a', 'g', 'p']),
    LanguageSubtag::new(['a', 'g', 'q']),
    LanguageSubtag::new(['a', 'g', 'r']),
    LanguageSubtag::new(['a', 'g', 's']),
    LanguageSubtag::new(['a', 'g', 't']),
    LanguageSubtag::new(['a', 'g', 'u']),
    LanguageSubtag::new(['a', 'g', 'v']),
    LanguageSubtag::new(['a', 'g', 'w']),
    LanguageSubtag::new(['a', 'g', 'x']),
    LanguageSubtag::new(['a', 'g', 'y']),
    LanguageSubtag::new(['a', 'g', 'z']),
    LanguageSubtag::new(['a', 'h', 'a']),
    LanguageSubtag::new(['a', 'h', 'b']),
    LanguageSubtag::new(['a', 'h', 'g']),
    LanguageSubtag::new(['a', 'h', 'h']),
    LanguageSubtag::new(['a', 'h', 'i']),
    LanguageSubtag::new(['a', 'h', 'k']),
    LanguageSubtag::new(['a', 'h', 'l']),
    LanguageSubtag::new(['a', 'h', 'm']),
    LanguageSubtag::new(['a', 'h', 'n']),
    LanguageSubtag::new(['a', 'h', 'o']),
    LanguageSubtag::new(['a', 'h', 'p']),
    LanguageSubtag::new(['a', 'h', 'r']),
    LanguageSubtag::new(['a', 'h', 's']),
    LanguageSubtag::new(['a', 'h', 't']),
    LanguageSubtag::new(['a', 'i', 'a']),
    LanguageSubtag::new(['a', 'i', 'b']),
    LanguageSubtag::new(['a', 'i', 'c']),
    LanguageSubtag::new(['a', 'i', 'd']),
    LanguageSubtag::new(['a', 'i', 'e']),
    LanguageSubtag::new(['a', 'i', 'f']),
    LanguageSubtag::new(['a', 'i', 'g']),
    LanguageSubtag::new(['a', 'i', 'h']),
    LanguageSubtag::new(['a', 'i', 'i']),
    LanguageSubtag::new(['a', 'i', 'j']),
    LanguageSubtag::new(['a', 'i', 'k']),
    LanguageSubtag::new(['a', 'i', 'l']),
    LanguageSubtag::new(['a', 'i', 'm']),
    LanguageSubtag::new(['a', 'i', 'n']),
    LanguageSubtag::new(['a', 'i', 'o']),
    LanguageSubtag::new(['a', 'i', 'p']),
    LanguageSubtag::new(['a', 'i', 'q']),
    LanguageSubtag::new(['a', 'i', 'r']),
    LanguageSubtag::new(['a', 'i', 's']),
    LanguageSubtag::new(['a', 'i', 't']),
    LanguageSubtag::new(['a', 'i', 'w']),
    LanguageSubtag::new(['a', 'i', 'x']),
    LanguageSubtag::new(['a', 'i', 'y']),
    LanguageSubtag::new(['a', 'j', 'a']),
    LanguageSubtag::new(['a', 'j', 'g']),
    LanguageSubtag::new(['a', 'j', 'i']),
    LanguageSubtag::new(['a', 'j', 'n']),
    LanguageSubtag::new(['a', 'j', 'p']),
    LanguageSubtag::new(['a', 'j', 't']),
    LanguageSubtag::new(['a', 'j', 'u']),
    LanguageSubtag::new(['a', 'j', 'w']),
    LanguageSubtag::new(['a', 'j', 'z']),
    LanguageSubtag::new(['a', 'k', ' ']),
    LanguageSubtag::new(['a', 'k', 'b']),
    LanguageSubtag::new(['a', 'k', 'c']),
    LanguageSubtag::new(['a', 'k', 'd']),
    LanguageSubtag::new(['a', 'k', 'e']),
    LanguageSubtag::new(['a', 'k', 'f']),
    LanguageSubtag::new(['a', 'k', 'g']),
    LanguageSubtag::new(['a', 'k', 'h']),
    LanguageSubtag::new(['a', 'k', 'i']),
    LanguageSubtag::new(['a', 'k', 'j']),
    LanguageSubtag::new(['a', 'k', 'k']),
    LanguageSubtag::new(['a', 'k', 'l']),
    LanguageSubtag::new(['a', 'k', 'm']),
    LanguageSubtag::new(['a', 'k', 'o']),
    LanguageSubtag::new(['a', 'k', 'p']),
    LanguageSubtag::new(['a', 'k', 'q']),
    LanguageSubtag::new(['a', 'k', 'r']),
    LanguageSubtag::new(['a', 'k', 's']),
    LanguageSubtag::new(['a', 'k', 't']),
    LanguageSubtag::new(['a', 'k', 'u']),
    LanguageSubtag::new(['a', 'k', 'v']),
    LanguageSubtag::new(['a', 'k', 'w']),
    LanguageSubtag::new(['a', 'k', 'x']),
    LanguageSubtag::new(['a', 'k', 'y']),
    LanguageSubtag::new(['a', 'k', 'z']),
    LanguageSubtag::new(['a', 'l', 'a']),
    LanguageSubtag::new(['a', 'l', 'c']),
    LanguageSubtag::new(['a', 'l', 'd']),
    LanguageSubtag::new(['a', 'l', 'e']),
    LanguageSubtag::new(['a', 'l', 'f']),
    LanguageSubtag::new(['a', 'l', 'g']),
    LanguageSubtag::new(['a', 'l', 'h']),
    LanguageSubtag::new(['a', 'l', 'i']),
    LanguageSubtag::new(['a', 'l', 'j']),
    LanguageSubtag::new(['a', 'l', 'k']),
    LanguageSubtag::new(['a', 'l', 'l']),
    LanguageSubtag::new(['a', 'l', 'm']),
    LanguageSubtag::new(['a', 'l', 'n']),
    LanguageSubtag::new(['a', 'l', 'o']),
    LanguageSubtag::new(['a', 'l', 'p']),
    LanguageSubtag::new(['a', 'l', 'q']),
    LanguageSubtag::new(['a', 'l', 'r']),
    LanguageSubtag::new(['a', 'l', 's']),
    LanguageSubtag::new(['a', 'l', 't']),
    LanguageSubtag::new(['a', 'l', 'u']),
    LanguageSubtag::new(['a', 'l', 'v']),
    LanguageSubtag::new(['a', 'l', 'w']),
    LanguageSubtag::new(['a', 'l', 'x']),
    LanguageSubtag::new(['a', 'l', 'y']),
    LanguageSubtag::new(['a', 'l', 'z']),
    LanguageSubtag::new(['a', 'm', ' ']),
    LanguageSubtag::new(['a', 'm', 'a']),
    LanguageSubtag::new(['a', 'm', 'b']),
    LanguageSubtag::new(['a', 'm', 'c']),
    LanguageSubtag::new(['a', 'm', 'e']),
    LanguageSubtag::new(['a', 'm', 'f']),
    LanguageSubtag::new(['a', 'm', 'g']),
    LanguageSubtag::new(['a', 'm', 'i']),
    LanguageSubtag::new(['a', 'm', 'j']),
    LanguageSubtag::new(['a', 'm', 'k']),
    LanguageSubtag::new(['a', 'm', 'l']),
    LanguageSubtag::new(['a', 'm', 'm']),
    LanguageSubtag::new(['a', 'm', 'n']),
    LanguageSubtag::new(['a', 'm', 'o']),
    LanguageSubtag::new(['a', 'm', 'p']),
    LanguageSubtag::new(['a', 'm', 'q']),
    LanguageSubtag::new(['a', 'm', 'r']),
    LanguageSubtag::new(['a', 'm', 's']),
    LanguageSubtag::new(['a', 'm', 't']),
    LanguageSubtag::new(['a', 'm', 'u']),
    LanguageSubtag::new(['a', 'm', 'v']),
    LanguageSubtag::new(['a', 'm', 'w']),
    LanguageSubtag::new(['a', 'm', 'x']),
    LanguageSubtag::new(['a', 'm', 'y']),
    LanguageSubtag::new(['a', 'm', 'z']),
    LanguageSubtag::new(['a', 'n', ' ']),
    LanguageSubtag::new(['a', 'n', 'a']),
    LanguageSubtag::new(['a', 'n', 'b']),
    LanguageSubtag::new(['a', 'n', 'c']),
    LanguageSubtag::new(['a', 'n', 'd']),
    LanguageSubtag::new(['a', 'n', 'e']),
    LanguageSubtag::new(['a', 'n', 'f']),
    LanguageSubtag::new(['a', 'n', 'g']),
    LanguageSubtag::new(['a', 'n', 'h']),
    LanguageSubtag::new(['a', 'n', 'i']),
    LanguageSubtag::new(['a', 'n', 'j']),
    LanguageSubtag::new(['a', 'n', 'k']),
    LanguageSubtag::new(['a', 'n', 'l']),
    LanguageSubtag::new(['a', 'n', 'm']),
    LanguageSubtag::new(['a', 'n', 'n']),
    LanguageSubtag::new(['a', 'n', 'o']),
    LanguageSubtag::new(['a', 'n', 'p']),
    LanguageSubtag::new(['a', 'n', 'q']),
    LanguageSubtag::new(['a', 'n', 'r']),
    LanguageSubtag::new(['a', 'n', 's']),
    LanguageSubtag::new(['a', 'n', 't']),
    LanguageSubtag::new(['a', 'n', 'u']),
    LanguageSubtag::new(['a', 'n', 'v']),
    LanguageSubtag::new(['a', 'n', 'w']),
    LanguageSubtag::new(['a', 'n', 'x']),
    LanguageSubtag::new(['a', 'n', 'y']),
    LanguageSubtag::new(['a', 'n', 'z']),
    LanguageSubtag::new(['a', 'o', 'a']),
    LanguageSubtag::new(['a', 'o', 'b']),
    LanguageSubtag::new(['a', 'o', 'c']),
    LanguageSubtag::new(['a', 'o', 'd']),
    LanguageSubtag::new(['a', 'o', 'e']),
    LanguageSubtag::new(['a', 'o', 'f']),
    LanguageSubtag::new(['a', 'o', 'g']),
    LanguageSubtag::new(['a', 'o', 'h']),
    LanguageSubtag::new(['a', 'o', 'i']),
    LanguageSubtag::new(['a', 'o', 'j']),
    LanguageSubtag::new(['a', 'o', 'k']),
    LanguageSubtag::new(['a', 'o', 'l']),
    LanguageSubtag::new(['a', 'o', 'm']),
    LanguageSubtag::new(['a', 'o', 'n']),
    LanguageSubtag::new(['a', 'o', 'r']),
    LanguageSubtag::new(['a', 'o', 's']),
    LanguageSubtag::new(['a', 'o', 't']),
    LanguageSubtag::new(['a', 'o', 'u']),
    LanguageSubtag::new(['a', 'o', 'x']),
    LanguageSubtag::new(['a', 'o', 'z']),
    LanguageSubtag::new(['a', 'p', 'a']),
    LanguageSubtag::new(['a', 'p', 'b']),
    LanguageSubtag::new(['a', 'p', 'c']),
    LanguageSubtag::new(['a', 'p', 'd']),
    LanguageSubtag::new(['a', 'p', 'e']),
    LanguageSubtag::new(['a', 'p', 'f']),
    LanguageSubtag::new(['a', 'p', 'g']),
    LanguageSubtag::new(['a', 'p', 'h']),
    LanguageSubtag::new(['a', 'p', 'i']),
    LanguageSubtag::new(['a', 'p', 'j']),
    LanguageSubtag::new(['a', 'p', 'k']),
    LanguageSubtag::new(['a', 'p', 'l']),
    LanguageSubtag::new(['a', 'p', 'm']),
    LanguageSubtag::new(['a', 'p', 'n']),
    LanguageSubtag::new(['a', 'p', 'o']),
    LanguageSubtag::new(['a', 'p', 'p']),
    LanguageSubtag::new(['a', 'p', 'q']),
    LanguageSubtag::new(['a', 'p', 'r']),
    LanguageSubtag::new(['a', 'p', 's']),
    LanguageSubtag::new(['a', 'p', 't']),
    LanguageSubtag::new(['a', 'p', 'u']),
    LanguageSubtag::new(['a', 'p', 'v']),
    LanguageSubtag::new(['a', 'p', 'w']),
    LanguageSubtag::new(['a', 'p', 'x']),
    LanguageSubtag::new(['a', 'p', 'y']),
    LanguageSubtag::new(['a', 'p', 'z']),
    LanguageSubtag::new(['a', 'q', 'a']),
    LanguageSubtag::new(['a', 'q', 'c']),
    LanguageSubtag::new(['a', 'q', 'd']),
    LanguageSubtag::new(['a', 'q', 'g']),
    LanguageSubtag::new(['a', 'q', 'k']),
    LanguageSubtag::new(['a', 'q', 'l']),
    LanguageSubtag::new(['a', 'q', 'm']),
    LanguageSubtag::new(['a', 'q', 'n']),
    LanguageSubtag::new(['a', 'q', 'p']),
    LanguageSubtag::new(['a', 'q', 'r']),
    LanguageSubtag::new(['a', 'q', 't']),
    LanguageSubtag::new(['a', 'q', 'z']),
    LanguageSubtag::new(['a', 'r', ' ']),
    LanguageSubtag::new(['a', 'r', 'b']),
    LanguageSubtag::new(['a', 'r', 'c']),
    LanguageSubtag::new(['a', 'r', 'd']),
    LanguageSubtag::new(['a', 'r', 'e']),
    LanguageSubtag::new(['a', 'r', 'h']),
    LanguageSubtag::new(['a', 'r', 'i']),
    LanguageSubtag::new(['a', 'r', 'j']),
    LanguageSubtag::new(['a', 'r', 'k']),
    LanguageSubtag::new(['a', 'r', 'l']),
    LanguageSubtag::new(['a', 'r', 'n']),
    LanguageSubtag::new(['a', 'r', 'o']),
    LanguageSubtag::new(['a', 'r', 'p']),
    LanguageSubtag::new(['a', 'r', 'q']),
    LanguageSubtag::new(['a', 'r', 'r']),
    LanguageSubtag::new(['a', 'r', 's']),
    LanguageSubtag::new(['a', 'r', 't']),
    LanguageSubtag::new(['a', 'r', 'u']),
    LanguageSubtag::new(['a', 'r', 'v']),
    LanguageSubtag::new(['a', 'r', 'w']),
    LanguageSubtag::new(['a', 'r', 'x']),
    LanguageSubtag::new(['a', 'r', 'y']),
    LanguageSubtag::new(['a', 'r', 'z']),
    LanguageSubtag::new(['a', 's', ' ']),
    LanguageSubtag::new(['a', 's', 'a']),
    LanguageSubtag::new(['a', 's', 'b']),
    LanguageSubtag::new(['a', 's', 'c']),
    LanguageSubtag::new(['a', 's', 'd']),
    LanguageSubtag::new(['a', 's', 'e']),
    LanguageSubtag::new(['a', 's', 'f']),
    LanguageSubtag::new(['a', 's', 'g']),
    LanguageSubtag::new(['a', 's', 'h']),
    LanguageSubtag::new(['a', 's', 'i']),
    LanguageSubtag::new(['a', 's', 'j']),
    LanguageSubtag::new(['a', 's', 'k']),
    LanguageSubtag::new(['a', 's', 'l']),
    LanguageSubtag::new(['a', 's', 'n']),
    LanguageSubtag::new(['a', 's', 'o']),
    LanguageSubtag::new(['a', 's', 'p']),
    LanguageSubtag::new(['a', 's', 'q']),
    LanguageSubtag::new(['a', 's', 'r']),
    LanguageSubtag::new(['a', 's', 's']),
    LanguageSubtag::new(['a', 's', 't']),
    LanguageSubtag::new(['a', 's', 'u']),
    LanguageSubtag::new(['a', 's', 'v']),
    LanguageSubtag::new(['a', 's', 'w']),
    LanguageSubtag::new(['a', 's', 'x']),
    LanguageSubtag::new(['a', 's', 'y']),
    LanguageSubtag::new(['a', 's', 'z']),
    LanguageSubtag::new(['a', 't', 'a']),
    LanguageSubtag::new(['a', 't', 'b']),
    LanguageSubtag::new(['a', 't', 'c']),
    LanguageSubtag::new(['a', 't', 'd']),
    LanguageSubtag::new(['a', 't', 'e']),
    LanguageSubtag::new(['a', 't', 'g']),
    LanguageSubtag::new(['a', 't', 'h']),
    LanguageSubtag::new(['a', 't', 'i']),
    LanguageSubtag::new(['a', 't', 'j']),
    LanguageSubtag::new(['a', 't', 'k']),
    LanguageSubtag::new(['a', 't', 'l']),
    LanguageSubtag::new(['a', 't', 'm']),
    LanguageSubtag::new(['a', 't', 'n']),
    LanguageSubtag::new(['a', 't', 'o']),
    LanguageSubtag::new(['a', 't', 'p']),
    LanguageSubtag::new(['a', 't', 'q']),
    LanguageSubtag::new(['a', 't', 'r']),
    LanguageSubtag::new(['a', 't', 's']),
    LanguageSubtag::new(['a', 't', 't']),
    LanguageSubtag::new(['a', 't', 'u']),
    LanguageSubtag::new(['a', 't', 'v']),
    LanguageSubtag::new(['a', 't', 'w']),
    LanguageSubtag::new(['a', 't', 'x']),
    LanguageSubtag::new(['a', 't', 'y']),
    LanguageSubtag::new(['a', 't', 'z']),
    LanguageSubtag::new(['a', 'u', 'a']),
    LanguageSubtag::new(['a', 'u', 'b']),
    LanguageSubtag::new(['a', 'u', 'c']),
    LanguageSubtag::new(['a', 'u', 'd']),
    LanguageSubtag::new(['a', 'u', 'e']),
    LanguageSubtag::new(['a', 'u', 'f']),
    LanguageSubtag::new(['a', 'u', 'g']),
    LanguageSubtag::new(['a', 'u', 'h']),
    LanguageSubtag::new(['a', 'u', 'i']),
    LanguageSubtag::new(['a', 'u', 'j']),
    LanguageSubtag::new(['a', 'u', 'k']),
    LanguageSubtag::new(['a', 'u', 'l']),
    LanguageSubtag::new(['a', 'u', 'm']),
    LanguageSubtag::new(['a', 'u', 'n']),
    LanguageSubtag::new(['a', 'u', 'o']),
    LanguageSubtag::new(['a', 'u', 'p']),
    LanguageSubtag::new(['a', 'u', 'q']),
    LanguageSubtag::new(['a', 'u', 'r']),
    LanguageSubtag::new(['a', 'u', 's']),
    LanguageSubtag::new(['a', 'u', 't']),
    LanguageSubtag::new(['a', 'u', 'u']),
    LanguageSubtag::new(['a', 'u', 'w']),
    LanguageSubtag::new(['a', 'u', 'x']),
    LanguageSubtag::new(['a', 'u', 'y']),
    LanguageSubtag::new(['a', 'u', 'z']),
    LanguageSubtag::new(['a', 'v', ' ']),
    LanguageSubtag::new(['a', 'v', 'b']),
    LanguageSubtag::new(['a', 'v', 'd']),
    LanguageSubtag::new(['a', 'v', 'i']),
    LanguageSubtag::new(['a', 'v', 'k']),
    LanguageSubtag::new(['a', 'v', 'l']),
    LanguageSubtag::new(['a', 'v', 'm']),
    LanguageSubtag::new(['a', 'v', 'n']),
    LanguageSubtag::new(['a', 'v', 'o']),
    LanguageSubtag::new(['a', 'v', 's']),
    LanguageSubtag::new(['a', 'v', 't']),
    LanguageSubtag::new(['a', 'v', 'u']),
    LanguageSubtag::new(['a', 'v', 'v']),
    LanguageSubtag::new(['a', 'w', 'a']),
    LanguageSubtag::new(['a', 'w', 'b']),
    LanguageSubtag::new(['a', 'w', 'c']),
    LanguageSubtag::new(['a', 'w', 'd']),
    LanguageSubtag::new(['a', 'w', 'e']),
    LanguageSubtag::new(['a', 'w', 'g']),
    LanguageSubtag::new(['a', 'w', 'h']),
    LanguageSubtag::new(['a', 'w', 'i']),
    LanguageSubtag::new(['a', 'w', 'k']),
    LanguageSubtag::new(['a', 'w', 'm']),
    LanguageSubtag::new(['a', 'w', 'n']),
    LanguageSubtag::new(['a', 'w', 'o']),
    LanguageSubtag::new(['a', 'w', 'r']),
    LanguageSubtag::new(['a', 'w', 's']),
    LanguageSubtag::new(['a', 'w', 't']),
    LanguageSubtag::new(['a', 'w', 'u']),
    LanguageSubtag::new(['a', 'w', 'v']),
    LanguageSubtag::new(['a', 'w', 'w']),
    LanguageSubtag::new(['a', 'w', 'x']),
    LanguageSubtag::new(['a', 'w', 'y']),
    LanguageSubtag::new(['a', 'x', 'b']),
    LanguageSubtag::new(['a', 'x', 'e']),
    LanguageSubtag::new(['a', 'x', 'g']),
    LanguageSubtag::new(['a', 'x', 'k']),
    LanguageSubtag::new(['a', 'x', 'l']),
    LanguageSubtag::new(['a', 'x', 'm']),
    LanguageSubtag::new(['a', 'x', 'x']),
    LanguageSubtag::new(['a', 'y', ' ']),
    LanguageSubtag::new(['a', 'y', 'a']),
    LanguageSubtag::new(['a', 'y', 'b']),
    LanguageSubtag::new(['a', 'y', 'c']),
    LanguageSubtag::new(['a', 'y', 'd']),
    LanguageSubtag::new(['a', 'y', 'e']),
    LanguageSubtag::new(['a', 'y', 'g']),
    LanguageSubtag::new(['a', 'y', 'h']),
    LanguageSubtag::new(['a', 'y', 'i']),
    LanguageSubtag::new(['a', 'y', 'k']),
    LanguageSubtag::new(['a', 'y', 'l']),
    LanguageSubtag::new(['a', 'y', 'n']),
    LanguageSubtag::new(['a', 'y', 'o']),
    LanguageSubtag::new(['a', 'y', 'p']),
    LanguageSubtag::new(['a', 'y', 'q']),
    LanguageSubtag::new(['a', 'y', 'r']),
    LanguageSubtag::new(['a', 'y', 's']),
    LanguageSubtag::new(['a', 'y', 't']),
    LanguageSubtag::new(['a', 'y', 'u']),
    LanguageSubtag::new(['a', 'y', 'x']),
    LanguageSubtag::new(['a', 'y', 'y']),
    LanguageSubtag::new(['a', 'y', 'z']),
    LanguageSubtag::new(['a', 'z', ' ']),
    LanguageSubtag::new(['a', 'z', 'a']),
    LanguageSubtag::new(['a', 'z', 'b']),
    LanguageSubtag::new(['a', 'z', 'c']),
    LanguageSubtag::new(['a', 'z', 'd']),
    LanguageSubtag::new(['a', 'z', 'g']),
    LanguageSubtag::new(['a', 'z', 'j']),
    LanguageSubtag::new(['a', 'z', 'm']),
    LanguageSubtag::new(['a', 'z', 'n']),
    LanguageSubtag::new(['a', 'z', 'o']),
    LanguageSubtag::new(['a', 'z', 't']),
    LanguageSubtag::new(['a', 'z', 'z']),
    LanguageSubtag::new(['b', 'a', ' ']),
    LanguageSubtag::new(['b', 'a', 'a']),
    LanguageSubtag::new(['b', 'a', 'b']),
    LanguageSubtag::new(['b', 'a', 'c']),
    LanguageSubtag::new(['b', 'a', 'd']),
    LanguageSubtag::new(['b', 'a', 'e']),
    LanguageSubtag::new(['b', 'a', 'f']),
    LanguageSubtag::new(['b', 'a', 'g']),
    LanguageSubtag::new(['b', 'a', 'h']),
    LanguageSubtag::new(['b', 'a', 'i']),
    LanguageSubtag::new(['b', 'a', 'j']),
    LanguageSubtag::new(['b', 'a', 'l']),
    LanguageSubtag::new(['b', 'a', 'n']),
    LanguageSubtag::new(['b', 'a', 'o']),
    LanguageSubtag::new(['b', 'a', 'p']),
    LanguageSubtag::new(['b', 'a', 'r']),
    LanguageSubtag::new(['b', 'a', 's']),
    LanguageSubtag::new(['b', 'a', 't']),
    LanguageSubtag::new(['b', 'a', 'u']),
    LanguageSubtag::new(['b', 'a', 'v']),
    LanguageSubtag::new(['b', 'a', 'w']),
    LanguageSubtag::new(['b', 'a', 'x']),
    LanguageSubtag::new(['b', 'a', 'y']),
    LanguageSubtag::new(['b', 'a', 'z']),
    LanguageSubtag::new(['b', 'b', 'a']),
    LanguageSubtag::new(['b', 'b', 'b']),
    LanguageSubtag::new(['b', 'b', 'c']),
    LanguageSubtag::new(['b', 'b', 'd']),
    LanguageSubtag::new(['b', 'b', 'e']),
    LanguageSubtag::new(['b', 'b', 'f']),
    LanguageSubtag::new(['b', 'b', 'g']),
    LanguageSubtag::new(['b', 'b', 'h']),
    LanguageSubtag::new(['b', 'b', 'i']),
    LanguageSubtag::new(['b', 'b', 'j']),
    LanguageSubtag::new(['b', 'b', 'k']),
    LanguageSubtag::new(['b', 'b', 'l']),
    LanguageSubtag::new(['b', 'b', 'm']),
    LanguageSubtag::new(['b', 'b', 'n']),
    LanguageSubtag::new(['b', 'b', 'o']),
    LanguageSubtag::new(['b', 'b', 'p']),
    LanguageSubtag::new(['b', 'b', 'q']),
    LanguageSubtag::new(['b', 'b', 'r']),
    LanguageSubtag::new(['b', 'b', 's']),
    LanguageSubtag::new(['b', 'b', 't']),
    LanguageSubtag::new(['b', 'b', 'u']),
    LanguageSubtag::new(['b', 'b', 'v']),
    LanguageSubtag::new(['b', 'b', 'w']),
    LanguageSubtag::new(['b', 'b', 'x']),
    LanguageSubtag::new(['b', 'b', 'y']),
    LanguageSubtag::new(['b', 'b', 'z']),
    LanguageSubtag::new(['b', 'c', 'a']),
    LanguageSubtag::new(['b', 'c', 'b']),
    LanguageSubtag::new(['b', 'c', 'c']),
    LanguageSubtag::new(['b', 'c', 'd']),
    LanguageSubtag::new(['b', 'c', 'e']),
    LanguageSubtag::new(['b', 'c', 'f']),
    LanguageSubtag::new(['b', 'c', 'g']),
    LanguageSubtag::new(['b', 'c', 'h']),
    LanguageSubtag::new(['b', 'c', 'i']),
    LanguageSubtag::new(['b', 'c', 'j']),
    LanguageSubtag::new(['b', 'c', 'k']),
    LanguageSubtag::new(['b', 'c', 'l']),
    LanguageSubtag::new(['b', 'c', 'm']),
    LanguageSubtag::new(['b', 'c', 'n']),
    LanguageSubtag::new(['b', 'c', 'o']),
    LanguageSubtag::new(['b', 'c', 'p']),
    LanguageSubtag::new(['b', 'c', 'q']),
    LanguageSubtag::new(['b', 'c', 'r']),
    LanguageSubtag::new(['b', 'c', 's']),
    LanguageSubtag::new(['b', 'c', 't']),
    LanguageSubtag::new(['b', 'c', 'u']),
    LanguageSubtag::new(['b', 'c', 'v']),
    LanguageSubtag::new(['b', 'c', 'w']),
    LanguageSubtag::new(['b', 'c', 'y']),
    LanguageSubtag::new(['b', 'c', 'z']),
    LanguageSubtag::new(['b', 'd', 'a']),
    LanguageSubtag::new(['b', 'd', 'b']),
    LanguageSubtag::new(['b', 'd', 'c']),
    LanguageSubtag::new(['b', 'd', 'd']),
    LanguageSubtag::new(['b', 'd', 'e']),
    LanguageSubtag::new(['b', 'd', 'f']),
    LanguageSubtag::new(['b', 'd', 'g']),
    LanguageSubtag::new(['b', 'd', 'h']),
    LanguageSubtag::new(['b', 'd', 'i']),
    LanguageSubtag::new(['b', 'd', 'j']),
    LanguageSubtag::new(['b', 'd', 'k']),
    LanguageSubtag::new(['b', 'd', 'l']),
    LanguageSubtag::new(['b', 'd', 'm']),
    LanguageSubtag::new(['b', 'd', 'n']),
    LanguageSubtag::new(['b', 'd', 'o']),
    LanguageSubtag::new(['b', 'd', 'p']),
    LanguageSubtag::new(['b', 'd', 'q']),
    LanguageSubtag::new(['b', 'd', 'r']),
    LanguageSubtag::new(['b', 'd', 's']),
    LanguageSubtag::new(['b', 'd', 't']),
    LanguageSubtag::new(['b', 'd', 'u']),
    LanguageSubtag::new(['b', 'd', 'v']),
    LanguageSubtag::new(['b', 'd', 'w']),
    LanguageSubtag::new(['b', 'd', 'x']),
    LanguageSubtag::new(['b', 'd', 'y']),
    LanguageSubtag::new(['b', 'd', 'z']),
    LanguageSubtag::new(['b', 'e', ' ']),
    LanguageSubtag::new(['b', 'e', 'a']),
    LanguageSubtag::new(['b', 'e', 'b']),
    LanguageSubtag::new(['b', 'e', 'c']),
    LanguageSubtag::new(['b', 'e', 'd']),
    LanguageSubtag::new(['b', 'e', 'e']),
    LanguageSubtag::new(['b', 'e', 'f']),
    LanguageSubtag::new(['b', 'e', 'g']),
    LanguageSubtag::new(['b', 'e', 'h']),
    LanguageSubtag::new(['b', 'e', 'i']),
    LanguageSubtag::new(['b', 'e', 'j']),
    LanguageSubtag::new(['b', 'e', 'k']),
    LanguageSubtag::new(['b', 'e', 'm']),
    LanguageSubtag::new(['b', 'e', 'o']),
    LanguageSubtag::new(['b', 'e', 'p']),
    LanguageSubtag::new(['b', 'e', 'q']),
    LanguageSubtag::new(['b', 'e', 'r']),
    LanguageSubtag::new(['b', 'e', 's']),
    LanguageSubtag::new(['b', 'e', 't']),
    LanguageSubtag::new(['b', 'e', 'u']),
    LanguageSubtag::new(['b', 'e', 'v']),
    LanguageSubtag::new(['b', 'e', 'w']),
    LanguageSubtag::new(['b', 'e', 'x']),
    LanguageSubtag::new(['b', 'e', 'y']),
    LanguageSubtag::new(['b', 'e', 'z']),
    LanguageSubtag::new(['b', 'f', 'a']),
    LanguageSubtag::new(['b', 'f', 'b']),
    LanguageSubtag::new(['b', 'f', 'c']),
    LanguageSubtag::new(['b', 'f', 'd']),
    LanguageSubtag::new(['b', 'f', 'e']),
    LanguageSubtag::new(['b', 'f', 'f']),
    LanguageSubtag::new(['b', 'f', 'g']),
    LanguageSubtag::new(['b', 'f', 'h']),
    LanguageSubtag::new(['b', 'f', 'i']),
    LanguageSubtag::new(['b', 'f', 'j']),
    LanguageSubtag::new(['b', 'f', 'k']),
    LanguageSubtag::new(['b', 'f', 'l']),
    LanguageSubtag::new(['b', 'f', 'm']),
    LanguageSubtag::new(['b', 'f', 'n']),
    LanguageSubtag::new(['b', 'f', 'o']),
    LanguageSubtag::new(['b', 'f', 'p']),
    LanguageSubtag::new(['b', 'f', 'q']),
    LanguageSubtag::new(['b', 'f', 'r']),
    LanguageSubtag::new(['b', 'f', 's']),
    LanguageSubtag::new(['b', 'f', 't']),
    LanguageSubtag::new(['b', 'f', 'u']),
    LanguageSubtag::new(['b', 'f', 'w']),
    LanguageSubtag::new(['b', 'f', 'x']),
    LanguageSubtag::new(['b', 'f', 'y']),
    LanguageSubtag::new(['b', 'f', 'z']),
    LanguageSubtag::new(['b', 'g', ' ']),
    LanguageSubtag::new(['b', 'g', 'a']),
    LanguageSubtag::new(['b', 'g', 'b']),
    LanguageSubtag::new(['b', 'g', 'c']),
    LanguageSubtag::new(['b', 'g', 'd']),
    LanguageSubtag::new(['b', 'g', 'e']),
    LanguageSubtag::new(['b', 'g', 'f']),
    LanguageSubtag::new(['b', 'g', 'g']),
    LanguageSubtag::new(['b', 'g', 'i']),
    LanguageSubtag::new(['b', 'g', 'j']),
    LanguageSubtag::new(['b', 'g', 'k']),
    LanguageSubtag::new(['b', 'g', 'l']),
    LanguageSubtag::new(['b', 'g', 'm']),
    LanguageSubtag::new(['b', 'g', 'n']),
    LanguageSubtag::new(['b', 'g', 'o']),
    LanguageSubtag::new(['b', 'g', 'p']),
    LanguageSubtag::new(['b', 'g', 'q']),
    LanguageSubtag::new(['b', 'g', 'r']),
    LanguageSubtag::new(['b', 'g', 's']),
    LanguageSubtag::new(['b', 'g', 't']),
    LanguageSubtag::new(['b', 'g', 'u']),
    LanguageSubtag::new(['b', 'g', 'v']),
    LanguageSubtag::new(['b', 'g', 'w']),
    LanguageSubtag::new(['b', 'g', 'x']),
    LanguageSubtag::new(['b', 'g', 'y']),
    LanguageSubtag::new(['b', 'g', 'z']),
    LanguageSubtag::new(['b', 'h', ' ']),
    LanguageSubtag::new(['b', 'h', 'a']),
    LanguageSubtag::new(['b', 'h', 'b']),
    LanguageSubtag::new(['b', 'h', 'c']),
    LanguageSubtag::new(['b', 'h', 'd']),
    LanguageSubtag::new(['b', 'h', 'e']),
    LanguageSubtag::new(['b', 'h', 'f']),
    LanguageSubtag::new(['b', 'h', 'g']),
    LanguageSubtag::new(['b', 'h', 'h']),
    LanguageSubtag::new(['b', 'h', 'i']),
    LanguageSubtag::new(['b', 'h', 'j']),
    LanguageSubtag::new(['b', 'h', 'k']),
    LanguageSubtag::new(['b', 'h', 'l']),
    LanguageSubtag::new(['b', 'h', 'm']),
    LanguageSubtag::new(['b', 'h', 'n']),
    LanguageSubtag::new(['b', 'h', 'o']),
    LanguageSubtag::new(['b', 'h', 'p']),
    LanguageSubtag::new(['b', 'h', 'q']),
    LanguageSubtag::new(['b', 'h', 'r']),
    LanguageSubtag::new(['b', 'h', 's']),
    LanguageSubtag::new(['b', 'h', 't']),
    LanguageSubtag::new(['b', 'h', 'u']),
    LanguageSubtag::new(['b', 'h', 'v']),
    LanguageSubtag::new(['b', 'h', 'w']),
    LanguageSubtag::new(['b', 'h', 'x']),
    LanguageSubtag::new(['b', 'h', 'y']),
    LanguageSubtag::new(['b', 'h', 'z']),
    LanguageSubtag::new(['b', 'i', ' ']),
    LanguageSubtag::new(['b', 'i', 'a']),
    LanguageSubtag::new(['b', 'i', 'b']),
    LanguageSubtag::new(['b', 'i', 'c']),
    LanguageSubtag::new(['b', 'i', 'd']),
    LanguageSubtag::new(['b', 'i', 'e']),
    LanguageSubtag::new(['b', 'i', 'f']),
    LanguageSubtag::new(['b', 'i', 'g']),
    LanguageSubtag::new(['b', 'i', 'j']),
    LanguageSubtag::new(['b', 'i', 'k']),
    LanguageSubtag::new(['b', 'i', 'l']),
    LanguageSubtag::new(['b', 'i', 'm']),
    LanguageSubtag::new(['b', 'i', 'n']),
    LanguageSubtag::new(['b', 'i', 'o']),
    LanguageSubtag::new(['b', 'i', 'p']),
    LanguageSubtag::new(['b', 'i', 'q']),
    LanguageSubtag::new(['b', 'i', 'r']),
    LanguageSubtag::new(['b', 'i', 't']),
    LanguageSubtag::new(['b', 'i', 'u']),
    LanguageSubtag::new(['b', 'i', 'v']),
    LanguageSubtag::new(['b', 'i', 'w']),
    LanguageSubtag::new(['b', 'i', 'x']),
    LanguageSubtag::new(['b', 'i', 'y']),
    LanguageSubtag::new(['b', 'i', 'z']),
    LanguageSubtag::new(['b', 'j', 'a']),
    LanguageSubtag::new(['b', 'j', 'b']),
    LanguageSubtag::new(['b', 'j', 'c']),
    LanguageSubtag::new(['b', 'j', 'd']),
    LanguageSubtag::new(['b', 'j', 'e']),
    LanguageSubtag::new(['b', 'j', 'f']),
    LanguageSubtag::new(['b', 'j', 'g']),
    LanguageSubtag::new(['b', 'j', 'h']),
    LanguageSubtag::new(['b', 'j', 'i']),
    LanguageSubtag::new(['b', 'j', 'j']),
    LanguageSubtag::new(['b', 'j', 'k']),
    LanguageSubtag::new(['b', 'j', 'l']),
    LanguageSubtag::new(['b', 'j', 'm']),
    LanguageSubtag::new(['b', 'j', 'n']),
    LanguageSubtag::new(['b', 'j', 'o']),
    LanguageSubtag::new(['b', 'j', 'p']),
    LanguageSubtag::new(['b', 'j', 'q']),
    LanguageSubtag::new(['b', 'j', 'r']),
    LanguageSubtag::new(['b', 'j', 's']),
    LanguageSubtag::new(['b', 'j', 't']),
    LanguageSubtag::new(['b', 'j', 'u']),
    LanguageSubtag::new(['b', 'j', 'v']),
    LanguageSubtag::new(['b', 'j', 'w']),
    LanguageSubtag::new(['b', 'j', 'x']),
    LanguageSubtag::new(['b', 'j', 'y']),
    LanguageSubtag::new(['b', 'j', 'z']),
    LanguageSubtag::new(['b', 'k', 'a']),
    LanguageSubtag::new(['b', 'k', 'b']),
    LanguageSubtag::new(['b', 'k', 'c']),
    LanguageSubtag::new(['b', 'k', 'd']),
    LanguageSubtag::new(['b', 'k', 'f']),
    LanguageSubtag::new(['b', 'k', 'g']),
    LanguageSubtag::new(['b', 'k', 'h']),
    LanguageSubtag::new(['b', 'k', 'i']),
    LanguageSubtag::new(['b', 'k', 'j']),
    LanguageSubtag::new(['b', 'k', 'k']),
    LanguageSubtag::new(['b', 'k', 'l']),
    LanguageSubtag::new(['b', 'k', 'm']),
    LanguageSubtag::new(['b', 'k', 'n']),
    LanguageSubtag::new(['b', 'k', 'o']),
    LanguageSubtag::new(['b', 'k', 'p']),
    LanguageSubtag::new(['b', 'k', 'q']),
    LanguageSubtag::new(['b', 'k', 'r']),
    LanguageSubtag::new(['b', 'k', 's']),
    LanguageSubtag::new(['b', 'k', 't']),
    LanguageSubtag::new(['b', 'k', 'u']),
    LanguageSubtag::new(['b', 'k', 'v']),
    LanguageSubtag::new(['b', 'k', 'w']),
    LanguageSubtag::new(['b', 'k', 'x']),
    LanguageSubtag::new(['b', 'k', 'y']),
    LanguageSubtag::new(['b', 'k', 'z']),
    LanguageSubtag::new(['b', 'l', 'a']),
    LanguageSubtag::new(['b', 'l', 'b']),
    LanguageSubtag::new(['b', 'l', 'c']),
    LanguageSubtag::new(['b', 'l', 'd']),
    LanguageSubtag::new(['b', 'l', 'e']),
    LanguageSubtag::new(['b', 'l', 'f']),
    LanguageSubtag::new(['b', 'l', 'g']),
    LanguageSubtag::new(['b', 'l', 'h']),
    LanguageSubtag::new(['b', 'l', 'i']),
    LanguageSubtag::new(['b', 'l', 'j']),
    LanguageSubtag::new(['b', 'l', 'k']),
    LanguageSubtag::new(['b', 'l', 'l']),
    LanguageSubtag::new(['b', 'l', 'm']),
    LanguageSubtag::new(['b', 'l', 'n']),
    LanguageSubtag::new(['b', 'l', 'o']),
    LanguageSubtag::new(['b', 'l', 'p']),
    LanguageSubtag::new(['b', 'l', 'q']),
    LanguageSubtag::new(['b', 'l', 'r']),
    LanguageSubtag::new(['b', 'l', 's']),
    LanguageSubtag::new(['b', 'l', 't']),
    LanguageSubtag::new(['b', 'l', 'v']),
    LanguageSubtag::new(['b', 'l', 'w']),
    LanguageSubtag::new(['b', 'l', 'x']),
    LanguageSubtag::new(['b', 'l', 'y']),
    LanguageSubtag::new(['b', 'l', 'z']),
    LanguageSubtag::new(['b', 'm', ' ']),
    LanguageSubtag::new(['b', 'm', 'a']),
    LanguageSubtag::new(['b', 'm', 'b']),
    LanguageSubtag::new(['b', 'm', 'c']),
    LanguageSubtag::new(['b', 'm', 'd']),
    LanguageSubtag::new(['b', 'm', 'e']),
    LanguageSubtag::new(['b', 'm', 'f']),
    LanguageSubtag::new(['b', 'm', 'g']),
    LanguageSubtag::new(['b', 'm', 'h']),
    LanguageSubtag::new(['b', 'm', 'i']),
    LanguageSubtag::new(['b', 'm', 'j']),
    LanguageSubtag::new(['b', 'm', 'k']),
    LanguageSubtag::new(['b', 'm', 'l']),
    LanguageSubtag::new(['b', 'm', 'm']),
    LanguageSubtag::new(['b', 'm', 'n']),
    LanguageSubtag::new(['b', 'm', 'o']),
    LanguageSubtag::new(['b', 'm', 'p']),
    LanguageSubtag::new(['b', 'm', 'q']),
    LanguageSubtag::new(['b', 'm', 'r']),
    LanguageSubtag::new(['b', 'm', 's']),
    LanguageSubtag::new(['b', 'm', 't']),
    LanguageSubtag::new(['b', 'm', 'u']),
    LanguageSubtag::new(['b', 'm', 'v']),
    LanguageSubtag::new(['b', 'm', 'w']),
    LanguageSubtag::new(['b', 'm', 'x']),
    LanguageSubtag::new(['b', 'm', 'y']),
    LanguageSubtag::new(['b', 'm', 'z']),
    LanguageSubtag::new(['b', 'n', ' ']),
    LanguageSubtag::new(['b', 'n', 'a']),
    LanguageSubtag::new(['b', 'n', 'b']),
    LanguageSubtag::new(['b', 'n', 'c']),
    LanguageSubtag::new(['b', 'n', 'd']),
    LanguageSubtag::new(['b', 'n', 'e']),
    LanguageSubtag::new(['b', 'n', 'f']),
    LanguageSubtag::new(['b', 'n', 'g']),
    LanguageSubtag::new(['b', 'n', 'i']),
    LanguageSubtag::new(['b', 'n', 'j']),
    LanguageSubtag::new(['b', 'n', 'k']),
    LanguageSubtag::new(['b', 'n', 'l']),
    LanguageSubtag::new(['b', 'n', 'm']),
    LanguageSubtag::new(['b', 'n', 'n']),
    LanguageSubtag::new(['b', 'n', 'o']),
    LanguageSubtag::new(['b', 'n', 'p']),
    LanguageSubtag::new(['b', 'n', 'q']),
    LanguageSubtag::new(['b', 'n', 'r']),
    LanguageSubtag::new(['b', 'n', 's']),
    LanguageSubtag::new(['b', 'n', 't']),
    LanguageSubtag::new(['b', 'n', 'u']),
    LanguageSubtag::new(['b', 'n', 'v']),
    LanguageSubtag::new(['b', 'n', 'w']),
    LanguageSubtag::new(['b', 'n', 'x']),
    LanguageSubtag::new(['b', 'n', 'y']),
    LanguageSubtag::new(['b', 'n', 'z']),
    LanguageSubtag::new(['b', 'o', ' ']),
    LanguageSubtag::new(['b', 'o', 'a']),
    LanguageSubtag::new(['b', 'o', 'b']),
    LanguageSubtag::new(['b', 'o', 'e']),
    LanguageSubtag::new(['b', 'o', 'f']),
    LanguageSubtag::new(['b', 'o', 'g']),
    LanguageSubtag::new(['b', 'o', 'h']),
    LanguageSubtag::new(['b', 'o', 'i']),
    LanguageSubtag::new(['b', 'o', 'j']),
    LanguageSubtag::new(['b', 'o', 'k']),
    LanguageSubtag::new(['b', 'o', 'l']),
    LanguageSubtag::new(['b', 'o', 'm']),
    LanguageSubtag::new(['b', 'o', 'n']),
    LanguageSubtag::new(['b', 'o', 'o']),
    LanguageSubtag::new(['b', 'o', 'p']),
    LanguageSubtag::new(['b', 'o', 'q']),
    LanguageSubtag::new(['b', 'o', 'r']),
    LanguageSubtag::new(['b', 'o', 't']),
    LanguageSubtag::new(['b', 'o', 'u']),
    LanguageSubtag::new(['b', 'o', 'v']),
    LanguageSubtag::new(['b', 'o', 'w']),
    LanguageSubtag::new(['b', 'o', 'x']),
    LanguageSubtag::new(['b', 'o', 'y']),
    LanguageSubtag::new(['b', 'o', 'z']),
    LanguageSubtag::new(['b', 'p', 'a']),
    LanguageSubtag::new(['b', 'p', 'b']),
    LanguageSubtag::new(['b', 'p', 'd']),
    LanguageSubtag::new(['b', 'p', 'e']),
    LanguageSubtag::new(['b', 'p', 'g']),
    LanguageSubtag::new(['b', 'p', 'h']),
    LanguageSubtag::new(['b', 'p', 'i']),
    LanguageSubtag::new(['b', 'p', 'j']),
    LanguageSubtag::new(['b', 'p', 'k']),
    LanguageSubtag::new(['b', 'p', 'l']),
    LanguageSubtag::new(['b', 'p', 'm']),
    LanguageSubtag::new(['b', 'p', 'n']),
    LanguageSubtag::new(['b', 'p', 'o']),
    LanguageSubtag::new(['b', 'p', 'p']),
    LanguageSubtag::new(['b', 'p', 'q']),
    LanguageSubtag::new(['b', 'p', 'r']),
    LanguageSubtag::new(['b', 'p', 's']),
    LanguageSubtag::new(['b', 'p', 't']),
    LanguageSubtag::new(['b', 'p', 'u']),
    LanguageSubtag::new(['b', 'p', 'v']),
    LanguageSubtag::new(['b', 'p', 'w']),
    LanguageSubtag::new(['b', 'p', 'x']),
    LanguageSubtag::new(['b', 'p', 'y']),
    LanguageSubtag::new(['b', 'p', 'z']),
    LanguageSubtag::new(['b', 'q', 'a']),
    LanguageSubtag::new(['b', 'q', 'b']),
    LanguageSubtag::new(['b', 'q', 'c']),
    LanguageSubtag::new(['b', 'q', 'd']),
    LanguageSubtag::new(['b', 'q', 'f']),
    LanguageSubtag::new(['b', 'q', 'g']),
    LanguageSubtag::new(['b', 'q', 'h']),
    LanguageSubtag::new(['b', 'q', 'i']),
    LanguageSubtag::new(['b', 'q', 'j']),
    LanguageSubtag::new(['b', 'q', 'k']),
    LanguageSubtag::new(['b', 'q', 'l']),
    LanguageSubtag::new(['b', 'q', 'm']),
    LanguageSubtag::new(['b', 'q', 'n']),
    LanguageSubtag::new(['b', 'q', 'o']),
    LanguageSubtag::new(['b', 'q', 'p']),
    LanguageSubtag::new(['b', 'q', 'q']),
    LanguageSubtag::new(['b', 'q', 'r']),
    LanguageSubtag::new(['b', 'q', 's']),
    LanguageSubtag::new(['b', 'q', 't']),
    LanguageSubtag::new(['b', 'q', 'u']),
    LanguageSubtag::new(['b', 'q', 'v']),
    LanguageSubtag::new(['b', 'q', 'w']),
    LanguageSubtag::new(['b', 'q', 'x']),
    LanguageSubtag::new(['b', 'q', 'y']),
    LanguageSubtag::new(['b', 'q', 'z']),
    LanguageSubtag::new(['b', 'r', ' ']),
    LanguageSubtag::new(['b', 'r', 'a']),
    LanguageSubtag::new(['b', 'r', 'b']),
    LanguageSubtag::new(['b', 'r', 'c']),
    LanguageSubtag::new(['b', 'r', 'd']),
    LanguageSubtag::new(['b', 'r', 'f']),
    LanguageSubtag::new(['b', 'r', 'g']),
    LanguageSubtag::new(['b', 'r', 'h']),
    LanguageSubtag::new(['b', 'r', 'i']),
    LanguageSubtag::new(['b', 'r', 'j']),
    LanguageSubtag::new(['b', 'r', 'k']),
    LanguageSubtag::new(['b', 'r', 'l']),
    LanguageSubtag::new(['b', 'r', 'm']),
    LanguageSubtag::new(['b', 'r', 'n']),
    LanguageSubtag::new(['b', 'r', 'o']),
    LanguageSubtag::new(['b', 'r', 'p']),
    LanguageSubtag::new(['b', 'r', 'q']),
    LanguageSubtag::new(['b', 'r', 'r']),
    LanguageSubtag::new(['b', 'r', 's']),
    LanguageSubtag::new(['b', 'r', 't']),
    LanguageSubtag::new(['b', 'r', 'u']),
    LanguageSubtag::new(['b', 'r', 'v']),
    LanguageSubtag::new(['b', 'r', 'w']),
    LanguageSubtag::new(['b', 'r', 'x']),
    LanguageSubtag::new(['b', 'r', 'y']),
    LanguageSubtag::new(['b', 'r', 'z']),
    LanguageSubtag::new(['b', 's', ' ']),
    LanguageSubtag::new(['b', 's', 'a']),
    LanguageSubtag::new(['b', 's', 'b']),
    LanguageSubtag::new(['b', 's', 'c']),
    LanguageSubtag::new(['b', 's', 'e']),
    LanguageSubtag::new(['b', 's', 'f']),
    LanguageSubtag::new(['b', 's', 'g']),
    LanguageSubtag::new(['b', 's', 'h']),
    LanguageSubtag::new(['b', 's', 'i']),
    LanguageSubtag::new(['b', 's', 'j']),
    LanguageSubtag::new(['b', 's', 'k']),
    LanguageSubtag::new(['b', 's', 'l']),
    LanguageSubtag::new(['b', 's', 'm']),
    LanguageSubtag::new(['b', 's', 'n']),
    LanguageSubtag::new(['b', 's', 'o']),
    LanguageSubtag::new(['b', 's', 'p']),
    LanguageSubtag::new(['b', 's', 'q']),
    LanguageSubtag::new(['b', 's', 'r']),
    LanguageSubtag::new(['b', 's', 's']),
    LanguageSubtag::new(['b', 's', 't']),
    LanguageSubtag::new(['b', 's', 'u']),
    LanguageSubtag::new(['b', 's', 'v']),
    LanguageSubtag::new(['b', 's', 'w']),
    LanguageSubtag::new(['b', 's', 'x']),
    LanguageSubtag::new(['b', 's', 'y']),
    LanguageSubtag::new(['b', 't', 'a']),
    LanguageSubtag::new(['b', 't', 'b']),
    LanguageSubtag::new(['b', 't', 'c']),
    LanguageSubtag::new(['b', 't', 'd']),
    LanguageSubtag::new(['b', 't', 'e']),
    LanguageSubtag::new(['b', 't', 'f']),
    LanguageSubtag::new(['b', 't', 'g']),
    LanguageSubtag::new(['b', 't', 'h']),
    LanguageSubtag::new(['b', 't', 'i']),
    LanguageSubtag::new(['b', 't', 'j']),
    LanguageSubtag::new(['b', 't', 'k']),
    LanguageSubtag::new(['b', 't', 'l']),
    LanguageSubtag::new(['b', 't', 'm']),
    LanguageSubtag::new(['b', 't', 'n']),
    LanguageSubtag::new(['b', 't', 'o']),
    LanguageSubtag::new(['b', 't', 'p']),
    LanguageSubtag::new(['b', 't', 'q']),
    LanguageSubtag::new(['b', 't', 'r']),
    LanguageSubtag::new(['b', 't', 's']),
    LanguageSubtag::new(['b', 't', 't']),
    LanguageSubtag::new(['b', 't', 'u']),
    LanguageSubtag::new(['b', 't', 'v']),
    LanguageSubtag::new(['b', 't', 'w']),
    LanguageSubtag::new(['b', 't', 'x']),
    LanguageSubtag::new(['b', 't', 'y']),
    LanguageSubtag::new(['b', 't', 'z']),
    LanguageSubtag::new(['b', 'u', 'a']),
    LanguageSubtag::new(['b', 'u', 'b']),
    LanguageSubtag::new(['b', 'u', 'c']),
    LanguageSubtag::new(['b', 'u', 'd']),
    LanguageSubtag::new(['b', 'u', 'e']),
    LanguageSubtag::new(['b', 'u', 'f']),
    LanguageSubtag::new(['b', 'u', 'g']),
    LanguageSubtag::new(['b', 'u', 'h']),
    LanguageSubtag::new(['b', 'u', 'i']),
    LanguageSubtag::new(['b', 'u', 'j']),
    LanguageSubtag::new(['b', 'u', 'k']),
    LanguageSubtag::new(['b', 'u', 'm']),
    LanguageSubtag::new(['b', 'u', 'n']),
    LanguageSubtag::new(['b', 'u', 'o']),
    LanguageSubtag::new(['b', 'u', 'p']),
    LanguageSubtag::new(['b', 'u', 'q']),
    LanguageSubtag::new(['b', 'u', 's']),
    LanguageSubtag::new(['b', 'u', 't']),
    LanguageSubtag::new(['b', 'u', 'u']),
    LanguageSubtag::new(['b', 'u', 'v']),
    LanguageSubtag::new(['b', 'u', 'w']),
    LanguageSubtag::new(['b', 'u', 'x']),
    LanguageSubtag::new(['b', 'u', 'y']),
    LanguageSubtag::new(['b', 'u', 'z']),
    LanguageSubtag::new(['b', 'v', 'a']),
    LanguageSubtag::new(['b', 'v', 'b']),
    LanguageSubtag::new(['b', 'v', 'c']),
    LanguageSubtag::new(['b', 'v', 'd']),
    LanguageSubtag::new(['b', 'v', 'e']),
    LanguageSubtag::new(['b', 'v', 'f']),
    LanguageSubtag::new(['b', 'v', 'g']),
    LanguageSubtag::new(['b', 'v', 'h']),
    LanguageSubtag::new(['b', 'v', 'i']),
    LanguageSubtag::new(['b', 'v', 'j']),
    LanguageSubtag::new(['b', 'v', 'k']),
    LanguageSubtag::new(['b', 'v', 'l']),
    LanguageSubtag::new(['b', 'v', 'm']),
    LanguageSubtag::new(['b', 'v', 'n']),
    LanguageSubtag::new(['b', 'v', 'o']),
    LanguageSubtag::new(['b', 'v', 'p']),
    LanguageSubtag::new(['b', 'v', 'q']),
    LanguageSubtag::new(['b', 'v', 'r']),
    LanguageSubtag::new(['b', 'v', 't']),
    LanguageSubtag::new(['b', 'v', 'u']),
    LanguageSubtag::new(['b', 'v', 'v']),
    LanguageSubtag::new(['b', 'v', 'w']),
    LanguageSubtag::new(['b', 'v', 'x']),
    LanguageSubtag::new(['b', 'v', 'y']),
    LanguageSubtag::new(['b', 'v', 'z']),
    LanguageSubtag::new(['b', 'w', 'a']),
    LanguageSubtag::new(['b', 'w', 'b']),
    LanguageSubtag::new(['b', 'w', 'c']),
    LanguageSubtag::new(['b', 'w', 'd']),
    LanguageSubtag::new(['b', 'w', 'e']),
    LanguageSubtag::new(['b', 'w', 'f']),
    LanguageSubtag::new(['b', 'w', 'g']),
    LanguageSubtag::new(['b', 'w', 'h']),
    LanguageSubtag::new(['b', 'w', 'i']),
    LanguageSubtag::new(['b', 'w', 'j']),
    LanguageSubtag::new(['b', 'w', 'k']),
    LanguageSubtag::new(['b', 'w', 'l']),
    LanguageSubtag::new(['b', 'w', 'm']),
    LanguageSubtag::new(['b', 'w', 'n']),
    LanguageSubtag::new(['b', 'w', 'o']),
    LanguageSubtag::new(['b', 'w', 'p']),
    LanguageSubtag::new(['b', 'w', 'q']),
    LanguageSubtag::new(['b', 'w', 'r']),
    LanguageSubtag::new(['b', 'w', 's']),
    LanguageSubtag::new(['b', 'w', 't']),
    LanguageSubtag::new(['b', 'w', 'u']),
    LanguageSubtag::new(['b', 'w', 'w']),
    LanguageSubtag::new(['b', 'w', 'x']),
    LanguageSubtag::new(['b', 'w', 'y']),
    LanguageSubtag::new(['b', 'w', 'z']),
    LanguageSubtag::new(['b', 'x', 'a']),
    LanguageSubtag::new(['b', 'x', 'b']),
    LanguageSubtag::new(['b', 'x', 'c']),
    LanguageSubtag::new(['b', 'x', 'd']),
    LanguageSubtag::new(['b', 'x', 'e']),
    LanguageSubtag::new(['b', 'x', 'f']),
    LanguageSubtag::new(['b', 'x', 'g']),
    LanguageSubtag::new(['b', 'x', 'h']),
    LanguageSubtag::new(['b', 'x', 'i']),
    LanguageSubtag::new(['b', 'x', 'j']),
    LanguageSubtag::new(['b', 'x', 'k']),
    LanguageSubtag::new(['b', 'x', 'l']),
    LanguageSubtag::new(['b', 'x', 'm']),
    LanguageSubtag::new(['b', 'x', 'n']),
    LanguageSubtag::new(['b', 'x', 'o']),
    LanguageSubtag::new(['b', 'x', 'p']),
    LanguageSubtag::new(['b', 'x', 'q']),
    LanguageSubtag::new(['b', 'x', 'r']),
    LanguageSubtag::new(['b', 'x', 's']),
    LanguageSubtag::new(['b', 'x', 'u']),
    LanguageSubtag::new(['b', 'x', 'v']),
    LanguageSubtag::new(['b', 'x', 'w']),
    LanguageSubtag::new(['b', 'x', 'x']),
    LanguageSubtag::new(['b', 'x', 'z']),
    LanguageSubtag::new(['b', 'y', 'a']),
    LanguageSubtag::new(['b', 'y', 'b']),
    LanguageSubtag::new(['b', 'y', 'c']),
    LanguageSubtag::new(['b', 'y', 'd']),
    LanguageSubtag::new(['b', 'y', 'e']),
    LanguageSubtag::new(['b', 'y', 'f']),
    LanguageSubtag::new(['b', 'y', 'g']),
    LanguageSubtag::new(['b', 'y', 'h']),
    LanguageSubtag::new(['b', 'y', 'i']),
    LanguageSubtag::new(['b', 'y', 'j']),
    LanguageSubtag::new(['b', 'y', 'k']),
    LanguageSubtag::new(['b', 'y', 'l']),
    LanguageSubtag::new(['b', 'y', 'm']),
    LanguageSubtag::new(['b', 'y', 'n']),
    LanguageSubtag::new(['b', 'y', 'o']),
    LanguageSubtag::new(['b', 'y', 'p']),
    LanguageSubtag::new(['b', 'y', 'q']),
    LanguageSubtag::new(['b', 'y', 'r']),
    LanguageSubtag::new(['b', 'y', 's']),
    LanguageSubtag::new(['b', 'y', 't']),
    LanguageSubtag::new(['b', 'y', 'v']),
    LanguageSubtag::new(['b', 'y', 'w']),
    LanguageSubtag::new(['b', 'y', 'x']),
    LanguageSubtag::new(['b', 'y', 'y']),
    LanguageSubtag::new(['b', 'y', 'z']),
    LanguageSubtag::new(['b', 'z', 'a']),
    LanguageSubtag::new(['b', 'z', 'b']),
    LanguageSubtag::new(['b', 'z', 'c']),
    LanguageSubtag::new(['b', 'z', 'd']),
    LanguageSubtag::new(['b', 'z', 'e']),
    LanguageSubtag::new(['b', 'z', 'f']),
    LanguageSubtag::new(['b', 'z', 'g']),
    LanguageSubtag::new(['b', 'z', 'h']),
    LanguageSubtag::new(['b', 'z', 'i']),
    LanguageSubtag::new(['b', 'z', 'j']),
    LanguageSubtag::new(['b', 'z', 'k']),
    LanguageSubtag::new(['b', 'z', 'l']),
    LanguageSubtag::new(['b', 'z', 'm']),
    LanguageSubtag::new(['b', 'z', 'n']),
    LanguageSubtag::new(['b', 'z', 'o']),
    LanguageSubtag::new(['b', 'z', 'p']),
    LanguageSubtag::new(['b', 'z', 'q']),
    LanguageSubtag::new(['b', 'z', 'r']),
    LanguageSubtag::new(['b', 'z', 's']),
    LanguageSubtag::new(['b', 'z', 't']),
    LanguageSubtag::new(['b', 'z', 'u']),
    LanguageSubtag::new(['b', 'z', 'v']),
    LanguageSubtag::new(['b', 'z', 'w']),
    LanguageSubtag::new(['b', 'z', 'x']),
    LanguageSubtag::new(['b', 'z', 'y']),
    LanguageSubtag::new(['b', 'z', 'z']),
    LanguageSubtag::new(['c', 'a', ' ']),
    LanguageSubtag::new(['c', 'a', 'a']),
    LanguageSubtag::new(['c', 'a', 'b']),
    LanguageSubtag::new(['c', 'a', 'c']),
    LanguageSubtag::new(['c', 'a', 'd']),
    LanguageSubtag::new(['c', 'a', 'e']),
    LanguageSubtag::new(['c', 'a', 'f']),
    LanguageSubtag::new(['c', 'a', 'g']),
    LanguageSubtag::new(['c', 'a', 'h']),
    LanguageSubtag::new(['c', 'a', 'i']),
    LanguageSubtag::new(['c', 'a', 'j']),
    LanguageSubtag::new(['c', 'a', 'k']),
    LanguageSubtag::new(['c', 'a', 'l']),
    LanguageSubtag::new(['c', 'a', 'm']),
    LanguageSubtag::new(['c', 'a', 'n']),
    LanguageSubtag::new(['c', 'a', 'o']),
    LanguageSubtag::new(['c', 'a', 'p']),
    LanguageSubtag::new(['c', 'a', 'q']),
    LanguageSubtag::new(['c', 'a', 'r']),
    LanguageSubtag::new(['c', 'a', 's']),
    LanguageSubtag::new(['c', 'a', 'u']),
    LanguageSubtag::new(['c', 'a', 'v']),
    LanguageSubtag::new(['c', 'a', 'w']),
    LanguageSubtag::new(['c', 'a', 'x']),
    LanguageSubtag::new(['c', 'a', 'y']),
    LanguageSubtag::new(['c', 'a', 'z']),
    LanguageSubtag::new(['c', 'b', 'a']),
    LanguageSubtag::new(['c', 'b', 'b']),
    LanguageSubtag::new(['c', 'b', 'c']),
    LanguageSubtag::new(['c', 'b', 'd']),
    LanguageSubtag::new(['c', 'b', 'e']),
    LanguageSubtag::new(['c', 'b', 'g']),
    LanguageSubtag::new(['c', 'b', 'h']),
    LanguageSubtag::new(['c', 'b', 'i']),
    LanguageSubtag::new(['c', 'b', 'j']),
    LanguageSubtag::new(['c', 'b', 'k']),
    LanguageSubtag::new(['c', 'b', 'l']),
    LanguageSubtag::new(['c', 'b', 'n']),
    LanguageSubtag::new(['c', 'b', 'o']),
    LanguageSubtag::new(['c', 'b', 'q']),
    LanguageSubtag::new(['c', 'b', 'r']),
    LanguageSubtag::new(['c', 'b', 's']),
    LanguageSubtag::new(['c', 'b', 't']),
    LanguageSubtag::new(['c', 'b', 'u']),
    LanguageSubtag::new(['c', 'b', 'v']),
    LanguageSubtag::new(['c', 'b', 'w']),
    LanguageSubtag::new(['c', 'b', 'y']),
    LanguageSubtag::new(['c', 'c', 'a']),
    LanguageSubtag::new(['c', 'c', 'c']),
    LanguageSubtag::new(['c', 'c', 'd']),
    LanguageSubtag::new(['c', 'c', 'e']),
    LanguageSubtag::new(['c', 'c', 'g']),
    LanguageSubtag::new(['c', 'c', 'h']),
    LanguageSubtag::new(['c', 'c', 'j']),
    LanguageSubtag::new(['c', 'c', 'l']),
    LanguageSubtag::new(['c', 'c', 'm']),
    LanguageSubtag::new(['c', 'c', 'n']),
    LanguageSubtag::new(['c', 'c', 'o']),
    LanguageSubtag::new(['c', 'c', 'p']),
    LanguageSubtag::new(['c', 'c', 'q']),
    LanguageSubtag::new(['c', 'c', 'r']),
    LanguageSubtag::new(['c', 'c', 's']),
    LanguageSubtag::new(['c', 'd', 'a']),
    LanguageSubtag::new(['c', 'd', 'c']),
    LanguageSubtag::new(['c', 'd', 'd']),
    LanguageSubtag::new(['c', 'd', 'e']),
    LanguageSubtag::new(['c', 'd', 'f']),
    LanguageSubtag::new(['c', 'd', 'g']),
    LanguageSubtag::new(['c', 'd', 'h']),
    LanguageSubtag::new(['c', 'd', 'i']),
    LanguageSubtag::new(['c', 'd', 'j']),
    LanguageSubtag::new(['c', 'd', 'm']),
    LanguageSubtag::new(['c', 'd', 'n']),
    LanguageSubtag::new(['c', 'd', 'o']),
    LanguageSubtag::new(['c', 'd', 'r']),
    LanguageSubtag::new(['c', 'd', 's']),
    LanguageSubtag::new(['c', 'd', 'y']),
    LanguageSubtag::new(['c', 'd', 'z']),
    LanguageSubtag::new(['c', 'e', ' ']),
    LanguageSubtag::new(['c', 'e', 'a']),
    LanguageSubtag::new(['c', 'e', 'b']),
    LanguageSubtag::new(['c', 'e', 'g']),
    LanguageSubtag::new(['c', 'e', 'k']),
    LanguageSubtag::new(['c', 'e', 'l']),
    LanguageSubtag::new(['c', 'e', 'n']),
    LanguageSubtag::new(['c', 'e', 't']),
    LanguageSubtag::new(['c', 'e', 'y']),
    LanguageSubtag::new(['c', 'f', 'a']),
    LanguageSubtag::new(['c', 'f', 'd']),
    LanguageSubtag::new(['c', 'f', 'g']),
    LanguageSubtag::new(['c', 'f', 'm']),
    LanguageSubtag::new(['c', 'g', 'a']),
    LanguageSubtag::new(['c', 'g', 'c']),
    LanguageSubtag::new(['c', 'g', 'g']),
    LanguageSubtag::new(['c', 'g', 'k']),
    LanguageSubtag::new(['c', 'h', ' ']),
    LanguageSubtag::new(['c', 'h', 'b']),
    LanguageSubtag::new(['c', 'h', 'c']),
    LanguageSubtag::new(['c', 'h', 'd']),
    LanguageSubtag::new(['c', 'h', 'f']),
    LanguageSubtag::new(['c', 'h', 'g']),
    LanguageSubtag::new(['c', 'h', 'h']),
    LanguageSubtag::new(['c', 'h', 'j']),
    LanguageSubtag::new(['c', 'h', 'k']),
    LanguageSubtag::new(['c', 'h', 'l']),
    LanguageSubtag::new(['c', 'h', 'm']),
    LanguageSubtag::new(['c', 'h', 'n']),
    LanguageSubtag::new(['c', 'h', 'o']),
    LanguageSubtag::new(['c', 'h', 'p']),
    LanguageSubtag::new(['c', 'h', 'q']),
    LanguageSubtag::new(['c', 'h', 'r']),
    LanguageSubtag::new(['c', 'h', 't']),
    LanguageSubtag::new(['c', 'h', 'w']),
    LanguageSubtag::new(['c', 'h', 'x']),
    LanguageSubtag::new(['c', 'h', 'y']),
    LanguageSubtag::new(['c', 'h', 'z']),
    LanguageSubtag::new(['c', 'i', 'a']),
    LanguageSubtag::new(['c', 'i', 'b']),
    LanguageSubtag::new(['c', 'i', 'c']),
    LanguageSubtag::new(['c', 'i', 'd']),
    LanguageSubtag::new(['c', 'i', 'e']),
    LanguageSubtag::new(['c', 'i', 'h']),
    LanguageSubtag::new(['c', 'i', 'k']),
    LanguageSubtag::new(['c', 'i', 'm']),
    LanguageSubtag::new(['c', 'i', 'n']),
    LanguageSubtag::new(['c', 'i', 'p']),
    LanguageSubtag::new(['c', 'i', 'r']),
    LanguageSubtag::new(['c', 'i', 'w']),
    LanguageSubtag::new(['c', 'i', 'y']),
    LanguageSubtag::new(['c', 'j', 'a']),
    LanguageSubtag::new(['c', 'j', 'e']),
    LanguageSubtag::new(['c', 'j', 'h']),
    LanguageSubtag::new(['c', 'j', 'i']),
    LanguageSubtag::new(['c', 'j', 'k']),
    LanguageSubtag::new(['c', 'j', 'm']),
    LanguageSubtag::new(['c', 'j', 'n']),
    LanguageSubtag::new(['c', 'j', 'o']),
    LanguageSubtag::new(['c', 'j', 'p']),
    LanguageSubtag::new(['c', 'j', 'r']),
    LanguageSubtag::new(['c', 'j', 's']),
    LanguageSubtag::new(['c', 'j', 'v']),
    LanguageSubtag::new(['c', 'j', 'y']),
    LanguageSubtag::new(['c', 'k', 'a']),
    LanguageSubtag::new(['c', 'k', 'b']),
    LanguageSubtag::new(['c', 'k', 'h']),
    LanguageSubtag::new(['c', 'k', 'l']),
    LanguageSubtag::new(['c', 'k', 'm']),
    LanguageSubtag::new(['c', 'k', 'n']),
    LanguageSubtag::new(['c', 'k', 'o']),
    LanguageSubtag::new(['c', 'k', 'q']),
    LanguageSubtag::new(['c', 'k', 'r']),
    LanguageSubtag::new(['c', 'k', 's']),
    LanguageSubtag::new(['c', 'k', 't']),
    LanguageSubtag::new(['c', 'k', 'u']),
    LanguageSubtag::new(['c', 'k', 'v']),
    LanguageSubtag::new(['c', 'k', 'x']),
    LanguageSubtag::new(['c', 'k', 'y']),
    LanguageSubtag::new(['c', 'k', 'z']),
    LanguageSubtag::new(['c', 'l', 'a']),
    LanguageSubtag::new(['c', 'l', 'c']),
    LanguageSubtag::new(['c', 'l', 'd']),
    LanguageSubtag::new(['c', 'l', 'e']),
    LanguageSubtag::new(['c', 'l', 'h']),
    LanguageSubtag::new(['c', 'l', 'i']),
    LanguageSubtag::new(['c', 'l', 'j']),
    LanguageSubtag::new(['c', 'l', 'k']),
    LanguageSubtag::new(['c', 'l', 'l']),
    LanguageSubtag::new(['c', 'l', 'm']),
    LanguageSubtag::new(['c', 'l', 'o']),
    LanguageSubtag::new(['c', 'l', 't']),
    LanguageSubtag::new(['c', 'l', 'u']),
    LanguageSubtag::new(['c', 'l', 'w']),
    LanguageSubtag::new(['c', 'l', 'y']),
    LanguageSubtag::new(['c', 'm', 'a']),
    LanguageSubtag::new(['c', 'm', 'c']),
    LanguageSubtag::new(['c', 'm', 'e']),
    LanguageSubtag::new(['c', 'm', 'g']),
    LanguageSubtag::new(['c', 'm', 'i']),
    LanguageSubtag::new(['c', 'm', 'k']),
    LanguageSubtag::new(['c', 'm', 'l']),
    LanguageSubtag::new(['c', 'm', 'm']),
    LanguageSubtag::new(['c', 'm', 'n']),
    LanguageSubtag::new(['c', 'm', 'o']),
    LanguageSubtag::new(['c', 'm', 'r']),
    LanguageSubtag::new(['c', 'm', 's']),
    LanguageSubtag::new(['c', 'm', 't']),
    LanguageSubtag::new(['c', 'n', 'a']),
    LanguageSubtag::new(['c', 'n', 'b']),
    LanguageSubtag::new(['c', 'n', 'c']),
    LanguageSubtag::new(['c', 'n', 'g']),
    LanguageSubtag::new(['c', 'n', 'h']),
    LanguageSubtag::new(['c', 'n', 'i']),
    LanguageSubtag::new(['c', 'n', 'k']),
    LanguageSubtag::new(['c', 'n', 'l']),
    LanguageSubtag::new(['c', 'n', 'o']),
    LanguageSubtag::new(['c', 'n', 'p']),
    LanguageSubtag::new(['c', 'n', 'r']),
    LanguageSubtag::new(['c', 'n', 's']),
    LanguageSubtag::new(['c', 'n', 't']),
    LanguageSubtag::new(['c', 'n', 'u']),
    LanguageSubtag::new(['c', 'n', 'w']),
    LanguageSubtag::new(['c', 'n', 'x']),
    LanguageSubtag::new(['c', 'o', ' ']),
    LanguageSubtag::new(['c', 'o', 'a']),
    LanguageSubtag::new(['c', 'o', 'b']),
    LanguageSubtag::new(['c', 'o', 'c']),
    LanguageSubtag::new(['c', 'o', 'd']),
    LanguageSubtag::new(['c', 'o', 'e']),
    LanguageSubtag::new(['c', 'o', 'f']),
    LanguageSubtag::new(['c', 'o', 'g']),
    LanguageSubtag::new(['c', 'o', 'h']),
    LanguageSubtag::new(['c', 'o', 'j']),
    LanguageSubtag::new(['c', 'o', 'k']),
    LanguageSubtag::new(['c', 'o', 'l']),
    LanguageSubtag::new(['c', 'o', 'm']),
    LanguageSubtag::new(['c', 'o', 'n']),
    LanguageSubtag::new(['c', 'o', 'o']),
    LanguageSubtag::new(['c', 'o', 'p']),
    LanguageSubtag::new(['c', 'o', 'q']),
    LanguageSubtag::new(['c', 'o', 't']),
    LanguageSubtag::new(['c', 'o', 'u']),
    LanguageSubtag::new(['c', 'o', 'v']),
    LanguageSubtag::new(['c', 'o', 'w']),
    LanguageSubtag::new(['c', 'o', 'x']),
    LanguageSubtag::new(['c', 'o', 'y']),
    LanguageSubtag::new(['c', 'o', 'z']),
    LanguageSubtag::new(['c', 'p', 'a']),
    LanguageSubtag::new(['c', 'p', 'b']),
    LanguageSubtag::new(['c', 'p', 'c']),
    LanguageSubtag::new(['c', 'p', 'e']),
    LanguageSubtag::new(['c', 'p', 'f']),
    LanguageSubtag::new(['c', 'p', 'g']),
    LanguageSubtag::new(['c', 'p', 'i']),
    LanguageSubtag::new(['c', 'p', 'n']),
    LanguageSubtag::new(['c', 'p', 'o']),
    LanguageSubtag::new(['c', 'p', 'p']),
    LanguageSubtag::new(['c', 'p', 's']),
    LanguageSubtag::new(['c', 'p', 'u']),
    LanguageSubtag::new(['c', 'p', 'x']),
    LanguageSubtag::new(['c', 'p', 'y']),
    LanguageSubtag::new(['c', 'q', 'd']),
    LanguageSubtag::new(['c', 'q', 'u']),
    LanguageSubtag::new(['c', 'r', ' ']),
    LanguageSubtag::new(['c', 'r', 'a']),
    LanguageSubtag::new(['c', 'r', 'b']),
    LanguageSubtag::new(['c', 'r', 'c']),
    LanguageSubtag::new(['c', 'r', 'd']),
    LanguageSubtag::new(['c', 'r', 'f']),
    LanguageSubtag::new(['c', 'r', 'g']),
    LanguageSubtag::new(['c', 'r', 'h']),
    LanguageSubtag::new(['c', 'r', 'i']),
    LanguageSubtag::new(['c', 'r', 'j']),
    LanguageSubtag::new(['c', 'r', 'k']),
    LanguageSubtag::new(['c', 'r', 'l']),
    LanguageSubtag::new(['c', 'r', 'm']),
    LanguageSubtag::new(['c', 'r', 'n']),
    LanguageSubtag::new(['c', 'r', 'o']),
    LanguageSubtag::new(['c', 'r', 'p']),
    LanguageSubtag::new(['c', 'r', 'q']),
    LanguageSubtag::new(['c', 'r', 'r']),
    LanguageSubtag::new(['c', 'r', 's']),
    LanguageSubtag::new(['c', 'r', 't']),
    LanguageSubtag::new(['c', 'r', 'v']),
    LanguageSubtag::new(['c', 'r', 'w']),
    LanguageSubtag::new(['c', 'r', 'x']),
    LanguageSubtag::new(['c', 'r', 'y']),
    LanguageSubtag::new(['c', 'r', 'z']),
    LanguageSubtag::new(['c', 's', ' ']),
    LanguageSubtag::new(['c', 's', 'a']),
    LanguageSubtag::new(['c', 's', 'b']),
    LanguageSubtag::new(['c', 's', 'c']),
    LanguageSubtag::new(['c', 's', 'd']),
    LanguageSubtag::new(['c', 's', 'e']),
    LanguageSubtag::new(['c', 's', 'f']),
    LanguageSubtag::new(['c', 's', 'g']),
    LanguageSubtag::new(['c', 's', 'h']),
    LanguageSubtag::new(['c', 's', 'i']),
    LanguageSubtag::new(['c', 's', 'j']),
    LanguageSubtag::new(['c', 's', 'k']),
    LanguageSubtag::new(['c', 's', 'l']),
    LanguageSubtag::new(['c', 's', 'm']),
    LanguageSubtag::new(['c', 's', 'n']),
    LanguageSubtag::new(['c', 's', 'o']),
    LanguageSubtag::new(['c', 's', 'p']),
    LanguageSubtag::new(['c', 's', 'q']),
    LanguageSubtag::new(['c', 's', 'r']),
    LanguageSubtag::new(['c', 's', 's']),
    LanguageSubtag::new(['c', 's', 't']),
    LanguageSubtag::new(['c', 's', 'u']),
    LanguageSubtag::new(['c', 's', 'v']),
    LanguageSubtag::new(['c', 's', 'w']),
    LanguageSubtag::new(['c', 's', 'x']),
    LanguageSubtag::new(['c', 's', 'y']),
    LanguageSubtag::new(['c', 's', 'z']),
    LanguageSubtag::new(['c', 't', 'a']),
    LanguageSubtag::new(['c', 't', 'c']),
    LanguageSubtag::new(['c', 't', 'd']),
    LanguageSubtag::new(['c', 't', 'e']),
    LanguageSubtag::new(['c', 't', 'g']),
    LanguageSubtag::new(['c', 't', 'h']),
    LanguageSubtag::new(['c', 't', 'l']),
    LanguageSubtag::new(['c', 't', 'm']),
    LanguageSubtag::new(['c', 't', 'n']),
    LanguageSubtag::new(['c', 't', 'o']),
    LanguageSubtag::new(['c', 't', 'p']),
    LanguageSubtag::new(['c', 't', 's']),
    LanguageSubtag::new(['c', 't', 't']),
    LanguageSubtag::new(['c', 't', 'u']),
    LanguageSubtag::new(['c', 't', 'y']),
    LanguageSubtag::new(['c', 't', 'z']),
    LanguageSubtag::new(['c', 'u', ' ']),
    LanguageSubtag::new(['c', 'u', 'a']),
    LanguageSubtag::new(['c', 'u', 'b']),
    LanguageSubtag::new(['c', 'u', 'c']),
    LanguageSubtag::new(['c', 'u', 'g']),
    LanguageSubtag::new(['c', 'u', 'h']),
    LanguageSubtag::new(['c', 'u', 'i']),
    LanguageSubtag::new(['c', 'u', 'j']),
    LanguageSubtag::new(['c', 'u', 'k']),
    LanguageSubtag::new(['c', 'u', 'l']),
    LanguageSubtag::new(['c', 'u', 'm']),
    LanguageSubtag::new(['c', 'u', 'o']),
    LanguageSubtag::new(['c', 'u', 'p']),
    LanguageSubtag::new(['c', 'u', 'q']),
    LanguageSubtag::new(['c', 'u', 'r']),
    LanguageSubtag::new(['c', 'u', 's']),
    LanguageSubtag::new(['c', 'u', 't']),
    LanguageSubtag::new(['c', 'u', 'u']),
    LanguageSubtag::new(['c', 'u', 'v']),
    LanguageSubtag::new(['c', 'u', 'w']),
    LanguageSubtag::new(['c', 'u', 'x']),
    LanguageSubtag::new(['c', 'u', 'y']),
    LanguageSubtag::new(['c', 'v', ' ']),
    LanguageSubtag::new(['c', 'v', 'g']),
    LanguageSubtag::new(['c', 'v', 'n']),
    LanguageSubtag::new(['c', 'w', 'a']),
    LanguageSubtag::new(['c', 'w', 'b']),
    LanguageSubtag::new(['c', 'w', 'd']),
    LanguageSubtag::new(['c', 'w', 'e']),
    LanguageSubtag::new(['c', 'w', 'g']),
    LanguageSubtag::new(['c', 'w', 't']),
    LanguageSubtag::new(['c', 'y', ' ']),
    LanguageSubtag::new(['c', 'y', 'a']),
    LanguageSubtag::new(['c', 'y', 'b']),
    LanguageSubtag::new(['c', 'y', 'o']),
    LanguageSubtag::new(['c', 'z', 'h']),
    LanguageSubtag::new(['c', 'z', 'k']),
    LanguageSubtag::new(['c', 'z', 'n']),
    LanguageSubtag::new(['c', 'z', 'o']),
    LanguageSubtag::new(['c', 'z', 't']),
    LanguageSubtag::new(['d', 'a', ' ']),
    LanguageSubtag::new(['d', 'a', 'a']),
    LanguageSubtag::new(['d', 'a', 'c']),
    LanguageSubtag::new(['d', 'a', 'd']),
    LanguageSubtag::new(['d', 'a', 'e']),
    LanguageSubtag::new(['d', 'a', 'f']),
    LanguageSubtag::new(['d', 'a', 'g']),
    LanguageSubtag::new(['d', 'a', 'h']),
    LanguageSubtag::new(['d', 'a', 'i']),
    LanguageSubtag::new(['d', 'a', 'j']),
    LanguageSubtag::new(['d', 'a', 'k']),
    LanguageSubtag::new(['d', 'a', 'l']),
    LanguageSubtag::new(['d', 'a', 'm']),
    LanguageSubtag::new(['d', 'a', 'o']),
    LanguageSubtag::new(['d', 'a', 'p']),
    LanguageSubtag::new(['d', 'a', 'q']),
    LanguageSubtag::new(['d', 'a', 'r']),
    LanguageSubtag::new(['d', 'a', 's']),
    LanguageSubtag::new(['d', 'a', 'u']),
    LanguageSubtag::new(['d', 'a', 'v']),
    LanguageSubtag::new(['d', 'a', 'w']),
    LanguageSubtag::new(['d', 'a', 'x']),
    LanguageSubtag::new(['d', 'a', 'y']),
    LanguageSubtag::new(['d', 'a', 'z']),
    LanguageSubtag::new(['d', 'b', 'a']),
    LanguageSubtag::new(['d', 'b', 'b']),
    LanguageSubtag::new(['d', 'b', 'd']),
    LanguageSubtag::new(['d', 'b', 'e']),
    LanguageSubtag::new(['d', 'b', 'f']),
    LanguageSubtag::new(['d', 'b', 'g']),
    LanguageSubtag::new(['d', 'b', 'i']),
    LanguageSubtag::new(['d', 'b', 'j']),
    LanguageSubtag::new(['d', 'b', 'l']),
    LanguageSubtag::new(['d', 'b', 'm']),
    LanguageSubtag::new(['d', 'b', 'n']),
    LanguageSubtag::new(['d', 'b', 'o']),
    LanguageSubtag::new(['d', 'b', 'p']),
    LanguageSubtag::new(['d', 'b', 'q']),
    LanguageSubtag::new(['d', 'b', 'r']),
    LanguageSubtag::new(['d', 'b', 't']),
    LanguageSubtag::new(['d', 'b', 'u']),
    LanguageSubtag::new(['d', 'b', 'v']),
    LanguageSubtag::new(['d', 'b', 'w']),
    LanguageSubtag::new(['d', 'b', 'y']),
    LanguageSubtag::new(['d', 'c', 'c']),
    LanguageSubtag::new(['d', 'c', 'r']),
    LanguageSubtag::new(['d', 'd', 'a']),
    LanguageSubtag::new(['d', 'd', 'd']),
    LanguageSubtag::new(['d', 'd', 'e']),
    LanguageSubtag::new(['d', 'd', 'g']),
    LanguageSubtag::new(['d', 'd', 'i']),
    LanguageSubtag::new(['d', 'd', 'j']),
    LanguageSubtag::new(['d', 'd', 'n']),
    LanguageSubtag::new(['d', 'd', 'o']),
    LanguageSubtag::new(['d', 'd', 'r']),
    LanguageSubtag::new(['d', 'd', 's']),
    LanguageSubtag::new(['d', 'd', 'w']),
    LanguageSubtag::new(['d', 'e', ' ']),
    LanguageSubtag::new(['d', 'e', 'c']),
    LanguageSubtag::new(['d', 'e', 'd']),
    LanguageSubtag::new(['d', 'e', 'e']),
    LanguageSubtag::new(['d', 'e', 'f']),
    LanguageSubtag::new(['d', 'e', 'g']),
    LanguageSubtag::new(['d', 'e', 'h']),
    LanguageSubtag::new(['d', 'e', 'i']),
    LanguageSubtag::new(['d', 'e', 'k']),
    LanguageSubtag::new(['d', 'e', 'l']),
    LanguageSubtag::new(['d', 'e', 'm']),
    LanguageSubtag::new(['d', 'e', 'n']),
    LanguageSubtag::new(['d', 'e', 'p']),
    LanguageSubtag::new(['d', 'e', 'q']),
    LanguageSubtag::new(['d', 'e', 'r']),
    LanguageSubtag::new(['d', 'e', 's']),
    LanguageSubtag::new(['d', 'e', 'v']),
    LanguageSubtag::new(['d', 'e', 'z']),
    LanguageSubtag::new(['d', 'g', 'a']),
    LanguageSubtag::new(['d', 'g', 'b']),
    LanguageSubtag::new(['d', 'g', 'c']),
    LanguageSubtag::new(['d', 'g', 'd']),
    LanguageSubtag::new(['d', 'g', 'e']),
    LanguageSubtag::new(['d', 'g', 'g']),
    LanguageSubtag::new(['d', 'g', 'h']),
    LanguageSubtag::new(['d', 'g', 'i']),
    LanguageSubtag::new(['d', 'g', 'k']),
    LanguageSubtag::new(['d', 'g', 'l']),
    LanguageSubtag::new(['d', 'g', 'n']),
    LanguageSubtag::new(['d', 'g', 'o']),
    LanguageSubtag::new(['d', 'g', 'r']),
    LanguageSubtag::new(['d', 'g', 's']),
    LanguageSubtag::new(['d', 'g', 't']),
    LanguageSubtag::new(['d', 'g', 'u']),
    LanguageSubtag::new(['d', 'g', 'w']),
    LanguageSubtag::new(['d', 'g', 'x']),
    LanguageSubtag::new(['d', 'g', 'z']),
    LanguageSubtag::new(['d', 'h', 'a']),
    LanguageSubtag::new(['d', 'h', 'd']),
    LanguageSubtag::new(['d', 'h', 'g']),
    LanguageSubtag::new(['d', 'h', 'i']),
    LanguageSubtag::new(['d', 'h', 'l']),
    LanguageSubtag::new(['d', 'h', 'm']),
    LanguageSubtag::new(['d', 'h', 'n']),
    LanguageSubtag::new(['d', 'h', 'o']),
    LanguageSubtag::new(['d', 'h', 'r']),
    LanguageSubtag::new(['d', 'h', 's']),
    LanguageSubtag::new(['d', 'h', 'u']),
    LanguageSubtag::new(['d', 'h', 'v']),
    LanguageSubtag::new(['d', 'h', 'w']),
    LanguageSubtag::new(['d', 'h', 'x']),
    LanguageSubtag::new(['d', 'i', 'a']),
    LanguageSubtag::new(['d', 'i', 'b']),
    LanguageSubtag::new(['d', 'i', 'c']),
    LanguageSubtag::new(['d', 'i', 'd']),
    LanguageSubtag::new(['d', 'i', 'f']),
    LanguageSubtag::new(['d', 'i', 'g']),
    LanguageSubtag::new(['d', 'i', 'h']),
    LanguageSubtag::new(['d', 'i', 'i']),
    LanguageSubtag::new(['d', 'i', 'j']),
    LanguageSubtag::new(['d', 'i', 'k']),
    LanguageSubtag::new(['d', 'i', 'l']),
    LanguageSubtag::new(['d', 'i', 'm']),
    LanguageSubtag::new(['d', 'i', 'n']),
    LanguageSubtag::new(['d', 'i', 'o']),
    LanguageSubtag::new(['d', 'i', 'p']),
    LanguageSubtag::new(['d', 'i', 'q']),
    LanguageSubtag::new(['d', 'i', 'r']),
    LanguageSubtag::new(['d', 'i', 's']),
    LanguageSubtag::new(['d', 'i', 't']),
    LanguageSubtag::new(['d', 'i', 'u']),
    LanguageSubtag::new(['d', 'i', 'w']),
    LanguageSubtag::new(['d', 'i', 'x']),
    LanguageSubtag::new(['d', 'i', 'y']),
    LanguageSubtag::new(['d', 'i', 'z']),
    LanguageSubtag::new(['d', 'j', 'a']),
    LanguageSubtag::new(['d', 'j', 'b']),
    LanguageSubtag::new(['d', 'j', 'c']),
    LanguageSubtag::new(['d', 'j', 'd']),
    LanguageSubtag::new(['d', 'j', 'e']),
    LanguageSubtag::new(['d', 'j', 'f']),
    LanguageSubtag::new(['d', 'j', 'i']),
    LanguageSubtag::new(['d', 'j', 'j']),
    LanguageSubtag::new(['d', 'j', 'k']),
    LanguageSubtag::new(['d', 'j', 'l']),
    LanguageSubtag::new(['d', 'j', 'm']),
    LanguageSubtag::new(['d', 'j', 'n']),
    LanguageSubtag::new(['d', 'j', 'o']),
    LanguageSubtag::new(['d', 'j', 'r']),
    LanguageSubtag::new(['d', 'j', 'u']),
    LanguageSubtag::new(['d', 'j', 'w']),
    LanguageSubtag::new(['d', 'k', 'a']),
    LanguageSubtag::new(['d', 'k', 'g']),
    LanguageSubtag::new(['d', 'k', 'k']),
    LanguageSubtag::new(['d', 'k', 'l']),
    LanguageSubtag::new(['d', 'k', 'r']),
    LanguageSubtag::new(['d', 'k', 's']),
    LanguageSubtag::new(['d', 'k', 'x']),
    LanguageSubtag::new(['d', 'l', 'g']),
    LanguageSubtag::new(['d', 'l', 'k']),
    LanguageSubtag::new(['d', 'l', 'm']),
    LanguageSubtag::new(['d', 'l', 'n']),
    LanguageSubtag::new(['d', 'm', 'a']),
    LanguageSubtag::new(['d', 'm', 'b']),
    LanguageSubtag::new(['d', 'm', 'c']),
    LanguageSubtag::new(['d', 'm', 'd']),
    LanguageSubtag::new(['d', 'm', 'e']),
    LanguageSubtag::new(['d', 'm', 'f']),
    LanguageSubtag::new(['d', 'm', 'g']),
    LanguageSubtag::new(['d', 'm', 'k']),
    LanguageSubtag::new(['d', 'm', 'l']),
    LanguageSubtag::new(['d', 'm', 'm']),
    LanguageSubtag::new(['d', 'm', 'n']),
    LanguageSubtag::new(['d', 'm', 'o']),
    LanguageSubtag::new(['d', 'm', 'r']),
    LanguageSubtag::new(['d', 'm', 's']),
    LanguageSubtag::new(['d', 'm', 'u']),
    LanguageSubtag::new(['d', 'm', 'v']),
    LanguageSubtag::new(['d', 'm', 'w']),
    LanguageSubtag::new(['d', 'm', 'x']),
    LanguageSubtag::new(['d', 'm', 'y']),
    LanguageSubtag::new(['d', 'n', 'a']),
    LanguageSubtag::new(['d', 'n', 'd']),
    LanguageSubtag::new(['d', 'n', 'e']),
    LanguageSubtag::new(['d', 'n', 'g']),
    LanguageSubtag::new(['d', 'n', 'i']),
    LanguageSubtag::new(['d', 'n', 'j']),
    LanguageSubtag::new(['d', 'n', 'k']),
    LanguageSubtag::new(['d', 'n', 'n']),
    LanguageSubtag::new(['d', 'n', 'o']),
    LanguageSubtag::new(['d', 'n', 'r']),
    LanguageSubtag::new(['d', 'n', 't']),
    LanguageSubtag::new(['d', 'n', 'u']),
    LanguageSubtag::new(['d', 'n', 'v']),
    LanguageSubtag::new(['d', 'n', 'w']),
    LanguageSubtag::new(['d', 'n', 'y']),
    LanguageSubtag::new(['d', 'o', 'a']),
    LanguageSubtag::new(['d', 'o', 'b']),
    LanguageSubtag::new(['d', 'o', 'c']),
    LanguageSubtag::new(['d', 'o', 'e']),
    LanguageSubtag::new(['d', 'o', 'f']),
    LanguageSubtag::new(['d', 'o', 'h']),
    LanguageSubtag::new(['d', 'o', 'i']),
    LanguageSubtag::new(['d', 'o', 'k']),
    LanguageSubtag::new(['d', 'o', 'l']),
    LanguageSubtag::new(['d', 'o', 'n']),
    LanguageSubtag::new(['d', 'o', 'o']),
    LanguageSubtag::new(['d', 'o', 'p']),
    LanguageSubtag::new(['d', 'o', 'q']),
    LanguageSubtag::new(['d', 'o', 'r']),
    LanguageSubtag::new(['d', 'o', 's']),
    LanguageSubtag::new(['d', 'o', 't']),
    LanguageSubtag::new(['d', 'o', 'v']),
    LanguageSubtag::new(['d', 'o', 'w']),
    LanguageSubtag::new(['d', 'o', 'x']),
    LanguageSubtag::new(['d', 'o', 'y']),
    LanguageSubtag::new(['d', 'o', 'z']),
    LanguageSubtag::new(['d', 'p', 'p']),
    LanguageSubtag::new(['d', 'r', 'a']),
    LanguageSubtag::new(['d', 'r', 'b']),
    LanguageSubtag::new(['d', 'r', 'c']),
    LanguageSubtag::new(['d', 'r', 'd']),
    LanguageSubtag::new(['d', 'r', 'e']),
    LanguageSubtag::new(['d', 'r', 'g']),
    LanguageSubtag::new(['d', 'r', 'h']),
    LanguageSubtag::new(['d', 'r', 'i']),
    LanguageSubtag::new(['d', 'r', 'l']),
    LanguageSubtag::new(['d', 'r', 'n']),
    LanguageSubtag::new(['d', 'r', 'o']),
    LanguageSubtag::new(['d', 'r', 'q']),
    LanguageSubtag::new(['d', 'r', 'r']),
    LanguageSubtag::new(['d', 'r', 's']),
    LanguageSubtag::new(['d', 'r', 't']),
    LanguageSubtag::new(['d', 'r', 'u']),
    LanguageSubtag::new(['d', 'r', 'w']),
    LanguageSubtag::new(['d', 'r', 'y']),
    LanguageSubtag::new(['d', 's', 'b']),
    LanguageSubtag::new(['d', 's', 'e']),
    LanguageSubtag::new(['d', 's', 'h']),
    LanguageSubtag::new(['d', 's', 'i']),
    LanguageSubtag::new(['d', 's', 'l']),
    LanguageSubtag::new(['d', 's', 'n']),
    LanguageSubtag::new(['d', 's', 'o']),
    LanguageSubtag::new(['d', 's', 'q']),
    LanguageSubtag::new(['d', 't', 'a']),
    LanguageSubtag::new(['d', 't', 'b']),
    LanguageSubtag::new(['d', 't', 'd']),
    LanguageSubtag::new(['d', 't', 'h']),
    LanguageSubtag::new(['d', 't', 'i']),
    LanguageSubtag::new(['d', 't', 'k']),
    LanguageSubtag::new(['d', 't', 'm']),
    LanguageSubtag::new(['d', 't', 'n']),
    LanguageSubtag::new(['d', 't', 'o']),
    LanguageSubtag::new(['d', 't', 'p']),
    LanguageSubtag::new(['d', 't', 'r']),
    LanguageSubtag::new(['d', 't', 's']),
    LanguageSubtag::new(['d', 't', 't']),
    LanguageSubtag::new(['d', 't', 'u']),
    LanguageSubtag::new(['d', 't', 'y']),
    LanguageSubtag::new(['d', 'u', 'a']),
    LanguageSubtag::new(['d', 'u', 'b']),
    LanguageSubtag::new(['d', 'u', 'c']),
    LanguageSubtag::new(['d', 'u', 'd']),
    LanguageSubtag::new(['d', 'u', 'e']),
    LanguageSubtag::new(['d', 'u', 'f']),
    LanguageSubtag::new(['d', 'u', 'g']),
    LanguageSubtag::new(['d', 'u', 'h']),
    LanguageSubtag::new(['d', 'u', 'i']),
    LanguageSubtag::new(['d', 'u', 'j']),
    LanguageSubtag::new(['d', 'u', 'k']),
    LanguageSubtag::new(['d', 'u', 'l']),
    LanguageSubtag::new(['d', 'u', 'm']),
    LanguageSubtag::new(['d', 'u', 'n']),
    LanguageSubtag::new(['d', 'u', 'o']),
    LanguageSubtag::new(['d', 'u', 'p']),
    LanguageSubtag::new(['d', 'u', 'q']),
    LanguageSubtag::new(['d', 'u', 'r']),
    LanguageSubtag::new(['d', 'u', 's']),
    LanguageSubtag::new(['d', 'u', 'u']),
    LanguageSubtag::new(['d', 'u', 'v']),
    LanguageSubtag::new(['d', 'u', 'w']),
    LanguageSubtag::new(['d', 'u', 'x']),
    LanguageSubtag::new(['d', 'u', 'y']),
    LanguageSubtag::new(['d', 'u', 'z']),
    LanguageSubtag::new(['d', 'v', ' ']),
    LanguageSubtag::new(['d', 'v', 'a']),
    LanguageSubtag::new(['d', 'w', 'a']),
    LanguageSubtag::new(['d', 'w', 'k']),
    LanguageSubtag::new(['d', 'w', 'l']),
    LanguageSubtag::new(['d', 'w', 'r']),
    LanguageSubtag::new(['d', 'w', 's']),
    LanguageSubtag::new(['d', 'w', 'u']),
    LanguageSubtag::new(['d', 'w', 'w']),
    LanguageSubtag::new(['d', 'w', 'y']),
    LanguageSubtag::new(['d', 'w', 'z']),
    LanguageSubtag::new(['d', 'y', 'a']),
    LanguageSubtag::new(['d', 'y', 'b']),
    LanguageSubtag::new(['d', 'y', 'd']),
    LanguageSubtag::new(['d', 'y', 'g']),
    LanguageSubtag::new(['d', 'y', 'i']),
    LanguageSubtag::new(['d', 'y', 'm']),
    LanguageSubtag::new(['d', 'y', 'n']),
    LanguageSubtag::new(['d', 'y', 'o']),
    LanguageSubtag::new(['d', 'y', 'u']),
    LanguageSubtag::new(['d', 'y', 'y']),
    LanguageSubtag::new(['d', 'z', ' ']),
    LanguageSubtag::new(['d', 'z', 'a']),
    LanguageSubtag::new(['d', 'z', 'd']),
    LanguageSubtag::new(['d', 'z', 'e']),
    LanguageSubtag::new(['d', 'z', 'g']),
    LanguageSubtag::new(['d', 'z', 'l']),
    LanguageSubtag::new(['d', 'z', 'n']),
    LanguageSubtag::new(['e', 'a', 'a']),
    LanguageSubtag::new(['e', 'b', 'c']),
    LanguageSubtag::new(['e', 'b', 'g']),
    LanguageSubtag::new(['e', 'b', 'k']),
    LanguageSubtag::new(['e', 'b', 'o']),
    LanguageSubtag::new(['e', 'b', 'r']),
    LanguageSubtag::new(['e', 'b', 'u']),
    LanguageSubtag::new(['e', 'c', 'r']),
    LanguageSubtag::new(['e', 'c', 's']),
    LanguageSubtag::new(['e', 'c', 'y']),
    LanguageSubtag::new(['e', 'e', ' ']),
    LanguageSubtag::new(['e', 'e', 'e']),
    LanguageSubtag::new(['e', 'f', 'a']),
    LanguageSubtag::new(['e', 'f', 'e']),
    LanguageSubtag::new(['e', 'f', 'i']),
    LanguageSubtag::new(['e', 'g', 'a']),
    LanguageSubtag::new(['e', 'g', 'l']),
    LanguageSubtag::new(['e', 'g', 'o']),
    LanguageSubtag::new(['e', 'g', 'x']),
    LanguageSubtag::new(['e', 'g', 'y']),
    LanguageSubtag::new(['e', 'h', 's']),
    LanguageSubtag::new(['e', 'h', 'u']),
    LanguageSubtag::new(['e', 'i', 'p']),
    LanguageSubtag::new(['e', 'i', 't']),
    LanguageSubtag::new(['e', 'i', 'v']),
    LanguageSubtag::new(['e', 'j', 'a']),
    LanguageSubtag::new(['e', 'k', 'a']),
    LanguageSubtag::new(['e', 'k', 'c']),
    LanguageSubtag::new(['e', 'k', 'e']),
    LanguageSubtag::new(['e', 'k', 'g']),
    LanguageSubtag::new(['e', 'k', 'i']),
    LanguageSubtag::new(['e', 'k', 'k']),
    LanguageSubtag::new(['e', 'k', 'l']),
    LanguageSubtag::new(['e', 'k', 'm']),
    LanguageSubtag::new(['e', 'k', 'o']),
    LanguageSubtag::new(['e', 'k', 'p']),
    LanguageSubtag::new(['e', 'k', 'r']),
    LanguageSubtag::new(['e', 'k', 'y']),
    LanguageSubtag::new(['e', 'l', ' ']),
    LanguageSubtag::new(['e', 'l', 'e']),
    LanguageSubtag::new(['e', 'l', 'h']),
    LanguageSubtag::new(['e', 'l', 'i']),
    LanguageSubtag::new(['e', 'l', 'k']),
    LanguageSubtag::new(['e', 'l', 'm']),
    LanguageSubtag::new(['e', 'l', 'o']),
    LanguageSubtag::new(['e', 'l', 'p']),
    LanguageSubtag::new(['e', 'l', 'u']),
    LanguageSubtag::new(['e', 'l', 'x']),
    LanguageSubtag::new(['e', 'm', 'a']),
    LanguageSubtag::new(['e', 'm', 'b']),
    LanguageSubtag::new(['e', 'm', 'e']),
    LanguageSubtag::new(['e', 'm', 'g']),
    LanguageSubtag::new(['e', 'm', 'i']),
    LanguageSubtag::new(['e', 'm', 'k']),
    LanguageSubtag::new(['e', 'm', 'm']),
    LanguageSubtag::new(['e', 'm', 'n']),
    LanguageSubtag::new(['e', 'm', 'o']),
    LanguageSubtag::new(['e', 'm', 'p']),
    LanguageSubtag::new(['e', 'm', 'q']),
    LanguageSubtag::new(['e', 'm', 's']),
    LanguageSubtag::new(['e', 'm', 'u']),
    LanguageSubtag::new(['e', 'm', 'w']),
    LanguageSubtag::new(['e', 'm', 'x']),
    LanguageSubtag::new(['e', 'm', 'y']),
    LanguageSubtag::new(['e', 'm', 'z']),
    LanguageSubtag::new(['e', 'n', ' ']),
    LanguageSubtag::new(['e', 'n', 'a']),
    LanguageSubtag::new(['e', 'n', 'b']),
    LanguageSubtag::new(['e', 'n', 'c']),
    LanguageSubtag::new(['e', 'n', 'd']),
    LanguageSubtag::new(['e', 'n', 'f']),
    LanguageSubtag::new(['e', 'n', 'h']),
    LanguageSubtag::new(['e', 'n', 'l']),
    LanguageSubtag::new(['e', 'n', 'm']),
    LanguageSubtag::new(['e', 'n', 'n']),
    LanguageSubtag::new(['e', 'n', 'o']),
    LanguageSubtag::new(['e', 'n', 'q']),
    LanguageSubtag::new(['e', 'n', 'r']),
    LanguageSubtag::new(['e', 'n', 'u']),
    LanguageSubtag::new(['e', 'n', 'v']),
    LanguageSubtag::new(['e', 'n', 'w']),
    LanguageSubtag::new(['e', 'n', 'x']),
    LanguageSubtag::new(['e', 'o', ' ']),
    LanguageSubtag::new(['e', 'o', 't']),
    LanguageSubtag::new(['e', 'p', 'i']),
    LanguageSubtag::new(['e', 'r', 'a']),
    LanguageSubtag::new(['e', 'r', 'g']),
    LanguageSubtag::new(['e', 'r', 'h']),
    LanguageSubtag::new(['e', 'r', 'i']),
    LanguageSubtag::new(['e', 'r', 'k']),
    LanguageSubtag::new(['e', 'r', 'o']),
    LanguageSubtag::new(['e', 'r', 'r']),
    LanguageSubtag::new(['e', 'r', 's']),
    LanguageSubtag::new(['e', 'r', 't']),
    LanguageSubtag::new(['e', 'r', 'w']),
    LanguageSubtag::new(['e', 's', ' ']),
    LanguageSubtag::new(['e', 's', 'e']),
    LanguageSubtag::new(['e', 's', 'g']),
    LanguageSubtag::new(['e', 's', 'h']),
    LanguageSubtag::new(['e', 's', 'i']),
    LanguageSubtag::new(['e', 's', 'k']),
    LanguageSubtag::new(['e', 's', 'l']),
    LanguageSubtag::new(['e', 's', 'm']),
    LanguageSubtag::new(['e', 's', 'n']),
    LanguageSubtag::new(['e', 's', 'o']),
    LanguageSubtag::new(['e', 's', 'q']),
    LanguageSubtag::new(['e', 's', 's']),
    LanguageSubtag::new(['e', 's', 'u']),
    LanguageSubtag::new(['e', 's', 'x']),
    LanguageSubtag::new(['e', 's', 'y']),
    LanguageSubtag::new(['e', 't', ' ']),
    LanguageSubtag::new(['e', 't', 'b']),
    LanguageSubtag::new(['e', 't', 'c']),
    LanguageSubtag::new(['e', 't', 'h']),
    LanguageSubtag::new(['e', 't', 'n']),
    LanguageSubtag::new(['e', 't', 'o']),
    LanguageSubtag::new(['e', 't', 'r']),
    LanguageSubtag::new(['e', 't', 's']),
    LanguageSubtag::new(['e', 't', 't']),
    LanguageSubtag::new(['e', 't', 'u']),
    LanguageSubtag::new(['e', 't', 'x']),
    LanguageSubtag::new(['e', 't', 'z']),
    LanguageSubtag::new(['e', 'u', ' ']),
    LanguageSubtag::new(['e', 'u', 'q']),
    LanguageSubtag::new(['e', 'v', 'e']),
    LanguageSubtag::new(['e', 'v', 'h']),
    LanguageSubtag::new(['e', 'v', 'n']),
    LanguageSubtag::new(['e', 'w', 'o']),
    LanguageSubtag::new(['e', 'x', 't']),
    LanguageSubtag::new(['e', 'y', 'a']),
    LanguageSubtag::new(['e', 'y', 'o']),
    LanguageSubtag::new(['e', 'z', 'a']),
    LanguageSubtag::new(['e', 'z', 'e']),
    LanguageSubtag::new(['f', 'a', ' ']),
    LanguageSubtag::new(['f', 'a', 'a']),
    LanguageSubtag::new(['f', 'a', 'b']),
    LanguageSubtag::new(['f', 'a', 'd']),
    LanguageSubtag::new(['f', 'a', 'f']),
    LanguageSubtag::new(['f', 'a', 'g']),
    LanguageSubtag::new(['f', 'a', 'h']),
    LanguageSubtag::new(['f', 'a', 'i']),
    LanguageSubtag::new(['f', 'a', 'j']),
    LanguageSubtag::new(['f', 'a', 'k']),
    LanguageSubtag::new(['f', 'a', 'l']),
    LanguageSubtag::new(['f', 'a', 'm']),
    LanguageSubtag::new(['f', 'a', 'n']),
    LanguageSubtag::new(['f', 'a', 'p']),
    LanguageSubtag::new(['f', 'a', 'r']),
    LanguageSubtag::new(['f', 'a', 't']),
    LanguageSubtag::new(['f', 'a', 'u']),
    LanguageSubtag::new(['f', 'a', 'x']),
    LanguageSubtag::new(['f', 'a', 'y']),
    LanguageSubtag::new(['f', 'a', 'z']),
    LanguageSubtag::new(['f', 'b', 'l']),
    LanguageSubtag::new(['f', 'c', 's']),
    LanguageSubtag::new(['f', 'e', 'r']),
    LanguageSubtag::new(['f', 'f', ' ']),
    LanguageSubtag::new(['f', 'f', 'i']),
    LanguageSubtag::new(['f', 'f', 'm']),
    LanguageSubtag::new(['f', 'g', 'r']),
    LanguageSubtag::new(['f', 'i', ' ']),
    LanguageSubtag::new(['f', 'i', 'a']),
    LanguageSubtag::new(['f', 'i', 'e']),
    LanguageSubtag::new(['f', 'i', 'f']),
    LanguageSubtag::new(['f', 'i', 'l']),
    LanguageSubtag::new(['f', 'i', 'p']),
    LanguageSubtag::new(['f', 'i', 'r']),
    LanguageSubtag::new(['f', 'i', 't']),
    LanguageSubtag::new(['f', 'i', 'u']),
    LanguageSubtag::new(['f', 'i', 'w']),
    LanguageSubtag::new(['f', 'j', ' ']),
    LanguageSubtag::new(['f', 'k', 'k']),
    LanguageSubtag::new(['f', 'k', 'v']),
    LanguageSubtag::new(['f', 'l', 'a']),
    LanguageSubtag::new(['f', 'l', 'h']),
    LanguageSubtag::new(['f', 'l', 'i']),
    LanguageSubtag::new(['f', 'l', 'l']),
    LanguageSubtag::new(['f', 'l', 'n']),
    LanguageSubtag::new(['f', 'l', 'r']),
    LanguageSubtag::new(['f', 'l', 'y']),
    LanguageSubtag::new(['f', 'm', 'p']),
    LanguageSubtag::new(['f', 'm', 'u']),
    LanguageSubtag::new(['f', 'n', 'b']),
    LanguageSubtag::new(['f', 'n', 'g']),
    LanguageSubtag::new(['f', 'n', 'i']),
    LanguageSubtag::new(['f', 'o', ' ']),
    LanguageSubtag::new(['f', 'o', 'd']),
    LanguageSubtag::new(['f', 'o', 'i']),
    LanguageSubtag::new(['f', 'o', 'm']),
    LanguageSubtag::new(['f', 'o', 'n']),
    LanguageSubtag::new(['f', 'o', 'r']),
    LanguageSubtag::new(['f', 'o', 's']),
    LanguageSubtag::new(['f', 'o', 'x']),
    LanguageSubtag::new(['f', 'p', 'e']),
    LanguageSubtag::new(['f', 'q', 's']),
    LanguageSubtag::new(['f', 'r', ' ']),
    LanguageSubtag::new(['f', 'r', 'c']),
    LanguageSubtag::new(['f', 'r', 'd']),
    LanguageSubtag::new(['f', 'r', 'k']),
    LanguageSubtag::new(['f', 'r', 'm']),
    LanguageSubtag::new(['f', 'r', 'o']),
    LanguageSubtag::new(['f', 'r', 'p']),
    LanguageSubtag::new(['f', 'r', 'q']),
    LanguageSubtag::new(['f', 'r', 'r']),
    LanguageSubtag::new(['f', 'r', 's']),
    LanguageSubtag::new(['f', 'r', 't']),
    LanguageSubtag::new(['f', 's', 'e']),
    LanguageSubtag::new(['f', 's', 'l']),
    LanguageSubtag::new(['f', 's', 's']),
    LanguageSubtag::new(['f', 'u', 'b']),
    LanguageSubtag::new(['f', 'u', 'c']),
    LanguageSubtag::new(['f', 'u', 'd']),
    LanguageSubtag::new(['f', 'u', 'e']),
    LanguageSubtag::new(['f', 'u', 'f']),
    LanguageSubtag::new(['f', 'u', 'h']),
    LanguageSubtag::new(['f', 'u', 'i']),
    LanguageSubtag::new(['f', 'u', 'j']),
    LanguageSubtag::new(['f', 'u', 'm']),
    LanguageSubtag::new(['f', 'u', 'n']),
    LanguageSubtag::new(['f', 'u', 'q']),
    LanguageSubtag::new(['f', 'u', 'r']),
    LanguageSubtag::new(['f', 'u', 't']),
    LanguageSubtag::new(['f', 'u', 'u']),
    LanguageSubtag::new(['f', 'u', 'v']),
    LanguageSubtag::new(['f', 'u', 'y']),
    LanguageSubtag::new(['f', 'v', 'r']),
    LanguageSubtag::new(['f', 'w', 'a']),
    LanguageSubtag::new(['f', 'w', 'e']),
    LanguageSubtag::new(['f', 'y', ' ']),
    LanguageSubtag::new(['g', 'a', ' ']),
    LanguageSubtag::new(['g', 'a', 'a']),
    LanguageSubtag::new(['g', 'a', 'b']),
    LanguageSubtag::new(['g', 'a', 'c']),
    LanguageSubtag::new(['g', 'a', 'd']),
    LanguageSubtag::new(['g', 'a', 'e']),
    LanguageSubtag::new(['g', 'a', 'f']),
    LanguageSubtag::new(['g', 'a', 'g']),
    LanguageSubtag::new(['g', 'a', 'h']),
    LanguageSubtag::new(['g', 'a', 'i']),
    LanguageSubtag::new(['g', 'a', 'j']),
    LanguageSubtag::new(['g', 'a', 'k']),
    LanguageSubtag::new(['g', 'a', 'l']),
    LanguageSubtag::new(['g', 'a', 'm']),
    LanguageSubtag::new(['g', 'a', 'n']),
    LanguageSubtag::new(['g', 'a', 'o']),
    LanguageSubtag::new(['g', 'a', 'p']),
    LanguageSubtag::new(['g', 'a', 'q']),
    LanguageSubtag::new(['g', 'a', 'r']),
    LanguageSubtag::new(['g', 'a', 's']),
    LanguageSubtag::new(['g', 'a', 't']),
    LanguageSubtag::new(['g', 'a', 'u']),
    LanguageSubtag::new(['g', 'a', 'v']),
    LanguageSubtag::new(['g', 'a', 'w']),
    LanguageSubtag::new(['g', 'a', 'x']),
    LanguageSubtag::new(['g', 'a', 'y']),
    LanguageSubtag::new(['g', 'a', 'z']),
    LanguageSubtag::new(['g', 'b', 'a']),
    LanguageSubtag::new(['g', 'b', 'b']),
    LanguageSubtag::new(['g', 'b', 'c']),
    LanguageSubtag::new(['g', 'b', 'd']),
    LanguageSubtag::new(['g', 'b', 'e']),
    LanguageSubtag::new(['g', 'b', 'f']),
    LanguageSubtag::new(['g', 'b', 'g']),
    LanguageSubtag::new(['g', 'b', 'h']),
    LanguageSubtag::new(['g', 'b', 'i']),
    LanguageSubtag::new(['g', 'b', 'j']),
    LanguageSubtag::new(['g', 'b', 'k']),
    LanguageSubtag::new(['g', 'b', 'l']),
    LanguageSubtag::new(['g', 'b', 'm']),
    LanguageSubtag::new(['g', 'b', 'n']),
    LanguageSubtag::new(['g', 'b', 'o']),
    LanguageSubtag::new(['g', 'b', 'p']),
    LanguageSubtag::new(['g', 'b', 'q']),
    LanguageSubtag::new(['g', 'b', 'r']),
    LanguageSubtag::new(['g', 'b', 's']),
    LanguageSubtag::new(['g', 'b', 'u']),
    LanguageSubtag::new(['g', 'b', 'v']),
    LanguageSubtag::new(['g', 'b', 'w']),
    LanguageSubtag::new(['g', 'b', 'x']),
    LanguageSubtag::new(['g', 'b', 'y']),
    LanguageSubtag::new(['g', 'b', 'z']),
    LanguageSubtag::new(['g', 'c', 'c']),
    LanguageSubtag::new(['g', 'c', 'd']),
    LanguageSubtag::new(['g', 'c', 'e']),
    LanguageSubtag::new(['g', 'c', 'f']),
    LanguageSubtag::new(['g', 'c', 'l']),
    LanguageSubtag::new(['g', 'c', 'n']),
    LanguageSubtag::new(['g', 'c', 'r']),
    LanguageSubtag::new(['g', 'c', 't']),
    LanguageSubtag::new(['g', 'd', ' ']),
    LanguageSubtag::new(['g', 'd', 'a']),
    LanguageSubtag::new(['g', 'd', 'b']),
    LanguageSubtag::new(['g', 'd', 'c']),
    LanguageSubtag::new(['g', 'd', 'd']),
    LanguageSubtag::new(['g', 'd', 'e']),
    LanguageSubtag::new(['g', 'd', 'f']),
    LanguageSubtag::new(['g', 'd', 'g']),
    LanguageSubtag::new(['g', 'd', 'h']),
    LanguageSubtag::new(['g', 'd', 'i']),
    LanguageSubtag::new(['g', 'd', 'j']),
    LanguageSubtag::new(['g', 'd', 'k']),
    LanguageSubtag::new(['g', 'd', 'l']),
    LanguageSubtag::new(['g', 'd', 'm']),
    LanguageSubtag::new(['g', 'd', 'n']),
    LanguageSubtag::new(['g', 'd', 'o']),
    LanguageSubtag::new(['g', 'd', 'q']),
    LanguageSubtag::new(['g', 'd', 'r']),
    LanguageSubtag::new(['g', 'd', 's']),
    LanguageSubtag::new(['g', 'd', 't']),
    LanguageSubtag::new(['g', 'd', 'u']),
    LanguageSubtag::new(['g', 'd', 'x']),
    LanguageSubtag::new(['g', 'e', 'a']),
    LanguageSubtag::new(['g', 'e', 'b']),
    LanguageSubtag::new(['g', 'e', 'c']),
    LanguageSubtag::new(['g', 'e', 'd']),
    LanguageSubtag::new(['g', 'e', 'f']),
    LanguageSubtag::new(['g', 'e', 'g']),
    LanguageSubtag::new(['g', 'e', 'h']),
    LanguageSubtag::new(['g', 'e', 'i']),
    LanguageSubtag::new(['g', 'e', 'j']),
    LanguageSubtag::new(['g', 'e', 'k']),
    LanguageSubtag::new(['g', 'e', 'l']),
    LanguageSubtag::new(['g', 'e', 'm']),
    LanguageSubtag::new(['g', 'e', 'q']),
    LanguageSubtag::new(['g', 'e', 's']),
    LanguageSubtag::new(['g', 'e', 'v']),
    LanguageSubtag::new(['g', 'e', 'w']),
    LanguageSubtag::new(['g', 'e', 'x']),
    LanguageSubtag::new(['g', 'e', 'y']),
    LanguageSubtag::new(['g', 'e', 'z']),
    LanguageSubtag::new(['g', 'f', 'k']),
    LanguageSubtag::new(['g', 'f', 't']),
    LanguageSubtag::new(['g', 'f', 'x']),
    LanguageSubtag::new(['g', 'g', 'a']),
    LanguageSubtag::new(['g', 'g', 'b']),
    LanguageSubtag::new(['g', 'g', 'd']),
    LanguageSubtag::new(['g', 'g', 'e']),
    LanguageSubtag::new(['g', 'g', 'g']),
    LanguageSubtag::new(['g', 'g', 'k']),
    LanguageSubtag::new(['g', 'g', 'l']),
    LanguageSubtag::new(['g', 'g', 'n']),
    LanguageSubtag::new(['g', 'g', 'o']),
    LanguageSubtag::new(['g', 'g', 'r']),
    LanguageSubtag::new(['g', 'g', 't']),
    LanguageSubtag::new(['g', 'g', 'u']),
    LanguageSubtag::new(['g', 'g', 'w']),
    LanguageSubtag::new(['g', 'h', 'a']),
    LanguageSubtag::new(['g', 'h', 'c']),
    LanguageSubtag::new(['g', 'h', 'e']),
    LanguageSubtag::new(['g', 'h', 'h']),
    LanguageSubtag::new(['g', 'h', 'k']),
    LanguageSubtag::new(['g', 'h', 'l']),
    LanguageSubtag::new(['g', 'h', 'n']),
    LanguageSubtag::new(['g', 'h', 'o']),
    LanguageSubtag::new(['g', 'h', 'r']),
    LanguageSubtag::new(['g', 'h', 's']),
    LanguageSubtag::new(['g', 'h', 't']),
    LanguageSubtag::new(['g', 'i', 'a']),
    LanguageSubtag::new(['g', 'i', 'b']),
    LanguageSubtag::new(['g', 'i', 'c']),
    LanguageSubtag::new(['g', 'i', 'd']),
    LanguageSubtag::new(['g', 'i', 'e']),
    LanguageSubtag::new(['g', 'i', 'g']),
    LanguageSubtag::new(['g', 'i', 'h']),
    LanguageSubtag::new(['g', 'i', 'i']),
    LanguageSubtag::new(['g', 'i', 'l']),
    LanguageSubtag::new(['g', 'i', 'm']),
    LanguageSubtag::new(['g', 'i', 'n']),
    LanguageSubtag::new(['g', 'i', 'o']),
    LanguageSubtag::new(['g', 'i', 'p']),
    LanguageSubtag::new(['g', 'i', 'q']),
    LanguageSubtag::new(['g', 'i', 'r']),
    LanguageSubtag::new(['g', 'i', 's']),
    LanguageSubtag::new(['g', 'i', 't']),
    LanguageSubtag::new(['g', 'i', 'u']),
    LanguageSubtag::new(['g', 'i', 'w']),
    LanguageSubtag::new(['g', 'i', 'x']),
    LanguageSubtag::new(['g', 'i', 'y']),
    LanguageSubtag::new(['g', 'i', 'z']),
    LanguageSubtag::new(['g', 'j', 'i']),
    LanguageSubtag::new(['g', 'j', 'k']),
    LanguageSubtag::new(['g', 'j', 'm']),
    LanguageSubtag::new(['g', 'j', 'n']),
    LanguageSubtag::new(['g', 'j', 'r']),
    LanguageSubtag::new(['g', 'j', 'u']),
    LanguageSubtag::new(['g', 'k', 'a']),
    LanguageSubtag::new(['g', 'k', 'd']),
    LanguageSubtag::new(['g', 'k', 'e']),
    LanguageSubtag::new(['g', 'k', 'n']),
    LanguageSubtag::new(['g', 'k', 'o']),
    LanguageSubtag::new(['g', 'k', 'p']),
    LanguageSubtag::new(['g', 'k', 'u']),
    LanguageSubtag::new(['g', 'l', ' ']),
    LanguageSubtag::new(['g', 'l', 'b']),
    LanguageSubtag::new(['g', 'l', 'c']),
    LanguageSubtag::new(['g', 'l', 'd']),
    LanguageSubtag::new(['g', 'l', 'h']),
    LanguageSubtag::new(['g', 'l', 'i']),
    LanguageSubtag::new(['g', 'l', 'j']),
    LanguageSubtag::new(['g', 'l', 'k']),
    LanguageSubtag::new(['g', 'l', 'l']),
    LanguageSubtag::new(['g', 'l', 'o']),
    LanguageSubtag::new(['g', 'l', 'r']),
    LanguageSubtag::new(['g', 'l', 'u']),
    LanguageSubtag::new(['g', 'l', 'w']),
    LanguageSubtag::new(['g', 'l', 'y']),
    LanguageSubtag::new(['g', 'm', 'a']),
    LanguageSubtag::new(['g', 'm', 'b']),
    LanguageSubtag::new(['g', 'm', 'd']),
    LanguageSubtag::new(['g', 'm', 'e']),
    LanguageSubtag::new(['g', 'm', 'g']),
    LanguageSubtag::new(['g', 'm', 'h']),
    LanguageSubtag::new(['g', 'm', 'l']),
    LanguageSubtag::new(['g', 'm', 'm']),
    LanguageSubtag::new(['g', 'm', 'n']),
    LanguageSubtag::new(['g', 'm', 'q']),
    LanguageSubtag::new(['g', 'm', 'r']),
    LanguageSubtag::new(['g', 'm', 'u']),
    LanguageSubtag::new(['g', 'm', 'v']),
    LanguageSubtag::new(['g', 'm', 'w']),
    LanguageSubtag::new(['g', 'm', 'x']),
    LanguageSubtag::new(['g', 'm', 'y']),
    LanguageSubtag::new(['g', 'm', 'z']),
    LanguageSubtag::new(['g', 'n', ' ']),
    LanguageSubtag::new(['g', 'n', 'a']),
    LanguageSubtag::new(['g', 'n', 'b']),
    LanguageSubtag::new(['g', 'n', 'c']),
    LanguageSubtag::new(['g', 'n', 'd']),
    LanguageSubtag::new(['g', 'n', 'e']),
    LanguageSubtag::new(['g', 'n', 'g']),
    LanguageSubtag::new(['g', 'n', 'h']),
    LanguageSubtag::new(['g', 'n', 'i']),
    LanguageSubtag::new(['g', 'n', 'j']),
    LanguageSubtag::new(['g', 'n', 'k']),
    LanguageSubtag::new(['g', 'n', 'l']),
    LanguageSubtag::new(['g', 'n', 'm']),
    LanguageSubtag::new(['g', 'n', 'n']),
    LanguageSubtag::new(['g', 'n', 'o']),
    LanguageSubtag::new(['g', 'n', 'q']),
    LanguageSubtag::new(['g', 'n', 'r']),
    LanguageSubtag::new(['g', 'n', 't']),
    LanguageSubtag::new(['g', 'n', 'u']),
    LanguageSubtag::new(['g', 'n', 'w']),
    LanguageSubtag::new(['g', 'n', 'z']),
    LanguageSubtag::new(['g', 'o', 'a']),
    LanguageSubtag::new(['g', 'o', 'b']),
    LanguageSubtag::new(['g', 'o', 'c']),
    LanguageSubtag::new(['g', 'o', 'd']),
    LanguageSubtag::new(['g', 'o', 'e']),
    LanguageSubtag::new(['g', 'o', 'f']),
    LanguageSubtag::new(['g', 'o', 'g']),
    LanguageSubtag::new(['g', 'o', 'h']),
    LanguageSubtag::new(['g', 'o', 'i']),
    LanguageSubtag::new(['g', 'o', 'j']),
    LanguageSubtag::new(['g', 'o', 'k']),
    LanguageSubtag::new(['g', 'o', 'l']),
    LanguageSubtag::new(['g', 'o', 'm']),
    LanguageSubtag::new(['g', 'o', 'n']),
    LanguageSubtag::new(['g', 'o', 'o']),
    LanguageSubtag::new(['g', 'o', 'p']),
    LanguageSubtag::new(['g', 'o', 'q']),
    LanguageSubtag::new(['g', 'o', 'r']),
    LanguageSubtag::new(['g', 'o', 's']),
    LanguageSubtag::new(['g', 'o', 't']),
    LanguageSubtag::new(['g', 'o', 'u']),
    LanguageSubtag::new(['g', 'o', 'w']),
    LanguageSubtag::new(['g', 'o', 'x']),
    LanguageSubtag::new(['g', 'o', 'y']),
    LanguageSubtag::new(['g', 'o', 'z']),
    LanguageSubtag::new(['g', 'p', 'a']),
    LanguageSubtag::new(['g', 'p', 'e']),
    LanguageSubtag::new(['g', 'p', 'n']),
    LanguageSubtag::new(['g', 'q', 'a']),
    LanguageSubtag::new(['g', 'q', 'i']),
    LanguageSubtag::new(['g', 'q', 'n']),
    LanguageSubtag::new(['g', 'q', 'r']),
    LanguageSubtag::new(['g', 'q', 'u']),
    LanguageSubtag::new(['g', 'r', 'a']),
    LanguageSubtag::new(['g', 'r', 'b']),
    LanguageSubtag::new(['g', 'r', 'c']),
    LanguageSubtag::new(['g', 'r', 'd']),
    LanguageSubtag::new(['g', 'r', 'g']),
    LanguageSubtag::new(['g', 'r', 'h']),
    LanguageSubtag::new(['g', 'r', 'i']),
    LanguageSubtag::new(['g', 'r', 'j']),
    LanguageSubtag::new(['g', 'r', 'k']),
    LanguageSubtag::new(['g', 'r', 'm']),
    LanguageSubtag::new(['g', 'r', 'o']),
    LanguageSubtag::new(['g', 'r', 'q']),
    LanguageSubtag::new(['g', 'r', 'r']),
    LanguageSubtag::new(['g', 'r', 's']),
    LanguageSubtag::new(['g', 'r', 't']),
    LanguageSubtag::new(['g', 'r', 'u']),
    LanguageSubtag::new(['g', 'r', 'v']),
    LanguageSubtag::new(['g', 'r', 'w']),
    LanguageSubtag::new(['g', 'r', 'x']),
    LanguageSubtag::new(['g', 'r', 'y']),
    LanguageSubtag::new(['g', 'r', 'z']),
    LanguageSubtag::new(['g', 's', 'e']),
    LanguageSubtag::new(['g', 's', 'g']),
    LanguageSubtag::new(['g', 's', 'l']),
    LanguageSubtag::new(['g', 's', 'm']),
    LanguageSubtag::new(['g', 's', 'n']),
    LanguageSubtag::new(['g', 's', 'o']),
    LanguageSubtag::new(['g', 's', 'p']),
    LanguageSubtag::new(['g', 's', 's']),
    LanguageSubtag::new(['g', 's', 'w']),
    LanguageSubtag::new(['g', 't', 'a']),
    LanguageSubtag::new(['g', 't', 'i']),
    LanguageSubtag::new(['g', 't', 'u']),
    LanguageSubtag::new(['g', 'u', ' ']),
    LanguageSubtag::new(['g', 'u', 'a']),
    LanguageSubtag::new(['g', 'u', 'b']),
    LanguageSubtag::new(['g', 'u', 'c']),
    LanguageSubtag::new(['g', 'u', 'd']),
    LanguageSubtag::new(['g', 'u', 'e']),
    LanguageSubtag::new(['g', 'u', 'f']),
    LanguageSubtag::new(['g', 'u', 'g']),
    LanguageSubtag::new(['g', 'u', 'h']),
    LanguageSubtag::new(['g', 'u', 'i']),
    LanguageSubtag::new(['g', 'u', 'k']),
    LanguageSubtag::new(['g', 'u', 'l']),
    LanguageSubtag::new(['g', 'u', 'm']),
    LanguageSubtag::new(['g', 'u', 'n']),
    LanguageSubtag::new(['g', 'u', 'o']),
    LanguageSubtag::new(['g', 'u', 'p']),
    LanguageSubtag::new(['g', 'u', 'q']),
    LanguageSubtag::new(['g', 'u', 'r']),
    LanguageSubtag::new(['g', 'u', 's']),
    LanguageSubtag::new(['g', 'u', 't']),
    LanguageSubtag::new(['g', 'u', 'u']),
    LanguageSubtag::new(['g', 'u', 'v']),
    LanguageSubtag::new(['g', 'u', 'w']),
    LanguageSubtag::new(['g', 'u', 'x']),
    LanguageSubtag::new(['g', 'u', 'z']),
    LanguageSubtag::new(['g', 'v', ' ']),
    LanguageSubtag::new(['g', 'v', 'a']),
    LanguageSubtag::new(['g', 'v', 'c']),
    LanguageSubtag::new(['g', 'v', 'e']),
    LanguageSubtag::new(['g', 'v', 'f']),
    LanguageSubtag::new(['g', 'v', 'j']),
    LanguageSubtag::new(['g', 'v', 'l']),
    LanguageSubtag::new(['g', 'v', 'm']),
    LanguageSubtag::new(['g', 'v', 'n']),
    LanguageSubtag::new(['g', 'v', 'o']),
    LanguageSubtag::new(['g', 'v', 'p']),
    LanguageSubtag::new(['g', 'v', 'r']),
    LanguageSubtag::new(['g', 'v', 's']),
    LanguageSubtag::new(['g', 'v', 'y']),
    LanguageSubtag::new(['g', 'w', 'a']),
    LanguageSubtag::new(['g', 'w', 'b']),
    LanguageSubtag::new(['g', 'w', 'c']),
    LanguageSubtag::new(['g', 'w', 'd']),
    LanguageSubtag::new(['g', 'w', 'e']),
    LanguageSubtag::new(['g', 'w', 'f']),
    LanguageSubtag::new(['g', 'w', 'g']),
    LanguageSubtag::new(['g', 'w', 'i']),
    LanguageSubtag::new(['g', 'w', 'j']),
    LanguageSubtag::new(['g', 'w', 'm']),
    LanguageSubtag::new(['g', 'w', 'n']),
    LanguageSubtag::new(['g', 'w', 'r']),
    LanguageSubtag::new(['g', 'w', 't']),
    LanguageSubtag::new(['g', 'w', 'u']),
    LanguageSubtag::new(['g', 'w', 'w']),
    LanguageSubtag::new(['g', 'w', 'x']),
    LanguageSubtag::new(['g', 'x', 'x']),
    LanguageSubtag::new(['g', 'y', 'a']),
    LanguageSubtag::new(['g', 'y', 'b']),
    LanguageSubtag::new(['g', 'y', 'd']),
    LanguageSubtag::new(['g', 'y', 'e']),
    LanguageSubtag::new(['g', 'y', 'f']),
    LanguageSubtag::new(['g', 'y', 'g']),
    LanguageSubtag::new(['g', 'y', 'i']),
    LanguageSubtag::new(['g', 'y', 'l']),
    LanguageSubtag::new(['g', 'y', 'm']),
    LanguageSubtag::new(['g', 'y', 'n']),
    LanguageSubtag::new(['g', 'y', 'o']),
    LanguageSubtag::new(['g', 'y', 'r']),
    LanguageSubtag::new(['g', 'y', 'y']),
    LanguageSubtag::new(['g', 'y', 'z']),
    LanguageSubtag::new(['g', 'z', 'a']),
    LanguageSubtag::new(['g', 'z', 'i']),
    LanguageSubtag::new(['g', 'z', 'n']),
    LanguageSubtag::new(['h', 'a', ' ']),
    LanguageSubtag::new(['h', 'a', 'a']),
    LanguageSubtag::new(['h', 'a', 'b']),
    LanguageSubtag::new(['h', 'a', 'c']),
    LanguageSubtag::new(['h', 'a', 'd']),
    LanguageSubtag::new(['h', 'a', 'e']),
    LanguageSubtag::new(['h', 'a', 'f']),
    LanguageSubtag::new(['h', 'a', 'g']),
    LanguageSubtag::new(['h', 'a', 'h']),
    LanguageSubtag::new(['h', 'a', 'i']),
    LanguageSubtag::new(['h', 'a', 'j']),
    LanguageSubtag::new(['h', 'a', 'k']),
    LanguageSubtag::new(['h', 'a', 'l']),
    LanguageSubtag::new(['h', 'a', 'm']),
    LanguageSubtag::new(['h', 'a', 'n']),
    LanguageSubtag::new(['h', 'a', 'o']),
    LanguageSubtag::new(['h', 'a', 'p']),
    LanguageSubtag::new(['h', 'a', 'q']),
    LanguageSubtag::new(['h', 'a', 'r']),
    LanguageSubtag::new(['h', 'a', 's']),
    LanguageSubtag::new(['h', 'a', 'v']),
    LanguageSubtag::new(['h', 'a', 'w']),
    LanguageSubtag::new(['h', 'a', 'x']),
    LanguageSubtag::new(['h', 'a', 'y']),
    LanguageSubtag::new(['h', 'a', 'z']),
    LanguageSubtag::new(['h', 'b', 'a']),
    LanguageSubtag::new(['h', 'b', 'b']),
    LanguageSubtag::new(['h', 'b', 'n']),
    LanguageSubtag::new(['h', 'b', 'o']),
    LanguageSubtag::new(['h', 'b', 'u']),
    LanguageSubtag::new(['h', 'c', 'a']),
    LanguageSubtag::new(['h', 'c', 'h']),
    LanguageSubtag::new(['h', 'd', 'n']),
    LanguageSubtag::new(['h', 'd', 's']),
    LanguageSubtag::new(['h', 'd', 'y']),
    LanguageSubtag::new(['h', 'e', ' ']),
    LanguageSubtag::new(['h', 'e', 'a']),
    LanguageSubtag::new(['h', 'e', 'd']),
    LanguageSubtag::new(['h', 'e', 'g']),
    LanguageSubtag::new(['h', 'e', 'h']),
    LanguageSubtag::new(['h', 'e', 'i']),
    LanguageSubtag::new(['h', 'e', 'm']),
    LanguageSubtag::new(['h', 'g', 'm']),
    LanguageSubtag::new(['h', 'g', 'w']),
    LanguageSubtag::new(['h', 'h', 'i']),
    LanguageSubtag::new(['h', 'h', 'r']),
    LanguageSubtag::new(['h', 'h', 'y']),
    LanguageSubtag::new(['h', 'i', ' ']),
    LanguageSubtag::new(['h', 'i', 'a']),
    LanguageSubtag::new(['h', 'i', 'b']),
    LanguageSubtag::new(['h', 'i', 'd']),
    LanguageSubtag::new(['h', 'i', 'f']),
    LanguageSubtag::new(['h', 'i', 'g']),
    LanguageSubtag::new(['h', 'i', 'h']),
    LanguageSubtag::new(['h', 'i', 'i']),
    LanguageSubtag::new(['h', 'i', 'j']),
    LanguageSubtag::new(['h', 'i', 'k']),
    LanguageSubtag::new(['h', 'i', 'l']),
    LanguageSubtag::new(['h', 'i', 'm']),
    LanguageSubtag::new(['h', 'i', 'o']),
    LanguageSubtag::new(['h', 'i', 'r']),
    LanguageSubtag::new(['h', 'i', 't']),
    LanguageSubtag::new(['h', 'i', 'w']),
    LanguageSubtag::new(['h', 'i', 'x']),
    LanguageSubtag::new(['h', 'j', 'i']),
    LanguageSubtag::new(['h', 'k', 'a']),
    LanguageSubtag::new(['h', 'k', 'e']),
    LanguageSubtag::new(['h', 'k', 'h']),
    LanguageSubtag::new(['h', 'k', 'k']),
    LanguageSubtag::new(['h', 'k', 'n']),
    LanguageSubtag::new(['h', 'k', 's']),
    LanguageSubtag::new(['h', 'l', 'a']),
    LanguageSubtag::new(['h', 'l', 'b']),
    LanguageSubtag::new(['h', 'l', 'd']),
    LanguageSubtag::new(['h', 'l', 'e']),
    LanguageSubtag::new(['h', 'l', 't']),
    LanguageSubtag::new(['h', 'l', 'u']),
    LanguageSubtag::new(['h', 'm', 'a']),
    LanguageSubtag::new(['h', 'm', 'b']),
    LanguageSubtag::new(['h', 'm', 'c']),
    LanguageSubtag::new(['h', 'm', 'd']),
    LanguageSubtag::new(['h', 'm', 'e']),
    LanguageSubtag::new(['h', 'm', 'f']),
    LanguageSubtag::new(['h', 'm', 'g']),
    LanguageSubtag::new(['h', 'm', 'h']),
    LanguageSubtag::new(['h', 'm', 'i']),
    LanguageSubtag::new(['h', 'm', 'j']),
    LanguageSubtag::new(['h', 'm', 'k']),
    LanguageSubtag::new(['h', 'm', 'l']),
    LanguageSubtag::new(['h', 'm', 'm']),
    LanguageSubtag::new(['h', 'm', 'n']),
    LanguageSubtag::new(['h', 'm', 'p']),
    LanguageSubtag::new(['h', 'm', 'q']),
    LanguageSubtag::new(['h', 'm', 'r']),
    LanguageSubtag::new(['h', 'm', 's']),
    LanguageSubtag::new(['h', 'm', 't']),
    LanguageSubtag::new(['h', 'm', 'u']),
    LanguageSubtag::new(['h', 'm', 'v']),
    LanguageSubtag::new(['h', 'm', 'w']),
    LanguageSubtag::new(['h', 'm', 'x']),
    LanguageSubtag::new(['h', 'm', 'y']),
    LanguageSubtag::new(['h', 'm', 'z']),
    LanguageSubtag::new(['h', 'n', 'a']),
    LanguageSubtag::new(['h', 'n', 'd']),
    LanguageSubtag::new(['h', 'n', 'e']),
    LanguageSubtag::new(['h', 'n', 'g']),
    LanguageSubtag::new(['h', 'n', 'h']),
    LanguageSubtag::new(['h', 'n', 'i']),
    LanguageSubtag::new(['h', 'n', 'j']),
    LanguageSubtag::new(['h', 'n', 'n']),
    LanguageSubtag::new(['h', 'n', 'o']),
    LanguageSubtag::new(['h', 'n', 's']),
    LanguageSubtag::new(['h', 'n', 'u']),
    LanguageSubtag::new(['h', 'o', ' ']),
    LanguageSubtag::new(['h', 'o', 'a']),
    LanguageSubtag::new(['h', 'o', 'b']),
    LanguageSubtag::new(['h', 'o', 'c']),
    LanguageSubtag::new(['h', 'o', 'd']),
    LanguageSubtag::new(['h', 'o', 'e']),
    LanguageSubtag::new(['h', 'o', 'h']),
    LanguageSubtag::new(['h', 'o', 'i']),
    LanguageSubtag::new(['h', 'o', 'j']),
    LanguageSubtag::new(['h', 'o', 'k']),
    LanguageSubtag::new(['h', 'o', 'l']),
    LanguageSubtag::new(['h', 'o', 'm']),
    LanguageSubtag::new(['h', 'o', 'o']),
    LanguageSubtag::new(['h', 'o', 'p']),
    LanguageSubtag::new(['h', 'o', 'r']),
    LanguageSubtag::new(['h', 'o', 's']),
    LanguageSubtag::new(['h', 'o', 't']),
    LanguageSubtag::new(['h', 'o', 'v']),
    LanguageSubtag::new(['h', 'o', 'w']),
    LanguageSubtag::new(['h', 'o', 'y']),
    LanguageSubtag::new(['h', 'o', 'z']),
    LanguageSubtag::new(['h', 'p', 'o']),
    LanguageSubtag::new(['h', 'p', 's']),
    LanguageSubtag::new(['h', 'r', ' ']),
    LanguageSubtag::new(['h', 'r', 'a']),
    LanguageSubtag::new(['h', 'r', 'c']),
    LanguageSubtag::new(['h', 'r', 'e']),
    LanguageSubtag::new(['h', 'r', 'k']),
    LanguageSubtag::new(['h', 'r', 'm']),
    LanguageSubtag::new(['h', 'r', 'o']),
    LanguageSubtag::new(['h', 'r', 'p']),
    LanguageSubtag::new(['h', 'r', 'r']),
    LanguageSubtag::new(['h', 'r', 't']),
    LanguageSubtag::new(['h', 'r', 'u']),
    LanguageSubtag::new(['h', 'r', 'w']),
    LanguageSubtag::new(['h', 'r', 'x']),
    LanguageSubtag::new(['h', 'r', 'z']),
    LanguageSubtag::new(['h', 's', 'b']),
    LanguageSubtag::new(['h', 's', 'h']),
    LanguageSubtag::new(['h', 's', 'l']),
    LanguageSubtag::new(['h', 's', 'n']),
    LanguageSubtag::new(['h', 's', 's']),
    LanguageSubtag::new(['h', 't', ' ']),
    LanguageSubtag::new(['h', 't', 'i']),
    LanguageSubtag::new(['h', 't', 'o']),
    LanguageSubtag::new(['h', 't', 's']),
    LanguageSubtag::new(['h', 't', 'u']),
    LanguageSubtag::new(['h', 't', 'x']),
    LanguageSubtag::new(['h', 'u', ' ']),
    LanguageSubtag::new(['h', 'u', 'b']),
    LanguageSubtag::new(['h', 'u', 'c']),
    LanguageSubtag::new(['h', 'u', 'd']),
    LanguageSubtag::new(['h', 'u', 'e']),
    LanguageSubtag::new(['h', 'u', 'f']),
    LanguageSubtag::new(['h', 'u', 'g']),
    LanguageSubtag::new(['h', 'u', 'h']),
    LanguageSubtag::new(['h', 'u', 'i']),
    LanguageSubtag::new(['h', 'u', 'j']),
    LanguageSubtag::new(['h', 'u', 'k']),
    LanguageSubtag::new(['h', 'u', 'l']),
    LanguageSubtag::new(['h', 'u', 'm']),
    LanguageSubtag::new(['h', 'u', 'o']),
    LanguageSubtag::new(['h', 'u', 'p']),
    LanguageSubtag::new(['h', 'u', 'q']),
    LanguageSubtag::new(['h', 'u', 'r']),
    LanguageSubtag::new(['h', 'u', 's']),
    LanguageSubtag::new(['h', 'u', 't']),
    LanguageSubtag::new(['h', 'u', 'u']),
    LanguageSubtag::new(['h', 'u', 'v']),
    LanguageSubtag::new(['h', 'u', 'w']),
    LanguageSubtag::new(['h', 'u', 'x']),
    LanguageSubtag::new(['h', 'u', 'y']),
    LanguageSubtag::new(['h', 'u', 'z']),
    LanguageSubtag::new(['h', 'v', 'c']),
    LanguageSubtag::new(['h', 'v', 'e']),
    LanguageSubtag::new(['h', 'v', 'k']),
    LanguageSubtag::new(['h', 'v', 'n']),
    LanguageSubtag::new(['h', 'v', 'v']),
    LanguageSubtag::new(['h', 'w', 'a']),
    LanguageSubtag::new(['h', 'w', 'c']),
    LanguageSubtag::new(['h', 'w', 'o']),
    LanguageSubtag::new(['h', 'y', ' ']),
    LanguageSubtag::new(['h', 'y', 'a']),
    LanguageSubtag::new(['h', 'y', 'w']),
    LanguageSubtag::new(['h', 'y', 'x']),
    LanguageSubtag::new(['h', 'z', ' ']),
    LanguageSubtag::new(['i', 'a', ' ']),
    LanguageSubtag::new(['i', 'a', 'i']),
    LanguageSubtag::new(['i', 'a', 'n']),
    LanguageSubtag::new(['i', 'a', 'p']),
    LanguageSubtag::new(['i', 'a', 'r']),
    LanguageSubtag::new(['i', 'b', 'a']),
    LanguageSubtag::new(['i', 'b', 'b']),
    LanguageSubtag::new(['i', 'b', 'd']),
    LanguageSubtag::new(['i', 'b', 'e']),
    LanguageSubtag::new(['i', 'b', 'g']),
    LanguageSubtag::new(['i', 'b', 'h']),
    LanguageSubtag::new(['i', 'b', 'i']),
    LanguageSubtag::new(['i', 'b', 'l']),
    LanguageSubtag::new(['i', 'b', 'm']),
    LanguageSubtag::new(['i', 'b', 'n']),
    LanguageSubtag::new(['i', 'b', 'r']),
    LanguageSubtag::new(['i', 'b', 'u']),
    LanguageSubtag::new(['i', 'b', 'y']),
    LanguageSubtag::new(['i', 'c', 'a']),
    LanguageSubtag::new(['i', 'c', 'h']),
    LanguageSubtag::new(['i', 'c', 'l']),
    LanguageSubtag::new(['i', 'c', 'r']),
    LanguageSubtag::new(['i', 'd', ' ']),
    LanguageSubtag::new(['i', 'd', 'a']),
    LanguageSubtag::new(['i', 'd', 'b']),
    LanguageSubtag::new(['i', 'd', 'c']),
    LanguageSubtag::new(['i', 'd', 'd']),
    LanguageSubtag::new(['i', 'd', 'e']),
    LanguageSubtag::new(['i', 'd', 'i']),
    LanguageSubtag::new(['i', 'd', 'r']),
    LanguageSubtag::new(['i', 'd', 's']),
    LanguageSubtag::new(['i', 'd', 't']),
    LanguageSubtag::new(['i', 'd', 'u']),
    LanguageSubtag::new(['i', 'e', ' ']),
    LanguageSubtag::new(['i', 'f', 'a']),
    LanguageSubtag::new(['i', 'f', 'b']),
    LanguageSubtag::new(['i', 'f', 'e']),
    LanguageSubtag::new(['i', 'f', 'f']),
    LanguageSubtag::new(['i', 'f', 'k']),
    LanguageSubtag::new(['i', 'f', 'm']),
    LanguageSubtag::new(['i', 'f', 'u']),
    LanguageSubtag::new(['i', 'f', 'y']),
    LanguageSubtag::new(['i', 'g', ' ']),
    LanguageSubtag::new(['i', 'g', 'b']),
    LanguageSubtag::new(['i', 'g', 'e']),
    LanguageSubtag::new(['i', 'g', 'g']),
    LanguageSubtag::new(['i', 'g', 'l']),
    LanguageSubtag::new(['i', 'g', 'm']),
    LanguageSubtag::new(['i', 'g', 'n']),
    LanguageSubtag::new(['i', 'g', 'o']),
    LanguageSubtag::new(['i', 'g', 's']),
    LanguageSubtag::new(['i', 'g', 'w']),
    LanguageSubtag::new(['i', 'h', 'b']),
    LanguageSubtag::new(['i', 'h', 'i']),
    LanguageSubtag::new(['i', 'h', 'p']),
    LanguageSubtag::new(['i', 'h', 'w']),
    LanguageSubtag::new(['i', 'i', ' ']),
    LanguageSubtag::new(['i', 'i', 'n']),
    LanguageSubtag::new(['i', 'i', 'r']),
    LanguageSubtag::new(['i', 'j', 'c']),
    LanguageSubtag::new(['i', 'j', 'e']),
    LanguageSubtag::new(['i', 'j', 'j']),
    LanguageSubtag::new(['i', 'j', 'n']),
    LanguageSubtag::new(['i', 'j', 'o']),
    LanguageSubtag::new(['i', 'j', 's']),
    LanguageSubtag::new(['i', 'k', ' ']),
    LanguageSubtag::new(['i', 'k', 'e']),
    LanguageSubtag::new(['i', 'k', 'i']),
    LanguageSubtag::new(['i', 'k', 'k']),
    LanguageSubtag::new(['i', 'k', 'l']),
    LanguageSubtag::new(['i', 'k', 'o']),
    LanguageSubtag::new(['i', 'k', 'p']),
    LanguageSubtag::new(['i', 'k', 'r']),
    LanguageSubtag::new(['i', 'k', 's']),
    LanguageSubtag::new(['i', 'k', 't']),
    LanguageSubtag::new(['i', 'k', 'v']),
    LanguageSubtag::new(['i', 'k', 'w']),
    LanguageSubtag::new(['i', 'k', 'x']),
    LanguageSubtag::new(['i', 'k', 'z']),
    LanguageSubtag::new(['i', 'l', 'a']),
    LanguageSubtag::new(['i', 'l', 'b']),
    LanguageSubtag::new(['i', 'l', 'g']),
    LanguageSubtag::new(['i', 'l', 'i']),
    LanguageSubtag::new(['i', 'l', 'k']),
    LanguageSubtag::new(['i', 'l', 'l']),
    LanguageSubtag::new(['i', 'l', 'm']),
    LanguageSubtag::new(['i', 'l', 'o']),
    LanguageSubtag::new(['i', 'l', 'p']),
    LanguageSubtag::new(['i', 'l', 's']),
    LanguageSubtag::new(['i', 'l', 'u']),
    LanguageSubtag::new(['i', 'l', 'v']),
    LanguageSubtag::new(['i', 'l', 'w']),
    LanguageSubtag::new(['i', 'm', 'a']),
    LanguageSubtag::new(['i', 'm', 'e']),
    LanguageSubtag::new(['i', 'm', 'i']),
    LanguageSubtag::new(['i', 'm', 'l']),
    LanguageSubtag::new(['i', 'm', 'n']),
    LanguageSubtag::new(['i', 'm', 'o']),
    LanguageSubtag::new(['i', 'm', 'r']),
    LanguageSubtag::new(['i', 'm', 's']),
    LanguageSubtag::new(['i', 'm', 'y']),
    LanguageSubtag::new(['i', 'n', ' ']),
    LanguageSubtag::new(['i', 'n', 'b']),
    LanguageSubtag::new(['i', 'n', 'c']),
    LanguageSubtag::new(['i', 'n', 'e']),
    LanguageSubtag::new(['i', 'n', 'g']),
    LanguageSubtag::new(['i', 'n', 'h']),
    LanguageSubtag::new(['i', 'n', 'j']),
    LanguageSubtag::new(['i', 'n', 'l']),
    LanguageSubtag::new(['i', 'n', 'm']),
    LanguageSubtag::new(['i', 'n', 'n']),
    LanguageSubtag::new(['i', 'n', 'o']),
    LanguageSubtag::new(['i', 'n', 'p']),
    LanguageSubtag::new(['i', 'n', 's']),
    LanguageSubtag::new(['i', 'n', 't']),
    LanguageSubtag::new(['i', 'n', 'z']),
    LanguageSubtag::new(['i', 'o', ' ']),
    LanguageSubtag::new(['i', 'o', 'r']),
    LanguageSubtag::new(['i', 'o', 'u']),
    LanguageSubtag::new(['i', 'o', 'w']),
    LanguageSubtag::new(['i', 'p', 'i']),
    LanguageSubtag::new(['i', 'p', 'o']),
    LanguageSubtag::new(['i', 'q', 'u']),
    LanguageSubtag::new(['i', 'q', 'w']),
    LanguageSubtag::new(['i', 'r', 'a']),
    LanguageSubtag::new(['i', 'r', 'e']),
    LanguageSubtag::new(['i', 'r', 'h']),
    LanguageSubtag::new(['i', 'r', 'i']),
    LanguageSubtag::new(['i', 'r', 'k']),
    LanguageSubtag::new(['i', 'r', 'n']),
    LanguageSubtag::new(['i', 'r', 'o']),
    LanguageSubtag::new(['i', 'r', 'r']),
    LanguageSubtag::new(['i', 'r', 'u']),
    LanguageSubtag::new(['i', 'r', 'x']),
    LanguageSubtag::new(['i', 'r', 'y']),
    LanguageSubtag::new(['i', 's', ' ']),
    LanguageSubtag::new(['i', 's', 'a']),
    LanguageSubtag::new(['i', 's', 'c']),
    LanguageSubtag::new(['i', 's', 'd']),
    LanguageSubtag::new(['i', 's', 'e']),
    LanguageSubtag::new(['i', 's', 'g']),
    LanguageSubtag::new(['i', 's', 'h']),
    LanguageSubtag::new(['i', 's', 'i']),
    LanguageSubtag::new(['i', 's', 'k']),
    LanguageSubtag::new(['i', 's', 'm']),
    LanguageSubtag::new(['i', 's', 'n']),
    LanguageSubtag::new(['i', 's', 'o']),
    LanguageSubtag::new(['i', 's', 'r']),
    LanguageSubtag::new(['i', 's', 't']),
    LanguageSubtag::new(['i', 's', 'u']),
    LanguageSubtag::new(['i', 't', ' ']),
    LanguageSubtag::new(['i', 't', 'b']),
    LanguageSubtag::new(['i', 't', 'c']),
    LanguageSubtag::new(['i', 't', 'd']),
    LanguageSubtag::new(['i', 't', 'e']),
    LanguageSubtag::new(['i', 't', 'i']),
    LanguageSubtag::new(['i', 't', 'k']),
    LanguageSubtag::new(['i', 't', 'l']),
    LanguageSubtag::new(['i', 't', 'm']),
    LanguageSubtag::new(['i', 't', 'o']),
    LanguageSubtag::new(['i', 't', 'r']),
    LanguageSubtag::new(['i', 't', 's']),
    LanguageSubtag::new(['i', 't', 't']),
    LanguageSubtag::new(['i', 't', 'v']),
    LanguageSubtag::new(['i', 't', 'w']),
    LanguageSubtag::new(['i', 't', 'x']),
    LanguageSubtag::new(['i', 't', 'y']),
    LanguageSubtag::new(['i', 't', 'z']),
    LanguageSubtag::new(['i', 'u', ' ']),
    LanguageSubtag::new(['i', 'u', 'm']),
    LanguageSubtag::new(['i', 'v', 'b']),
    LanguageSubtag::new(['i', 'v', 'v']),
    LanguageSubtag::new(['i', 'w', ' ']),
    LanguageSubtag::new(['i', 'w', 'k']),
    LanguageSubtag::new(['i', 'w', 'm']),
    LanguageSubtag::new(['i', 'w', 'o']),
    LanguageSubtag::new(['i', 'w', 's']),
    LanguageSubtag::new(['i', 'x', 'c']),
    LanguageSubtag::new(['i', 'x', 'l']),
    LanguageSubtag::new(['i', 'y', 'a']),
    LanguageSubtag::new(['i', 'y', 'o']),
    LanguageSubtag::new(['i', 'y', 'x']),
    LanguageSubtag::new(['i', 'z', 'h']),
    LanguageSubtag::new(['i', 'z', 'i']),
    LanguageSubtag::new(['i', 'z', 'r']),
    LanguageSubtag::new(['i', 'z', 'z']),
    LanguageSubtag::new(['j', 'a', ' ']),
    LanguageSubtag::new(['j', 'a', 'a']),
    LanguageSubtag::new(['j', 'a', 'b']),
    LanguageSubtag::new(['j', 'a', 'c']),
    LanguageSubtag::new(['j', 'a', 'd']),
    LanguageSubtag::new(['j', 'a', 'e']),
    LanguageSubtag::new(['j', 'a', 'f']),
    LanguageSubtag::new(['j', 'a', 'h']),
    LanguageSubtag::new(['j', 'a', 'j']),
    LanguageSubtag::new(['j', 'a', 'k']),
    LanguageSubtag::new(['j', 'a', 'l']),
    LanguageSubtag::new(['j', 'a', 'm']),
    LanguageSubtag::new(['j', 'a', 'n']),
    LanguageSubtag::new(['j', 'a', 'o']),
    LanguageSubtag::new(['j', 'a', 'q']),
    LanguageSubtag::new(['j', 'a', 'r']),
    LanguageSubtag::new(['j', 'a', 's']),
    LanguageSubtag::new(['j', 'a', 't']),
    LanguageSubtag::new(['j', 'a', 'u']),
    LanguageSubtag::new(['j', 'a', 'x']),
    LanguageSubtag::new(['j', 'a', 'y']),
    LanguageSubtag::new(['j', 'a', 'z']),
    LanguageSubtag::new(['j', 'b', 'e']),
    LanguageSubtag::new(['j', 'b', 'i']),
    LanguageSubtag::new(['j', 'b', 'j']),
    LanguageSubtag::new(['j', 'b', 'k']),
    LanguageSubtag::new(['j', 'b', 'm']),
    LanguageSubtag::new(['j', 'b', 'n']),
    LanguageSubtag::new(['j', 'b', 'o']),
    LanguageSubtag::new(['j', 'b', 'r']),
    LanguageSubtag::new(['j', 'b', 't']),
    LanguageSubtag::new(['j', 'b', 'u']),
    LanguageSubtag::new(['j', 'b', 'w']),
    LanguageSubtag::new(['j', 'c', 's']),
    LanguageSubtag::new(['j', 'c', 't']),
    LanguageSubtag::new(['j', 'd', 'a']),
    LanguageSubtag::new(['j', 'd', 'g']),
    LanguageSubtag::new(['j', 'd', 't']),
    LanguageSubtag::new(['j', 'e', 'b']),
    LanguageSubtag::new(['j', 'e', 'e']),
    LanguageSubtag::new(['j', 'e', 'g']),
    LanguageSubtag::new(['j', 'e', 'h']),
    LanguageSubtag::new(['j', 'e', 'i']),
    LanguageSubtag::new(['j', 'e', 'k']),
    LanguageSubtag::new(['j', 'e', 'l']),
    LanguageSubtag::new(['j', 'e', 'n']),
    LanguageSubtag::new(['j', 'e', 'r']),
    LanguageSubtag::new(['j', 'e', 't']),
    LanguageSubtag::new(['j', 'e', 'u']),
    LanguageSubtag::new(['j', 'g', 'b']),
    LanguageSubtag::new(['j', 'g', 'e']),
    LanguageSubtag::new(['j', 'g', 'k']),
    LanguageSubtag::new(['j', 'g', 'o']),
    LanguageSubtag::new(['j', 'h', 'i']),
    LanguageSubtag::new(['j', 'h', 's']),
    LanguageSubtag::new(['j', 'i', ' ']),
    LanguageSubtag::new(['j', 'i', 'a']),
    LanguageSubtag::new(['j', 'i', 'b']),
    LanguageSubtag::new(['j', 'i', 'c']),
    LanguageSubtag::new(['j', 'i', 'd']),
    LanguageSubtag::new(['j', 'i', 'e']),
    LanguageSubtag::new(['j', 'i', 'g']),
    LanguageSubtag::new(['j', 'i', 'h']),
    LanguageSubtag::new(['j', 'i', 'i']),
    LanguageSubtag::new(['j', 'i', 'l']),
    LanguageSubtag::new(['j', 'i', 'm']),
    LanguageSubtag::new(['j', 'i', 'o']),
    LanguageSubtag::new(['j', 'i', 'q']),
    LanguageSubtag::new(['j', 'i', 't']),
    LanguageSubtag::new(['j', 'i', 'u']),
    LanguageSubtag::new(['j', 'i', 'v']),
    LanguageSubtag::new(['j', 'i', 'y']),
    LanguageSubtag::new(['j', 'j', 'e']),
    LanguageSubtag::new(['j', 'j', 'r']),
    LanguageSubtag::new(['j', 'k', 'a']),
    LanguageSubtag::new(['j', 'k', 'm']),
    LanguageSubtag::new(['j', 'k', 'o']),
    LanguageSubtag::new(['j', 'k', 'p']),
    LanguageSubtag::new(['j', 'k', 'r']),
    LanguageSubtag::new(['j', 'k', 's']),
    LanguageSubtag::new(['j', 'k', 'u']),
    LanguageSubtag::new(['j', 'l', 'e']),
    LanguageSubtag::new(['j', 'l', 's']),
    LanguageSubtag::new(['j', 'm', 'a']),
    LanguageSubtag::new(['j', 'm', 'b']),
    LanguageSubtag::new(['j', 'm', 'c']),
    LanguageSubtag::new(['j', 'm', 'd']),
    LanguageSubtag::new(['j', 'm', 'i']),
    LanguageSubtag::new(['j', 'm', 'l']),
    LanguageSubtag::new(['j', 'm', 'n']),
    LanguageSubtag::new(['j', 'm', 'r']),
    LanguageSubtag::new(['j', 'm', 's']),
    LanguageSubtag::new(['j', 'm', 'w']),
    LanguageSubtag::new(['j', 'm', 'x']),
    LanguageSubtag::new(['j', 'n', 'a']),
    LanguageSubtag::new(['j', 'n', 'd']),
    LanguageSubtag::new(['j', 'n', 'g']),
    LanguageSubtag::new(['j', 'n', 'i']),
    LanguageSubtag::new(['j', 'n', 'j']),
    LanguageSubtag::new(['j', 'n', 'l']),
    LanguageSubtag::new(['j', 'n', 's']),
    LanguageSubtag::new(['j', 'o', 'b']),
    LanguageSubtag::new(['j', 'o', 'd']),
    LanguageSubtag::new(['j', 'o', 'g']),
    LanguageSubtag::new(['j', 'o', 'r']),
    LanguageSubtag::new(['j', 'o', 's']),
    LanguageSubtag::new(['j', 'o', 'w']),
    LanguageSubtag::new(['j', 'p', 'a']),
    LanguageSubtag::new(['j', 'p', 'r']),
    LanguageSubtag::new(['j', 'p', 'x']),
    LanguageSubtag::new(['j', 'q', 'r']),
    LanguageSubtag::new(['j', 'r', 'a']),
    LanguageSubtag::new(['j', 'r', 'b']),
    LanguageSubtag::new(['j', 'r', 'r']),
    LanguageSubtag::new(['j', 'r', 't']),
    LanguageSubtag::new(['j', 'r', 'u']),
    LanguageSubtag::new(['j', 's', 'l']),
    LanguageSubtag::new(['j', 'u', 'a']),
    LanguageSubtag::new(['j', 'u', 'b']),
    LanguageSubtag::new(['j', 'u', 'c']),
    LanguageSubtag::new(['j', 'u', 'd']),
    LanguageSubtag::new(['j', 'u', 'h']),
    LanguageSubtag::new(['j', 'u', 'i']),
    LanguageSubtag::new(['j', 'u', 'k']),
    LanguageSubtag::new(['j', 'u', 'l']),
    LanguageSubtag::new(['j', 'u', 'm']),
    LanguageSubtag::new(['j', 'u', 'n']),
    LanguageSubtag::new(['j', 'u', 'o']),
    LanguageSubtag::new(['j', 'u', 'p']),
    LanguageSubtag::new(['j', 'u', 'r']),
    LanguageSubtag::new(['j', 'u', 's']),
    LanguageSubtag::new(['j', 'u', 't']),
    LanguageSubtag::new(['j', 'u', 'u']),
    LanguageSubtag::new(['j', 'u', 'w']),
    LanguageSubtag::new(['j', 'u', 'y']),
    LanguageSubtag::new(['j', 'v', ' ']),
    LanguageSubtag::new(['j', 'v', 'd']),
    LanguageSubtag::new(['j', 'v', 'n']),
    LanguageSubtag::new(['j', 'w', ' ']),
    LanguageSubtag::new(['j', 'w', 'i']),
    LanguageSubtag::new(['j', 'y', 'a']),
    LanguageSubtag::new(['j', 'y', 'e']),
    LanguageSubtag::new(['j', 'y', 'y']),
    LanguageSubtag::new(['k', 'a', ' ']),
    LanguageSubtag::new(['k', 'a', 'a']),
    LanguageSubtag::new(['k', 'a', 'b']),
    LanguageSubtag::new(['k', 'a', 'c']),
    LanguageSubtag::new(['k', 'a', 'd']),
    LanguageSubtag::new(['k', 'a', 'e']),
    LanguageSubtag::new(['k', 'a', 'f']),
    LanguageSubtag::new(['k', 'a', 'g']),
    LanguageSubtag::new(['k', 'a', 'h']),
    LanguageSubtag::new(['k', 'a', 'i']),
    LanguageSubtag::new(['k', 'a', 'j']),
    LanguageSubtag::new(['k', 'a', 'k']),
    LanguageSubtag::new(['k', 'a', 'm']),
    LanguageSubtag::new(['k', 'a', 'o']),
    LanguageSubtag::new(['k', 'a', 'p']),
    LanguageSubtag::new(['k', 'a', 'q']),
    LanguageSubtag::new(['k', 'a', 'r']),
    LanguageSubtag::new(['k', 'a', 'v']),
    LanguageSubtag::new(['k', 'a', 'w']),
    LanguageSubtag::new(['k', 'a', 'x']),
    LanguageSubtag::new(['k', 'a', 'y']),
    LanguageSubtag::new(['k', 'b', 'a']),
    LanguageSubtag::new(['k', 'b', 'b']),
    LanguageSubtag::new(['k', 'b', 'c']),
    LanguageSubtag::new(['k', 'b', 'd']),
    LanguageSubtag::new(['k', 'b', 'e']),
    LanguageSubtag::new(['k', 'b', 'f']),
    LanguageSubtag::new(['k', 'b', 'g']),
    LanguageSubtag::new(['k', 'b', 'h']),
    LanguageSubtag::new(['k', 'b', 'i']),
    LanguageSubtag::new(['k', 'b', 'j']),
    LanguageSubtag::new(['k', 'b', 'k']),
    LanguageSubtag::new(['k', 'b', 'l']),
    LanguageSubtag::new(['k', 'b', 'm']),
    LanguageSubtag::new(['k', 'b', 'n']),
    LanguageSubtag::new(['k', 'b', 'o']),
    LanguageSubtag::new(['k', 'b', 'p']),
    LanguageSubtag::new(['k', 'b', 'q']),
    LanguageSubtag::new(['k', 'b', 'r']),
    LanguageSubtag::new(['k', 'b', 's']),
    LanguageSubtag::new(['k', 'b', 't']),
    LanguageSubtag::new(['k', 'b', 'u']),
    LanguageSubtag::new(['k', 'b', 'v']),
    LanguageSubtag::new(['k', 'b', 'w']),
    LanguageSubtag::new(['k', 'b', 'x']),
    LanguageSubtag::new(['k', 'b', 'y']),
    LanguageSubtag::new(['k', 'b', 'z']),
    LanguageSubtag::new(['k', 'c', 'a']),
    LanguageSubtag::new(['k', 'c', 'b']),
    LanguageSubtag::new(['k', 'c', 'c']),
    LanguageSubtag::new(['k', 'c', 'd']),
    LanguageSubtag::new(['k', 'c', 'e']),
    LanguageSubtag::new(['k', 'c', 'f']),
    LanguageSubtag::new(['k', 'c', 'g']),
    LanguageSubtag::new(['k', 'c', 'h']),
    LanguageSubtag::new(['k', 'c', 'i']),
    LanguageSubtag::new(['k', 'c', 'j']),
    LanguageSubtag::new(['k', 'c', 'k']),
    LanguageSubtag::new(['k', 'c', 'l']),
    LanguageSubtag::new(['k', 'c', 'm']),
    LanguageSubtag::new(['k', 'c', 'n']),
    LanguageSubtag::new(['k', 'c', 'o']),
    LanguageSubtag::new(['k', 'c', 'p']),
    LanguageSubtag::new(['k', 'c', 'q']),
    LanguageSubtag::new(['k', 'c', 'r']),
    LanguageSubtag::new(['k', 'c', 's']),
    LanguageSubtag::new(['k', 'c', 't']),
    LanguageSubtag::new(['k', 'c', 'u']),
    LanguageSubtag::new(['k', 'c', 'v']),
    LanguageSubtag::new(['k', 'c', 'w']),
    LanguageSubtag::new(['k', 'c', 'x']),
    LanguageSubtag::new(['k', 'c', 'y']),
    LanguageSubtag::new(['k', 'c', 'z']),
    LanguageSubtag::new(['k', 'd', 'a']),
    LanguageSubtag::new(['k', 'd', 'c']),
    LanguageSubtag::new(['k', 'd', 'd']),
    LanguageSubtag::new(['k', 'd', 'e']),
    LanguageSubtag::new(['k', 'd', 'f']),
    LanguageSubtag::new(['k', 'd', 'g']),
    LanguageSubtag::new(['k', 'd', 'h']),
    LanguageSubtag::new(['k', 'd', 'i']),
    LanguageSubtag::new(['k', 'd', 'j']),
    LanguageSubtag::new(['k', 'd', 'k']),
    LanguageSubtag::new(['k', 'd', 'l']),
    LanguageSubtag::new(['k', 'd', 'm']),
    LanguageSubtag::new(['k', 'd', 'n']),
    LanguageSubtag::new(['k', 'd', 'o']),
    LanguageSubtag::new(['k', 'd', 'p']),
    LanguageSubtag::new(['k', 'd', 'q']),
    LanguageSubtag::new(['k', 'd', 'r']),
    LanguageSubtag::new(['k', 'd', 't']),
    LanguageSubtag::new(['k', 'd', 'u']),
    LanguageSubtag::new(['k', 'd', 'v']),
    LanguageSubtag::new(['k', 'd', 'w']),
    LanguageSubtag::new(['k', 'd', 'x']),
    LanguageSubtag::new(['k', 'd', 'y']),
    LanguageSubtag::new(['k', 'd', 'z']),
    LanguageSubtag::new(['k', 'e', 'a']),
    LanguageSubtag::new(['k', 'e', 'b']),
    LanguageSubtag::new(['k', 'e', 'c']),
    LanguageSubtag::new(['k', 'e', 'd']),
    LanguageSubtag::new(['k', 'e', 'e']),
    LanguageSubtag::new(['k', 'e', 'f']),
    LanguageSubtag::new(['k', 'e', 'g']),
    LanguageSubtag::new(['k', 'e', 'h']),
    LanguageSubtag::new(['k', 'e', 'i']),
    LanguageSubtag::new(['k', 'e', 'j']),
    LanguageSubtag::new(['k', 'e', 'k']),
    LanguageSubtag::new(['k', 'e', 'l']),
    LanguageSubtag::new(['k', 'e', 'm']),
    LanguageSubtag::new(['k', 'e', 'n']),
    LanguageSubtag::new(['k', 'e', 'o']),
    LanguageSubtag::new(['k', 'e', 'p']),
    LanguageSubtag::new(['k', 'e', 'q']),
    LanguageSubtag::new(['k', 'e', 'r']),
    LanguageSubtag::new(['k', 'e', 's']),
    LanguageSubtag::new(['k', 'e', 't']),
    LanguageSubtag::new(['k', 'e', 'u']),
    LanguageSubtag::new(['k', 'e', 'v']),
    LanguageSubtag::new(['k', 'e', 'w']),
    LanguageSubtag::new(['k', 'e', 'x']),
    LanguageSubtag::new(['k', 'e', 'y']),
    LanguageSubtag::new(['k', 'e', 'z']),
    LanguageSubtag::new(['k', 'f', 'a']),
    LanguageSubtag::new(['k', 'f', 'b']),
    LanguageSubtag::new(['k', 'f', 'c']),
    LanguageSubtag::new(['k', 'f', 'd']),
    LanguageSubtag::new(['k', 'f', 'e']),
    LanguageSubtag::new(['k', 'f', 'f']),
    LanguageSubtag::new(['k', 'f', 'g']),
    LanguageSubtag::new(['k', 'f', 'h']),
    LanguageSubtag::new(['k', 'f', 'i']),
    LanguageSubtag::new(['k', 'f', 'j']),
    LanguageSubtag::new(['k', 'f', 'k']),
    LanguageSubtag::new(['k', 'f', 'l']),
    LanguageSubtag::new(['k', 'f', 'm']),
    LanguageSubtag::new(['k', 'f', 'n']),
    LanguageSubtag::new(['k', 'f', 'o']),
    LanguageSubtag::new(['k', 'f', 'p']),
    LanguageSubtag::new(['k', 'f', 'q']),
    LanguageSubtag::new(['k', 'f', 'r']),
    LanguageSubtag::new(['k', 'f', 's']),
    LanguageSubtag::new(['k', 'f', 't']),
    LanguageSubtag::new(['k', 'f', 'u']),
    LanguageSubtag::new(['k', 'f', 'v']),
    LanguageSubtag::new(['k', 'f', 'w']),
    LanguageSubtag::new(['k', 'f', 'x']),
    LanguageSubtag::new(['k', 'f', 'y']),
    LanguageSubtag::new(['k', 'f', 'z']),
    LanguageSubtag::new(['k', 'g', ' ']),
    LanguageSubtag::new(['k', 'g', 'a']),
    LanguageSubtag::new(['k', 'g', 'b']),
    LanguageSubtag::new(['k', 'g', 'c']),
    LanguageSubtag::new(['k', 'g', 'd']),
    LanguageSubtag::new(['k', 'g', 'e']),
    LanguageSubtag::new(['k', 'g', 'f']),
    LanguageSubtag::new(['k', 'g', 'g']),
    LanguageSubtag::new(['k', 'g', 'h']),
    LanguageSubtag::new(['k', 'g', 'i']),
    LanguageSubtag::new(['k', 'g', 'j']),
    LanguageSubtag::new(['k', 'g', 'k']),
    LanguageSubtag::new(['k', 'g', 'l']),
    LanguageSubtag::new(['k', 'g', 'm']),
    LanguageSubtag::new(['k', 'g', 'n']),
    LanguageSubtag::new(['k', 'g', 'o']),
    LanguageSubtag::new(['k', 'g', 'p']),
    LanguageSubtag::new(['k', 'g', 'q']),
    LanguageSubtag::new(['k', 'g', 'r']),
    LanguageSubtag::new(['k', 'g', 's']),
    LanguageSubtag::new(['k', 'g', 't']),
    LanguageSubtag::new(['k', 'g', 'u']),
    LanguageSubtag::new(['k', 'g', 'v']),
    LanguageSubtag::new(['k', 'g', 'w']),
    LanguageSubtag::new(['k', 'g', 'x']),
    LanguageSubtag::new(['k', 'g', 'y']),
    LanguageSubtag::new(['k', 'h', 'a']),
    LanguageSubtag::new(['k', 'h', 'b']),
    LanguageSubtag::new(['k', 'h', 'c']),
    LanguageSubtag::new(['k', 'h', 'd']),
    LanguageSubtag::new(['k', 'h', 'e']),
    LanguageSubtag::new(['k', 'h', 'f']),
    LanguageSubtag::new(['k', 'h', 'g']),
    LanguageSubtag::new(['k', 'h', 'h']),
    LanguageSubtag::new(['k', 'h', 'i']),
    LanguageSubtag::new(['k', 'h', 'j']),
    LanguageSubtag::new(['k', 'h', 'k']),
    LanguageSubtag::new(['k', 'h', 'l']),
    LanguageSubtag::new(['k', 'h', 'n']),
    LanguageSubtag::new(['k', 'h', 'o']),
    LanguageSubtag::new(['k', 'h', 'p']),
    LanguageSubtag::new(['k', 'h', 'q']),
    LanguageSubtag::new(['k', 'h', 'r']),
    LanguageSubtag::new(['k', 'h', 's']),
    LanguageSubtag::new(['k', 'h', 't']),
    LanguageSubtag::new(['k', 'h', 'u']),
    LanguageSubtag::new(['k', 'h', 'v']),
    LanguageSubtag::new(['k', 'h', 'w']),
    LanguageSubtag::new(['k', 'h', 'x']),
    LanguageSubtag::new(['k', 'h', 'y']),
    LanguageSubtag::new(['k', 'h', 'z']),
    LanguageSubtag::new(['k', 'i', ' ']),
    LanguageSubtag::new(['k', 'i', 'a']),
    LanguageSubtag::new(['k', 'i', 'b']),
    LanguageSubtag::new(['k', 'i', 'c']),
    LanguageSubtag::new(['k', 'i', 'd']),
    LanguageSubtag::new(['k', 'i', 'e']),
    LanguageSubtag::new(['k', 'i', 'f']),
    LanguageSubtag::new(['k', 'i', 'g']),
    LanguageSubtag::new(['k', 'i', 'h']),
    LanguageSubtag::new(['k', 'i', 'i']),
    LanguageSubtag::new(['k', 'i', 'j']),
    LanguageSubtag::new(['k', 'i', 'l']),
    LanguageSubtag::new(['k', 'i', 'm']),
    LanguageSubtag::new(['k', 'i', 'o']),
    LanguageSubtag::new(['k', 'i', 'p']),
    LanguageSubtag::new(['k', 'i', 'q']),
    LanguageSubtag::new(['k', 'i', 's']),
    LanguageSubtag::new(['k', 'i', 't']),
    LanguageSubtag::new(['k', 'i', 'u']),
    LanguageSubtag::new(['k', 'i', 'v']),
    LanguageSubtag::new(['k', 'i', 'w']),
    LanguageSubtag::new(['k', 'i', 'x']),
    LanguageSubtag::new(['k', 'i', 'y']),
    LanguageSubtag::new(['k', 'i', 'z']),
    LanguageSubtag::new(['k', 'j', ' ']),
    LanguageSubtag::new(['k', 'j', 'a']),
    LanguageSubtag::new(['k', 'j', 'b']),
    LanguageSubtag::new(['k', 'j', 'c']),
    LanguageSubtag::new(['k', 'j', 'd']),
    LanguageSubtag::new(['k', 'j', 'e']),
    LanguageSubtag::new(['k', 'j', 'f']),
    LanguageSubtag::new(['k', 'j', 'g']),
    LanguageSubtag::new(['k', 'j', 'h']),
    LanguageSubtag::new(['k', 'j', 'i']),
    LanguageSubtag::new(['k', 'j', 'j']),
    LanguageSubtag::new(['k', 'j', 'k']),
    LanguageSubtag::new(['k', 'j', 'l']),
    LanguageSubtag::new(['k', 'j', 'm']),
    LanguageSubtag::new(['k', 'j', 'n']),
    LanguageSubtag::new(['k', 'j', 'o']),
    LanguageSubtag::new(['k', 'j', 'p']),
    LanguageSubtag::new(['k', 'j', 'q']),
    LanguageSubtag::new(['k', 'j', 'r']),
    LanguageSubtag::new(['k', 'j', 's']),
    LanguageSubtag::new(['k', 'j', 't']),
    LanguageSubtag::new(['k', 'j', 'u']),
    LanguageSubtag::new(['k', 'j', 'v']),
    LanguageSubtag::new(['k', 'j', 'x']),
    LanguageSubtag::new(['k', 'j', 'y']),
    LanguageSubtag::new(['k', 'j', 'z']),
    LanguageSubtag::new(['k', 'k', ' ']),
    LanguageSubtag::new(['k', 'k', 'a']),
    LanguageSubtag::new(['k', 'k', 'b']),
    LanguageSubtag::new(['k', 'k', 'c']),
    LanguageSubtag::new(['k', 'k', 'd']),
    LanguageSubtag::new(['k', 'k', 'e']),
    LanguageSubtag::new(['k', 'k', 'f']),
    LanguageSubtag::new(['k', 'k', 'g']),
    LanguageSubtag::new(['k', 'k', 'h']),
    LanguageSubtag::new(['k', 'k', 'i']),
    LanguageSubtag::new(['k', 'k', 'j']),
    LanguageSubtag::new(['k', 'k', 'k']),
    LanguageSubtag::new(['k', 'k', 'l']),
    LanguageSubtag::new(['k', 'k', 'm']),
    LanguageSubtag::new(['k', 'k', 'n']),
    LanguageSubtag::new(['k', 'k', 'o']),
    LanguageSubtag::new(['k', 'k', 'p']),
    LanguageSubtag::new(['k', 'k', 'q']),
    LanguageSubtag::new(['k', 'k', 'r']),
    LanguageSubtag::new(['k', 'k', 's']),
    LanguageSubtag::new(['k', 'k', 't']),
    LanguageSubtag::new(['k', 'k', 'u']),
    LanguageSubtag::new(['k', 'k', 'v']),
    LanguageSubtag::new(['k', 'k', 'w']),
    LanguageSubtag::new(['k', 'k', 'x']),
    LanguageSubtag::new(['k', 'k', 'y']),
    LanguageSubtag::new(['k', 'k', 'z']),
    LanguageSubtag::new(['k', 'l', ' ']),
    LanguageSubtag::new(['k', 'l', 'a']),
    LanguageSubtag::new(['k', 'l', 'b']),
    LanguageSubtag::new(['k', 'l', 'c']),
    LanguageSubtag::new(['k', 'l', 'd']),
    LanguageSubtag::new(['k', 'l', 'e']),
    LanguageSubtag::new(['k', 'l', 'f']),
    LanguageSubtag::new(['k', 'l', 'g']),
    LanguageSubtag::new(['k', 'l', 'h']),
    LanguageSubtag::new(['k', 'l', 'i']),
    LanguageSubtag::new(['k', 'l', 'j']),
    LanguageSubtag::new(['k', 'l', 'k']),
    LanguageSubtag::new(['k', 'l', 'l']),
    LanguageSubtag::new(['k', 'l', 'm']),
    LanguageSubtag::new(['k', 'l', 'n']),
    LanguageSubtag::new(['k', 'l', 'o']),
    LanguageSubtag::new(['k', 'l', 'p']),
    LanguageSubtag::new(['k', 'l', 'q']),
    LanguageSubtag::new(['k', 'l', 'r']),
    LanguageSubtag::new(['k', 'l', 's']),
    LanguageSubtag::new(['k', 'l', 't']),
    LanguageSubtag::new(['k', 'l', 'u']),
    LanguageSubtag::new(['k', 'l', 'v']),
    LanguageSubtag::new(['k', 'l', 'w']),
    LanguageSubtag::new(['k', 'l', 'x']),
    LanguageSubtag::new(['k', 'l', 'y']),
    LanguageSubtag::new(['k', 'l', 'z']),
    LanguageSubtag::new(['k', 'm', ' ']),
    LanguageSubtag::new(['k', 'm', 'a']),
    LanguageSubtag::new(['k', 'm', 'b']),
    LanguageSubtag::new(['k', 'm', 'c']),
    LanguageSubtag::new(['k', 'm', 'd']),
    LanguageSubtag::new(['k', 'm', 'e']),
    LanguageSubtag::new(['k', 'm', 'f']),
    LanguageSubtag::new(['k', 'm', 'g']),
    LanguageSubtag::new(['k', 'm', 'h']),
    LanguageSubtag::new(['k', 'm', 'i']),
    LanguageSubtag::new(['k', 'm', 'j']),
    LanguageSubtag::new(['k', 'm', 'k']),
    LanguageSubtag::new(['k', 'm', 'l']),
    LanguageSubtag::new(['k', 'm', 'm']),
    LanguageSubtag::new(['k', 'm', 'n']),
    LanguageSubtag::new(['k', 'm', 'o']),
    LanguageSubtag::new(['k', 'm', 'p']),
    LanguageSubtag::new(['k', 'm', 'q']),
    LanguageSubtag::new(['k', 'm', 'r']),
    LanguageSubtag::new(['k', 'm', 's']),
    LanguageSubtag::new(['k', 'm', 't']),
    LanguageSubtag::new(['k', 'm', 'u']),
    LanguageSubtag::new(['k', 'm', 'v']),
    LanguageSubtag::new(['k', 'm', 'w']),
    LanguageSubtag::new(['k', 'm', 'x']),
    LanguageSubtag::new(['k', 'm', 'y']),
    LanguageSubtag::new(['k', 'm', 'z']),
    LanguageSubtag::new(['k', 'n', ' ']),
    LanguageSubtag::new(['k', 'n', 'a']),
    LanguageSubtag::new(['k', 'n', 'b']),
    LanguageSubtag::new(['k', 'n', 'c']),
    LanguageSubtag::new(['k', 'n', 'd']),
    LanguageSubtag::new(['k', 'n', 'e']),
    LanguageSubtag::new(['k', 'n', 'f']),
    LanguageSubtag::new(['k', 'n', 'g']),
    LanguageSubtag::new(['k', 'n', 'i']),
    LanguageSubtag::new(['k', 'n', 'j']),
    LanguageSubtag::new(['k', 'n', 'k']),
    LanguageSubtag::new(['k', 'n', 'l']),
    LanguageSubtag::new(['k', 'n', 'm']),
    LanguageSubtag::new(['k', 'n', 'n']),
    LanguageSubtag::new(['k', 'n', 'o']),
    LanguageSubtag::new(['k', 'n', 'p']),
    LanguageSubtag::new(['k', 'n', 'q']),
    LanguageSubtag::new(['k', 'n', 'r']),
    LanguageSubtag::new(['k', 'n', 's']),
    LanguageSubtag::new(['k', 'n', 't']),
    LanguageSubtag::new(['k', 'n', 'u']),
    LanguageSubtag::new(['k', 'n', 'v']),
    LanguageSubtag::new(['k', 'n', 'w']),
    LanguageSubtag::new(['k', 'n', 'x']),
    LanguageSubtag::new(['k', 'n', 'y']),
    LanguageSubtag::new(['k', 'n', 'z']),
    LanguageSubtag::new(['k', 'o', ' ']),
    LanguageSubtag::new(['k', 'o', 'a']),
    LanguageSubtag::new(['k', 'o', 'c']),
    LanguageSubtag::new(['k', 'o', 'd']),
    LanguageSubtag::new(['k', 'o', 'e']),
    LanguageSubtag::new(['k', 'o', 'f']),
    LanguageSubtag::new(['k', 'o', 'g']),
    LanguageSubtag::new(['k', 'o', 'h']),
    LanguageSubtag::new(['k', 'o', 'i']),
    LanguageSubtag::new(['k', 'o', 'j']),
    LanguageSubtag::new(['k', 'o', 'k']),
    LanguageSubtag::new(['k', 'o', 'l']),
    LanguageSubtag::new(['k', 'o', 'o']),
    LanguageSubtag::new(['k', 'o', 'p']),
    LanguageSubtag::new(['k', 'o', 'q']),
    LanguageSubtag::new(['k', 'o', 's']),
    LanguageSubtag::new(['k', 'o', 't']),
    LanguageSubtag::new(['k', 'o', 'u']),
    LanguageSubtag::new(['k', 'o', 'v']),
    LanguageSubtag::new(['k', 'o', 'w']),
    LanguageSubtag::new(['k', 'o', 'x']),
    LanguageSubtag::new(['k', 'o', 'y']),
    LanguageSubtag::new(['k', 'o', 'z']),
    LanguageSubtag::new(['k', 'p', 'a']),
    LanguageSubtag::new(['k', 'p', 'b']),
    LanguageSubtag::new(['k', 'p', 'c']),
    LanguageSubtag::new(['k', 'p', 'd']),
    LanguageSubtag::new(['k', 'p', 'e']),
    LanguageSubtag::new(['k', 'p', 'f']),
    LanguageSubtag::new(['k', 'p', 'g']),
    LanguageSubtag::new(['k', 'p', 'h']),
    LanguageSubtag::new(['k', 'p', 'i']),
    LanguageSubtag::new(['k', 'p', 'j']),
    LanguageSubtag::new(['k', 'p', 'k']),
    LanguageSubtag::new(['k', 'p', 'l']),
    LanguageSubtag::new(['k', 'p', 'm']),
    LanguageSubtag::new(['k', 'p', 'n']),
    LanguageSubtag::new(['k', 'p', 'o']),
    LanguageSubtag::new(['k', 'p', 'p']),
    LanguageSubtag::new(['k', 'p', 'q']),
    LanguageSubtag::new(['k', 'p', 'r']),
    LanguageSubtag::new(['k', 'p', 's']),
    LanguageSubtag::new(['k', 'p', 't']),
    LanguageSubtag::new(['k', 'p', 'u']),
    LanguageSubtag::new(['k', 'p', 'v']),
    LanguageSubtag::new(['k', 'p', 'w']),
    LanguageSubtag::new(['k', 'p', 'x']),
    LanguageSubtag::new(['k', 'p', 'y']),
    LanguageSubtag::new(['k', 'p', 'z']),
    LanguageSubtag::new(['k', 'q', 'a']),
    LanguageSubtag::new(['k', 'q', 'b']),
    LanguageSubtag::new(['k', 'q', 'c']),
    LanguageSubtag::new(['k', 'q', 'd']),
    LanguageSubtag::new(['k', 'q', 'e']),
    LanguageSubtag::new(['k', 'q', 'f']),
    LanguageSubtag::new(['k', 'q', 'g']),
    LanguageSubtag::new(['k', 'q', 'h']),
    LanguageSubtag::new(['k', 'q', 'i']),
    LanguageSubtag::new(['k', 'q', 'j']),
    LanguageSubtag::new(['k', 'q', 'k']),
    LanguageSubtag::new(['k', 'q', 'l']),
    LanguageSubtag::new(['k', 'q', 'm']),
    LanguageSubtag::new(['k', 'q', 'n']),
    LanguageSubtag::new(['k', 'q', 'o']),
    LanguageSubtag::new(['k', 'q', 'p']),
    LanguageSubtag::new(['k', 'q', 'q']),
    LanguageSubtag::new(['k', 'q', 'r']),
    LanguageSubtag::new(['k', 'q', 's']),
    LanguageSubtag::new(['k', 'q', 't']),
    LanguageSubtag::new(['k', 'q', 'u']),
    LanguageSubtag::new(['k', 'q', 'v']),
    LanguageSubtag::new(['k', 'q', 'w']),
    LanguageSubtag::new(['k', 'q', 'x']),
    LanguageSubtag::new(['k', 'q', 'y']),
    LanguageSubtag::new(['k', 'q', 'z']),
    LanguageSubtag::new(['k', 'r', ' ']),
    LanguageSubtag::new(['k', 'r', 'a']),
    LanguageSubtag::new(['k', 'r', 'b']),
    LanguageSubtag::new(['k', 'r', 'c']),
    LanguageSubtag::new(['k', 'r', 'd']),
    LanguageSubtag::new(['k', 'r', 'e']),
    LanguageSubtag::new(['k', 'r', 'f']),
    LanguageSubtag::new(['k', 'r', 'h']),
    LanguageSubtag::new(['k', 'r', 'i']),
    LanguageSubtag::new(['k', 'r', 'j']),
    LanguageSubtag::new(['k', 'r', 'k']),
    LanguageSubtag::new(['k', 'r', 'l']),
    LanguageSubtag::new(['k', 'r', 'm']),
    LanguageSubtag::new(['k', 'r', 'n']),
    LanguageSubtag::new(['k', 'r', 'o']),
    LanguageSubtag::new(['k', 'r', 'p']),
    LanguageSubtag::new(['k', 'r', 'r']),
    LanguageSubtag::new(['k', 'r', 's']),
    LanguageSubtag::new(['k', 'r', 't']),
    LanguageSubtag::new(['k', 'r', 'u']),
    LanguageSubtag::new(['k', 'r', 'v']),
    LanguageSubtag::new(['k', 'r', 'w']),
    LanguageSubtag::new(['k', 'r', 'x']),
    LanguageSubtag::new(['k', 'r', 'y']),
    LanguageSubtag::new(['k', 'r', 'z']),
    LanguageSubtag::new(['k', 's', ' ']),
    LanguageSubtag::new(['k', 's', 'a']),
    LanguageSubtag::new(['k', 's', 'b']),
    LanguageSubtag::new(['k', 's', 'c']),
    LanguageSubtag::new(['k', 's', 'd']),
    LanguageSubtag::new(['k', 's', 'e']),
    LanguageSubtag::new(['k', 's', 'f']),
    LanguageSubtag::new(['k', 's', 'g']),
    LanguageSubtag::new(['k', 's', 'h']),
    LanguageSubtag::new(['k', 's', 'i']),
    LanguageSubtag::new(['k', 's', 'j']),
    LanguageSubtag::new(['k', 's', 'k']),
    LanguageSubtag::new(['k', 's', 'l']),
    LanguageSubtag::new(['k', 's', 'm']),
    LanguageSubtag::new(['k', 's', 'n']),
    LanguageSubtag::new(['k', 's', 'o']),
    LanguageSubtag::new(['k', 's', 'p']),
    LanguageSubtag::new(['k', 's', 'q']),
    LanguageSubtag::new(['k', 's', 'r']),
    LanguageSubtag::new(['k', 's', 's']),
    LanguageSubtag::new(['k', 's', 't']),
    LanguageSubtag::new(['k', 's', 'u']),
    LanguageSubtag::new(['k', 's', 'v']),
    LanguageSubtag::new(['k', 's', 'w']),
    LanguageSubtag::new(['k', 's', 'x']),
    LanguageSubtag::new(['k', 's', 'y']),
    LanguageSubtag::new(['k', 's', 'z']),
    LanguageSubtag::new(['k', 't', 'a']),
    LanguageSubtag::new(['k', 't', 'b']),
    LanguageSubtag::new(['k', 't', 'c']),
    LanguageSubtag::new(['k', 't', 'd']),
    LanguageSubtag::new(['k', 't', 'e']),
    LanguageSubtag::new(['k', 't', 'f']),
    LanguageSubtag::new(['k', 't', 'g']),
    LanguageSubtag::new(['k', 't', 'h']),
    LanguageSubtag::new(['k', 't', 'i']),
    LanguageSubtag::new(['k', 't', 'j']),
    LanguageSubtag::new(['k', 't', 'k']),
    LanguageSubtag::new(['k', 't', 'l']),
    LanguageSubtag::new(['k', 't', 'm']),
    LanguageSubtag::new(['k', 't', 'n']),
    LanguageSubtag::new(['k', 't', 'o']),
    LanguageSubtag::new(['k', 't', 'p']),
    LanguageSubtag::new(['k', 't', 'q']),
    LanguageSubtag::new(['k', 't', 'r']),
    LanguageSubtag::new(['k', 't', 's']),
    LanguageSubtag::new(['k', 't', 't']),
    LanguageSubtag::new(['k', 't', 'u']),
    LanguageSubtag::new(['k', 't', 'v']),
    LanguageSubtag::new(['k', 't', 'w']),
    LanguageSubtag::new(['k', 't', 'x']),
    LanguageSubtag::new(['k', 't', 'y']),
    LanguageSubtag::new(['k', 't', 'z']),
    LanguageSubtag::new(['k', 'u', ' ']),
    LanguageSubtag::new(['k', 'u', 'b']),
    LanguageSubtag::new(['k', 'u', 'c']),
    LanguageSubtag::new(['k', 'u', 'd']),
    LanguageSubtag::new(['k', 'u', 'e']),
    LanguageSubtag::new(['k', 'u', 'f']),
    LanguageSubtag::new(['k', 'u', 'g']),
    LanguageSubtag::new(['k', 'u', 'h']),
    LanguageSubtag::new(['k', 'u', 'i']),
    LanguageSubtag::new(['k', 'u', 'j']),
    LanguageSubtag::new(['k', 'u', 'k']),
    LanguageSubtag::new(['k', 'u', 'l']),
    LanguageSubtag::new(['k', 'u', 'm']),
    LanguageSubtag::new(['k', 'u', 'n']),
    LanguageSubtag::new(['k', 'u', 'o']),
    LanguageSubtag::new(['k', 'u', 'p']),
    LanguageSubtag::new(['k', 'u', 'q']),
    LanguageSubtag::new(['k', 'u', 's']),
    LanguageSubtag::new(['k', 'u', 't']),
    LanguageSubtag::new(['k', 'u', 'u']),
    LanguageSubtag::new(['k', 'u', 'v']),
    LanguageSubtag::new(['k', 'u', 'w']),
    LanguageSubtag::new(['k', 'u', 'x']),
    LanguageSubtag::new(['k', 'u', 'y']),
    LanguageSubtag::new(['k', 'u', 'z']),
    LanguageSubtag::new(['k', 'v', ' ']),
    LanguageSubtag::new(['k', 'v', 'a']),
    LanguageSubtag::new(['k', 'v', 'b']),
    LanguageSubtag::new(['k', 'v', 'c']),
    LanguageSubtag::new(['k', 'v', 'd']),
    LanguageSubtag::new(['k', 'v', 'e']),
    LanguageSubtag::new(['k', 'v', 'f']),
    LanguageSubtag::new(['k', 'v', 'g']),
    LanguageSubtag::new(['k', 'v', 'h']),
    LanguageSubtag::new(['k', 'v', 'i']),
    LanguageSubtag::new(['k', 'v', 'j']),
    LanguageSubtag::new(['k', 'v', 'k']),
    LanguageSubtag::new(['k', 'v', 'l']),
    LanguageSubtag::new(['k', 'v', 'm']),
    LanguageSubtag::new(['k', 'v', 'n']),
    LanguageSubtag::new(['k', 'v', 'o']),
    LanguageSubtag::new(['k', 'v', 'p']),
    LanguageSubtag::new(['k', 'v', 'q']),
    LanguageSubtag::new(['k', 'v', 'r']),
    LanguageSubtag::new(['k', 'v', 's']),
    LanguageSubtag::new(['k', 'v', 't']),
    LanguageSubtag::new(['k', 'v', 'u']),
    LanguageSubtag::new(['k', 'v', 'v']),
    LanguageSubtag::new(['k', 'v', 'w']),
    LanguageSubtag::new(['k', 'v', 'x']),
    LanguageSubtag::new(['k', 'v', 'y']),
    LanguageSubtag::new(['k', 'v', 'z']),
    LanguageSubtag::new(['k', 'w', ' ']),
    LanguageSubtag::new(['k', 'w', 'a']),
    LanguageSubtag::new(['k', 'w', 'b']),
    LanguageSubtag::new(['k', 'w', 'c']),
    LanguageSubtag::new(['k', 'w', 'd']),
    LanguageSubtag::new(['k', 'w', 'e']),
    LanguageSubtag::new(['k', 'w', 'f']),
    LanguageSubtag::new(['k', 'w', 'g']),
    LanguageSubtag::new(['k', 'w', 'h']),
    LanguageSubtag::new(['k', 'w', 'i']),
    LanguageSubtag::new(['k', 'w', 'j']),
    LanguageSubtag::new(['k', 'w', 'k']),
    LanguageSubtag::new(['k', 'w', 'l']),
    LanguageSubtag::new(['k', 'w', 'm']),
    LanguageSubtag::new(['k', 'w', 'n']),
    LanguageSubtag::new(['k', 'w', 'o']),
    LanguageSubtag::new(['k', 'w', 'p']),
    LanguageSubtag::new(['k', 'w', 'q']),
    LanguageSubtag::new(['k', 'w', 'r']),
    LanguageSubtag::new(['k', 'w', 's']),
    LanguageSubtag::new(['k', 'w', 't']),
    LanguageSubtag::new(['k', 'w', 'u']),
    LanguageSubtag::new(['k', 'w', 'v']),
    LanguageSubtag::new(['k', 'w', 'w']),
    LanguageSubtag::new(['k', 'w', 'x']),
    LanguageSubtag::new(['k', 'w', 'y']),
    LanguageSubtag::new(['k', 'w', 'z']),
    LanguageSubtag::new(['k', 'x', 'a']),
    LanguageSubtag::new(['k', 'x', 'b']),
    LanguageSubtag::new(['k', 'x', 'c']),
    LanguageSubtag::new(['k', 'x', 'd']),
    LanguageSubtag::new(['k', 'x', 'e']),
    LanguageSubtag::new(['k', 'x', 'f']),
    LanguageSubtag::new(['k', 'x', 'h']),
    LanguageSubtag::new(['k', 'x', 'i']),
    LanguageSubtag::new(['k', 'x', 'j']),
    LanguageSubtag::new(['k', 'x', 'k']),
    LanguageSubtag::new(['k', 'x', 'l']),
    LanguageSubtag::new(['k', 'x', 'm']),
    LanguageSubtag::new(['k', 'x', 'n']),
    LanguageSubtag::new(['k', 'x', 'o']),
    LanguageSubtag::new(['k', 'x', 'p']),
    LanguageSubtag::new(['k', 'x', 'q']),
    LanguageSubtag::new(['k', 'x', 'r']),
    LanguageSubtag::new(['k', 'x', 's']),
    LanguageSubtag::new(['k', 'x', 't']),
    LanguageSubtag::new(['k', 'x', 'u']),
    LanguageSubtag::new(['k', 'x', 'v']),
    LanguageSubtag::new(['k', 'x', 'w']),
    LanguageSubtag::new(['k', 'x', 'x']),
    LanguageSubtag::new(['k', 'x', 'y']),
    LanguageSubtag::new(['k', 'x', 'z']),
    LanguageSubtag::new(['k', 'y', ' ']),
    LanguageSubtag::new(['k', 'y', 'a']),
    LanguageSubtag::new(['k', 'y', 'b']),
    LanguageSubtag::new(['k', 'y', 'c']),
    LanguageSubtag::new(['k', 'y', 'd']),
    LanguageSubtag::new(['k', 'y', 'e']),
    LanguageSubtag::new(['k', 'y', 'f']),
    LanguageSubtag::new(['k', 'y', 'g']),
    LanguageSubtag::new(['k', 'y', 'h']),
    LanguageSubtag::new(['k', 'y', 'i']),
    LanguageSubtag::new(['k', 'y', 'j']),
    LanguageSubtag::new(['k', 'y', 'k']),
    LanguageSubtag::new(['k', 'y', 'l']),
    LanguageSubtag::new(['k', 'y', 'm']),
    LanguageSubtag::new(['k', 'y', 'n']),
    LanguageSubtag::new(['k', 'y', 'o']),
    LanguageSubtag::new(['k', 'y', 'p']),
    LanguageSubtag::new(['k', 'y', 'q']),
    LanguageSubtag::new(['k', 'y', 'r']),
    LanguageSubtag::new(['k', 'y', 's']),
    LanguageSubtag::new(['k', 'y', 't']),
    LanguageSubtag::new(['k', 'y', 'u']),
    LanguageSubtag::new(['k', 'y', 'v']),
    LanguageSubtag::new(['k', 'y', 'w']),
    LanguageSubtag::new(['k', 'y', 'x']),
    LanguageSubtag::new(['k', 'y', 'y']),
    LanguageSubtag::new(['k', 'y', 'z']),
    LanguageSubtag::new(['k', 'z', 'a']),
    LanguageSubtag::new(['k', 'z', 'b']),
    LanguageSubtag::new(['k', 'z', 'c']),
    LanguageSubtag::new(['k', 'z', 'd']),
    LanguageSubtag::new(['k', 'z', 'e']),
    LanguageSubtag::new(['k', 'z', 'f']),
    LanguageSubtag::new(['k', 'z', 'g']),
    LanguageSubtag::new(['k', 'z', 'h']),
    LanguageSubtag::new(['k', 'z', 'i']),
    LanguageSubtag::new(['k', 'z', 'j']),
    LanguageSubtag::new(['k', 'z', 'k']),
    LanguageSubtag::new(['k', 'z', 'l']),
    LanguageSubtag::new(['k', 'z', 'm']),
    LanguageSubtag::new(['k', 'z', 'n']),
    LanguageSubtag::new(['k', 'z', 'o']),
    LanguageSubtag::new(['k', 'z', 'p']),
    LanguageSubtag::new(['k', 'z', 'q']),
    LanguageSubtag::new(['k', 'z', 'r']),
    LanguageSubtag::new(['k', 'z', 's']),
    LanguageSubtag::new(['k', 'z', 't']),
    LanguageSubtag::new(['k', 'z', 'u']),
    LanguageSubtag::new(['k', 'z', 'v']),
    LanguageSubtag::new(['k', 'z', 'w']),
    LanguageSubtag::new(['k', 'z', 'x']),
    LanguageSubtag::new(['k', 'z', 'y']),
    LanguageSubtag::new(['k', 'z', 'z']),
    LanguageSubtag::new(['l', 'a', ' ']),
    LanguageSubtag::new(['l', 'a', 'a']),
    LanguageSubtag::new(['l', 'a', 'b']),
    LanguageSubtag::new(['l', 'a', 'c']),
    LanguageSubtag::new(['l', 'a', 'd']),
    LanguageSubtag::new(['l', 'a', 'e']),
    LanguageSubtag::new(['l', 'a', 'f']),
    LanguageSubtag::new(['l', 'a', 'g']),
    LanguageSubtag::new(['l', 'a', 'h']),
    LanguageSubtag::new(['l', 'a', 'i']),
    LanguageSubtag::new(['l', 'a', 'j']),
    LanguageSubtag::new(['l', 'a', 'k']),
    LanguageSubtag::new(['l', 'a', 'l']),
    LanguageSubtag::new(['l', 'a', 'm']),
    LanguageSubtag::new(['l', 'a', 'n']),
    LanguageSubtag::new(['l', 'a', 'p']),
    LanguageSubtag::new(['l', 'a', 'q']),
    LanguageSubtag::new(['l', 'a', 'r']),
    LanguageSubtag::new(['l', 'a', 's']),
    LanguageSubtag::new(['l', 'a', 'u']),
    LanguageSubtag::new(['l', 'a', 'w']),
    LanguageSubtag::new(['l', 'a', 'x']),
    LanguageSubtag::new(['l', 'a', 'y']),
    LanguageSubtag::new(['l', 'a', 'z']),
    LanguageSubtag::new(['l', 'b', ' ']),
    LanguageSubtag::new(['l', 'b', 'a']),
    LanguageSubtag::new(['l', 'b', 'b']),
    LanguageSubtag::new(['l', 'b', 'c']),
    LanguageSubtag::new(['l', 'b', 'e']),
    LanguageSubtag::new(['l', 'b', 'f']),
    LanguageSubtag::new(['l', 'b', 'g']),
    LanguageSubtag::new(['l', 'b', 'i']),
    LanguageSubtag::new(['l', 'b', 'j']),
    LanguageSubtag::new(['l', 'b', 'k']),
    LanguageSubtag::new(['l', 'b', 'l']),
    LanguageSubtag::new(['l', 'b', 'm']),
    LanguageSubtag::new(['l', 'b', 'n']),
    LanguageSubtag::new(['l', 'b', 'o']),
    LanguageSubtag::new(['l', 'b', 'q']),
    LanguageSubtag::new(['l', 'b', 'r']),
    LanguageSubtag::new(['l', 'b', 's']),
    LanguageSubtag::new(['l', 'b', 't']),
    LanguageSubtag::new(['l', 'b', 'u']),
    LanguageSubtag::new(['l', 'b', 'v']),
    LanguageSubtag::new(['l', 'b', 'w']),
    LanguageSubtag::new(['l', 'b', 'x']),
    LanguageSubtag::new(['l', 'b', 'y']),
    LanguageSubtag::new(['l', 'b', 'z']),
    LanguageSubtag::new(['l', 'c', 'c']),
    LanguageSubtag::new(['l', 'c', 'd']),
    LanguageSubtag::new(['l', 'c', 'e']),
    LanguageSubtag::new(['l', 'c', 'f']),
    LanguageSubtag::new(['l', 'c', 'h']),
    LanguageSubtag::new(['l', 'c', 'l']),
    LanguageSubtag::new(['l', 'c', 'm']),
    LanguageSubtag::new(['l', 'c', 'p']),
    LanguageSubtag::new(['l', 'c', 'q']),
    LanguageSubtag::new(['l', 'c', 's']),
    LanguageSubtag::new(['l', 'd', 'a']),
    LanguageSubtag::new(['l', 'd', 'b']),
    LanguageSubtag::new(['l', 'd', 'd']),
    LanguageSubtag::new(['l', 'd', 'g']),
    LanguageSubtag::new(['l', 'd', 'h']),
    LanguageSubtag::new(['l', 'd', 'i']),
    LanguageSubtag::new(['l', 'd', 'j']),
    LanguageSubtag::new(['l', 'd', 'k']),
    LanguageSubtag::new(['l', 'd', 'l']),
    LanguageSubtag::new(['l', 'd', 'm']),
    LanguageSubtag::new(['l', 'd', 'n']),
    LanguageSubtag::new(['l', 'd', 'o']),
    LanguageSubtag::new(['l', 'd', 'p']),
    LanguageSubtag::new(['l', 'd', 'q']),
    LanguageSubtag::new(['l', 'e', 'a']),
    LanguageSubtag::new(['l', 'e', 'b']),
    LanguageSubtag::new(['l', 'e', 'c']),
    LanguageSubtag::new(['l', 'e', 'd']),
    LanguageSubtag::new(['l', 'e', 'e']),
    LanguageSubtag::new(['l', 'e', 'f']),
    LanguageSubtag::new(['l', 'e', 'g']),
    LanguageSubtag::new(['l', 'e', 'h']),
    LanguageSubtag::new(['l', 'e', 'i']),
    LanguageSubtag::new(['l', 'e', 'j']),
    LanguageSubtag::new(['l', 'e', 'k']),
    LanguageSubtag::new(['l', 'e', 'l']),
    LanguageSubtag::new(['l', 'e', 'm']),
    LanguageSubtag::new(['l', 'e', 'n']),
    LanguageSubtag::new(['l', 'e', 'o']),
    LanguageSubtag::new(['l', 'e', 'p']),
    LanguageSubtag::new(['l', 'e', 'q']),
    LanguageSubtag::new(['l', 'e', 'r']),
    LanguageSubtag::new(['l', 'e', 's']),
    LanguageSubtag::new(['l', 'e', 't']),
    LanguageSubtag::new(['l', 'e', 'u']),
    LanguageSubtag::new(['l', 'e', 'v']),
    LanguageSubtag::new(['l', 'e', 'w']),
    LanguageSubtag::new(['l', 'e', 'x']),
    LanguageSubtag::new(['l', 'e', 'y']),
    LanguageSubtag::new(['l', 'e', 'z']),
    LanguageSubtag::new(['l', 'f', 'a']),
    LanguageSubtag::new(['l', 'f', 'n']),
    LanguageSubtag::new(['l', 'g', ' ']),
    LanguageSubtag::new(['l', 'g', 'a']),
    LanguageSubtag::new(['l', 'g', 'b']),
    LanguageSubtag::new(['l', 'g', 'g']),
    LanguageSubtag::new(['l', 'g', 'h']),
    LanguageSubtag::new(['l', 'g', 'i']),
    LanguageSubtag::new(['l', 'g', 'k']),
    LanguageSubtag::new(['l', 'g', 'l']),
    LanguageSubtag::new(['l', 'g', 'm']),
    LanguageSubtag::new(['l', 'g', 'n']),
    LanguageSubtag::new(['l', 'g', 'q']),
    LanguageSubtag::new(['l', 'g', 'r']),
    LanguageSubtag::new(['l', 'g', 't']),
    LanguageSubtag::new(['l', 'g', 'u']),
    LanguageSubtag::new(['l', 'g', 'z']),
    LanguageSubtag::new(['l', 'h', 'a']),
    LanguageSubtag::new(['l', 'h', 'h']),
    LanguageSubtag::new(['l', 'h', 'i']),
    LanguageSubtag::new(['l', 'h', 'l']),
    LanguageSubtag::new(['l', 'h', 'm']),
    LanguageSubtag::new(['l', 'h', 'n']),
    LanguageSubtag::new(['l', 'h', 'p']),
    LanguageSubtag::new(['l', 'h', 's']),
    LanguageSubtag::new(['l', 'h', 't']),
    LanguageSubtag::new(['l', 'h', 'u']),
    LanguageSubtag::new(['l', 'i', ' ']),
    LanguageSubtag::new(['l', 'i', 'a']),
    LanguageSubtag::new(['l', 'i', 'b']),
    LanguageSubtag::new(['l', 'i', 'c']),
    LanguageSubtag::new(['l', 'i', 'd']),
    LanguageSubtag::new(['l', 'i', 'e']),
    LanguageSubtag::new(['l', 'i', 'f']),
    LanguageSubtag::new(['l', 'i', 'g']),
    LanguageSubtag::new(['l', 'i', 'h']),
    LanguageSubtag::new(['l', 'i', 'i']),
    LanguageSubtag::new(['l', 'i', 'j']),
    LanguageSubtag::new(['l', 'i', 'k']),
    LanguageSubtag::new(['l', 'i', 'l']),
    LanguageSubtag::new(['l', 'i', 'o']),
    LanguageSubtag::new(['l', 'i', 'p']),
    LanguageSubtag::new(['l', 'i', 'q']),
    LanguageSubtag::new(['l', 'i', 'r']),
    LanguageSubtag::new(['l', 'i', 's']),
    LanguageSubtag::new(['l', 'i', 'u']),
    LanguageSubtag::new(['l', 'i', 'v']),
    LanguageSubtag::new(['l', 'i', 'w']),
    LanguageSubtag::new(['l', 'i', 'x']),
    LanguageSubtag::new(['l', 'i', 'y']),
    LanguageSubtag::new(['l', 'i', 'z']),
    LanguageSubtag::new(['l', 'j', 'a']),
    LanguageSubtag::new(['l', 'j', 'e']),
    LanguageSubtag::new(['l', 'j', 'i']),
    LanguageSubtag::new(['l', 'j', 'l']),
    LanguageSubtag::new(['l', 'j', 'p']),
    LanguageSubtag::new(['l', 'j', 'w']),
    LanguageSubtag::new(['l', 'j', 'x']),
    LanguageSubtag::new(['l', 'k', 'a']),
    LanguageSubtag::new(['l', 'k', 'b']),
    LanguageSubtag::new(['l', 'k', 'c']),
    LanguageSubtag::new(['l', 'k', 'd']),
    LanguageSubtag::new(['l', 'k', 'e']),
    LanguageSubtag::new(['l', 'k', 'h']),
    LanguageSubtag::new(['l', 'k', 'i']),
    LanguageSubtag::new(['l', 'k', 'j']),
    LanguageSubtag::new(['l', 'k', 'l']),
    LanguageSubtag::new(['l', 'k', 'm']),
    LanguageSubtag::new(['l', 'k', 'n']),
    LanguageSubtag::new(['l', 'k', 'o']),
    LanguageSubtag::new(['l', 'k', 'r']),
    LanguageSubtag::new(['l', 'k', 's']),
    LanguageSubtag::new(['l', 'k', 't']),
    LanguageSubtag::new(['l', 'k', 'u']),
    LanguageSubtag::new(['l', 'k', 'y']),
    LanguageSubtag::new(['l', 'l', 'a']),
    LanguageSubtag::new(['l', 'l', 'b']),
    LanguageSubtag::new(['l', 'l', 'c']),
    LanguageSubtag::new(['l', 'l', 'd']),
    LanguageSubtag::new(['l', 'l', 'e']),
    LanguageSubtag::new(['l', 'l', 'f']),
    LanguageSubtag::new(['l', 'l', 'g']),
    LanguageSubtag::new(['l', 'l', 'h']),
    LanguageSubtag::new(['l', 'l', 'i']),
    LanguageSubtag::new(['l', 'l', 'j']),
    LanguageSubtag::new(['l', 'l', 'k']),
    LanguageSubtag::new(['l', 'l', 'l']),
    LanguageSubtag::new(['l', 'l', 'm']),
    LanguageSubtag::new(['l', 'l', 'n']),
    LanguageSubtag::new(['l', 'l', 'o']),
    LanguageSubtag::new(['l', 'l', 'p']),
    LanguageSubtag::new(['l', 'l', 'q']),
    LanguageSubtag::new(['l', 'l', 's']),
    LanguageSubtag::new(['l', 'l', 'u']),
    LanguageSubtag::new(['l', 'l', 'x']),
    LanguageSubtag::new(['l', 'm', 'a']),
    LanguageSubtag::new(['l', 'm', 'b']),
    LanguageSubtag::new(['l', 'm', 'c']),
    LanguageSubtag::new(['l', 'm', 'd']),
    LanguageSubtag::new(['l', 'm', 'e']),
    LanguageSubtag::new(['l', 'm', 'f']),
    LanguageSubtag::new(['l', 'm', 'g']),
    LanguageSubtag::new(['l', 'm', 'h']),
    LanguageSubtag::new(['l', 'm', 'i']),
    LanguageSubtag::new(['l', 'm', 'j']),
    LanguageSubtag::new(['l', 'm', 'k']),
    LanguageSubtag::new(['l', 'm', 'l']),
    LanguageSubtag::new(['l', 'm', 'm']),
    LanguageSubtag::new(['l', 'm', 'n']),
    LanguageSubtag::new(['l', 'm', 'o']),
    LanguageSubtag::new(['l', 'm', 'p']),
    LanguageSubtag::new(['l', 'm', 'q']),
    LanguageSubtag::new(['l', 'm', 'r']),
    LanguageSubtag::new(['l', 'm', 'u']),
    LanguageSubtag::new(['l', 'm', 'v']),
    LanguageSubtag::new(['l', 'm', 'w']),
    LanguageSubtag::new(['l', 'm', 'x']),
    LanguageSubtag::new(['l', 'm', 'y']),
    LanguageSubtag::new(['l', 'm', 'z']),
    LanguageSubtag::new(['l', 'n', ' ']),
    LanguageSubtag::new(['l', 'n', 'a']),
    LanguageSubtag::new(['l', 'n', 'b']),
    LanguageSubtag::new(['l', 'n', 'd']),
    LanguageSubtag::new(['l', 'n', 'g']),
    LanguageSubtag::new(['l', 'n', 'h']),
    LanguageSubtag::new(['l', 'n', 'i']),
    LanguageSubtag::new(['l', 'n', 'j']),
    LanguageSubtag::new(['l', 'n', 'l']),
    LanguageSubtag::new(['l', 'n', 'm']),
    LanguageSubtag::new(['l', 'n', 'n']),
    LanguageSubtag::new(['l', 'n', 'o']),
    LanguageSubtag::new(['l', 'n', 's']),
    LanguageSubtag::new(['l', 'n', 'u']),
    LanguageSubtag::new(['l', 'n', 'w']),
    LanguageSubtag::new(['l', 'n', 'z']),
    LanguageSubtag::new(['l', 'o', ' ']),
    LanguageSubtag::new(['l', 'o', 'a']),
    LanguageSubtag::new(['l', 'o', 'b']),
    LanguageSubtag::new(['l', 'o', 'c']),
    LanguageSubtag::new(['l', 'o', 'e']),
    LanguageSubtag::new(['l', 'o', 'f']),
    LanguageSubtag::new(['l', 'o', 'g']),
    LanguageSubtag::new(['l', 'o', 'h']),
    LanguageSubtag::new(['l', 'o', 'i']),
    LanguageSubtag::new(['l', 'o', 'j']),
    LanguageSubtag::new(['l', 'o', 'k']),
    LanguageSubtag::new(['l', 'o', 'l']),
    LanguageSubtag::new(['l', 'o', 'm']),
    LanguageSubtag::new(['l', 'o', 'n']),
    LanguageSubtag::new(['l', 'o', 'o']),
    LanguageSubtag::new(['l', 'o', 'p']),
    LanguageSubtag::new(['l', 'o', 'q']),
    LanguageSubtag::new(['l', 'o', 'r']),
    LanguageSubtag::new(['l', 'o', 's']),
    LanguageSubtag::new(['l', 'o', 't']),
    LanguageSubtag::new(['l', 'o', 'u']),
    LanguageSubtag::new(['l', 'o', 'v']),
    LanguageSubtag::new(['l', 'o', 'w']),
    LanguageSubtag::new(['l', 'o', 'x']),
    LanguageSubtag::new(['l', 'o', 'y']),
    LanguageSubtag::new(['l', 'o', 'z']),
    LanguageSubtag::new(['l', 'p', 'a']),
    LanguageSubtag::new(['l', 'p', 'e']),
    LanguageSubtag::new(['l', 'p', 'n']),
    LanguageSubtag::new(['l', 'p', 'o']),
    LanguageSubtag::new(['l', 'p', 'x']),
    LanguageSubtag::new(['l', 'r', 'a']),
    LanguageSubtag::new(['l', 'r', 'c']),
    LanguageSubtag::new(['l', 'r', 'e']),
    LanguageSubtag::new(['l', 'r', 'g']),
    LanguageSubtag::new(['l', 'r', 'i']),
    LanguageSubtag::new(['l', 'r', 'k']),
    LanguageSubtag::new(['l', 'r', 'l']),
    LanguageSubtag::new(['l', 'r', 'm']),
    LanguageSubtag::new(['l', 'r', 'n']),
    LanguageSubtag::new(['l', 'r', 'o']),
    LanguageSubtag::new(['l', 'r', 'r']),
    LanguageSubtag::new(['l', 'r', 't']),
    LanguageSubtag::new(['l', 'r', 'v']),
    LanguageSubtag::new(['l', 'r', 'z']),
    LanguageSubtag::new(['l', 's', 'a']),
    LanguageSubtag::new(['l', 's', 'b']),
    LanguageSubtag::new(['l', 's', 'd']),
    LanguageSubtag::new(['l', 's', 'e']),
    LanguageSubtag::new(['l', 's', 'g']),
    LanguageSubtag::new(['l', 's', 'h']),
    LanguageSubtag::new(['l', 's', 'i']),
    LanguageSubtag::new(['l', 's', 'l']),
    LanguageSubtag::new(['l', 's', 'm']),
    LanguageSubtag::new(['l', 's', 'n']),
    LanguageSubtag::new(['l', 's', 'o']),
    LanguageSubtag::new(['l', 's', 'p']),
    LanguageSubtag::new(['l', 's', 'r']),
    LanguageSubtag::new(['l', 's', 's']),
    LanguageSubtag::new(['l', 's', 't']),
    LanguageSubtag::new(['l', 's', 'v']),
    LanguageSubtag::new(['l', 's', 'y']),
    LanguageSubtag::new(['l', 't', ' ']),
    LanguageSubtag::new(['l', 't', 'c']),
    LanguageSubtag::new(['l', 't', 'g']),
    LanguageSubtag::new(['l', 't', 'h']),
    LanguageSubtag::new(['l', 't', 'i']),
    LanguageSubtag::new(['l', 't', 'n']),
    LanguageSubtag::new(['l', 't', 'o']),
    LanguageSubtag::new(['l', 't', 's']),
    LanguageSubtag::new(['l', 't', 'u']),
    LanguageSubtag::new(['l', 'u', ' ']),
    LanguageSubtag::new(['l', 'u', 'a']),
    LanguageSubtag::new(['l', 'u', 'c']),
    LanguageSubtag::new(['l', 'u', 'd']),
    LanguageSubtag::new(['l', 'u', 'e']),
    LanguageSubtag::new(['l', 'u', 'f']),
    LanguageSubtag::new(['l', 'u', 'i']),
    LanguageSubtag::new(['l', 'u', 'j']),
    LanguageSubtag::new(['l', 'u', 'k']),
    LanguageSubtag::new(['l', 'u', 'l']),
    LanguageSubtag::new(['l', 'u', 'm']),
    LanguageSubtag::new(['l', 'u', 'n']),
    LanguageSubtag::new(['l', 'u', 'o']),
    LanguageSubtag::new(['l', 'u', 'p']),
    LanguageSubtag::new(['l', 'u', 'q']),
    LanguageSubtag::new(['l', 'u', 'r']),
    LanguageSubtag::new(['l', 'u', 's']),
    LanguageSubtag::new(['l', 'u', 't']),
    LanguageSubtag::new(['l', 'u', 'u']),
    LanguageSubtag::new(['l', 'u', 'v']),
    LanguageSubtag::new(['l', 'u', 'w']),
    LanguageSubtag::new(['l', 'u', 'y']),
    LanguageSubtag::new(['l', 'u', 'z']),
    LanguageSubtag::new(['l', 'v', ' ']),
    LanguageSubtag::new(['l', 'v', 'a']),
    LanguageSubtag::new(['l', 'v', 'i']),
    LanguageSubtag::new(['l', 'v', 'k']),
    LanguageSubtag::new(['l', 'v', 's']),
    LanguageSubtag::new(['l', 'v', 'u']),
    LanguageSubtag::new(['l', 'w', 'a']),
    LanguageSubtag::new(['l', 'w', 'e']),
    LanguageSubtag::new(['l', 'w', 'g']),
    LanguageSubtag::new(['l', 'w', 'h']),
    LanguageSubtag::new(['l', 'w', 'l']),
    LanguageSubtag::new(['l', 'w', 'm']),
    LanguageSubtag::new(['l', 'w', 'o']),
    LanguageSubtag::new(['l', 'w', 's']),
    LanguageSubtag::new(['l', 'w', 't']),
    LanguageSubtag::new(['l', 'w', 'u']),
    LanguageSubtag::new(['l', 'w', 'w']),
    LanguageSubtag::new(['l', 'x', 'm']),
    LanguageSubtag::new(['l', 'y', 'a']),
    LanguageSubtag::new(['l', 'y', 'g']),
    LanguageSubtag::new(['l', 'y', 'n']),
    LanguageSubtag::new(['l', 'z', 'h']),
    LanguageSubtag::new(['l', 'z', 'l']),
    LanguageSubtag::new(['l', 'z', 'n']),
    LanguageSubtag::new(['l', 'z', 'z']),
    LanguageSubtag::new(['m', 'a', 'a']),
    LanguageSubtag::new(['m', 'a', 'b']),
    LanguageSubtag::new(['m', 'a', 'd']),
    LanguageSubtag::new(['m', 'a', 'e']),
    LanguageSubtag::new(['m', 'a', 'f']),
    LanguageSubtag::new(['m', 'a', 'g']),
    LanguageSubtag::new(['m', 'a', 'i']),
    LanguageSubtag::new(['m', 'a', 'j']),
    LanguageSubtag::new(['m', 'a', 'k']),
    LanguageSubtag::new(['m', 'a', 'm']),
    LanguageSubtag::new(['m', 'a', 'n']),
    LanguageSubtag::new(['m', 'a', 'p']),
    LanguageSubtag::new(['m', 'a', 'q']),
    LanguageSubtag::new(['m', 'a', 's']),
    LanguageSubtag::new(['m', 'a', 't']),
    LanguageSubtag::new(['m', 'a', 'u']),
    LanguageSubtag::new(['m', 'a', 'v']),
    LanguageSubtag::new(['m', 'a', 'w']),
    LanguageSubtag::new(['m', 'a', 'x']),
    LanguageSubtag::new(['m', 'a', 'z']),
    LanguageSubtag::new(['m', 'b', 'a']),
    LanguageSubtag::new(['m', 'b', 'b']),
    LanguageSubtag::new(['m', 'b', 'c']),
    LanguageSubtag::new(['m', 'b', 'd']),
    LanguageSubtag::new(['m', 'b', 'e']),
    LanguageSubtag::new(['m', 'b', 'f']),
    LanguageSubtag::new(['m', 'b', 'h']),
    LanguageSubtag::new(['m', 'b', 'i']),
    LanguageSubtag::new(['m', 'b', 'j']),
    LanguageSubtag::new(['m', 'b', 'k']),
    LanguageSubtag::new(['m', 'b', 'l']),
    LanguageSubtag::new(['m', 'b', 'm']),
    LanguageSubtag::new(['m', 'b', 'n']),
    LanguageSubtag::new(['m', 'b', 'o']),
    LanguageSubtag::new(['m', 'b', 'p']),
    LanguageSubtag::new(['m', 'b', 'q']),
    LanguageSubtag::new(['m', 'b', 'r']),
    LanguageSubtag::new(['m', 'b', 's']),
    LanguageSubtag::new(['m', 'b', 't']),
    LanguageSubtag::new(['m', 'b', 'u']),
    LanguageSubtag::new(['m', 'b', 'v']),
    LanguageSubtag::new(['m', 'b', 'w']),
    LanguageSubtag::new(['m', 'b', 'x']),
    LanguageSubtag::new(['m', 'b', 'y']),
    LanguageSubtag::new(['m', 'b', 'z']),
    LanguageSubtag::new(['m', 'c', 'a']),
    LanguageSubtag::new(['m', 'c', 'b']),
    LanguageSubtag::new(['m', 'c', 'c']),
    LanguageSubtag::new(['m', 'c', 'd']),
    LanguageSubtag::new(['m', 'c', 'e']),
    LanguageSubtag::new(['m', 'c', 'f']),
    LanguageSubtag::new(['m', 'c', 'g']),
    LanguageSubtag::new(['m', 'c', 'h']),
    LanguageSubtag::new(['m', 'c', 'i']),
    LanguageSubtag::new(['m', 'c', 'j']),
    LanguageSubtag::new(['m', 'c', 'k']),
    LanguageSubtag::new(['m', 'c', 'l']),
    LanguageSubtag::new(['m', 'c', 'm']),
    LanguageSubtag::new(['m', 'c', 'n']),
    LanguageSubtag::new(['m', 'c', 'o']),
    LanguageSubtag::new(['m', 'c', 'p']),
    LanguageSubtag::new(['m', 'c', 'q']),
    LanguageSubtag::new(['m', 'c', 'r']),
    LanguageSubtag::new(['m', 'c', 's']),
    LanguageSubtag::new(['m', 'c', 't']),
    LanguageSubtag::new(['m', 'c', 'u']),
    LanguageSubtag::new(['m', 'c', 'v']),
    LanguageSubtag::new(['m', 'c', 'w']),
    LanguageSubtag::new(['m', 'c', 'x']),
    LanguageSubtag::new(['m', 'c', 'y']),
    LanguageSubtag::new(['m', 'c', 'z']),
    LanguageSubtag::new(['m', 'd', 'a']),
    LanguageSubtag::new(['m', 'd', 'b']),
    LanguageSubtag::new(['m', 'd', 'c']),
    LanguageSubtag::new(['m', 'd', 'd']),
    LanguageSubtag::new(['m', 'd', 'e']),
    LanguageSubtag::new(['m', 'd', 'f']),
    LanguageSubtag::new(['m', 'd', 'g']),
    LanguageSubtag::new(['m', 'd', 'h']),
    LanguageSubtag::new(['m', 'd', 'i']),
    LanguageSubtag::new(['m', 'd', 'j']),
    LanguageSubtag::new(['m', 'd', 'k']),
    LanguageSubtag::new(['m', 'd', 'l']),
    LanguageSubtag::new(['m', 'd', 'm']),
    LanguageSubtag::new(['m', 'd', 'n']),
    LanguageSubtag::new(['m', 'd', 'p']),
    LanguageSubtag::new(['m', 'd', 'q']),
    LanguageSubtag::new(['m', 'd', 'r']),
    LanguageSubtag::new(['m', 'd', 's']),
    LanguageSubtag::new(['m', 'd', 't']),
    LanguageSubtag::new(['m', 'd', 'u']),
    LanguageSubtag::new(['m', 'd', 'v']),
    LanguageSubtag::new(['m', 'd', 'w']),
    LanguageSubtag::new(['m', 'd', 'x']),
    LanguageSubtag::new(['m', 'd', 'y']),
    LanguageSubtag::new(['m', 'd', 'z']),
    LanguageSubtag::new(['m', 'e', 'a']),
    LanguageSubtag::new(['m', 'e', 'b']),
    LanguageSubtag::new(['m', 'e', 'c']),
    LanguageSubtag::new(['m', 'e', 'd']),
    LanguageSubtag::new(['m', 'e', 'e']),
    LanguageSubtag::new(['m', 'e', 'f']),
    LanguageSubtag::new(['m', 'e', 'g']),
    LanguageSubtag::new(['m', 'e', 'h']),
    LanguageSubtag::new(['m', 'e', 'i']),
    LanguageSubtag::new(['m', 'e', 'j']),
    LanguageSubtag::new(['m', 'e', 'k']),
    LanguageSubtag::new(['m', 'e', 'l']),
    LanguageSubtag::new(['m', 'e', 'm']),
    LanguageSubtag::new(['m', 'e', 'n']),
    LanguageSubtag::new(['m', 'e', 'o']),
    LanguageSubtag::new(['m', 'e', 'p']),
    LanguageSubtag::new(['m', 'e', 'q']),
    LanguageSubtag::new(['m', 'e', 'r']),
    LanguageSubtag::new(['m', 'e', 's']),
    LanguageSubtag::new(['m', 'e', 't']),
    LanguageSubtag::new(['m', 'e', 'u']),
    LanguageSubtag::new(['m', 'e', 'v']),
    LanguageSubtag::new(['m', 'e', 'w']),
    LanguageSubtag::new(['m', 'e', 'y']),
    LanguageSubtag::new(['m', 'e', 'z']),
    LanguageSubtag::new(['m', 'f', 'a']),
    LanguageSubtag::new(['m', 'f', 'b']),
    LanguageSubtag::new(['m', 'f', 'c']),
    LanguageSubtag::new(['m', 'f', 'd']),
    LanguageSubtag::new(['m', 'f', 'e']),
    LanguageSubtag::new(['m', 'f', 'f']),
    LanguageSubtag::new(['m', 'f', 'g']),
    LanguageSubtag::new(['m', 'f', 'h']),
    LanguageSubtag::new(['m', 'f', 'i']),
    LanguageSubtag::new(['m', 'f', 'j']),
    LanguageSubtag::new(['m', 'f', 'k']),
    LanguageSubtag::new(['m', 'f', 'l']),
    LanguageSubtag::new(['m', 'f', 'm']),
    LanguageSubtag::new(['m', 'f', 'n']),
    LanguageSubtag::new(['m', 'f', 'o']),
    LanguageSubtag::new(['m', 'f', 'p']),
    LanguageSubtag::new(['m', 'f', 'q']),
    LanguageSubtag::new(['m', 'f', 'r']),
    LanguageSubtag::new(['m', 'f', 's']),
    LanguageSubtag::new(['m', 'f', 't']),
    LanguageSubtag::new(['m', 'f', 'u']),
    LanguageSubtag::new(['m', 'f', 'v']),
    LanguageSubtag::new(['m', 'f', 'w']),
    LanguageSubtag::new(['m', 'f', 'x']),
    LanguageSubtag::new(['m', 'f', 'y']),
    LanguageSubtag::new(['m', 'f', 'z']),
    LanguageSubtag::new(['m', 'g', ' ']),
    LanguageSubtag::new(['m', 'g', 'a']),
    LanguageSubtag::new(['m', 'g', 'b']),
    LanguageSubtag::new(['m', 'g', 'c']),
    LanguageSubtag::new(['m', 'g', 'd']),
    LanguageSubtag::new(['m', 'g', 'e']),
    LanguageSubtag::new(['m', 'g', 'f']),
    LanguageSubtag::new(['m', 'g', 'g']),
    LanguageSubtag::new(['m', 'g', 'h']),
    LanguageSubtag::new(['m', 'g', 'i']),
    LanguageSubtag::new(['m', 'g', 'j']),
    LanguageSubtag::new(['m', 'g', 'k']),
    LanguageSubtag::new(['m', 'g', 'l']),
    LanguageSubtag::new(['m', 'g', 'm']),
    LanguageSubtag::new(['m', 'g', 'n']),
    LanguageSubtag::new(['m', 'g', 'o']),
    LanguageSubtag::new(['m', 'g', 'p']),
    LanguageSubtag::new(['m', 'g', 'q']),
    LanguageSubtag::new(['m', 'g', 'r']),
    LanguageSubtag::new(['m', 'g', 's']),
    LanguageSubtag::new(['m', 'g', 't']),
    LanguageSubtag::new(['m', 'g', 'u']),
    LanguageSubtag::new(['m', 'g', 'v']),
    LanguageSubtag::new(['m', 'g', 'w']),
    LanguageSubtag::new(['m', 'g', 'x']),
    LanguageSubtag::new(['m', 'g', 'y']),
    LanguageSubtag::new(['m', 'g', 'z']),
    LanguageSubtag::new(['m', 'h', ' ']),
    LanguageSubtag::new(['m', 'h', 'a']),
    LanguageSubtag::new(['m', 'h', 'b']),
    LanguageSubtag::new(['m', 'h', 'c']),
    LanguageSubtag::new(['m', 'h', 'd']),
    LanguageSubtag::new(['m', 'h', 'e']),
    LanguageSubtag::new(['m', 'h', 'f']),
    LanguageSubtag::new(['m', 'h', 'g']),
    LanguageSubtag::new(['m', 'h', 'h']),
    LanguageSubtag::new(['m', 'h', 'i']),
    LanguageSubtag::new(['m', 'h', 'j']),
    LanguageSubtag::new(['m', 'h', 'k']),
    LanguageSubtag::new(['m', 'h', 'l']),
    LanguageSubtag::new(['m', 'h', 'm']),
    LanguageSubtag::new(['m', 'h', 'n']),
    LanguageSubtag::new(['m', 'h', 'o']),
    LanguageSubtag::new(['m', 'h', 'p']),
    LanguageSubtag::new(['m', 'h', 'q']),
    LanguageSubtag::new(['m', 'h', 'r']),
    LanguageSubtag::new(['m', 'h', 's']),
    LanguageSubtag::new(['m', 'h', 't']),
    LanguageSubtag::new(['m', 'h', 'u']),
    LanguageSubtag::new(['m', 'h', 'w']),
    LanguageSubtag::new(['m', 'h', 'x']),
    LanguageSubtag::new(['m', 'h', 'y']),
    LanguageSubtag::new(['m', 'h', 'z']),
    LanguageSubtag::new(['m', 'i', ' ']),
    LanguageSubtag::new(['m', 'i', 'a']),
    LanguageSubtag::new(['m', 'i', 'b']),
    LanguageSubtag::new(['m', 'i', 'c']),
    LanguageSubtag::new(['m', 'i', 'd']),
    LanguageSubtag::new(['m', 'i', 'e']),
    LanguageSubtag::new(['m', 'i', 'f']),
    LanguageSubtag::new(['m', 'i', 'g']),
    LanguageSubtag::new(['m', 'i', 'h']),
    LanguageSubtag::new(['m', 'i', 'i']),
    LanguageSubtag::new(['m', 'i', 'j']),
    LanguageSubtag::new(['m', 'i', 'k']),
    LanguageSubtag::new(['m', 'i', 'l']),
    LanguageSubtag::new(['m', 'i', 'm']),
    LanguageSubtag::new(['m', 'i', 'n']),
    LanguageSubtag::new(['m', 'i', 'o']),
    LanguageSubtag::new(['m', 'i', 'p']),
    LanguageSubtag::new(['m', 'i', 'q']),
    LanguageSubtag::new(['m', 'i', 'r']),
    LanguageSubtag::new(['m', 'i', 's']),
    LanguageSubtag::new(['m', 'i', 't']),
    LanguageSubtag::new(['m', 'i', 'u']),
    LanguageSubtag::new(['m', 'i', 'w']),
    LanguageSubtag::new(['m', 'i', 'x']),
    LanguageSubtag::new(['m', 'i', 'y']),
    LanguageSubtag::new(['m', 'i', 'z']),
    LanguageSubtag::new(['m', 'j', 'a']),
    LanguageSubtag::new(['m', 'j', 'b']),
    LanguageSubtag::new(['m', 'j', 'c']),
    LanguageSubtag::new(['m', 'j', 'd']),
    LanguageSubtag::new(['m', 'j', 'e']),
    LanguageSubtag::new(['m', 'j', 'g']),
    LanguageSubtag::new(['m', 'j', 'h']),
    LanguageSubtag::new(['m', 'j', 'i']),
    LanguageSubtag::new(['m', 'j', 'j']),
    LanguageSubtag::new(['m', 'j', 'k']),
    LanguageSubtag::new(['m', 'j', 'l']),
    LanguageSubtag::new(['m', 'j', 'm']),
    LanguageSubtag::new(['m', 'j', 'n']),
    LanguageSubtag::new(['m', 'j', 'o']),
    LanguageSubtag::new(['m', 'j', 'p']),
    LanguageSubtag::new(['m', 'j', 'q']),
    LanguageSubtag::new(['m', 'j', 'r']),
    LanguageSubtag::new(['m', 'j', 's']),
    LanguageSubtag::new(['m', 'j', 't']),
    LanguageSubtag::new(['m', 'j', 'u']),
    LanguageSubtag::new(['m', 'j', 'v']),
    LanguageSubtag::new(['m', 'j', 'w']),
    LanguageSubtag::new(['m', 'j', 'x']),
    LanguageSubtag::new(['m', 'j', 'y']),
    LanguageSubtag::new(['m', 'j', 'z']),
    LanguageSubtag::new(['m', 'k', ' ']),
    LanguageSubtag::new(['m', 'k', 'a']),
    LanguageSubtag::new(['m', 'k', 'b']),
    LanguageSubtag::new(['m', 'k', 'c']),
    LanguageSubtag::new(['m', 'k', 'e']),
    LanguageSubtag::new(['m', 'k', 'f']),
    LanguageSubtag::new(['m', 'k', 'g']),
    LanguageSubtag::new(['m', 'k', 'h']),
    LanguageSubtag::new(['m', 'k', 'i']),
    LanguageSubtag::new(['m', 'k', 'j']),
    LanguageSubtag::new(['m', 'k', 'k']),
    LanguageSubtag::new(['m', 'k', 'l']),
    LanguageSubtag::new(['m', 'k', 'm']),
    LanguageSubtag::new(['m', 'k', 'n']),
    LanguageSubtag::new(['m', 'k', 'o']),
    LanguageSubtag::new(['m', 'k', 'p']),
    LanguageSubtag::new(['m', 'k', 'q']),
    LanguageSubtag::new(['m', 'k', 'r']),
    LanguageSubtag::new(['m', 'k', 's']),
    LanguageSubtag::new(['m', 'k', 't']),
    LanguageSubtag::new(['m', 'k', 'u']),
    LanguageSubtag::new(['m', 'k', 'v']),
    LanguageSubtag::new(['m', 'k', 'w']),
    LanguageSubtag::new(['m', 'k', 'x']),
    LanguageSubtag::new(['m', 'k', 'y']),
    LanguageSubtag::new(['m', 'k', 'z']),
    LanguageSubtag::new(['m', 'l', ' ']),
    LanguageSubtag::new(['m', 'l', 'a']),
    LanguageSubtag::new(['m', 'l', 'b']),
    LanguageSubtag::new(['m', 'l', 'c']),
    LanguageSubtag::new(['m', 'l', 'd']),
    LanguageSubtag::new(['m', 'l', 'e']),
    LanguageSubtag::new(['m', 'l', 'f']),
    LanguageSubtag::new(['m', 'l', 'h']),
    LanguageSubtag::new(['m', 'l', 'i']),
    LanguageSubtag::new(['m', 'l', 'j']),
    LanguageSubtag::new(['m', 'l', 'k']),
    LanguageSubtag::new(['m', 'l', 'l']),
    LanguageSubtag::new(['m', 'l', 'm']),
    LanguageSubtag::new(['m', 'l', 'n']),
    LanguageSubtag::new(['m', 'l', 'o']),
    LanguageSubtag::new(['m', 'l', 'p']),
    LanguageSubtag::new(['m', 'l', 'q']),
    LanguageSubtag::new(['m', 'l', 'r']),
    LanguageSubtag::new(['m', 'l', 's']),
    LanguageSubtag::new(['m', 'l', 'u']),
    LanguageSubtag::new(['m', 'l', 'v']),
    LanguageSubtag::new(['m', 'l', 'w']),
    LanguageSubtag::new(['m', 'l', 'x']),
    LanguageSubtag::new(['m', 'l', 'z']),
    LanguageSubtag::new(['m', 'm', 'a']),
    LanguageSubtag::new(['m', 'm', 'b']),
    LanguageSubtag::new(['m', 'm', 'c']),
    LanguageSubtag::new(['m', 'm', 'd']),
    LanguageSubtag::new(['m', 'm', 'e']),
    LanguageSubtag::new(['m', 'm', 'f']),
    LanguageSubtag::new(['m', 'm', 'g']),
    LanguageSubtag::new(['m', 'm', 'h']),
    LanguageSubtag::new(['m', 'm', 'i']),
    LanguageSubtag::new(['m', 'm', 'j']),
    LanguageSubtag::new(['m', 'm', 'k']),
    LanguageSubtag::new(['m', 'm', 'l']),
    LanguageSubtag::new(['m', 'm', 'm']),
    LanguageSubtag::new(['m', 'm', 'n']),
    LanguageSubtag::new(['m', 'm', 'o']),
    LanguageSubtag::new(['m', 'm', 'p']),
    LanguageSubtag::new(['m', 'm', 'q']),
    LanguageSubtag::new(['m', 'm', 'r']),
    LanguageSubtag::new(['m', 'm', 't']),
    LanguageSubtag::new(['m', 'm', 'u']),
    LanguageSubtag::new(['m', 'm', 'v']),
    LanguageSubtag::new(['m', 'm', 'w']),
    LanguageSubtag::new(['m', 'm', 'x']),
    LanguageSubtag::new(['m', 'm', 'y']),
    LanguageSubtag::new(['m', 'm', 'z']),
    LanguageSubtag::new(['m', 'n', ' ']),
    LanguageSubtag::new(['m', 'n', 'a']),
    LanguageSubtag::new(['m', 'n', 'b']),
    LanguageSubtag::new(['m', 'n', 'c']),
    LanguageSubtag::new(['m', 'n', 'd']),
    LanguageSubtag::new(['m', 'n', 'e']),
    LanguageSubtag::new(['m', 'n', 'f']),
    LanguageSubtag::new(['m', 'n', 'g']),
    LanguageSubtag::new(['m', 'n', 'h']),
    LanguageSubtag::new(['m', 'n', 'i']),
    LanguageSubtag::new(['m', 'n', 'j']),
    LanguageSubtag::new(['m', 'n', 'k']),
    LanguageSubtag::new(['m', 'n', 'l']),
    LanguageSubtag::new(['m', 'n', 'm']),
    LanguageSubtag::new(['m', 'n', 'n']),
    LanguageSubtag::new(['m', 'n', 'o']),
    LanguageSubtag::new(['m', 'n', 'p']),
    LanguageSubtag::new(['m', 'n', 'q']),
    LanguageSubtag::new(['m', 'n', 'r']),
    LanguageSubtag::new(['m', 'n', 's']),
    LanguageSubtag::new(['m', 'n', 't']),
    LanguageSubtag::new(['m', 'n', 'u']),
    LanguageSubtag::new(['m', 'n', 'v']),
    LanguageSubtag::new(['m', 'n', 'w']),
    LanguageSubtag::new(['m', 'n', 'x']),
    LanguageSubtag::new(['m', 'n', 'y']),
    LanguageSubtag::new(['m', 'n', 'z']),
    LanguageSubtag::new(['m', 'o', ' ']),
    LanguageSubtag::new(['m', 'o', 'a']),
    LanguageSubtag::new(['m', 'o', 'c']),
    LanguageSubtag::new(['m', 'o', 'd']),
    LanguageSubtag::new(['m', 'o', 'e']),
    LanguageSubtag::new(['m', 'o', 'f']),
    LanguageSubtag::new(['m', 'o', 'g']),
    LanguageSubtag::new(['m', 'o', 'h']),
    LanguageSubtag::new(['m', 'o', 'i']),
    LanguageSubtag::new(['m', 'o', 'j']),
    LanguageSubtag::new(['m', 'o', 'k']),
    LanguageSubtag::new(['m', 'o', 'm']),
    LanguageSubtag::new(['m', 'o', 'o']),
    LanguageSubtag::new(['m', 'o', 'p']),
    LanguageSubtag::new(['m', 'o', 'q']),
    LanguageSubtag::new(['m', 'o', 'r']),
    LanguageSubtag::new(['m', 'o', 's']),
    LanguageSubtag::new(['m', 'o', 't']),
    LanguageSubtag::new(['m', 'o', 'u']),
    LanguageSubtag::new(['m', 'o', 'v']),
    LanguageSubtag::new(['m', 'o', 'w']),
    LanguageSubtag::new(['m', 'o', 'x']),
    LanguageSubtag::new(['m', 'o', 'y']),
    LanguageSubtag::new(['m', 'o', 'z']),
    LanguageSubtag::new(['m', 'p', 'a']),
    LanguageSubtag::new(['m', 'p', 'b']),
    LanguageSubtag::new(['m', 'p', 'c']),
    LanguageSubtag::new(['m', 'p', 'd']),
    LanguageSubtag::new(['m', 'p', 'e']),
    LanguageSubtag::new(['m', 'p', 'g']),
    LanguageSubtag::new(['m', 'p', 'h']),
    LanguageSubtag::new(['m', 'p', 'i']),
    LanguageSubtag::new(['m', 'p', 'j']),
    LanguageSubtag::new(['m', 'p', 'k']),
    LanguageSubtag::new(['m', 'p', 'l']),
    LanguageSubtag::new(['m', 'p', 'm']),
    LanguageSubtag::new(['m', 'p', 'n']),
    LanguageSubtag::new(['m', 'p', 'o']),
    LanguageSubtag::new(['m', 'p', 'p']),
    LanguageSubtag::new(['m', 'p', 'q']),
    LanguageSubtag::new(['m', 'p', 'r']),
    LanguageSubtag::new(['m', 'p', 's']),
    LanguageSubtag::new(['m', 'p', 't']),
    LanguageSubtag::new(['m', 'p', 'u']),
    LanguageSubtag::new(['m', 'p', 'v']),
    LanguageSubtag::new(['m', 'p', 'w']),
    LanguageSubtag::new(['m', 'p', 'x']),
    LanguageSubtag::new(['m', 'p', 'y']),
    LanguageSubtag::new(['m', 'p', 'z']),
    LanguageSubtag::new(['m', 'q', 'a']),
    LanguageSubtag::new(['m', 'q', 'b']),
    LanguageSubtag::new(['m', 'q', 'c']),
    LanguageSubtag::new(['m', 'q', 'e']),
    LanguageSubtag::new(['m', 'q', 'f']),
    LanguageSubtag::new(['m', 'q', 'g']),
    LanguageSubtag::new(['m', 'q', 'h']),
    LanguageSubtag::new(['m', 'q', 'i']),
    LanguageSubtag::new(['m', 'q', 'j']),
    LanguageSubtag::new(['m', 'q', 'k']),
    LanguageSubtag::new(['m', 'q', 'l']),
    LanguageSubtag::new(['m', 'q', 'm']),
    LanguageSubtag::new(['m', 'q', 'n']),
    LanguageSubtag::new(['m', 'q', 'o']),
    LanguageSubtag::new(['m', 'q', 'p']),
    LanguageSubtag::new(['m', 'q', 'q']),
    LanguageSubtag::new(['m', 'q', 'r']),
    LanguageSubtag::new(['m', 'q', 's']),
    LanguageSubtag::new(['m', 'q', 't']),
    LanguageSubtag::new(['m', 'q', 'u']),
    LanguageSubtag::new(['m', 'q', 'v']),
    LanguageSubtag::new(['m', 'q', 'w']),
    LanguageSubtag::new(['m', 'q', 'x']),
    LanguageSubtag::new(['m', 'q', 'y']),
    LanguageSubtag::new(['m', 'q', 'z']),
    LanguageSubtag::new(['m', 'r', ' ']),
    LanguageSubtag::new(['m', 'r', 'a']),
    LanguageSubtag::new(['m', 'r', 'b']),
    LanguageSubtag::new(['m', 'r', 'c']),
    LanguageSubtag::new(['m', 'r', 'd']),
    LanguageSubtag::new(['m', 'r', 'e']),
    LanguageSubtag::new(['m', 'r', 'f']),
    LanguageSubtag::new(['m', 'r', 'g']),
    LanguageSubtag::new(['m', 'r', 'h']),
    LanguageSubtag::new(['m', 'r', 'j']),
    LanguageSubtag::new(['m', 'r', 'k']),
    LanguageSubtag::new(['m', 'r', 'l']),
    LanguageSubtag::new(['m', 'r', 'm']),
    LanguageSubtag::new(['m', 'r', 'n']),
    LanguageSubtag::new(['m', 'r', 'o']),
    LanguageSubtag::new(['m', 'r', 'p']),
    LanguageSubtag::new(['m', 'r', 'q']),
    LanguageSubtag::new(['m', 'r', 'r']),
    LanguageSubtag::new(['m', 'r', 's']),
    LanguageSubtag::new(['m', 'r', 't']),
    LanguageSubtag::new(['m', 'r', 'u']),
    LanguageSubtag::new(['m', 'r', 'v']),
    LanguageSubtag::new(['m', 'r', 'w']),
    LanguageSubtag::new(['m', 'r', 'x']),
    LanguageSubtag::new(['m', 'r', 'y']),
    LanguageSubtag::new(['m', 'r', 'z']),
    LanguageSubtag::new(['m', 's', ' ']),
    LanguageSubtag::new(['m', 's', 'b']),
    LanguageSubtag::new(['m', 's', 'c']),
    LanguageSubtag::new(['m', 's', 'd']),
    LanguageSubtag::new(['m', 's', 'e']),
    LanguageSubtag::new(['m', 's', 'f']),
    LanguageSubtag::new(['m', 's', 'g']),
    LanguageSubtag::new(['m', 's', 'h']),
    LanguageSubtag::new(['m', 's', 'i']),
    LanguageSubtag::new(['m', 's', 'j']),
    LanguageSubtag::new(['m', 's', 'k']),
    LanguageSubtag::new(['m', 's', 'l']),
    LanguageSubtag::new(['m', 's', 'm']),
    LanguageSubtag::new(['m', 's', 'n']),
    LanguageSubtag::new(['m', 's', 'o']),
    LanguageSubtag::new(['m', 's', 'p']),
    LanguageSubtag::new(['m', 's', 'q']),
    LanguageSubtag::new(['m', 's', 'r']),
    LanguageSubtag::new(['m', 's', 's']),
    LanguageSubtag::new(['m', 's', 't']),
    LanguageSubtag::new(['m', 's', 'u']),
    LanguageSubtag::new(['m', 's', 'v']),
    LanguageSubtag::new(['m', 's', 'w']),
    LanguageSubtag::new(['m', 's', 'x']),
    LanguageSubtag::new(['m', 's', 'y']),
    LanguageSubtag::new(['m', 's', 'z']),
    LanguageSubtag::new(['m', 't', ' ']),
    LanguageSubtag::new(['m', 't', 'a']),
    LanguageSubtag::new(['m', 't', 'b']),
    LanguageSubtag::new(['m', 't', 'c']),
    LanguageSubtag::new(['m', 't', 'd']),
    LanguageSubtag::new(['m', 't', 'e']),
    LanguageSubtag::new(['m', 't', 'f']),
    LanguageSubtag::new(['m', 't', 'g']),
    LanguageSubtag::new(['m', 't', 'h']),
    LanguageSubtag::new(['m', 't', 'i']),
    LanguageSubtag::new(['m', 't', 'j']),
    LanguageSubtag::new(['m', 't', 'k']),
    LanguageSubtag::new(['m', 't', 'l']),
    LanguageSubtag::new(['m', 't', 'm']),
    LanguageSubtag::new(['m', 't', 'n']),
    LanguageSubtag::new(['m', 't', 'o']),
    LanguageSubtag::new(['m', 't', 'p']),
    LanguageSubtag::new(['m', 't', 'q']),
    LanguageSubtag::new(['m', 't', 'r']),
    LanguageSubtag::new(['m', 't', 's']),
    LanguageSubtag::new(['m', 't', 't']),
    LanguageSubtag::new(['m', 't', 'u']),
    LanguageSubtag::new(['m', 't', 'v']),
    LanguageSubtag::new(['m', 't', 'w']),
    LanguageSubtag::new(['m', 't', 'x']),
    LanguageSubtag::new(['m', 't', 'y']),
    LanguageSubtag::new(['m', 'u', 'a']),
    LanguageSubtag::new(['m', 'u', 'b']),
    LanguageSubtag::new(['m', 'u', 'c']),
    LanguageSubtag::new(['m', 'u', 'd']),
    LanguageSubtag::new(['m', 'u', 'e']),
    LanguageSubtag::new(['m', 'u', 'g']),
    LanguageSubtag::new(['m', 'u', 'h']),
    LanguageSubtag::new(['m', 'u', 'i']),
    LanguageSubtag::new(['m', 'u', 'j']),
    LanguageSubtag::new(['m', 'u', 'k']),
    LanguageSubtag::new(['m', 'u', 'l']),
    LanguageSubtag::new(['m', 'u', 'm']),
    LanguageSubtag::new(['m', 'u', 'n']),
    LanguageSubtag::new(['m', 'u', 'o']),
    LanguageSubtag::new(['m', 'u', 'p']),
    LanguageSubtag::new(['m', 'u', 'q']),
    LanguageSubtag::new(['m', 'u', 'r']),
    LanguageSubtag::new(['m', 'u', 's']),
    LanguageSubtag::new(['m', 'u', 't']),
    LanguageSubtag::new(['m', 'u', 'u']),
    LanguageSubtag::new(['m', 'u', 'v']),
    LanguageSubtag::new(['m', 'u', 'x']),
    LanguageSubtag::new(['m', 'u', 'y']),
    LanguageSubtag::new(['m', 'u', 'z']),
    LanguageSubtag::new(['m', 'v', 'a']),
    LanguageSubtag::new(['m', 'v', 'b']),
    LanguageSubtag::new(['m', 'v', 'd']),
    LanguageSubtag::new(['m', 'v', 'e']),
    LanguageSubtag::new(['m', 'v', 'f']),
    LanguageSubtag::new(['m', 'v', 'g']),
    LanguageSubtag::new(['m', 'v', 'h']),
    LanguageSubtag::new(['m', 'v', 'i']),
    LanguageSubtag::new(['m', 'v', 'k']),
    LanguageSubtag::new(['m', 'v', 'l']),
    LanguageSubtag::new(['m', 'v', 'm']),
    LanguageSubtag::new(['m', 'v', 'n']),
    LanguageSubtag::new(['m', 'v', 'o']),
    LanguageSubtag::new(['m', 'v', 'p']),
    LanguageSubtag::new(['m', 'v', 'q']),
    LanguageSubtag::new(['m', 'v', 'r']),
    LanguageSubtag::new(['m', 'v', 's']),
    LanguageSubtag::new(['m', 'v', 't']),
    LanguageSubtag::new(['m', 'v', 'u']),
    LanguageSubtag::new(['m', 'v', 'v']),
    LanguageSubtag::new(['m', 'v', 'w']),
    LanguageSubtag::new(['m', 'v', 'x']),
    LanguageSubtag::new(['m', 'v', 'y']),
    LanguageSubtag::new(['m', 'v', 'z']),
    LanguageSubtag::new(['m', 'w', 'a']),
    LanguageSubtag::new(['m', 'w', 'b']),
    LanguageSubtag::new(['m', 'w', 'c']),
    LanguageSubtag::new(['m', 'w', 'd']),
    LanguageSubtag::new(['m', 'w', 'e']),
    LanguageSubtag::new(['m', 'w', 'f']),
    LanguageSubtag::new(['m', 'w', 'g']),
    LanguageSubtag::new(['m', 'w', 'h']),
    LanguageSubtag::new(['m', 'w', 'i']),
    LanguageSubtag::new(['m', 'w', 'j']),
    LanguageSubtag::new(['m', 'w', 'k']),
    LanguageSubtag::new(['m', 'w', 'l']),
    LanguageSubtag::new(['m', 'w', 'm']),
    LanguageSubtag::new(['m', 'w', 'n']),
    LanguageSubtag::new(['m', 'w', 'o']),
    LanguageSubtag::new(['m', 'w', 'p']),
    LanguageSubtag::new(['m', 'w', 'q']),
    LanguageSubtag::new(['m', 'w', 'r']),
    LanguageSubtag::new(['m', 'w', 's']),
    LanguageSubtag::new(['m', 'w', 't']),
    LanguageSubtag::new(['m', 'w', 'u']),
    LanguageSubtag::new(['m', 'w', 'v']),
    LanguageSubtag::new(['m', 'w', 'w']),
    LanguageSubtag::new(['m', 'w', 'x']),
    LanguageSubtag::new(['m', 'w', 'y']),
    LanguageSubtag::new(['m', 'w', 'z']),
    LanguageSubtag::new(['m', 'x', 'a']),
    LanguageSubtag::new(['m', 'x', 'b']),
    LanguageSubtag::new(['m', 'x', 'c']),
    LanguageSubtag::new(['m', 'x', 'd']),
    LanguageSubtag::new(['m', 'x', 'e']),
    LanguageSubtag::new(['m', 'x', 'f']),
    LanguageSubtag::new(['m', 'x', 'g']),
    LanguageSubtag::new(['m', 'x', 'h']),
    LanguageSubtag::new(['m', 'x', 'i']),
    LanguageSubtag::new(['m', 'x', 'j']),
    LanguageSubtag::new(['m', 'x', 'k']),
    LanguageSubtag::new(['m', 'x', 'l']),
    LanguageSubtag::new(['m', 'x', 'm']),
    LanguageSubtag::new(['m', 'x', 'n']),
    LanguageSubtag::new(['m', 'x', 'o']),
    LanguageSubtag::new(['m', 'x', 'p']),
    LanguageSubtag::new(['m', 'x', 'q']),
    LanguageSubtag::new(['m', 'x', 'r']),
    LanguageSubtag::new(['m', 'x', 's']),
    LanguageSubtag::new(['m', 'x', 't']),
    LanguageSubtag::new(['m', 'x', 'u']),
    LanguageSubtag::new(['m', 'x', 'v']),
    LanguageSubtag::new(['m', 'x', 'w']),
    LanguageSubtag::new(['m', 'x', 'x']),
    LanguageSubtag::new(['m', 'x', 'y']),
    LanguageSubtag::new(['m', 'x', 'z']),
    LanguageSubtag::new(['m', 'y', ' ']),
    LanguageSubtag::new(['m', 'y', 'b']),
    LanguageSubtag::new(['m', 'y', 'c']),
    LanguageSubtag::new(['m', 'y', 'd']),
    LanguageSubtag::new(['m', 'y', 'e']),
    LanguageSubtag::new(['m', 'y', 'f']),
    LanguageSubtag::new(['m', 'y', 'g']),
    LanguageSubtag::new(['m', 'y', 'h']),
    LanguageSubtag::new(['m', 'y', 'i']),
    LanguageSubtag::new(['m', 'y', 'j']),
    LanguageSubtag::new(['m', 'y', 'k']),
    LanguageSubtag::new(['m', 'y', 'l']),
    LanguageSubtag::new(['m', 'y', 'm']),
    LanguageSubtag::new(['m', 'y', 'n']),
    LanguageSubtag::new(['m', 'y', 'o']),
    LanguageSubtag::new(['m', 'y', 'p']),
    LanguageSubtag::new(['m', 'y', 'q']),
    LanguageSubtag::new(['m', 'y', 'r']),
    LanguageSubtag::new(['m', 'y', 's']),
    LanguageSubtag::new(['m', 'y', 't']),
    LanguageSubtag::new(['m', 'y', 'u']),
    LanguageSubtag::new(['m', 'y', 'v']),
    LanguageSubtag::new(['m', 'y', 'w']),
    LanguageSubtag::new(['m', 'y', 'x']),
    LanguageSubtag::new(['m', 'y', 'y']),
    LanguageSubtag::new(['m', 'y', 'z']),
    LanguageSubtag::new(['m', 'z', 'a']),
    LanguageSubtag::new(['m', 'z', 'b']),
    LanguageSubtag::new(['m', 'z', 'c']),
    LanguageSubtag::new(['m', 'z', 'd']),
    LanguageSubtag::new(['m', 'z', 'e']),
    LanguageSubtag::new(['m', 'z', 'g']),
    LanguageSubtag::new(['m', 'z', 'h']),
    LanguageSubtag::new(['m', 'z', 'i']),
    LanguageSubtag::new(['m', 'z', 'j']),
    LanguageSubtag::new(['m', 'z', 'k']),
    LanguageSubtag::new(['m', 'z', 'l']),
    LanguageSubtag::new(['m', 'z', 'm']),
    LanguageSubtag::new(['m', 'z', 'n']),
    LanguageSubtag::new(['m', 'z', 'o']),
    LanguageSubtag::new(['m', 'z', 'p']),
    LanguageSubtag::new(['m', 'z', 'q']),
    LanguageSubtag::new(['m', 'z', 'r']),
    LanguageSubtag::new(['m', 'z', 's']),
    LanguageSubtag::new(['m', 'z', 't']),
    LanguageSubtag::new(['m', 'z', 'u']),
    LanguageSubtag::new(['m', 'z', 'v']),
    LanguageSubtag::new(['m', 'z', 'w']),
    LanguageSubtag::new(['m', 'z', 'x']),
    LanguageSubtag::new(['m', 'z', 'y']),
    LanguageSubtag::new(['m', 'z', 'z']),
    LanguageSubtag::new(['n', 'a', ' ']),
    LanguageSubtag::new(['n', 'a', 'a']),
    LanguageSubtag::new(['n', 'a', 'b']),
    LanguageSubtag::new(['n', 'a', 'c']),
    LanguageSubtag::new(['n', 'a', 'd']),
    LanguageSubtag::new(['n', 'a', 'e']),
    LanguageSubtag::new(['n', 'a', 'f']),
    LanguageSubtag::new(['n', 'a', 'g']),
    LanguageSubtag::new(['n', 'a', 'h']),
    LanguageSubtag::new(['n', 'a', 'i']),
    LanguageSubtag::new(['n', 'a', 'j']),
    LanguageSubtag::new(['n', 'a', 'k']),
    LanguageSubtag::new(['n', 'a', 'l']),
    LanguageSubtag::new(['n', 'a', 'm']),
    LanguageSubtag::new(['n', 'a', 'n']),
    LanguageSubtag::new(['n', 'a', 'o']),
    LanguageSubtag::new(['n', 'a', 'p']),
    LanguageSubtag::new(['n', 'a', 'q']),
    LanguageSubtag::new(['n', 'a', 'r']),
    LanguageSubtag::new(['n', 'a', 's']),
    LanguageSubtag::new(['n', 'a', 't']),
    LanguageSubtag::new(['n', 'a', 'w']),
    LanguageSubtag::new(['n', 'a', 'x']),
    LanguageSubtag::new(['n', 'a', 'y']),
    LanguageSubtag::new(['n', 'a', 'z']),
    LanguageSubtag::new(['n', 'b', ' ']),
    LanguageSubtag::new(['n', 'b', 'a']),
    LanguageSubtag::new(['n', 'b', 'b']),
    LanguageSubtag::new(['n', 'b', 'c']),
    LanguageSubtag::new(['n', 'b', 'd']),
    LanguageSubtag::new(['n', 'b', 'e']),
    LanguageSubtag::new(['n', 'b', 'f']),
    LanguageSubtag::new(['n', 'b', 'g']),
    LanguageSubtag::new(['n', 'b', 'h']),
    LanguageSubtag::new(['n', 'b', 'i']),
    LanguageSubtag::new(['n', 'b', 'j']),
    LanguageSubtag::new(['n', 'b', 'k']),
    LanguageSubtag::new(['n', 'b', 'm']),
    LanguageSubtag::new(['n', 'b', 'n']),
    LanguageSubtag::new(['n', 'b', 'o']),
    LanguageSubtag::new(['n', 'b', 'p']),
    LanguageSubtag::new(['n', 'b', 'q']),
    LanguageSubtag::new(['n', 'b', 'r']),
    LanguageSubtag::new(['n', 'b', 's']),
    LanguageSubtag::new(['n', 'b', 't']),
    LanguageSubtag::new(['n', 'b', 'u']),
    LanguageSubtag::new(['n', 'b', 'v']),
    LanguageSubtag::new(['n', 'b', 'w']),
    LanguageSubtag::new(['n', 'b', 'x']),
    LanguageSubtag::new(['n', 'b', 'y']),
    LanguageSubtag::new(['n', 'c', 'a']),
    LanguageSubtag::new(['n', 'c', 'b']),
    LanguageSubtag::new(['n', 'c', 'c']),
    LanguageSubtag::new(['n', 'c', 'd']),
    LanguageSubtag::new(['n', 'c', 'e']),
    LanguageSubtag::new(['n', 'c', 'f']),
    LanguageSubtag::new(['n', 'c', 'g']),
    LanguageSubtag::new(['n', 'c', 'h']),
    LanguageSubtag::new(['n', 'c', 'i']),
    LanguageSubtag::new(['n', 'c', 'j']),
    LanguageSubtag::new(['n', 'c', 'k']),
    LanguageSubtag::new(['n', 'c', 'l']),
    LanguageSubtag::new(['n', 'c', 'm']),
    LanguageSubtag::new(['n', 'c', 'n']),
    LanguageSubtag::new(['n', 'c', 'o']),
    LanguageSubtag::new(['n', 'c', 'p']),
    LanguageSubtag::new(['n', 'c', 'q']),
    LanguageSubtag::new(['n', 'c', 'r']),
    LanguageSubtag::new(['n', 'c', 's']),
    LanguageSubtag::new(['n', 'c', 't']),
    LanguageSubtag::new(['n', 'c', 'u']),
    LanguageSubtag::new(['n', 'c', 'x']),
    LanguageSubtag::new(['n', 'c', 'z']),
    LanguageSubtag::new(['n', 'd', ' ']),
    LanguageSubtag::new(['n', 'd', 'a']),
    LanguageSubtag::new(['n', 'd', 'b']),
    LanguageSubtag::new(['n', 'd', 'c']),
    LanguageSubtag::new(['n', 'd', 'd']),
    LanguageSubtag::new(['n', 'd', 'f']),
    LanguageSubtag::new(['n', 'd', 'g']),
    LanguageSubtag::new(['n', 'd', 'h']),
    LanguageSubtag::new(['n', 'd', 'i']),
    LanguageSubtag::new(['n', 'd', 'j']),
    LanguageSubtag::new(['n', 'd', 'k']),
    LanguageSubtag::new(['n', 'd', 'l']),
    LanguageSubtag::new(['n', 'd', 'm']),
    LanguageSubtag::new(['n', 'd', 'n']),
    LanguageSubtag::new(['n', 'd', 'p']),
    LanguageSubtag::new(['n', 'd', 'q']),
    LanguageSubtag::new(['n', 'd', 'r']),
    LanguageSubtag::new(['n', 'd', 's']),
    LanguageSubtag::new(['n', 'd', 't']),
    LanguageSubtag::new(['n', 'd', 'u']),
    LanguageSubtag::new(['n', 'd', 'v']),
    LanguageSubtag::new(['n', 'd', 'w']),
    LanguageSubtag::new(['n', 'd', 'x']),
    LanguageSubtag::new(['n', 'd', 'y']),
    LanguageSubtag::new(['n', 'd', 'z']),
    LanguageSubtag::new(['n', 'e', ' ']),
    LanguageSubtag::new(['n', 'e', 'a']),
    LanguageSubtag::new(['n', 'e', 'b']),
    LanguageSubtag::new(['n', 'e', 'c']),
    LanguageSubtag::new(['n', 'e', 'd']),
    LanguageSubtag::new(['n', 'e', 'e']),
    LanguageSubtag::new(['n', 'e', 'f']),
    LanguageSubtag::new(['n', 'e', 'g']),
    LanguageSubtag::new(['n', 'e', 'h']),
    LanguageSubtag::new(['n', 'e', 'i']),
    LanguageSubtag::new(['n', 'e', 'j']),
    LanguageSubtag::new(['n', 'e', 'k']),
    LanguageSubtag::new(['n', 'e', 'm']),
    LanguageSubtag::new(['n', 'e', 'n']),
    LanguageSubtag::new(['n', 'e', 'o']),
    LanguageSubtag::new(['n', 'e', 'q']),
    LanguageSubtag::new(['n', 'e', 'r']),
    LanguageSubtag::new(['n', 'e', 's']),
    LanguageSubtag::new(['n', 'e', 't']),
    LanguageSubtag::new(['n', 'e', 'u']),
    LanguageSubtag::new(['n', 'e', 'v']),
    LanguageSubtag::new(['n', 'e', 'w']),
    LanguageSubtag::new(['n', 'e', 'x']),
    LanguageSubtag::new(['n', 'e', 'y']),
    LanguageSubtag::new(['n', 'e', 'z']),
    LanguageSubtag::new(['n', 'f', 'a']),
    LanguageSubtag::new(['n', 'f', 'd']),
    LanguageSubtag::new(['n', 'f', 'l']),
    LanguageSubtag::new(['n', 'f', 'r']),
    LanguageSubtag::new(['n', 'f', 'u']),
    LanguageSubtag::new(['n', 'g', ' ']),
    LanguageSubtag::new(['n', 'g', 'a']),
    LanguageSubtag::new(['n', 'g', 'b']),
    LanguageSubtag::new(['n', 'g', 'c']),
    LanguageSubtag::new(['n', 'g', 'd']),
    LanguageSubtag::new(['n', 'g', 'e']),
    LanguageSubtag::new(['n', 'g', 'f']),
    LanguageSubtag::new(['n', 'g', 'g']),
    LanguageSubtag::new(['n', 'g', 'h']),
    LanguageSubtag::new(['n', 'g', 'i']),
    LanguageSubtag::new(['n', 'g', 'j']),
    LanguageSubtag::new(['n', 'g', 'k']),
    LanguageSubtag::new(['n', 'g', 'l']),
    LanguageSubtag::new(['n', 'g', 'm']),
    LanguageSubtag::new(['n', 'g', 'n']),
    LanguageSubtag::new(['n', 'g', 'o']),
    LanguageSubtag::new(['n', 'g', 'p']),
    LanguageSubtag::new(['n', 'g', 'q']),
    LanguageSubtag::new(['n', 'g', 'r']),
    LanguageSubtag::new(['n', 'g', 's']),
    LanguageSubtag::new(['n', 'g', 't']),
    LanguageSubtag::new(['n', 'g', 'u']),
    LanguageSubtag::new(['n', 'g', 'v']),
    LanguageSubtag::new(['n', 'g', 'w']),
    LanguageSubtag::new(['n', 'g', 'x']),
    LanguageSubtag::new(['n', 'g', 'y']),
    LanguageSubtag::new(['n', 'g', 'z']),
    LanguageSubtag::new(['n', 'h', 'a']),
    LanguageSubtag::new(['n', 'h', 'b']),
    LanguageSubtag::new(['n', 'h', 'c']),
    LanguageSubtag::new(['n', 'h', 'd']),
    LanguageSubtag::new(['n', 'h', 'e']),
    LanguageSubtag::new(['n', 'h', 'f']),
    LanguageSubtag::new(['n', 'h', 'g']),
    LanguageSubtag::new(['n', 'h', 'h']),
    LanguageSubtag::new(['n', 'h', 'i']),
    LanguageSubtag::new(['n', 'h', 'k']),
    LanguageSubtag::new(['n', 'h', 'm']),
    LanguageSubtag::new(['n', 'h', 'n']),
    LanguageSubtag::new(['n', 'h', 'o']),
    LanguageSubtag::new(['n', 'h', 'p']),
    LanguageSubtag::new(['n', 'h', 'q']),
    LanguageSubtag::new(['n', 'h', 'r']),
    LanguageSubtag::new(['n', 'h', 't']),
    LanguageSubtag::new(['n', 'h', 'u']),
    LanguageSubtag::new(['n', 'h', 'v']),
    LanguageSubtag::new(['n', 'h', 'w']),
    LanguageSubtag::new(['n', 'h', 'x']),
    LanguageSubtag::new(['n', 'h', 'y']),
    LanguageSubtag::new(['n', 'h', 'z']),
    LanguageSubtag::new(['n', 'i', 'a']),
    LanguageSubtag::new(['n', 'i', 'b']),
    LanguageSubtag::new(['n', 'i', 'c']),
    LanguageSubtag::new(['n', 'i', 'd']),
    LanguageSubtag::new(['n', 'i', 'e']),
    LanguageSubtag::new(['n', 'i', 'f']),
    LanguageSubtag::new(['n', 'i', 'g']),
    LanguageSubtag::new(['n', 'i', 'h']),
    LanguageSubtag::new(['n', 'i', 'i']),
    LanguageSubtag::new(['n', 'i', 'j']),
    LanguageSubtag::new(['n', 'i', 'k']),
    LanguageSubtag::new(['n', 'i', 'l']),
    LanguageSubtag::new(['n', 'i', 'm']),
    LanguageSubtag::new(['n', 'i', 'n']),
    LanguageSubtag::new(['n', 'i', 'o']),
    LanguageSubtag::new(['n', 'i', 'q']),
    LanguageSubtag::new(['n', 'i', 'r']),
    LanguageSubtag::new(['n', 'i', 's']),
    LanguageSubtag::new(['n', 'i', 't']),
    LanguageSubtag::new(['n', 'i', 'u']),
    LanguageSubtag::new(['n', 'i', 'v']),
    LanguageSubtag::new(['n', 'i', 'w']),
    LanguageSubtag::new(['n', 'i', 'x']),
    LanguageSubtag::new(['n', 'i', 'y']),
    LanguageSubtag::new(['n', 'i', 'z']),
    LanguageSubtag::new(['n', 'j', 'a']),
    LanguageSubtag::new(['n', 'j', 'b']),
    LanguageSubtag::new(['n', 'j', 'd']),
    LanguageSubtag::new(['n', 'j', 'h']),
    LanguageSubtag::new(['n', 'j', 'i']),
    LanguageSubtag::new(['n', 'j', 'j']),
    LanguageSubtag::new(['n', 'j', 'l']),
    LanguageSubtag::new(['n', 'j', 'm']),
    LanguageSubtag::new(['n', 'j', 'n']),
    LanguageSubtag::new(['n', 'j', 'o']),
    LanguageSubtag::new(['n', 'j', 'r']),
    LanguageSubtag::new(['n', 'j', 's']),
    LanguageSubtag::new(['n', 'j', 't']),
    LanguageSubtag::new(['n', 'j', 'u']),
    LanguageSubtag::new(['n', 'j', 'x']),
    LanguageSubtag::new(['n', 'j', 'y']),
    LanguageSubtag::new(['n', 'j', 'z']),
    LanguageSubtag::new(['n', 'k', 'a']),
    LanguageSubtag::new(['n', 'k', 'b']),
    LanguageSubtag::new(['n', 'k', 'c']),
    LanguageSubtag::new(['n', 'k', 'd']),
    LanguageSubtag::new(['n', 'k', 'e']),
    LanguageSubtag::new(['n', 'k', 'f']),
    LanguageSubtag::new(['n', 'k', 'g']),
    LanguageSubtag::new(['n', 'k', 'h']),
    LanguageSubtag::new(['n', 'k', 'i']),
    LanguageSubtag::new(['n', 'k', 'j']),
    LanguageSubtag::new(['n', 'k', 'k']),
    LanguageSubtag::new(['n', 'k', 'm']),
    LanguageSubtag::new(['n', 'k', 'n']),
    LanguageSubtag::new(['n', 'k', 'o']),
    LanguageSubtag::new(['n', 'k', 'p']),
    LanguageSubtag::new(['n', 'k', 'q']),
    LanguageSubtag::new(['n', 'k', 'r']),
    LanguageSubtag::new(['n', 'k', 's']),
    LanguageSubtag::new(['n', 'k', 't']),
    LanguageSubtag::new(['n', 'k', 'u']),
    LanguageSubtag::new(['n', 'k', 'v']),
    LanguageSubtag::new(['n', 'k', 'w']),
    LanguageSubtag::new(['n', 'k', 'x']),
    LanguageSubtag::new(['n', 'k', 'z']),
    LanguageSubtag::new(['n', 'l', ' ']),
    LanguageSubtag::new(['n', 'l', 'a']),
    LanguageSubtag::new(['n', 'l', 'c']),
    LanguageSubtag::new(['n', 'l', 'e']),
    LanguageSubtag::new(['n', 'l', 'g']),
    LanguageSubtag::new(['n', 'l', 'i']),
    LanguageSubtag::new(['n', 'l', 'j']),
    LanguageSubtag::new(['n', 'l', 'k']),
    LanguageSubtag::new(['n', 'l', 'l']),
    LanguageSubtag::new(['n', 'l', 'm']),
    LanguageSubtag::new(['n', 'l', 'n']),
    LanguageSubtag::new(['n', 'l', 'o']),
    LanguageSubtag::new(['n', 'l', 'q']),
    LanguageSubtag::new(['n', 'l', 'r']),
    LanguageSubtag::new(['n', 'l', 'u']),
    LanguageSubtag::new(['n', 'l', 'v']),
    LanguageSubtag::new(['n', 'l', 'w']),
    LanguageSubtag::new(['n', 'l', 'x']),
    LanguageSubtag::new(['n', 'l', 'y']),
    LanguageSubtag::new(['n', 'l', 'z']),
    LanguageSubtag::new(['n', 'm', 'a']),
    LanguageSubtag::new(['n', 'm', 'b']),
    LanguageSubtag::new(['n', 'm', 'c']),
    LanguageSubtag::new(['n', 'm', 'd']),
    LanguageSubtag::new(['n', 'm', 'e']),
    LanguageSubtag::new(['n', 'm', 'f']),
    LanguageSubtag::new(['n', 'm', 'g']),
    LanguageSubtag::new(['n', 'm', 'h']),
    LanguageSubtag::new(['n', 'm', 'i']),
    LanguageSubtag::new(['n', 'm', 'j']),
    LanguageSubtag::new(['n', 'm', 'k']),
    LanguageSubtag::new(['n', 'm', 'l']),
    LanguageSubtag::new(['n', 'm', 'm']),
    LanguageSubtag::new(['n', 'm', 'n']),
    LanguageSubtag::new(['n', 'm', 'o']),
    LanguageSubtag::new(['n', 'm', 'p']),
    LanguageSubtag::new(['n', 'm', 'q']),
    LanguageSubtag::new(['n', 'm', 'r']),
    LanguageSubtag::new(['n', 'm', 's']),
    LanguageSubtag::new(['n', 'm', 't']),
    LanguageSubtag::new(['n', 'm', 'u']),
    LanguageSubtag::new(['n', 'm', 'v']),
    LanguageSubtag::new(['n', 'm', 'w']),
    LanguageSubtag::new(['n', 'm', 'x']),
    LanguageSubtag::new(['n', 'm', 'y']),
    LanguageSubtag::new(['n', 'm', 'z']),
    LanguageSubtag::new(['n', 'n', ' ']),
    LanguageSubtag::new(['n', 'n', 'a']),
    LanguageSubtag::new(['n', 'n', 'b']),
    LanguageSubtag::new(['n', 'n', 'c']),
    LanguageSubtag::new(['n', 'n', 'd']),
    LanguageSubtag::new(['n', 'n', 'e']),
    LanguageSubtag::new(['n', 'n', 'f']),
    LanguageSubtag::new(['n', 'n', 'g']),
    LanguageSubtag::new(['n', 'n', 'h']),
    LanguageSubtag::new(['n', 'n', 'i']),
    LanguageSubtag::new(['n', 'n', 'j']),
    LanguageSubtag::new(['n', 'n', 'k']),
    LanguageSubtag::new(['n', 'n', 'l']),
    LanguageSubtag::new(['n', 'n', 'm']),
    LanguageSubtag::new(['n', 'n', 'n']),
    LanguageSubtag::new(['n', 'n', 'p']),
    LanguageSubtag::new(['n', 'n', 'q']),
    LanguageSubtag::new(['n', 'n', 'r']),
    LanguageSubtag::new(['n', 'n', 's']),
    LanguageSubtag::new(['n', 'n', 't']),
    LanguageSubtag::new(['n', 'n', 'u']),
    LanguageSubtag::new(['n', 'n', 'v']),
    LanguageSubtag::new(['n', 'n', 'w']),
    LanguageSubtag::new(['n', 'n', 'x']),
    LanguageSubtag::new(['n', 'n', 'y']),
    LanguageSubtag::new(['n', 'n', 'z']),
    LanguageSubtag::new(['n', 'o', ' ']),
    LanguageSubtag::new(['n', 'o', 'a']),
    LanguageSubtag::new(['n', 'o', 'c']),
    LanguageSubtag::new(['n', 'o', 'd']),
    LanguageSubtag::new(['n', 'o', 'e']),
    LanguageSubtag::new(['n', 'o', 'f']),
    LanguageSubtag::new(['n', 'o', 'g']),
    LanguageSubtag::new(['n', 'o', 'h']),
    LanguageSubtag::new(['n', 'o', 'i']),
    LanguageSubtag::new(['n', 'o', 'j']),
    LanguageSubtag::new(['n', 'o', 'k']),
    LanguageSubtag::new(['n', 'o', 'l']),
    LanguageSubtag::new(['n', 'o', 'm']),
    LanguageSubtag::new(['n', 'o', 'n']),
    LanguageSubtag::new(['n', 'o', 'o']),
    LanguageSubtag::new(['n', 'o', 'p']),
    LanguageSubtag::new(['n', 'o', 'q']),
    LanguageSubtag::new(['n', 'o', 's']),
    LanguageSubtag::new(['n', 'o', 't']),
    LanguageSubtag::new(['n', 'o', 'u']),
    LanguageSubtag::new(['n', 'o', 'v']),
    LanguageSubtag::new(['n', 'o', 'w']),
    LanguageSubtag::new(['n', 'o', 'y']),
    LanguageSubtag::new(['n', 'o', 'z']),
    LanguageSubtag::new(['n', 'p', 'a']),
    LanguageSubtag::new(['n', 'p', 'b']),
    LanguageSubtag::new(['n', 'p', 'g']),
    LanguageSubtag::new(['n', 'p', 'h']),
    LanguageSubtag::new(['n', 'p', 'i']),
    LanguageSubtag::new(['n', 'p', 'l']),
    LanguageSubtag::new(['n', 'p', 'n']),
    LanguageSubtag::new(['n', 'p', 'o']),
    LanguageSubtag::new(['n', 'p', 's']),
    LanguageSubtag::new(['n', 'p', 'u']),
    LanguageSubtag::new(['n', 'p', 'x']),
    LanguageSubtag::new(['n', 'p', 'y']),
    LanguageSubtag::new(['n', 'q', 'g']),
    LanguageSubtag::new(['n', 'q', 'k']),
    LanguageSubtag::new(['n', 'q', 'l']),
    LanguageSubtag::new(['n', 'q', 'm']),
    LanguageSubtag::new(['n', 'q', 'n']),
    LanguageSubtag::new(['n', 'q', 'o']),
    LanguageSubtag::new(['n', 'q', 'q']),
    LanguageSubtag::new(['n', 'q', 't']),
    LanguageSubtag::new(['n', 'q', 'y']),
    LanguageSubtag::new(['n', 'r', ' ']),
    LanguageSubtag::new(['n', 'r', 'a']),
    LanguageSubtag::new(['n', 'r', 'b']),
    LanguageSubtag::new(['n', 'r', 'c']),
    LanguageSubtag::new(['n', 'r', 'e']),
    LanguageSubtag::new(['n', 'r', 'f']),
    LanguageSubtag::new(['n', 'r', 'g']),
    LanguageSubtag::new(['n', 'r', 'i']),
    LanguageSubtag::new(['n', 'r', 'k']),
    LanguageSubtag::new(['n', 'r', 'l']),
    LanguageSubtag::new(['n', 'r', 'm']),
    LanguageSubtag::new(['n', 'r', 'n']),
    LanguageSubtag::new(['n', 'r', 'p']),
    LanguageSubtag::new(['n', 'r', 'r']),
    LanguageSubtag::new(['n', 'r', 't']),
    LanguageSubtag::new(['n', 'r', 'u']),
    LanguageSubtag::new(['n', 'r', 'x']),
    LanguageSubtag::new(['n', 'r', 'z']),
    LanguageSubtag::new(['n', 's', 'a']),
    LanguageSubtag::new(['n', 's', 'b']),
    LanguageSubtag::new(['n', 's', 'c']),
    LanguageSubtag::new(['n', 's', 'd']),
    LanguageSubtag::new(['n', 's', 'e']),
    LanguageSubtag::new(['n', 's', 'f']),
    LanguageSubtag::new(['n', 's', 'g']),
    LanguageSubtag::new(['n', 's', 'h']),
    LanguageSubtag::new(['n', 's', 'i']),
    LanguageSubtag::new(['n', 's', 'k']),
    LanguageSubtag::new(['n', 's', 'l']),
    LanguageSubtag::new(['n', 's', 'm']),
    LanguageSubtag::new(['n', 's', 'n']),
    LanguageSubtag::new(['n', 's', 'o']),
    LanguageSubtag::new(['n', 's', 'p']),
    LanguageSubtag::new(['n', 's', 'q']),
    LanguageSubtag::new(['n', 's', 'r']),
    LanguageSubtag::new(['n', 's', 's']),
    LanguageSubtag::new(['n', 's', 't']),
    LanguageSubtag::new(['n', 's', 'u']),
    LanguageSubtag::new(['n', 's', 'v']),
    LanguageSubtag::new(['n', 's', 'w']),
    LanguageSubtag::new(['n', 's', 'x']),
    LanguageSubtag::new(['n', 's', 'y']),
    LanguageSubtag::new(['n', 's', 'z']),
    LanguageSubtag::new(['n', 't', 'd']),
    LanguageSubtag::new(['n', 't', 'e']),
    LanguageSubtag::new(['n', 't', 'g']),
    LanguageSubtag::new(['n', 't', 'i']),
    LanguageSubtag::new(['n', 't', 'j']),
    LanguageSubtag::new(['n', 't', 'k']),
    LanguageSubtag::new(['n', 't', 'm']),
    LanguageSubtag::new(['n', 't', 'o']),
    LanguageSubtag::new(['n', 't', 'p']),
    LanguageSubtag::new(['n', 't', 'r']),
    LanguageSubtag::new(['n', 't', 's']),
    LanguageSubtag::new(['n', 't', 'u']),
    LanguageSubtag::new(['n', 't', 'w']),
    LanguageSubtag::new(['n', 't', 'x']),
    LanguageSubtag::new(['n', 't', 'y']),
    LanguageSubtag::new(['n', 't', 'z']),
    LanguageSubtag::new(['n', 'u', 'a']),
    LanguageSubtag::new(['n', 'u', 'b']),
    LanguageSubtag::new(['n', 'u', 'c']),
    LanguageSubtag::new(['n', 'u', 'd']),
    LanguageSubtag::new(['n', 'u', 'e']),
    LanguageSubtag::new(['n', 'u', 'f']),
    LanguageSubtag::new(['n', 'u', 'g']),
    LanguageSubtag::new(['n', 'u', 'h']),
    LanguageSubtag::new(['n', 'u', 'i']),
    LanguageSubtag::new(['n', 'u', 'j']),
    LanguageSubtag::new(['n', 'u', 'k']),
    LanguageSubtag::new(['n', 'u', 'l']),
    LanguageSubtag::new(['n', 'u', 'm']),
    LanguageSubtag::new(['n', 'u', 'n']),
    LanguageSubtag::new(['n', 'u', 'o']),
    LanguageSubtag::new(['n', 'u', 'p']),
    LanguageSubtag::new(['n', 'u', 'q']),
    LanguageSubtag::new(['n', 'u', 'r']),
    LanguageSubtag::new(['n', 'u', 's']),
    LanguageSubtag::new(['n', 'u', 't']),
    LanguageSubtag::new(['n', 'u', 'u']),
    LanguageSubtag::new(['n', 'u', 'v']),
    LanguageSubtag::new(['n', 'u', 'w']),
    LanguageSubtag::new(['n', 'u', 'x']),
    LanguageSubtag::new(['n', 'u', 'y']),
    LanguageSubtag::new(['n', 'u', 'z']),
    LanguageSubtag::new(['n', 'v', ' ']),
    LanguageSubtag::new(['n', 'v', 'h']),
    LanguageSubtag::new(['n', 'v', 'm']),
    LanguageSubtag::new(['n', 'v', 'o']),
    LanguageSubtag::new(['n', 'w', 'a']),
    LanguageSubtag::new(['n', 'w', 'b']),
    LanguageSubtag::new(['n', 'w', 'c']),
    LanguageSubtag::new(['n', 'w', 'e']),
    LanguageSubtag::new(['n', 'w', 'g']),
    LanguageSubtag::new(['n', 'w', 'i']),
    LanguageSubtag::new(['n', 'w', 'm']),
    LanguageSubtag::new(['n', 'w', 'o']),
    LanguageSubtag::new(['n', 'w', 'r']),
    LanguageSubtag::new(['n', 'w', 'x']),
    LanguageSubtag::new(['n', 'w', 'y']),
    LanguageSubtag::new(['n', 'x', 'a']),
    LanguageSubtag::new(['n', 'x', 'd']),
    LanguageSubtag::new(['n', 'x', 'e']),
    LanguageSubtag::new(['n', 'x', 'g']),
    LanguageSubtag::new(['n', 'x', 'i']),
    LanguageSubtag::new(['n', 'x', 'k']),
    LanguageSubtag::new(['n', 'x', 'l']),
    LanguageSubtag::new(['n', 'x', 'm']),
    LanguageSubtag::new(['n', 'x', 'n']),
    LanguageSubtag::new(['n', 'x', 'o']),
    LanguageSubtag::new(['n', 'x', 'q']),
    LanguageSubtag::new(['n', 'x', 'r']),
    LanguageSubtag::new(['n', 'x', 'u']),
    LanguageSubtag::new(['n', 'x', 'x']),
    LanguageSubtag::new(['n', 'y', ' ']),
    LanguageSubtag::new(['n', 'y', 'b']),
    LanguageSubtag::new(['n', 'y', 'c']),
    LanguageSubtag::new(['n', 'y', 'd']),
    LanguageSubtag::new(['n', 'y', 'e']),
    LanguageSubtag::new(['n', 'y', 'f']),
    LanguageSubtag::new(['n', 'y', 'g']),
    LanguageSubtag::new(['n', 'y', 'h']),
    LanguageSubtag::new(['n', 'y', 'i']),
    LanguageSubtag::new(['n', 'y', 'j']),
    LanguageSubtag::new(['n', 'y', 'k']),
    LanguageSubtag::new(['n', 'y', 'l']),
    LanguageSubtag::new(['n', 'y', 'm']),
    LanguageSubtag::new(['n', 'y', 'n']),
    LanguageSubtag::new(['n', 'y', 'o']),
    LanguageSubtag::new(['n', 'y', 'p']),
    LanguageSubtag::new(['n', 'y', 'q']),
    LanguageSubtag::new(['n', 'y', 'r']),
    LanguageSubtag::new(['n', 'y', 's']),
    LanguageSubtag::new(['n', 'y', 't']),
    LanguageSubtag::new(['n', 'y', 'u']),
    LanguageSubtag::new(['n', 'y', 'v']),
    LanguageSubtag::new(['n', 'y', 'w']),
    LanguageSubtag::new(['n', 'y', 'x']),
    LanguageSubtag::new(['n', 'y', 'y']),
    LanguageSubtag::new(['n', 'z', 'a']),
    LanguageSubtag::new(['n', 'z', 'b']),
    LanguageSubtag::new(['n', 'z', 'd']),
    LanguageSubtag::new(['n', 'z', 'i']),
    LanguageSubtag::new(['n', 'z', 'k']),
    LanguageSubtag::new(['n', 'z', 'm']),
    LanguageSubtag::new(['n', 'z', 's']),
    LanguageSubtag::new(['n', 'z', 'u']),
    LanguageSubtag::new(['n', 'z', 'y']),
    LanguageSubtag::new(['n', 'z', 'z']),
    LanguageSubtag::new(['o', 'a', 'a']),
    LanguageSubtag::new(['o', 'a', 'c']),
    LanguageSubtag::new(['o', 'a', 'r']),
    LanguageSubtag::new(['o', 'a', 'v']),
    LanguageSubtag::new(['o', 'b', 'i']),
    LanguageSubtag::new(['o', 'b', 'k']),
    LanguageSubtag::new(['o', 'b', 'l']),
    LanguageSubtag::new(['o', 'b', 'm']),
    LanguageSubtag::new(['o', 'b', 'o']),
    LanguageSubtag::new(['o', 'b', 'r']),
    LanguageSubtag::new(['o', 'b', 't']),
    LanguageSubtag::new(['o', 'b', 'u']),
    LanguageSubtag::new(['o', 'c', ' ']),
    LanguageSubtag::new(['o', 'c', 'a']),
    LanguageSubtag::new(['o', 'c', 'h']),
    LanguageSubtag::new(['o', 'c', 'm']),
    LanguageSubtag::new(['o', 'c', 'o']),
    LanguageSubtag::new(['o', 'c', 'u']),
    LanguageSubtag::new(['o', 'd', 'a']),
    LanguageSubtag::new(['o', 'd', 'k']),
    LanguageSubtag::new(['o', 'd', 't']),
    LanguageSubtag::new(['o', 'd', 'u']),
    LanguageSubtag::new(['o', 'f', 'o']),
    LanguageSubtag::new(['o', 'f', 's']),
    LanguageSubtag::new(['o', 'f', 'u']),
    LanguageSubtag::new(['o', 'g', 'b']),
    LanguageSubtag::new(['o', 'g', 'c']),
    LanguageSubtag::new(['o', 'g', 'e']),
    LanguageSubtag::new(['o', 'g', 'g']),
    LanguageSubtag::new(['o', 'g', 'o']),
    LanguageSubtag::new(['o', 'g', 'u']),
    LanguageSubtag::new(['o', 'h', 't']),
    LanguageSubtag::new(['o', 'h', 'u']),
    LanguageSubtag::new(['o', 'i', 'a']),
    LanguageSubtag::new(['o', 'i', 'n']),
    LanguageSubtag::new(['o', 'j', ' ']),
    LanguageSubtag::new(['o', 'j', 'b']),
    LanguageSubtag::new(['o', 'j', 'c']),
    LanguageSubtag::new(['o', 'j', 'g']),
    LanguageSubtag::new(['o', 'j', 'p']),
    LanguageSubtag::new(['o', 'j', 's']),
    LanguageSubtag::new(['o', 'j', 'v']),
    LanguageSubtag::new(['o', 'j', 'w']),
    LanguageSubtag::new(['o', 'k', 'a']),
    LanguageSubtag::new(['o', 'k', 'b']),
    LanguageSubtag::new(['o', 'k', 'c']),
    LanguageSubtag::new(['o', 'k', 'd']),
    LanguageSubtag::new(['o', 'k', 'e']),
    LanguageSubtag::new(['o', 'k', 'g']),
    LanguageSubtag::new(['o', 'k', 'h']),
    LanguageSubtag::new(['o', 'k', 'i']),
    LanguageSubtag::new(['o', 'k', 'j']),
    LanguageSubtag::new(['o', 'k', 'k']),
    LanguageSubtag::new(['o', 'k', 'l']),
    LanguageSubtag::new(['o', 'k', 'm']),
    LanguageSubtag::new(['o', 'k', 'n']),
    LanguageSubtag::new(['o', 'k', 'o']),
    LanguageSubtag::new(['o', 'k', 'r']),
    LanguageSubtag::new(['o', 'k', 's']),
    LanguageSubtag::new(['o', 'k', 'u']),
    LanguageSubtag::new(['o', 'k', 'v']),
    LanguageSubtag::new(['o', 'k', 'x']),
    LanguageSubtag::new(['o', 'k', 'z']),
    LanguageSubtag::new(['o', 'l', 'a']),
    LanguageSubtag::new(['o', 'l', 'd']),
    LanguageSubtag::new(['o', 'l', 'e']),
    LanguageSubtag::new(['o', 'l', 'k']),
    LanguageSubtag::new(['o', 'l', 'm']),
    LanguageSubtag::new(['o', 'l', 'o']),
    LanguageSubtag::new(['o', 'l', 'r']),
    LanguageSubtag::new(['o', 'l', 't']),
    LanguageSubtag::new(['o', 'l', 'u']),
    LanguageSubtag::new(['o', 'm', ' ']),
    LanguageSubtag::new(['o', 'm', 'a']),
    LanguageSubtag::new(['o', 'm', 'b']),
    LanguageSubtag::new(['o', 'm', 'c']),
    LanguageSubtag::new(['o', 'm', 'e']),
    LanguageSubtag::new(['o', 'm', 'g']),
    LanguageSubtag::new(['o', 'm', 'i']),
    LanguageSubtag::new(['o', 'm', 'k']),
    LanguageSubtag::new(['o', 'm', 'l']),
    LanguageSubtag::new(['o', 'm', 'n']),
    LanguageSubtag::new(['o', 'm', 'o']),
    LanguageSubtag::new(['o', 'm', 'p']),
    LanguageSubtag::new(['o', 'm', 'q']),
    LanguageSubtag::new(['o', 'm', 'r']),
    LanguageSubtag::new(['o', 'm', 't']),
    LanguageSubtag::new(['o', 'm', 'u']),
    LanguageSubtag::new(['o', 'm', 'v']),
    LanguageSubtag::new(['o', 'm', 'w']),
    LanguageSubtag::new(['o', 'm', 'x']),
    LanguageSubtag::new(['o', 'm', 'y']),
    LanguageSubtag::new(['o', 'n', 'a']),
    LanguageSubtag::new(['o', 'n', 'b']),
    LanguageSubtag::new(['o', 'n', 'e']),
    LanguageSubtag::new(['o', 'n', 'g']),
    LanguageSubtag::new(['o', 'n', 'i']),
    LanguageSubtag::new(['o', 'n', 'j']),
    LanguageSubtag::new(['o', 'n', 'k']),
    LanguageSubtag::new(['o', 'n', 'n']),
    LanguageSubtag::new(['o', 'n', 'o']),
    LanguageSubtag::new(['o', 'n', 'p']),
    LanguageSubtag::new(['o', 'n', 'r']),
    LanguageSubtag::new(['o', 'n', 's']),
    LanguageSubtag::new(['o', 'n', 't']),
    LanguageSubtag::new(['o', 'n', 'u']),
    LanguageSubtag::new(['o', 'n', 'w']),
    LanguageSubtag::new(['o', 'n', 'x']),
    LanguageSubtag::new(['o', 'o', 'd']),
    LanguageSubtag::new(['o', 'o', 'g']),
    LanguageSubtag::new(['o', 'o', 'n']),
    LanguageSubtag::new(['o', 'o', 'r']),
    LanguageSubtag::new(['o', 'o', 's']),
    LanguageSubtag::new(['o', 'p', 'a']),
    LanguageSubtag::new(['o', 'p', 'k']),
    LanguageSubtag::new(['o', 'p', 'm']),
    LanguageSubtag::new(['o', 'p', 'o']),
    LanguageSubtag::new(['o', 'p', 't']),
    LanguageSubtag::new(['o', 'p', 'y']),
    LanguageSubtag::new(['o', 'r', ' ']),
    LanguageSubtag::new(['o', 'r', 'a']),
    LanguageSubtag::new(['o', 'r', 'c']),
    LanguageSubtag::new(['o', 'r', 'e']),
    LanguageSubtag::new(['o', 'r', 'g']),
    LanguageSubtag::new(['o', 'r', 'h']),
    LanguageSubtag::new(['o', 'r', 'n']),
    LanguageSubtag::new(['o', 'r', 'o']),
    LanguageSubtag::new(['o', 'r', 'r']),
    LanguageSubtag::new(['o', 'r', 's']),
    LanguageSubtag::new(['o', 'r', 't']),
    LanguageSubtag::new(['o', 'r', 'u']),
    LanguageSubtag::new(['o', 'r', 'v']),
    LanguageSubtag::new(['o', 'r', 'w']),
    LanguageSubtag::new(['o', 'r', 'x']),
    LanguageSubtag::new(['o', 'r', 'y']),
    LanguageSubtag::new(['o', 'r', 'z']),
    LanguageSubtag::new(['o', 's', ' ']),
    LanguageSubtag::new(['o', 's', 'a']),
    LanguageSubtag::new(['o', 's', 'c']),
    LanguageSubtag::new(['o', 's', 'i']),
    LanguageSubtag::new(['o', 's', 'n']),
    LanguageSubtag::new(['o', 's', 'o']),
    LanguageSubtag::new(['o', 's', 'p']),
    LanguageSubtag::new(['o', 's', 't']),
    LanguageSubtag::new(['o', 's', 'u']),
    LanguageSubtag::new(['o', 's', 'x']),
    LanguageSubtag::new(['o', 't', 'a']),
    LanguageSubtag::new(['o', 't', 'b']),
    LanguageSubtag::new(['o', 't', 'd']),
    LanguageSubtag::new(['o', 't', 'e']),
    LanguageSubtag::new(['o', 't', 'i']),
    LanguageSubtag::new(['o', 't', 'k']),
    LanguageSubtag::new(['o', 't', 'l']),
    LanguageSubtag::new(['o', 't', 'm']),
    LanguageSubtag::new(['o', 't', 'n']),
    LanguageSubtag::new(['o', 't', 'o']),
    LanguageSubtag::new(['o', 't', 'q']),
    LanguageSubtag::new(['o', 't', 'r']),
    LanguageSubtag::new(['o', 't', 's']),
    LanguageSubtag::new(['o', 't', 't']),
    LanguageSubtag::new(['o', 't', 'u']),
    LanguageSubtag::new(['o', 't', 'w']),
    LanguageSubtag::new(['o', 't', 'x']),
    LanguageSubtag::new(['o', 't', 'y']),
    LanguageSubtag::new(['o', 't', 'z']),
    LanguageSubtag::new(['o', 'u', 'a']),
    LanguageSubtag::new(['o', 'u', 'b']),
    LanguageSubtag::new(['o', 'u', 'e']),
    LanguageSubtag::new(['o', 'u', 'i']),
    LanguageSubtag::new(['o', 'u', 'm']),
    LanguageSubtag::new(['o', 'u', 'n']),
    LanguageSubtag::new(['o', 'v', 'd']),
    LanguageSubtag::new(['o', 'w', 'i']),
    LanguageSubtag::new(['o', 'w', 'l']),
    LanguageSubtag::new(['o', 'y', 'b']),
    LanguageSubtag::new(['o', 'y', 'd']),
    LanguageSubtag::new(['o', 'y', 'm']),
    LanguageSubtag::new(['o', 'y', 'y']),
    LanguageSubtag::new(['o', 'z', 'm']),
    LanguageSubtag::new(['p', 'a', ' ']),
    LanguageSubtag::new(['p', 'a', 'a']),
    LanguageSubtag::new(['p', 'a', 'b']),
    LanguageSubtag::new(['p', 'a', 'c']),
    LanguageSubtag::new(['p', 'a', 'd']),
    LanguageSubtag::new(['p', 'a', 'e']),
    LanguageSubtag::new(['p', 'a', 'f']),
    LanguageSubtag::new(['p', 'a', 'g']),
    LanguageSubtag::new(['p', 'a', 'h']),
    LanguageSubtag::new(['p', 'a', 'i']),
    LanguageSubtag::new(['p', 'a', 'k']),
    LanguageSubtag::new(['p', 'a', 'l']),
    LanguageSubtag::new(['p', 'a', 'm']),
    LanguageSubtag::new(['p', 'a', 'o']),
    LanguageSubtag::new(['p', 'a', 'p']),
    LanguageSubtag::new(['p', 'a', 'q']),
    LanguageSubtag::new(['p', 'a', 'r']),
    LanguageSubtag::new(['p', 'a', 's']),
    LanguageSubtag::new(['p', 'a', 't']),
    LanguageSubtag::new(['p', 'a', 'u']),
    LanguageSubtag::new(['p', 'a', 'v']),
    LanguageSubtag::new(['p', 'a', 'w']),
    LanguageSubtag::new(['p', 'a', 'x']),
    LanguageSubtag::new(['p', 'a', 'y']),
    LanguageSubtag::new(['p', 'a', 'z']),
    LanguageSubtag::new(['p', 'b', 'b']),
    LanguageSubtag::new(['p', 'b', 'c']),
    LanguageSubtag::new(['p', 'b', 'e']),
    LanguageSubtag::new(['p', 'b', 'f']),
    LanguageSubtag::new(['p', 'b', 'g']),
    LanguageSubtag::new(['p', 'b', 'h']),
    LanguageSubtag::new(['p', 'b', 'i']),
    LanguageSubtag::new(['p', 'b', 'l']),
    LanguageSubtag::new(['p', 'b', 'm']),
    LanguageSubtag::new(['p', 'b', 'n']),
    LanguageSubtag::new(['p', 'b', 'o']),
    LanguageSubtag::new(['p', 'b', 'p']),
    LanguageSubtag::new(['p', 'b', 'r']),
    LanguageSubtag::new(['p', 'b', 's']),
    LanguageSubtag::new(['p', 'b', 't']),
    LanguageSubtag::new(['p', 'b', 'u']),
    LanguageSubtag::new(['p', 'b', 'v']),
    LanguageSubtag::new(['p', 'b', 'y']),
    LanguageSubtag::new(['p', 'b', 'z']),
    LanguageSubtag::new(['p', 'c', 'a']),
    LanguageSubtag::new(['p', 'c', 'b']),
    LanguageSubtag::new(['p', 'c', 'c']),
    LanguageSubtag::new(['p', 'c', 'd']),
    LanguageSubtag::new(['p', 'c', 'e']),
    LanguageSubtag::new(['p', 'c', 'f']),
    LanguageSubtag::new(['p', 'c', 'g']),
    LanguageSubtag::new(['p', 'c', 'h']),
    LanguageSubtag::new(['p', 'c', 'i']),
    LanguageSubtag::new(['p', 'c', 'j']),
    LanguageSubtag::new(['p', 'c', 'k']),
    LanguageSubtag::new(['p', 'c', 'l']),
    LanguageSubtag::new(['p', 'c', 'm']),
    LanguageSubtag::new(['p', 'c', 'n']),
    LanguageSubtag::new(['p', 'c', 'p']),
    LanguageSubtag::new(['p', 'c', 'r']),
    LanguageSubtag::new(['p', 'c', 'w']),
    LanguageSubtag::new(['p', 'd', 'a']),
    LanguageSubtag::new(['p', 'd', 'c']),
    LanguageSubtag::new(['p', 'd', 'i']),
    LanguageSubtag::new(['p', 'd', 'n']),
    LanguageSubtag::new(['p', 'd', 'o']),
    LanguageSubtag::new(['p', 'd', 't']),
    LanguageSubtag::new(['p', 'd', 'u']),
    LanguageSubtag::new(['p', 'e', 'a']),
    LanguageSubtag::new(['p', 'e', 'b']),
    LanguageSubtag::new(['p', 'e', 'd']),
    LanguageSubtag::new(['p', 'e', 'e']),
    LanguageSubtag::new(['p', 'e', 'f']),
    LanguageSubtag::new(['p', 'e', 'g']),
    LanguageSubtag::new(['p', 'e', 'h']),
    LanguageSubtag::new(['p', 'e', 'i']),
    LanguageSubtag::new(['p', 'e', 'j']),
    LanguageSubtag::new(['p', 'e', 'k']),
    LanguageSubtag::new(['p', 'e', 'l']),
    LanguageSubtag::new(['p', 'e', 'm']),
    LanguageSubtag::new(['p', 'e', 'o']),
    LanguageSubtag::new(['p', 'e', 'p']),
    LanguageSubtag::new(['p', 'e', 'q']),
    LanguageSubtag::new(['p', 'e', 's']),
    LanguageSubtag::new(['p', 'e', 'v']),
    LanguageSubtag::new(['p', 'e', 'x']),
    LanguageSubtag::new(['p', 'e', 'y']),
    LanguageSubtag::new(['p', 'e', 'z']),
    LanguageSubtag::new(['p', 'f', 'a']),
    LanguageSubtag::new(['p', 'f', 'e']),
    LanguageSubtag::new(['p', 'f', 'l']),
    LanguageSubtag::new(['p', 'g', 'a']),
    LanguageSubtag::new(['p', 'g', 'd']),
    LanguageSubtag::new(['p', 'g', 'g']),
    LanguageSubtag::new(['p', 'g', 'i']),
    LanguageSubtag::new(['p', 'g', 'k']),
    LanguageSubtag::new(['p', 'g', 'l']),
    LanguageSubtag::new(['p', 'g', 'n']),
    LanguageSubtag::new(['p', 'g', 's']),
    LanguageSubtag::new(['p', 'g', 'u']),
    LanguageSubtag::new(['p', 'g', 'y']),
    LanguageSubtag::new(['p', 'g', 'z']),
    LanguageSubtag::new(['p', 'h', 'a']),
    LanguageSubtag::new(['p', 'h', 'd']),
    LanguageSubtag::new(['p', 'h', 'g']),
    LanguageSubtag::new(['p', 'h', 'h']),
    LanguageSubtag::new(['p', 'h', 'i']),
    LanguageSubtag::new(['p', 'h', 'k']),
    LanguageSubtag::new(['p', 'h', 'l']),
    LanguageSubtag::new(['p', 'h', 'm']),
    LanguageSubtag::new(['p', 'h', 'n']),
    LanguageSubtag::new(['p', 'h', 'o']),
    LanguageSubtag::new(['p', 'h', 'q']),
    LanguageSubtag::new(['p', 'h', 'r']),
    LanguageSubtag::new(['p', 'h', 't']),
    LanguageSubtag::new(['p', 'h', 'u']),
    LanguageSubtag::new(['p', 'h', 'v']),
    LanguageSubtag::new(['p', 'h', 'w']),
    LanguageSubtag::new(['p', 'i', ' ']),
    LanguageSubtag::new(['p', 'i', 'a']),
    LanguageSubtag::new(['p', 'i', 'b']),
    LanguageSubtag::new(['p', 'i', 'c']),
    LanguageSubtag::new(['p', 'i', 'd']),
    LanguageSubtag::new(['p', 'i', 'e']),
    LanguageSubtag::new(['p', 'i', 'f']),
    LanguageSubtag::new(['p', 'i', 'g']),
    LanguageSubtag::new(['p', 'i', 'h']),
    LanguageSubtag::new(['p', 'i', 'i']),
    LanguageSubtag::new(['p', 'i', 'j']),
    LanguageSubtag::new(['p', 'i', 'l']),
    LanguageSubtag::new(['p', 'i', 'm']),
    LanguageSubtag::new(['p', 'i', 'n']),
    LanguageSubtag::new(['p', 'i', 'o']),
    LanguageSubtag::new(['p', 'i', 'p']),
    LanguageSubtag::new(['p', 'i', 'r']),
    LanguageSubtag::new(['p', 'i', 's']),
    LanguageSubtag::new(['p', 'i', 't']),
    LanguageSubtag::new(['p', 'i', 'u']),
    LanguageSubtag::new(['p', 'i', 'v']),
    LanguageSubtag::new(['p', 'i', 'w']),
    LanguageSubtag::new(['p', 'i', 'x']),
    LanguageSubtag::new(['p', 'i', 'y']),
    LanguageSubtag::new(['p', 'i', 'z']),
    LanguageSubtag::new(['p', 'j', 't']),
    LanguageSubtag::new(['p', 'k', 'a']),
    LanguageSubtag::new(['p', 'k', 'b']),
    LanguageSubtag::new(['p', 'k', 'c']),
    LanguageSubtag::new(['p', 'k', 'g']),
    LanguageSubtag::new(['p', 'k', 'h']),
    LanguageSubtag::new(['p', 'k', 'n']),
    LanguageSubtag::new(['p', 'k', 'o']),
    LanguageSubtag::new(['p', 'k', 'p']),
    LanguageSubtag::new(['p', 'k', 'r']),
    LanguageSubtag::new(['p', 'k', 's']),
    LanguageSubtag::new(['p', 'k', 't']),
    LanguageSubtag::new(['p', 'k', 'u']),
    LanguageSubtag::new(['p', 'l', ' ']),
    LanguageSubtag::new(['p', 'l', 'a']),
    LanguageSubtag::new(['p', 'l', 'b']),
    LanguageSubtag::new(['p', 'l', 'c']),
    LanguageSubtag::new(['p', 'l', 'd']),
    LanguageSubtag::new(['p', 'l', 'e']),
    LanguageSubtag::new(['p', 'l', 'f']),
    LanguageSubtag::new(['p', 'l', 'g']),
    LanguageSubtag::new(['p', 'l', 'h']),
    LanguageSubtag::new(['p', 'l', 'j']),
    LanguageSubtag::new(['p', 'l', 'k']),
    LanguageSubtag::new(['p', 'l', 'l']),
    LanguageSubtag::new(['p', 'l', 'n']),
    LanguageSubtag::new(['p', 'l', 'o']),
    LanguageSubtag::new(['p', 'l', 'p']),
    LanguageSubtag::new(['p', 'l', 'q']),
    LanguageSubtag::new(['p', 'l', 'r']),
    LanguageSubtag::new(['p', 'l', 's']),
    LanguageSubtag::new(['p', 'l', 't']),
    LanguageSubtag::new(['p', 'l', 'u']),
    LanguageSubtag::new(['p', 'l', 'v']),
    LanguageSubtag::new(['p', 'l', 'w']),
    LanguageSubtag::new(['p', 'l', 'y']),
    LanguageSubtag::new(['p', 'l', 'z']),
    LanguageSubtag::new(['p', 'm', 'a']),
    LanguageSubtag::new(['p', 'm', 'b']),
    LanguageSubtag::new(['p', 'm', 'c']),
    LanguageSubtag::new(['p', 'm', 'd']),
    LanguageSubtag::new(['p', 'm', 'e']),
    LanguageSubtag::new(['p', 'm', 'f']),
    LanguageSubtag::new(['p', 'm', 'h']),
    LanguageSubtag::new(['p', 'm', 'i']),
    LanguageSubtag::new(['p', 'm', 'j']),
    LanguageSubtag::new(['p', 'm', 'k']),
    LanguageSubtag::new(['p', 'm', 'l']),
    LanguageSubtag::new(['p', 'm', 'm']),
    LanguageSubtag::new(['p', 'm', 'n']),
    LanguageSubtag::new(['p', 'm', 'o']),
    LanguageSubtag::new(['p', 'm', 'q']),
    LanguageSubtag::new(['p', 'm', 'r']),
    LanguageSubtag::new(['p', 'm', 's']),
    LanguageSubtag::new(['p', 'm', 't']),
    LanguageSubtag::new(['p', 'm', 'u']),
    LanguageSubtag::new(['p', 'm', 'w']),
    LanguageSubtag::new(['p', 'm', 'x']),
    LanguageSubtag::new(['p', 'm', 'y']),
    LanguageSubtag::new(['p', 'm', 'z']),
    LanguageSubtag::new(['p', 'n', 'a']),
    LanguageSubtag::new(['p', 'n', 'b']),
    LanguageSubtag::new(['p', 'n', 'c']),
    LanguageSubtag::new(['p', 'n', 'd']),
    LanguageSubtag::new(['p', 'n', 'e']),
    LanguageSubtag::new(['p', 'n', 'g']),
    LanguageSubtag::new(['p', 'n', 'h']),
    LanguageSubtag::new(['p', 'n', 'i']),
    LanguageSubtag::new(['p', 'n', 'j']),
    LanguageSubtag::new(['p', 'n', 'k']),
    LanguageSubtag::new(['p', 'n', 'l']),
    LanguageSubtag::new(['p', 'n', 'm']),
    LanguageSubtag::new(['p', 'n', 'n']),
    LanguageSubtag::new(['p', 'n', 'o']),
    LanguageSubtag::new(['p', 'n', 'p']),
    LanguageSubtag::new(['p', 'n', 'q']),
    LanguageSubtag::new(['p', 'n', 'r']),
    LanguageSubtag::new(['p', 'n', 's']),
    LanguageSubtag::new(['p', 'n', 't']),
    LanguageSubtag::new(['p', 'n', 'u']),
    LanguageSubtag::new(['p', 'n', 'v']),
    LanguageSubtag::new(['p', 'n', 'w']),
    LanguageSubtag::new(['p', 'n', 'x']),
    LanguageSubtag::new(['p', 'n', 'y']),
    LanguageSubtag::new(['p', 'n', 'z']),
    LanguageSubtag::new(['p', 'o', 'c']),
    LanguageSubtag::new(['p', 'o', 'd']),
    LanguageSubtag::new(['p', 'o', 'e']),
    LanguageSubtag::new(['p', 'o', 'f']),
    LanguageSubtag::new(['p', 'o', 'g']),
    LanguageSubtag::new(['p', 'o', 'h']),
    LanguageSubtag::new(['p', 'o', 'i']),
    LanguageSubtag::new(['p', 'o', 'k']),
    LanguageSubtag::new(['p', 'o', 'm']),
    LanguageSubtag::new(['p', 'o', 'n']),
    LanguageSubtag::new(['p', 'o', 'o']),
    LanguageSubtag::new(['p', 'o', 'p']),
    LanguageSubtag::new(['p', 'o', 'q']),
    LanguageSubtag::new(['p', 'o', 's']),
    LanguageSubtag::new(['p', 'o', 't']),
    LanguageSubtag::new(['p', 'o', 'v']),
    LanguageSubtag::new(['p', 'o', 'w']),
    LanguageSubtag::new(['p', 'o', 'x']),
    LanguageSubtag::new(['p', 'o', 'y']),
    LanguageSubtag::new(['p', 'o', 'z']),
    LanguageSubtag::new(['p', 'p', 'a']),
    LanguageSubtag::new(['p', 'p', 'e']),
    LanguageSubtag::new(['p', 'p', 'i']),
    LanguageSubtag::new(['p', 'p', 'k']),
    LanguageSubtag::new(['p', 'p', 'l']),
    LanguageSubtag::new(['p', 'p', 'm']),
    LanguageSubtag::new(['p', 'p', 'n']),
    LanguageSubtag::new(['p', 'p', 'o']),
    LanguageSubtag::new(['p', 'p', 'p']),
    LanguageSubtag::new(['p', 'p', 'q']),
    LanguageSubtag::new(['p', 'p', 'r']),
    LanguageSubtag::new(['p', 'p', 's']),
    LanguageSubtag::new(['p', 'p', 't']),
    LanguageSubtag::new(['p', 'p', 'u']),
    LanguageSubtag::new(['p', 'q', 'a']),
    LanguageSubtag::new(['p', 'q', 'e']),
    LanguageSubtag::new(['p', 'q', 'm']),
    LanguageSubtag::new(['p', 'q', 'w']),
    LanguageSubtag::new(['p', 'r', 'a']),
    LanguageSubtag::new(['p', 'r', 'b']),
    LanguageSubtag::new(['p', 'r', 'c']),
    LanguageSubtag::new(['p', 'r', 'd']),
    LanguageSubtag::new(['p', 'r', 'e']),
    LanguageSubtag::new(['p', 'r', 'f']),
    LanguageSubtag::new(['p', 'r', 'g']),
    LanguageSubtag::new(['p', 'r', 'h']),
    LanguageSubtag::new(['p', 'r', 'i']),
    LanguageSubtag::new(['p', 'r', 'k']),
    LanguageSubtag::new(['p', 'r', 'l']),
    LanguageSubtag::new(['p', 'r', 'm']),
    LanguageSubtag::new(['p', 'r', 'n']),
    LanguageSubtag::new(['p', 'r', 'o']),
    LanguageSubtag::new(['p', 'r', 'p']),
    LanguageSubtag::new(['p', 'r', 'q']),
    LanguageSubtag::new(['p', 'r', 'r']),
    LanguageSubtag::new(['p', 'r', 's']),
    LanguageSubtag::new(['p', 'r', 't']),
    LanguageSubtag::new(['p', 'r', 'u']),
    LanguageSubtag::new(['p', 'r', 'w']),
    LanguageSubtag::new(['p', 'r', 'x']),
    LanguageSubtag::new(['p', 'r', 'y']),
    LanguageSubtag::new(['p', 'r', 'z']),
    LanguageSubtag::new(['p', 's', ' ']),
    LanguageSubtag::new(['p', 's', 'a']),
    LanguageSubtag::new(['p', 's', 'c']),
    LanguageSubtag::new(['p', 's', 'd']),
    LanguageSubtag::new(['p', 's', 'e']),
    LanguageSubtag::new(['p', 's', 'g']),
    LanguageSubtag::new(['p', 's', 'h']),
    LanguageSubtag::new(['p', 's', 'i']),
    LanguageSubtag::new(['p', 's', 'l']),
    LanguageSubtag::new(['p', 's', 'm']),
    LanguageSubtag::new(['p', 's', 'n']),
    LanguageSubtag::new(['p', 's', 'o']),
    LanguageSubtag::new(['p', 's', 'p']),
    LanguageSubtag::new(['p', 's', 'q']),
    LanguageSubtag::new(['p', 's', 'r']),
    LanguageSubtag::new(['p', 's', 's']),
    LanguageSubtag::new(['p', 's', 't']),
    LanguageSubtag::new(['p', 's', 'u']),
    LanguageSubtag::new(['p', 's', 'w']),
    LanguageSubtag::new(['p', 's', 'y']),
    LanguageSubtag::new(['p', 't', ' ']),
    LanguageSubtag::new(['p', 't', 'a']),
    LanguageSubtag::new(['p', 't', 'h']),
    LanguageSubtag::new(['p', 't', 'i']),
    LanguageSubtag::new(['p', 't', 'n']),
    LanguageSubtag::new(['p', 't', 'o']),
    LanguageSubtag::new(['p', 't', 'p']),
    LanguageSubtag::new(['p', 't', 'q']),
    LanguageSubtag::new(['p', 't', 'r']),
    LanguageSubtag::new(['p', 't', 't']),
    LanguageSubtag::new(['p', 't', 'u']),
    LanguageSubtag::new(['p', 't', 'v']),
    LanguageSubtag::new(['p', 't', 'w']),
    LanguageSubtag::new(['p', 't', 'y']),
    LanguageSubtag::new(['p', 'u', 'a']),
    LanguageSubtag::new(['p', 'u', 'b']),
    LanguageSubtag::new(['p', 'u', 'c']),
    LanguageSubtag::new(['p', 'u', 'd']),
    LanguageSubtag::new(['p', 'u', 'e']),
    LanguageSubtag::new(['p', 'u', 'f']),
    LanguageSubtag::new(['p', 'u', 'g']),
    LanguageSubtag::new(['p', 'u', 'i']),
    LanguageSubtag::new(['p', 'u', 'j']),
    LanguageSubtag::new(['p', 'u', 'k']),
    LanguageSubtag::new(['p', 'u', 'm']),
    LanguageSubtag::new(['p', 'u', 'o']),
    LanguageSubtag::new(['p', 'u', 'p']),
    LanguageSubtag::new(['p', 'u', 'q']),
    LanguageSubtag::new(['p', 'u', 'r']),
    LanguageSubtag::new(['p', 'u', 't']),
    LanguageSubtag::new(['p', 'u', 'u']),
    LanguageSubtag::new(['p', 'u', 'w']),
    LanguageSubtag::new(['p', 'u', 'x']),
    LanguageSubtag::new(['p', 'u', 'y']),
    LanguageSubtag::new(['p', 'u', 'z']),
    LanguageSubtag::new(['p', 'w', 'a']),
    LanguageSubtag::new(['p', 'w', 'b']),
    LanguageSubtag::new(['p', 'w', 'g']),
    LanguageSubtag::new(['p', 'w', 'i']),
    LanguageSubtag::new(['p', 'w', 'm']),
    LanguageSubtag::new(['p', 'w', 'n']),
    LanguageSubtag::new(['p', 'w', 'o']),
    LanguageSubtag::new(['p', 'w', 'r']),
    LanguageSubtag::new(['p', 'w', 'w']),
    LanguageSubtag::new(['p', 'x', 'm']),
    LanguageSubtag::new(['p', 'y', 'e']),
    LanguageSubtag::new(['p', 'y', 'm']),
    LanguageSubtag::new(['p', 'y', 'n']),
    LanguageSubtag::new(['p', 'y', 's']),
    LanguageSubtag::new(['p', 'y', 'u']),
    LanguageSubtag::new(['p', 'y', 'x']),
    LanguageSubtag::new(['p', 'y', 'y']),
    LanguageSubtag::new(['p', 'z', 'n']),
    LanguageSubtag::new(['q', 'u', ' ']),
    LanguageSubtag::new(['q', 'u', 'a']),
    LanguageSubtag::new(['q', 'u', 'b']),
    LanguageSubtag::new(['q', 'u', 'c']),
    LanguageSubtag::new(['q', 'u', 'd']),
    LanguageSubtag::new(['q', 'u', 'f']),
    LanguageSubtag::new(['q', 'u', 'g']),
    LanguageSubtag::new(['q', 'u', 'h']),
    LanguageSubtag::new(['q', 'u', 'i']),
    LanguageSubtag::new(['q', 'u', 'k']),
    LanguageSubtag::new(['q', 'u', 'l']),
    LanguageSubtag::new(['q', 'u', 'm']),
    LanguageSubtag::new(['q', 'u', 'n']),
    LanguageSubtag::new(['q', 'u', 'p']),
    LanguageSubtag::new(['q', 'u', 'q']),
    LanguageSubtag::new(['q', 'u', 'r']),
    LanguageSubtag::new(['q', 'u', 's']),
    LanguageSubtag::new(['q', 'u', 'v']),
    LanguageSubtag::new(['q', 'u', 'w']),
    LanguageSubtag::new(['q', 'u', 'x']),
    LanguageSubtag::new(['q', 'u', 'y']),
    LanguageSubtag::new(['q', 'u', 'z']),
    LanguageSubtag::new(['q', 'v', 'a']),
    LanguageSubtag::new(['q', 'v', 'c']),
    LanguageSubtag::new(['q', 'v', 'e']),
    LanguageSubtag::new(['q', 'v', 'h']),
    LanguageSubtag::new(['q', 'v', 'i']),
    LanguageSubtag::new(['q', 'v', 'j']),
    LanguageSubtag::new(['q', 'v', 'l']),
    LanguageSubtag::new(['q', 'v', 'm']),
    LanguageSubtag::new(['q', 'v', 'n']),
    LanguageSubtag::new(['q', 'v', 'o']),
    LanguageSubtag::new(['q', 'v', 'p']),
    LanguageSubtag::new(['q', 'v', 's']),
    LanguageSubtag::new(['q', 'v', 'w']),
    LanguageSubtag::new(['q', 'v', 'y']),
    LanguageSubtag::new(['q', 'v', 'z']),
    LanguageSubtag::new(['q', 'w', 'a']),
    LanguageSubtag::new(['q', 'w', 'c']),
    LanguageSubtag::new(['q', 'w', 'e']),
    LanguageSubtag::new(['q', 'w', 'h']),
    LanguageSubtag::new(['q', 'w', 'm']),
    LanguageSubtag::new(['q', 'w', 's']),
    LanguageSubtag::new(['q', 'w', 't']),
    LanguageSubtag::new(['q', 'x', 'a']),
    LanguageSubtag::new(['q', 'x', 'c']),
    LanguageSubtag::new(['q', 'x', 'h']),
    LanguageSubtag::new(['q', 'x', 'l']),
    LanguageSubtag::new(['q', 'x', 'n']),
    LanguageSubtag::new(['q', 'x', 'o']),
    LanguageSubtag::new(['q', 'x', 'p']),
    LanguageSubtag::new(['q', 'x', 'q']),
    LanguageSubtag::new(['q', 'x', 'r']),
    LanguageSubtag::new(['q', 'x', 's']),
    LanguageSubtag::new(['q', 'x', 't']),
    LanguageSubtag::new(['q', 'x', 'u']),
    LanguageSubtag::new(['q', 'x', 'w']),
    LanguageSubtag::new(['q', 'y', 'a']),
    LanguageSubtag::new(['q', 'y', 'p']),
    LanguageSubtag::new(['r', 'a', 'a']),
    LanguageSubtag::new(['r', 'a', 'b']),
    LanguageSubtag::new(['r', 'a', 'c']),
    LanguageSubtag::new(['r', 'a', 'd']),
    LanguageSubtag::new(['r', 'a', 'f']),
    LanguageSubtag::new(['r', 'a', 'g']),
    LanguageSubtag::new(['r', 'a', 'h']),
    LanguageSubtag::new(['r', 'a', 'i']),
    LanguageSubtag::new(['r', 'a', 'j']),
    LanguageSubtag::new(['r', 'a', 'k']),
    LanguageSubtag::new(['r', 'a', 'l']),
    LanguageSubtag::new(['r', 'a', 'm']),
    LanguageSubtag::new(['r', 'a', 'n']),
    LanguageSubtag::new(['r', 'a', 'o']),
    LanguageSubtag::new(['r', 'a', 'p']),
    LanguageSubtag::new(['r', 'a', 'q']),
    LanguageSubtag::new(['r', 'a', 'r']),
    LanguageSubtag::new(['r', 'a', 's']),
    LanguageSubtag::new(['r', 'a', 't']),
    LanguageSubtag::new(['r', 'a', 'u']),
    LanguageSubtag::new(['r', 'a', 'v']),
    LanguageSubtag::new(['r', 'a', 'w']),
    LanguageSubtag::new(['r', 'a', 'x']),
    LanguageSubtag::new(['r', 'a', 'y']),
    LanguageSubtag::new(['r', 'a', 'z']),
    LanguageSubtag::new(['r', 'b', 'b']),
    LanguageSubtag::new(['r', 'b', 'k']),
    LanguageSubtag::new(['r', 'b', 'l']),
    LanguageSubtag::new(['r', 'b', 'p']),
    LanguageSubtag::new(['r', 'c', 'f']),
    LanguageSubtag::new(['r', 'd', 'b']),
    LanguageSubtag::new(['r', 'e', 'a']),
    LanguageSubtag::new(['r', 'e', 'b']),
    LanguageSubtag::new(['r', 'e', 'e']),
    LanguageSubtag::new(['r', 'e', 'g']),
    LanguageSubtag::new(['r', 'e', 'i']),
    LanguageSubtag::new(['r', 'e', 'j']),
    LanguageSubtag::new(['r', 'e', 'l']),
    LanguageSubtag::new(['r', 'e', 'm']),
    LanguageSubtag::new(['r', 'e', 'n']),
    LanguageSubtag::new(['r', 'e', 'r']),
    LanguageSubtag::new(['r', 'e', 's']),
    LanguageSubtag::new(['r', 'e', 't']),
    LanguageSubtag::new(['r', 'e', 'y']),
    LanguageSubtag::new(['r', 'g', 'a']),
    LanguageSubtag::new(['r', 'g', 'e']),
    LanguageSubtag::new(['r', 'g', 'k']),
    LanguageSubtag::new(['r', 'g', 'n']),
    LanguageSubtag::new(['r', 'g', 'r']),
    LanguageSubtag::new(['r', 'g', 's']),
    LanguageSubtag::new(['r', 'g', 'u']),
    LanguageSubtag::new(['r', 'h', 'g']),
    LanguageSubtag::new(['r', 'h', 'p']),
    LanguageSubtag::new(['r', 'i', 'a']),
    LanguageSubtag::new(['r', 'i', 'e']),
    LanguageSubtag::new(['r', 'i', 'f']),
    LanguageSubtag::new(['r', 'i', 'l']),
    LanguageSubtag::new(['r', 'i', 'm']),
    LanguageSubtag::new(['r', 'i', 'n']),
    LanguageSubtag::new(['r', 'i', 'r']),
    LanguageSubtag::new(['r', 'i', 't']),
    LanguageSubtag::new(['r', 'i', 'u']),
    LanguageSubtag::new(['r', 'j', 'g']),
    LanguageSubtag::new(['r', 'j', 'i']),
    LanguageSubtag::new(['r', 'j', 's']),
    LanguageSubtag::new(['r', 'k', 'a']),
    LanguageSubtag::new(['r', 'k', 'b']),
    LanguageSubtag::new(['r', 'k', 'h']),
    LanguageSubtag::new(['r', 'k', 'i']),
    LanguageSubtag::new(['r', 'k', 'm']),
    LanguageSubtag::new(['r', 'k', 't']),
    LanguageSubtag::new(['r', 'k', 'w']),
    LanguageSubtag::new(['r', 'm', ' ']),
    LanguageSubtag::new(['r', 'm', 'a']),
    LanguageSubtag::new(['r', 'm', 'b']),
    LanguageSubtag::new(['r', 'm', 'c']),
    LanguageSubtag::new(['r', 'm', 'd']),
    LanguageSubtag::new(['r', 'm', 'e']),
    LanguageSubtag::new(['r', 'm', 'f']),
    LanguageSubtag::new(['r', 'm', 'g']),
    LanguageSubtag::new(['r', 'm', 'h']),
    LanguageSubtag::new(['r', 'm', 'i']),
    LanguageSubtag::new(['r', 'm', 'k']),
    LanguageSubtag::new(['r', 'm', 'l']),
    LanguageSubtag::new(['r', 'm', 'm']),
    LanguageSubtag::new(['r', 'm', 'n']),
    LanguageSubtag::new(['r', 'm', 'o']),
    LanguageSubtag::new(['r', 'm', 'p']),
    LanguageSubtag::new(['r', 'm', 'q']),
    LanguageSubtag::new(['r', 'm', 'r']),
    LanguageSubtag::new(['r', 'm', 's']),
    LanguageSubtag::new(['r', 'm', 't']),
    LanguageSubtag::new(['r', 'm', 'u']),
    LanguageSubtag::new(['r', 'm', 'v']),
    LanguageSubtag::new(['r', 'm', 'w']),
    LanguageSubtag::new(['r', 'm', 'x']),
    LanguageSubtag::new(['r', 'm', 'y']),
    LanguageSubtag::new(['r', 'm', 'z']),
    LanguageSubtag::new(['r', 'n', ' ']),
    LanguageSubtag::new(['r', 'n', 'a']),
    LanguageSubtag::new(['r', 'n', 'd']),
    LanguageSubtag::new(['r', 'n', 'g']),
    LanguageSubtag::new(['r', 'n', 'l']),
    LanguageSubtag::new(['r', 'n', 'n']),
    LanguageSubtag::new(['r', 'n', 'p']),
    LanguageSubtag::new(['r', 'n', 'r']),
    LanguageSubtag::new(['r', 'n', 'w']),
    LanguageSubtag::new(['r', 'o', ' ']),
    LanguageSubtag::new(['r', 'o', 'a']),
    LanguageSubtag::new(['r', 'o', 'b']),
    LanguageSubtag::new(['r', 'o', 'c']),
    LanguageSubtag::new(['r', 'o', 'd']),
    LanguageSubtag::new(['r', 'o', 'e']),
    LanguageSubtag::new(['r', 'o', 'f']),
    LanguageSubtag::new(['r', 'o', 'g']),
    LanguageSubtag::new(['r', 'o', 'l']),
    LanguageSubtag::new(['r', 'o', 'm']),
    LanguageSubtag::new(['r', 'o', 'o']),
    LanguageSubtag::new(['r', 'o', 'p']),
    LanguageSubtag::new(['r', 'o', 'r']),
    LanguageSubtag::new(['r', 'o', 'u']),
    LanguageSubtag::new(['r', 'o', 'w']),
    LanguageSubtag::new(['r', 'p', 'n']),
    LanguageSubtag::new(['r', 'p', 't']),
    LanguageSubtag::new(['r', 'r', 'i']),
    LanguageSubtag::new(['r', 'r', 'o']),
    LanguageSubtag::new(['r', 'r', 't']),
    LanguageSubtag::new(['r', 's', 'b']),
    LanguageSubtag::new(['r', 's', 'i']),
    LanguageSubtag::new(['r', 's', 'l']),
    LanguageSubtag::new(['r', 's', 'm']),
    LanguageSubtag::new(['r', 't', 'c']),
    LanguageSubtag::new(['r', 't', 'h']),
    LanguageSubtag::new(['r', 't', 'm']),
    LanguageSubtag::new(['r', 't', 's']),
    LanguageSubtag::new(['r', 't', 'w']),
    LanguageSubtag::new(['r', 'u', ' ']),
    LanguageSubtag::new(['r', 'u', 'b']),
    LanguageSubtag::new(['r', 'u', 'c']),
    LanguageSubtag::new(['r', 'u', 'e']),
    LanguageSubtag::new(['r', 'u', 'f']),
    LanguageSubtag::new(['r', 'u', 'g']),
    LanguageSubtag::new(['r', 'u', 'h']),
    LanguageSubtag::new(['r', 'u', 'i']),
    LanguageSubtag::new(['r', 'u', 'k']),
    LanguageSubtag::new(['r', 'u', 'o']),
    LanguageSubtag::new(['r', 'u', 'p']),
    LanguageSubtag::new(['r', 'u', 'q']),
    LanguageSubtag::new(['r', 'u', 't']),
    LanguageSubtag::new(['r', 'u', 'u']),
    LanguageSubtag::new(['r', 'u', 'y']),
    LanguageSubtag::new(['r', 'u', 'z']),
    LanguageSubtag::new(['r', 'w', ' ']),
    LanguageSubtag::new(['r', 'w', 'a']),
    LanguageSubtag::new(['r', 'w', 'k']),
    LanguageSubtag::new(['r', 'w', 'l']),
    LanguageSubtag::new(['r', 'w', 'm']),
    LanguageSubtag::new(['r', 'w', 'o']),
    LanguageSubtag::new(['r', 'w', 'r']),
    LanguageSubtag::new(['r', 'x', 'd']),
    LanguageSubtag::new(['r', 'x', 'w']),
    LanguageSubtag::new(['r', 'y', 'n']),
    LanguageSubtag::new(['r', 'y', 's']),
    LanguageSubtag::new(['r', 'y', 'u']),
    LanguageSubtag::new(['r', 'z', 'h']),
    LanguageSubtag::new(['s', 'a', ' ']),
    LanguageSubtag::new(['s', 'a', 'a']),
    LanguageSubtag::new(['s', 'a', 'b']),
    LanguageSubtag::new(['s', 'a', 'c']),
    LanguageSubtag::new(['s', 'a', 'd']),
    LanguageSubtag::new(['s', 'a', 'e']),
    LanguageSubtag::new(['s', 'a', 'f']),
    LanguageSubtag::new(['s', 'a', 'h']),
    LanguageSubtag::new(['s', 'a', 'i']),
    LanguageSubtag::new(['s', 'a', 'j']),
    LanguageSubtag::new(['s', 'a', 'k']),
    LanguageSubtag::new(['s', 'a', 'l']),
    LanguageSubtag::new(['s', 'a', 'm']),
    LanguageSubtag::new(['s', 'a', 'o']),
    LanguageSubtag::new(['s', 'a', 'p']),
    LanguageSubtag::new(['s', 'a', 'q']),
    LanguageSubtag::new(['s', 'a', 'r']),
    LanguageSubtag::new(['s', 'a', 's']),
    LanguageSubtag::new(['s', 'a', 't']),
    LanguageSubtag::new(['s', 'a', 'u']),
    LanguageSubtag::new(['s', 'a', 'v']),
    LanguageSubtag::new(['s', 'a', 'w']),
    LanguageSubtag::new(['s', 'a', 'x']),
    LanguageSubtag::new(['s', 'a', 'y']),
    LanguageSubtag::new(['s', 'a', 'z']),
    LanguageSubtag::new(['s', 'b', 'a']),
    LanguageSubtag::new(['s', 'b', 'b']),
    LanguageSubtag::new(['s', 'b', 'c']),
    LanguageSubtag::new(['s', 'b', 'd']),
    LanguageSubtag::new(['s', 'b', 'e']),
    LanguageSubtag::new(['s', 'b', 'f']),
    LanguageSubtag::new(['s', 'b', 'g']),
    LanguageSubtag::new(['s', 'b', 'h']),
    LanguageSubtag::new(['s', 'b', 'i']),
    LanguageSubtag::new(['s', 'b', 'j']),
    LanguageSubtag::new(['s', 'b', 'k']),
    LanguageSubtag::new(['s', 'b', 'l']),
    LanguageSubtag::new(['s', 'b', 'm']),
    LanguageSubtag::new(['s', 'b', 'n']),
    LanguageSubtag::new(['s', 'b', 'o']),
    LanguageSubtag::new(['s', 'b', 'p']),
    LanguageSubtag::new(['s', 'b', 'q']),
    LanguageSubtag::new(['s', 'b', 'r']),
    LanguageSubtag::new(['s', 'b', 's']),
    LanguageSubtag::new(['s', 'b', 't']),
    LanguageSubtag::new(['s', 'b', 'u']),
    LanguageSubtag::new(['s', 'b', 'v']),
    LanguageSubtag::new(['s', 'b', 'w']),
    LanguageSubtag::new(['s', 'b', 'x']),
    LanguageSubtag::new(['s', 'b', 'y']),
    LanguageSubtag::new(['s', 'b', 'z']),
    LanguageSubtag::new(['s', 'c', ' ']),
    LanguageSubtag::new(['s', 'c', 'a']),
    LanguageSubtag::new(['s', 'c', 'b']),
    LanguageSubtag::new(['s', 'c', 'e']),
    LanguageSubtag::new(['s', 'c', 'f']),
    LanguageSubtag::new(['s', 'c', 'g']),
    LanguageSubtag::new(['s', 'c', 'h']),
    LanguageSubtag::new(['s', 'c', 'i']),
    LanguageSubtag::new(['s', 'c', 'k']),
    LanguageSubtag::new(['s', 'c', 'l']),
    LanguageSubtag::new(['s', 'c', 'n']),
    LanguageSubtag::new(['s', 'c', 'o']),
    LanguageSubtag::new(['s', 'c', 'p']),
    LanguageSubtag::new(['s', 'c', 'q']),
    LanguageSubtag::new(['s', 'c', 's']),
    LanguageSubtag::new(['s', 'c', 't']),
    LanguageSubtag::new(['s', 'c', 'u']),
    LanguageSubtag::new(['s', 'c', 'v']),
    LanguageSubtag::new(['s', 'c', 'w']),
    LanguageSubtag::new(['s', 'c', 'x']),
    LanguageSubtag::new(['s', 'd', ' ']),
    LanguageSubtag::new(['s', 'd', 'a']),
    LanguageSubtag::new(['s', 'd', 'b']),
    LanguageSubtag::new(['s', 'd', 'c']),
    LanguageSubtag::new(['s', 'd', 'e']),
    LanguageSubtag::new(['s', 'd', 'f']),
    LanguageSubtag::new(['s', 'd', 'g']),
    LanguageSubtag::new(['s', 'd', 'h']),
    LanguageSubtag::new(['s', 'd', 'j']),
    LanguageSubtag::new(['s', 'd', 'k']),
    LanguageSubtag::new(['s', 'd', 'l']),
    LanguageSubtag::new(['s', 'd', 'm']),
    LanguageSubtag::new(['s', 'd', 'n']),
    LanguageSubtag::new(['s', 'd', 'o']),
    LanguageSubtag::new(['s', 'd', 'p']),
    LanguageSubtag::new(['s', 'd', 'q']),
    LanguageSubtag::new(['s', 'd', 'r']),
    LanguageSubtag::new(['s', 'd', 's']),
    LanguageSubtag::new(['s', 'd', 't']),
    LanguageSubtag::new(['s', 'd', 'u']),
    LanguageSubtag::new(['s', 'd', 'v']),
    LanguageSubtag::new(['s', 'd', 'x']),
    LanguageSubtag::new(['s', 'd', 'z']),
    LanguageSubtag::new(['s', 'e', ' ']),
    LanguageSubtag::new(['s', 'e', 'a']),
    LanguageSubtag::new(['s', 'e', 'b']),
    LanguageSubtag::new(['s', 'e', 'c']),
    LanguageSubtag::new(['s', 'e', 'd']),
    LanguageSubtag::new(['s', 'e', 'e']),
    LanguageSubtag::new(['s', 'e', 'f']),
    LanguageSubtag::new(['s', 'e', 'g']),
    LanguageSubtag::new(['s', 'e', 'h']),
    LanguageSubtag::new(['s', 'e', 'i']),
    LanguageSubtag::new(['s', 'e', 'j']),
    LanguageSubtag::new(['s', 'e', 'k']),
    LanguageSubtag::new(['s', 'e', 'l']),
    LanguageSubtag::new(['s', 'e', 'm']),
    LanguageSubtag::new(['s', 'e', 'n']),
    LanguageSubtag::new(['s', 'e', 'o']),
    LanguageSubtag::new(['s', 'e', 'p']),
    LanguageSubtag::new(['s', 'e', 'q']),
    LanguageSubtag::new(['s', 'e', 'r']),
    LanguageSubtag::new(['s', 'e', 's']),
    LanguageSubtag::new(['s', 'e', 't']),
    LanguageSubtag::new(['s', 'e', 'u']),
    LanguageSubtag::new(['s', 'e', 'v']),
    LanguageSubtag::new(['s', 'e', 'w']),
    LanguageSubtag::new(['s', 'e', 'y']),
    LanguageSubtag::new(['s', 'e', 'z']),
    LanguageSubtag::new(['s', 'f', 'b']),
    LanguageSubtag::new(['s', 'f', 'e']),
    LanguageSubtag::new(['s', 'f', 'm']),
    LanguageSubtag::new(['s', 'f', 's']),
    LanguageSubtag::new(['s', 'f', 'w']),
    LanguageSubtag::new(['s', 'g', ' ']),
    LanguageSubtag::new(['s', 'g', 'a']),
    LanguageSubtag::new(['s', 'g', 'b']),
    LanguageSubtag::new(['s', 'g', 'c']),
    LanguageSubtag::new(['s', 'g', 'd']),
    LanguageSubtag::new(['s', 'g', 'e']),
    LanguageSubtag::new(['s', 'g', 'g']),
    LanguageSubtag::new(['s', 'g', 'h']),
    LanguageSubtag::new(['s', 'g', 'i']),
    LanguageSubtag::new(['s', 'g', 'j']),
    LanguageSubtag::new(['s', 'g', 'k']),
    LanguageSubtag::new(['s', 'g', 'l']),
    LanguageSubtag::new(['s', 'g', 'm']),
    LanguageSubtag::new(['s', 'g', 'n']),
    LanguageSubtag::new(['s', 'g', 'o']),
    LanguageSubtag::new(['s', 'g', 'p']),
    LanguageSubtag::new(['s', 'g', 'r']),
    LanguageSubtag::new(['s', 'g', 's']),
    LanguageSubtag::new(['s', 'g', 't']),
    LanguageSubtag::new(['s', 'g', 'u']),
    LanguageSubtag::new(['s', 'g', 'w']),
    LanguageSubtag::new(['s', 'g', 'x']),
    LanguageSubtag::new(['s', 'g', 'y']),
    LanguageSubtag::new(['s', 'g', 'z']),
    LanguageSubtag::new(['s', 'h', ' ']),
    LanguageSubtag::new(['s', 'h', 'a']),
    LanguageSubtag::new(['s', 'h', 'b']),
    LanguageSubtag::new(['s', 'h', 'c']),
    LanguageSubtag::new(['s', 'h', 'd']),
    LanguageSubtag::new(['s', 'h', 'e']),
    LanguageSubtag::new(['s', 'h', 'g']),
    LanguageSubtag::new(['s', 'h', 'h']),
    LanguageSubtag::new(['s', 'h', 'i']),
    LanguageSubtag::new(['s', 'h', 'j']),
    LanguageSubtag::new(['s', 'h', 'k']),
    LanguageSubtag::new(['s', 'h', 'l']),
    LanguageSubtag::new(['s', 'h', 'm']),
    LanguageSubtag::new(['s', 'h', 'n']),
    LanguageSubtag::new(['s', 'h', 'o']),
    LanguageSubtag::new(['s', 'h', 'p']),
    LanguageSubtag::new(['s', 'h', 'q']),
    LanguageSubtag::new(['s', 'h', 'r']),
    LanguageSubtag::new(['s', 'h', 's']),
    LanguageSubtag::new(['s', 'h', 't']),
    LanguageSubtag::new(['s', 'h', 'u']),
    LanguageSubtag::new(['s', 'h', 'v']),
    LanguageSubtag::new(['s', 'h', 'w']),
    LanguageSubtag::new(['s', 'h', 'x']),
    LanguageSubtag::new(['s', 'h', 'y']),
    LanguageSubtag::new(['s', 'h', 'z']),
    LanguageSubtag::new(['s', 'i', ' ']),
    LanguageSubtag::new(['s', 'i', 'a']),
    LanguageSubtag::new(['s', 'i', 'b']),
    LanguageSubtag::new(['s', 'i', 'd']),
    LanguageSubtag::new(['s', 'i', 'e']),
    LanguageSubtag::new(['s', 'i', 'f']),
    LanguageSubtag::new(['s', 'i', 'g']),
    LanguageSubtag::new(['s', 'i', 'h']),
    LanguageSubtag::new(['s', 'i', 'i']),
    LanguageSubtag::new(['s', 'i', 'j']),
    LanguageSubtag::new(['s', 'i', 'k']),
    LanguageSubtag::new(['s', 'i', 'l']),
    LanguageSubtag::new(['s', 'i', 'm']),
    LanguageSubtag::new(['s', 'i', 'o']),
    LanguageSubtag::new(['s', 'i', 'p']),
    LanguageSubtag::new(['s', 'i', 'q']),
    LanguageSubtag::new(['s', 'i', 'r']),
    LanguageSubtag::new(['s', 'i', 's']),
    LanguageSubtag::new(['s', 'i', 't']),
    LanguageSubtag::new(['s', 'i', 'u']),
    LanguageSubtag::new(['s', 'i', 'v']),
    LanguageSubtag::new(['s', 'i', 'w']),
    LanguageSubtag::new(['s', 'i', 'x']),
    LanguageSubtag::new(['s', 'i', 'y']),
    LanguageSubtag::new(['s', 'i', 'z']),
    LanguageSubtag::new(['s', 'j', 'a']),
    LanguageSubtag::new(['s', 'j', 'b']),
    LanguageSubtag::new(['s', 'j', 'd']),
    LanguageSubtag::new(['s', 'j', 'e']),
    LanguageSubtag::new(['s', 'j', 'g']),
    LanguageSubtag::new(['s', 'j', 'k']),
    LanguageSubtag::new(['s', 'j', 'l']),
    LanguageSubtag::new(['s', 'j', 'm']),
    LanguageSubtag::new(['s', 'j', 'n']),
    LanguageSubtag::new(['s', 'j', 'o']),
    LanguageSubtag::new(['s', 'j', 'p']),
    LanguageSubtag::new(['s', 'j', 'r']),
    LanguageSubtag::new(['s', 'j', 's']),
    LanguageSubtag::new(['s', 'j', 't']),
    LanguageSubtag::new(['s', 'j', 'u']),
    LanguageSubtag::new(['s', 'j', 'w']),
    LanguageSubtag::new(['s', 'k', ' ']),
    LanguageSubtag::new(['s', 'k', 'a']),
    LanguageSubtag::new(['s', 'k', 'b']),
    LanguageSubtag::new(['s', 'k', 'c']),
    LanguageSubtag::new(['s', 'k', 'd']),
    LanguageSubtag::new(['s', 'k', 'e']),
    LanguageSubtag::new(['s', 'k', 'f']),
    LanguageSubtag::new(['s', 'k', 'g']),
    LanguageSubtag::new(['s', 'k', 'h']),
    LanguageSubtag::new(['s', 'k', 'i']),
    LanguageSubtag::new(['s', 'k', 'j']),
    LanguageSubtag::new(['s', 'k', 'k']),
    LanguageSubtag::new(['s', 'k', 'm']),
    LanguageSubtag::new(['s', 'k', 'n']),
    LanguageSubtag::new(['s', 'k', 'o']),
    LanguageSubtag::new(['s', 'k', 'p']),
    LanguageSubtag::new(['s', 'k', 'q']),
    LanguageSubtag::new(['s', 'k', 'r']),
    LanguageSubtag::new(['s', 'k', 's']),
    LanguageSubtag::new(['s', 'k', 't']),
    LanguageSubtag::new(['s', 'k', 'u']),
    LanguageSubtag::new(['s', 'k', 'v']),
    LanguageSubtag::new(['s', 'k', 'w']),
    LanguageSubtag::new(['s', 'k', 'x']),
    LanguageSubtag::new(['s', 'k', 'y']),
    LanguageSubtag::new(['s', 'k', 'z']),
    LanguageSubtag::new(['s', 'l', ' ']),
    LanguageSubtag::new(['s', 'l', 'a']),
    LanguageSubtag::new(['s', 'l', 'c']),
    LanguageSubtag::new(['s', 'l', 'd']),
    LanguageSubtag::new(['s', 'l', 'e']),
    LanguageSubtag::new(['s', 'l', 'f']),
    LanguageSubtag::new(['s', 'l', 'g']),
    LanguageSubtag::new(['s', 'l', 'h']),
    LanguageSubtag::new(['s', 'l', 'i']),
    LanguageSubtag::new(['s', 'l', 'j']),
    LanguageSubtag::new(['s', 'l', 'l']),
    LanguageSubtag::new(['s', 'l', 'm']),
    LanguageSubtag::new(['s', 'l', 'n']),
    LanguageSubtag::new(['s', 'l', 'p']),
    LanguageSubtag::new(['s', 'l', 'q']),
    LanguageSubtag::new(['s', 'l', 'r']),
    LanguageSubtag::new(['s', 'l', 's']),
    LanguageSubtag::new(['s', 'l', 't']),
    LanguageSubtag::new(['s', 'l', 'u']),
    LanguageSubtag::new(['s', 'l', 'w']),
    LanguageSubtag::new(['s', 'l', 'x']),
    LanguageSubtag::new(['s', 'l', 'y']),
    LanguageSubtag::new(['s', 'l', 'z']),
    LanguageSubtag::new(['s', 'm', ' ']),
    LanguageSubtag::new(['s', 'm', 'a']),
    LanguageSubtag::new(['s', 'm', 'b']),
    LanguageSubtag::new(['s', 'm', 'c']),
    LanguageSubtag::new(['s', 'm', 'd']),
    LanguageSubtag::new(['s', 'm', 'f']),
    LanguageSubtag::new(['s', 'm', 'g']),
    LanguageSubtag::new(['s', 'm', 'h']),
    LanguageSubtag::new(['s', 'm', 'i']),
    LanguageSubtag::new(['s', 'm', 'j']),
    LanguageSubtag::new(['s', 'm', 'k']),
    LanguageSubtag::new(['s', 'm', 'l']),
    LanguageSubtag::new(['s', 'm', 'm']),
    LanguageSubtag::new(['s', 'm', 'n']),
    LanguageSubtag::new(['s', 'm', 'p']),
    LanguageSubtag::new(['s', 'm', 'q']),
    LanguageSubtag::new(['s', 'm', 'r']),
    LanguageSubtag::new(['s', 'm', 's']),
    LanguageSubtag::new(['s', 'm', 't']),
    LanguageSubtag::new(['s', 'm', 'u']),
    LanguageSubtag::new(['s', 'm', 'v']),
    LanguageSubtag::new(['s', 'm', 'w']),
    LanguageSubtag::new(['s', 'm', 'x']),
    LanguageSubtag::new(['s', 'm', 'y']),
    LanguageSubtag::new(['s', 'm', 'z']),
    LanguageSubtag::new(['s', 'n', ' ']),
    LanguageSubtag::new(['s', 'n', 'b']),
    LanguageSubtag::new(['s', 'n', 'c']),
    LanguageSubtag::new(['s', 'n', 'e']),
    LanguageSubtag::new(['s', 'n', 'f']),
    LanguageSubtag::new(['s', 'n', 'g']),
    LanguageSubtag::new(['s', 'n', 'h']),
    LanguageSubtag::new(['s', 'n', 'i']),
    LanguageSubtag::new(['s', 'n', 'j']),
    LanguageSubtag::new(['s', 'n', 'k']),
    LanguageSubtag::new(['s', 'n', 'l']),
    LanguageSubtag::new(['s', 'n', 'm']),
    LanguageSubtag::new(['s', 'n', 'n']),
    LanguageSubtag::new(['s', 'n', 'o']),
    LanguageSubtag::new(['s', 'n', 'p']),
    LanguageSubtag::new(['s', 'n', 'q']),
    LanguageSubtag::new(['s', 'n', 'r']),
    LanguageSubtag::new(['s', 'n', 's']),
    LanguageSubtag::new(['s', 'n', 'u']),
    LanguageSubtag::new(['s', 'n', 'v']),
    LanguageSubtag::new(['s', 'n', 'w']),
    LanguageSubtag::new(['s', 'n', 'x']),
    LanguageSubtag::new(['s', 'n', 'y']),
    LanguageSubtag::new(['s', 'n', 'z']),
    LanguageSubtag::new(['s', 'o', ' ']),
    LanguageSubtag::new(['s', 'o', 'a']),
    LanguageSubtag::new(['s', 'o', 'b']),
    LanguageSubtag::new(['s', 'o', 'c']),
    LanguageSubtag::new(['s', 'o', 'd']),
    LanguageSubtag::new(['s', 'o', 'e']),
    LanguageSubtag::new(['s', 'o', 'g']),
    LanguageSubtag::new(['s', 'o', 'h']),
    LanguageSubtag::new(['s', 'o', 'i']),
    LanguageSubtag::new(['s', 'o', 'j']),
    LanguageSubtag::new(['s', 'o', 'k']),
    LanguageSubtag::new(['s', 'o', 'l']),
    LanguageSubtag::new(['s', 'o', 'n']),
    LanguageSubtag::new(['s', 'o', 'o']),
    LanguageSubtag::new(['s', 'o', 'p']),
    LanguageSubtag::new(['s', 'o', 'q']),
    LanguageSubtag::new(['s', 'o', 'r']),
    LanguageSubtag::new(['s', 'o', 's']),
    LanguageSubtag::new(['s', 'o', 'u']),
    LanguageSubtag::new(['s', 'o', 'v']),
    LanguageSubtag::new(['s', 'o', 'w']),
    LanguageSubtag::new(['s', 'o', 'x']),
    LanguageSubtag::new(['s', 'o', 'y']),
    LanguageSubtag::new(['s', 'o', 'z']),
    LanguageSubtag::new(['s', 'p', 'b']),
    LanguageSubtag::new(['s', 'p', 'c']),
    LanguageSubtag::new(['s', 'p', 'd']),
    LanguageSubtag::new(['s', 'p', 'e']),
    LanguageSubtag::new(['s', 'p', 'g']),
    LanguageSubtag::new(['s', 'p', 'i']),
    LanguageSubtag::new(['s', 'p', 'k']),
    LanguageSubtag::new(['s', 'p', 'l']),
    LanguageSubtag::new(['s', 'p', 'm']),
    LanguageSubtag::new(['s', 'p', 'n']),
    LanguageSubtag::new(['s', 'p', 'o']),
    LanguageSubtag::new(['s', 'p', 'p']),
    LanguageSubtag::new(['s', 'p', 'q']),
    LanguageSubtag::new(['s', 'p', 'r']),
    LanguageSubtag::new(['s', 'p', 's']),
    LanguageSubtag::new(['s', 'p', 't']),
    LanguageSubtag::new(['s', 'p', 'u']),
    LanguageSubtag::new(['s', 'p', 'v']),
    LanguageSubtag::new(['s', 'p', 'x']),
    LanguageSubtag::new(['s', 'p', 'y']),
    LanguageSubtag::new(['s', 'q', ' ']),
    LanguageSubtag::new(['s', 'q', 'a']),
    LanguageSubtag::new(['s', 'q', 'h']),
    LanguageSubtag::new(['s', 'q', 'j']),
    LanguageSubtag::new(['s', 'q', 'k']),
    LanguageSubtag::new(['s', 'q', 'm']),
    LanguageSubtag::new(['s', 'q', 'n']),
    LanguageSubtag::new(['s', 'q', 'o']),
    LanguageSubtag::new(['s', 'q', 'q']),
    LanguageSubtag::new(['s', 'q', 'r']),
    LanguageSubtag::new(['s', 'q', 's']),
    LanguageSubtag::new(['s', 'q', 't']),
    LanguageSubtag::new(['s', 'q', 'u']),
    LanguageSubtag::new(['s', 'q', 'x']),
    LanguageSubtag::new(['s', 'r', ' ']),
    LanguageSubtag::new(['s', 'r', 'a']),
    LanguageSubtag::new(['s', 'r', 'b']),
    LanguageSubtag::new(['s', 'r', 'c']),
    LanguageSubtag::new(['s', 'r', 'e']),
    LanguageSubtag::new(['s', 'r', 'f']),
    LanguageSubtag::new(['s', 'r', 'g']),
    LanguageSubtag::new(['s', 'r', 'h']),
    LanguageSubtag::new(['s', 'r', 'i']),
    LanguageSubtag::new(['s', 'r', 'k']),
    LanguageSubtag::new(['s', 'r', 'l']),
    LanguageSubtag::new(['s', 'r', 'm']),
    LanguageSubtag::new(['s', 'r', 'n']),
    LanguageSubtag::new(['s', 'r', 'o']),
    LanguageSubtag::new(['s', 'r', 'q']),
    LanguageSubtag::new(['s', 'r', 'r']),
    LanguageSubtag::new(['s', 'r', 's']),
    LanguageSubtag::new(['s', 'r', 't']),
    LanguageSubtag::new(['s', 'r', 'u']),
    LanguageSubtag::new(['s', 'r', 'v']),
    LanguageSubtag::new(['s', 'r', 'w']),
    LanguageSubtag::new(['s', 'r', 'x']),
    LanguageSubtag::new(['s', 'r', 'y']),
    LanguageSubtag::new(['s', 'r', 'z']),
    LanguageSubtag::new(['s', 's', ' ']),
    LanguageSubtag::new(['s', 's', 'a']),
    LanguageSubtag::new(['s', 's', 'b']),
    LanguageSubtag::new(['s', 's', 'c']),
    LanguageSubtag::new(['s', 's', 'd']),
    LanguageSubtag::new(['s', 's', 'e']),
    LanguageSubtag::new(['s', 's', 'f']),
    LanguageSubtag::new(['s', 's', 'g']),
    LanguageSubtag::new(['s', 's', 'h']),
    LanguageSubtag::new(['s', 's', 'i']),
    LanguageSubtag::new(['s', 's', 'j']),
    LanguageSubtag::new(['s', 's', 'k']),
    LanguageSubtag::new(['s', 's', 'l']),
    LanguageSubtag::new(['s', 's', 'm']),
    LanguageSubtag::new(['s', 's', 'n']),
    LanguageSubtag::new(['s', 's', 'o']),
    LanguageSubtag::new(['s', 's', 'p']),
    LanguageSubtag::new(['s', 's', 'q']),
    LanguageSubtag::new(['s', 's', 'r']),
    LanguageSubtag::new(['s', 's', 's']),
    LanguageSubtag::new(['s', 's', 't']),
    LanguageSubtag::new(['s', 's', 'u']),
    LanguageSubtag::new(['s', 's', 'v']),
    LanguageSubtag::new(['s', 's', 'x']),
    LanguageSubtag::new(['s', 's', 'y']),
    LanguageSubtag::new(['s', 's', 'z']),
    LanguageSubtag::new(['s', 't', ' ']),
    LanguageSubtag::new(['s', 't', 'a']),
    LanguageSubtag::new(['s', 't', 'b']),
    LanguageSubtag::new(['s', 't', 'd']),
    LanguageSubtag::new(['s', 't', 'e']),
    LanguageSubtag::new(['s', 't', 'f']),
    LanguageSubtag::new(['s', 't', 'g']),
    LanguageSubtag::new(['s', 't', 'h']),
    LanguageSubtag::new(['s', 't', 'i']),
    LanguageSubtag::new(['s', 't', 'j']),
    LanguageSubtag::new(['s', 't', 'k']),
    LanguageSubtag::new(['s', 't', 'l']),
    LanguageSubtag::new(['s', 't', 'm']),
    LanguageSubtag::new(['s', 't', 'n']),
    LanguageSubtag::new(['s', 't', 'o']),
    LanguageSubtag::new(['s', 't', 'p']),
    LanguageSubtag::new(['s', 't', 'q']),
    LanguageSubtag::new(['s', 't', 'r']),
    LanguageSubtag::new(['s', 't', 's']),
    LanguageSubtag::new(['s', 't', 't']),
    LanguageSubtag::new(['s', 't', 'u']),
    LanguageSubtag::new(['s', 't', 'v']),
    LanguageSubtag::new(['s', 't', 'w']),
    LanguageSubtag::new(['s', 't', 'y']),
    LanguageSubtag::new(['s', 'u', ' ']),
    LanguageSubtag::new(['s', 'u', 'a']),
    LanguageSubtag::new(['s', 'u', 'b']),
    LanguageSubtag::new(['s', 'u', 'c']),
    LanguageSubtag::new(['s', 'u', 'e']),
    LanguageSubtag::new(['s', 'u', 'g']),
    LanguageSubtag::new(['s', 'u', 'i']),
    LanguageSubtag::new(['s', 'u', 'j']),
    LanguageSubtag::new(['s', 'u', 'k']),
    LanguageSubtag::new(['s', 'u', 'l']),
    LanguageSubtag::new(['s', 'u', 'm']),
    LanguageSubtag::new(['s', 'u', 'o']),
    LanguageSubtag::new(['s', 'u', 'q']),
    LanguageSubtag::new(['s', 'u', 'r']),
    LanguageSubtag::new(['s', 'u', 's']),
    LanguageSubtag::new(['s', 'u', 't']),
    LanguageSubtag::new(['s', 'u', 'v']),
    LanguageSubtag::new(['s', 'u', 'w']),
    LanguageSubtag::new(['s', 'u', 'x']),
    LanguageSubtag::new(['s', 'u', 'y']),
    LanguageSubtag::new(['s', 'u', 'z']),
    LanguageSubtag::new(['s', 'v', ' ']),
    LanguageSubtag::new(['s', 'v', 'a']),
    LanguageSubtag::new(['s', 'v', 'b']),
    LanguageSubtag::new(['s', 'v', 'c']),
    LanguageSubtag::new(['s', 'v', 'e']),
    LanguageSubtag::new(['s', 'v', 'k']),
    LanguageSubtag::new(['s', 'v', 'm']),
    LanguageSubtag::new(['s', 'v', 'r']),
    LanguageSubtag::new(['s', 'v', 's']),
    LanguageSubtag::new(['s', 'v', 'x']),
    LanguageSubtag::new(['s', 'w', ' ']),
    LanguageSubtag::new(['s', 'w', 'b']),
    LanguageSubtag::new(['s', 'w', 'c']),
    LanguageSubtag::new(['s', 'w', 'f']),
    LanguageSubtag::new(['s', 'w', 'g']),
    LanguageSubtag::new(['s', 'w', 'h']),
    LanguageSubtag::new(['s', 'w', 'i']),
    LanguageSubtag::new(['s', 'w', 'j']),
    LanguageSubtag::new(['s', 'w', 'k']),
    LanguageSubtag::new(['s', 'w', 'l']),
    LanguageSubtag::new(['s', 'w', 'm']),
    LanguageSubtag::new(['s', 'w', 'n']),
    LanguageSubtag::new(['s', 'w', 'o']),
    LanguageSubtag::new(['s', 'w', 'p']),
    LanguageSubtag::new(['s', 'w', 'q']),
    LanguageSubtag::new(['s', 'w', 'r']),
    LanguageSubtag::new(['s', 'w', 's']),
    LanguageSubtag::new(['s', 'w', 't']),
    LanguageSubtag::new(['s', 'w', 'u']),
    LanguageSubtag::new(['s', 'w', 'v']),
    LanguageSubtag::new(['s', 'w', 'w']),
    LanguageSubtag::new(['s', 'w', 'x']),
    LanguageSubtag::new(['s', 'w', 'y']),
    LanguageSubtag::new(['s', 'x', 'b']),
    LanguageSubtag::new(['s', 'x', 'c']),
    LanguageSubtag::new(['s', 'x', 'e']),
    LanguageSubtag::new(['s', 'x', 'g']),
    LanguageSubtag::new(['s', 'x', 'k']),
    LanguageSubtag::new(['s', 'x', 'l']),
    LanguageSubtag::new(['s', 'x', 'm']),
    LanguageSubtag::new(['s', 'x', 'n']),
    LanguageSubtag::new(['s', 'x', 'o']),
    LanguageSubtag::new(['s', 'x', 'r']),
    LanguageSubtag::new(['s', 'x', 's']),
    LanguageSubtag::new(['s', 'x', 'u']),
    LanguageSubtag::new(['s', 'x', 'w']),
    LanguageSubtag::new(['s', 'y', 'a']),
    LanguageSubtag::new(['s', 'y', 'b']),
    LanguageSubtag::new(['s', 'y', 'c']),
    LanguageSubtag::new(['s', 'y', 'd']),
    LanguageSubtag::new(['s', 'y', 'i']),
    LanguageSubtag::new(['s', 'y', 'k']),
    LanguageSubtag::new(['s', 'y', 'l']),
    LanguageSubtag::new(['s', 'y', 'm']),
    LanguageSubtag::new(['s', 'y', 'n']),
    LanguageSubtag::new(['s', 'y', 'o']),
    LanguageSubtag::new(['s', 'y', 'r']),
    LanguageSubtag::new(['s', 'y', 's']),
    LanguageSubtag::new(['s', 'y', 'w']),
    LanguageSubtag::new(['s', 'y', 'x']),
    LanguageSubtag::new(['s', 'y', 'y']),
    LanguageSubtag::new(['s', 'z', 'a']),
    LanguageSubtag::new(['s', 'z', 'b']),
    LanguageSubtag::new(['s', 'z', 'c']),
    LanguageSubtag::new(['s', 'z', 'd']),
    LanguageSubtag::new(['s', 'z', 'e']),
    LanguageSubtag::new(['s', 'z', 'g']),
    LanguageSubtag::new(['s', 'z', 'l']),
    LanguageSubtag::new(['s', 'z', 'n']),
    LanguageSubtag::new(['s', 'z', 'p']),
    LanguageSubtag::new(['s', 'z', 's']),
    LanguageSubtag::new(['s', 'z', 'v']),
    LanguageSubtag::new(['s', 'z', 'w']),
    LanguageSubtag::new(['s', 'z', 'y']),
    LanguageSubtag::new(['t', 'a', ' ']),
    LanguageSubtag::new(['t', 'a', 'a']),
    LanguageSubtag::new(['t', 'a', 'b']),
    LanguageSubtag::new(['t', 'a', 'c']),
    LanguageSubtag::new(['t', 'a', 'd']),
    LanguageSubtag::new(['t', 'a', 'e']),
    LanguageSubtag::new(['t', 'a', 'f']),
    LanguageSubtag::new(['t', 'a', 'g']),
    LanguageSubtag::new(['t', 'a', 'i']),
    LanguageSubtag::new(['t', 'a', 'j']),
    LanguageSubtag::new(['t', 'a', 'k']),
    LanguageSubtag::new(['t', 'a', 'l']),
    LanguageSubtag::new(['t', 'a', 'n']),
    LanguageSubtag::new(['t', 'a', 'o']),
    LanguageSubtag::new(['t', 'a', 'p']),
    LanguageSubtag::new(['t', 'a', 'q']),
    LanguageSubtag::new(['t', 'a', 'r']),
    LanguageSubtag::new(['t', 'a', 's']),
    LanguageSubtag::new(['t', 'a', 'u']),
    LanguageSubtag::new(['t', 'a', 'v']),
    LanguageSubtag::new(['t', 'a', 'w']),
    LanguageSubtag::new(['t', 'a', 'x']),
    LanguageSubtag::new(['t', 'a', 'y']),
    LanguageSubtag::new(['t', 'a', 'z']),
    LanguageSubtag::new(['t', 'b', 'a']),
    LanguageSubtag::new(['t', 'b', 'b']),
    LanguageSubtag::new(['t', 'b', 'c']),
    LanguageSubtag::new(['t', 'b', 'd']),
    LanguageSubtag::new(['t', 'b', 'e']),
    LanguageSubtag::new(['t', 'b', 'f']),
    LanguageSubtag::new(['t', 'b', 'g']),
    LanguageSubtag::new(['t', 'b', 'h']),
    LanguageSubtag::new(['t', 'b', 'i']),
    LanguageSubtag::new(['t', 'b', 'j']),
    LanguageSubtag::new(['t', 'b', 'k']),
    LanguageSubtag::new(['t', 'b', 'l']),
    LanguageSubtag::new(['t', 'b', 'm']),
    LanguageSubtag::new(['t', 'b', 'n']),
    LanguageSubtag::new(['t', 'b', 'o']),
    LanguageSubtag::new(['t', 'b', 'p']),
    LanguageSubtag::new(['t', 'b', 'q']),
    LanguageSubtag::new(['t', 'b', 'r']),
    LanguageSubtag::new(['t', 'b', 's']),
    LanguageSubtag::new(['t', 'b', 't']),
    LanguageSubtag::new(['t', 'b', 'u']),
    LanguageSubtag::new(['t', 'b', 'v']),
    LanguageSubtag::new(['t', 'b', 'w']),
    LanguageSubtag::new(['t', 'b', 'x']),
    LanguageSubtag::new(['t', 'b', 'y']),
    LanguageSubtag::new(['t', 'b', 'z']),
    LanguageSubtag::new(['t', 'c', 'a']),
    LanguageSubtag::new(['t', 'c', 'b']),
    LanguageSubtag::new(['t', 'c', 'c']),
    LanguageSubtag::new(['t', 'c', 'd']),
    LanguageSubtag::new(['t', 'c', 'e']),
    LanguageSubtag::new(['t', 'c', 'f']),
    LanguageSubtag::new(['t', 'c', 'g']),
    LanguageSubtag::new(['t', 'c', 'h']),
    LanguageSubtag::new(['t', 'c', 'i']),
    LanguageSubtag::new(['t', 'c', 'k']),
    LanguageSubtag::new(['t', 'c', 'l']),
    LanguageSubtag::new(['t', 'c', 'm']),
    LanguageSubtag::new(['t', 'c', 'n']),
    LanguageSubtag::new(['t', 'c', 'o']),
    LanguageSubtag::new(['t', 'c', 'p']),
    LanguageSubtag::new(['t', 'c', 'q']),
    LanguageSubtag::new(['t', 'c', 's']),
    LanguageSubtag::new(['t', 'c', 't']),
    LanguageSubtag::new(['t', 'c', 'u']),
    LanguageSubtag::new(['t', 'c', 'w']),
    LanguageSubtag::new(['t', 'c', 'x']),
    LanguageSubtag::new(['t', 'c', 'y']),
    LanguageSubtag::new(['t', 'c', 'z']),
    LanguageSubtag::new(['t', 'd', 'a']),
    LanguageSubtag::new(['t', 'd', 'b']),
    LanguageSubtag::new(['t', 'd', 'c']),
    LanguageSubtag::new(['t', 'd', 'd']),
    LanguageSubtag::new(['t', 'd', 'e']),
    LanguageSubtag::new(['t', 'd', 'f']),
    LanguageSubtag::new(['t', 'd', 'g']),
    LanguageSubtag::new(['t', 'd', 'h']),
    LanguageSubtag::new(['t', 'd', 'i']),
    LanguageSubtag::new(['t', 'd', 'j']),
    LanguageSubtag::new(['t', 'd', 'k']),
    LanguageSubtag::new(['t', 'd', 'l']),
    LanguageSubtag::new(['t', 'd', 'm']),
    LanguageSubtag::new(['t', 'd', 'n']),
    LanguageSubtag::new(['t', 'd', 'o']),
    LanguageSubtag::new(['t', 'd', 'q']),
    LanguageSubtag::new(['t', 'd', 'r']),
    LanguageSubtag::new(['t', 'd', 's']),
    LanguageSubtag::new(['t', 'd', 't']),
    LanguageSubtag::new(['t', 'd', 'u']),
    LanguageSubtag::new(['t', 'd', 'v']),
    LanguageSubtag::new(['t', 'd', 'x']),
    LanguageSubtag::new(['t', 'd', 'y']),
    LanguageSubtag::new(['t', 'e', ' ']),
    LanguageSubtag::new(['t', 'e', 'a']),
    LanguageSubtag::new(['t', 'e', 'b']),
    LanguageSubtag::new(['t', 'e', 'c']),
    LanguageSubtag::new(['t', 'e', 'd']),
    LanguageSubtag::new(['t', 'e', 'e']),
    LanguageSubtag::new(['t', 'e', 'f']),
    LanguageSubtag::new(['t', 'e', 'g']),
    LanguageSubtag::new(['t', 'e', 'h']),
    LanguageSubtag::new(['t', 'e', 'i']),
    LanguageSubtag::new(['t', 'e', 'k']),
    LanguageSubtag::new(['t', 'e', 'm']),
    LanguageSubtag::new(['t', 'e', 'n']),
    LanguageSubtag::new(['t', 'e', 'o']),
    LanguageSubtag::new(['t', 'e', 'p']),
    LanguageSubtag::new(['t', 'e', 'q']),
    LanguageSubtag::new(['t', 'e', 'r']),
    LanguageSubtag::new(['t', 'e', 's']),
    LanguageSubtag::new(['t', 'e', 't']),
    LanguageSubtag::new(['t', 'e', 'u']),
    LanguageSubtag::new(['t', 'e', 'v']),
    LanguageSubtag::new(['t', 'e', 'w']),
    LanguageSubtag::new(['t', 'e', 'x']),
    LanguageSubtag::new(['t', 'e', 'y']),
    LanguageSubtag::new(['t', 'e', 'z']),
    LanguageSubtag::new(['t', 'f', 'i']),
    LanguageSubtag::new(['t', 'f', 'n']),
    LanguageSubtag::new(['t', 'f', 'o']),
    LanguageSubtag::new(['t', 'f', 'r']),
    LanguageSubtag::new(['t', 'f', 't']),
    LanguageSubtag::new(['t', 'g', ' ']),
    LanguageSubtag::new(['t', 'g', 'a']),
    LanguageSubtag::new(['t', 'g', 'b']),
    LanguageSubtag::new(['t', 'g', 'c']),
    LanguageSubtag::new(['t', 'g', 'd']),
    LanguageSubtag::new(['t', 'g', 'e']),
    LanguageSubtag::new(['t', 'g', 'f']),
    LanguageSubtag::new(['t', 'g', 'g']),
    LanguageSubtag::new(['t', 'g', 'h']),
    LanguageSubtag::new(['t', 'g', 'i']),
    LanguageSubtag::new(['t', 'g', 'j']),
    LanguageSubtag::new(['t', 'g', 'n']),
    LanguageSubtag::new(['t', 'g', 'o']),
    LanguageSubtag::new(['t', 'g', 'p']),
    LanguageSubtag::new(['t', 'g', 'q']),
    LanguageSubtag::new(['t', 'g', 'r']),
    LanguageSubtag::new(['t', 'g', 's']),
    LanguageSubtag::new(['t', 'g', 't']),
    LanguageSubtag::new(['t', 'g', 'u']),
    LanguageSubtag::new(['t', 'g', 'v']),
    LanguageSubtag::new(['t', 'g', 'w']),
    LanguageSubtag::new(['t', 'g', 'x']),
    LanguageSubtag::new(['t', 'g', 'y']),
    LanguageSubtag::new(['t', 'g', 'z']),
    LanguageSubtag::new(['t', 'h', ' ']),
    LanguageSubtag::new(['t', 'h', 'c']),
    LanguageSubtag::new(['t', 'h', 'd']),
    LanguageSubtag::new(['t', 'h', 'e']),
    LanguageSubtag::new(['t', 'h', 'f']),
    LanguageSubtag::new(['t', 'h', 'h']),
    LanguageSubtag::new(['t', 'h', 'i']),
    LanguageSubtag::new(['t', 'h', 'k']),
    LanguageSubtag::new(['t', 'h', 'l']),
    LanguageSubtag::new(['t', 'h', 'm']),
    LanguageSubtag::new(['t', 'h', 'n']),
    LanguageSubtag::new(['t', 'h', 'p']),
    LanguageSubtag::new(['t', 'h', 'q']),
    LanguageSubtag::new(['t', 'h', 'r']),
    LanguageSubtag::new(['t', 'h', 's']),
    LanguageSubtag::new(['t', 'h', 't']),
    LanguageSubtag::new(['t', 'h', 'u']),
    LanguageSubtag::new(['t', 'h', 'v']),
    LanguageSubtag::new(['t', 'h', 'w']),
    LanguageSubtag::new(['t', 'h', 'x']),
    LanguageSubtag::new(['t', 'h', 'y']),
    LanguageSubtag::new(['t', 'h', 'z']),
    LanguageSubtag::new(['t', 'i', ' ']),
    LanguageSubtag::new(['t', 'i', 'a']),
    LanguageSubtag::new(['t', 'i', 'c']),
    LanguageSubtag::new(['t', 'i', 'd']),
    LanguageSubtag::new(['t', 'i', 'e']),
    LanguageSubtag::new(['t', 'i', 'f']),
    LanguageSubtag::new(['t', 'i', 'g']),
    LanguageSubtag::new(['t', 'i', 'h']),
    LanguageSubtag::new(['t', 'i', 'i']),
    LanguageSubtag::new(['t', 'i', 'j']),
    LanguageSubtag::new(['t', 'i', 'k']),
    LanguageSubtag::new(['t', 'i', 'l']),
    LanguageSubtag::new(['t', 'i', 'm']),
    LanguageSubtag::new(['t', 'i', 'n']),
    LanguageSubtag::new(['t', 'i', 'o']),
    LanguageSubtag::new(['t', 'i', 'p']),
    LanguageSubtag::new(['t', 'i', 'q']),
    LanguageSubtag::new(['t', 'i', 's']),
    LanguageSubtag::new(['t', 'i', 't']),
    LanguageSubtag::new(['t', 'i', 'u']),
    LanguageSubtag::new(['t', 'i', 'v']),
    LanguageSubtag::new(['t', 'i', 'w']),
    LanguageSubtag::new(['t', 'i', 'x']),
    LanguageSubtag::new(['t', 'i', 'y']),
    LanguageSubtag::new(['t', 'i', 'z']),
    LanguageSubtag::new(['t', 'j', 'a']),
    LanguageSubtag::new(['t', 'j', 'g']),
    LanguageSubtag::new(['t', 'j', 'i']),
    LanguageSubtag::new(['t', 'j', 'j']),
    LanguageSubtag::new(['t', 'j', 'l']),
    LanguageSubtag::new(['t', 'j', 'm']),
    LanguageSubtag::new(['t', 'j', 'n']),
    LanguageSubtag::new(['t', 'j', 'o']),
    LanguageSubtag::new(['t', 'j', 'p']),
    LanguageSubtag::new(['t', 'j', 's']),
    LanguageSubtag::new(['t', 'j', 'u']),
    LanguageSubtag::new(['t', 'j', 'w']),
    LanguageSubtag::new(['t', 'k', ' ']),
    LanguageSubtag::new(['t', 'k', 'a']),
    LanguageSubtag::new(['t', 'k', 'b']),
    LanguageSubtag::new(['t', 'k', 'd']),
    LanguageSubtag::new(['t', 'k', 'e']),
    LanguageSubtag::new(['t', 'k', 'f']),
    LanguageSubtag::new(['t', 'k', 'g']),
    LanguageSubtag::new(['t', 'k', 'k']),
    LanguageSubtag::new(['t', 'k', 'l']),
    LanguageSubtag::new(['t', 'k', 'm']),
    LanguageSubtag::new(['t', 'k', 'n']),
    LanguageSubtag::new(['t', 'k', 'p']),
    LanguageSubtag::new(['t', 'k', 'q']),
    LanguageSubtag::new(['t', 'k', 'r']),
    LanguageSubtag::new(['t', 'k', 's']),
    LanguageSubtag::new(['t', 'k', 't']),
    LanguageSubtag::new(['t', 'k', 'u']),
    LanguageSubtag::new(['t', 'k', 'v']),
    LanguageSubtag::new(['t', 'k', 'w']),
    LanguageSubtag::new(['t', 'k', 'x']),
    LanguageSubtag::new(['t', 'k', 'z']),
    LanguageSubtag::new(['t', 'l', ' ']),
    LanguageSubtag::new(['t', 'l', 'a']),
    LanguageSubtag::new(['t', 'l', 'b']),
    LanguageSubtag::new(['t', 'l', 'c']),
    LanguageSubtag::new(['t', 'l', 'd']),
    LanguageSubtag::new(['t', 'l', 'f']),
    LanguageSubtag::new(['t', 'l', 'g']),
    LanguageSubtag::new(['t', 'l', 'h']),
    LanguageSubtag::new(['t', 'l', 'i']),
    LanguageSubtag::new(['t', 'l', 'j']),
    LanguageSubtag::new(['t', 'l', 'k']),
    LanguageSubtag::new(['t', 'l', 'l']),
    LanguageSubtag::new(['t', 'l', 'm']),
    LanguageSubtag::new(['t', 'l', 'n']),
    LanguageSubtag::new(['t', 'l', 'o']),
    LanguageSubtag::new(['t', 'l', 'p']),
    LanguageSubtag::new(['t', 'l', 'q']),
    LanguageSubtag::new(['t', 'l', 'r']),
    LanguageSubtag::new(['t', 'l', 's']),
    LanguageSubtag::new(['t', 'l', 't']),
    LanguageSubtag::new(['t', 'l', 'u']),
    LanguageSubtag::new(['t', 'l', 'v']),
    LanguageSubtag::new(['t', 'l', 'w']),
    LanguageSubtag::new(['t', 'l', 'x']),
    LanguageSubtag::new(['t', 'l', 'y']),
    LanguageSubtag::new(['t', 'm', 'a']),
    LanguageSubtag::new(['t', 'm', 'b']),
    LanguageSubtag::new(['t', 'm', 'c']),
    LanguageSubtag::new(['t', 'm', 'd']),
    LanguageSubtag::new(['t', 'm', 'e']),
    LanguageSubtag::new(['t', 'm', 'f']),
    LanguageSubtag::new(['t', 'm', 'g']),
    LanguageSubtag::new(['t', 'm', 'h']),
    LanguageSubtag::new(['t', 'm', 'i']),
    LanguageSubtag::new(['t', 'm', 'j']),
    LanguageSubtag::new(['t', 'm', 'k']),
    LanguageSubtag::new(['t', 'm', 'l']),
    LanguageSubtag::new(['t', 'm', 'm']),
    LanguageSubtag::new(['t', 'm', 'n']),
    LanguageSubtag::new(['t', 'm', 'o']),
    LanguageSubtag::new(['t', 'm', 'p']),
    LanguageSubtag::new(['t', 'm', 'q']),
    LanguageSubtag::new(['t', 'm', 'r']),
    LanguageSubtag::new(['t', 'm', 's']),
    LanguageSubtag::new(['t', 'm', 't']),
    LanguageSubtag::new(['t', 'm', 'u']),
    LanguageSubtag::new(['t', 'm', 'v']),
    LanguageSubtag::new(['t', 'm', 'w']),
    LanguageSubtag::new(['t', 'm', 'y']),
    LanguageSubtag::new(['t', 'm', 'z']),
    LanguageSubtag::new(['t', 'n', ' ']),
    LanguageSubtag::new(['t', 'n', 'a']),
    LanguageSubtag::new(['t', 'n', 'b']),
    LanguageSubtag::new(['t', 'n', 'c']),
    LanguageSubtag::new(['t', 'n', 'd']),
    LanguageSubtag::new(['t', 'n', 'e']),
    LanguageSubtag::new(['t', 'n', 'f']),
    LanguageSubtag::new(['t', 'n', 'g']),
    LanguageSubtag::new(['t', 'n', 'h']),
    LanguageSubtag::new(['t', 'n', 'i']),
    LanguageSubtag::new(['t', 'n', 'k']),
    LanguageSubtag::new(['t', 'n', 'l']),
    LanguageSubtag::new(['t', 'n', 'm']),
    LanguageSubtag::new(['t', 'n', 'n']),
    LanguageSubtag::new(['t', 'n', 'o']),
    LanguageSubtag::new(['t', 'n', 'p']),
    LanguageSubtag::new(['t', 'n', 'q']),
    LanguageSubtag::new(['t', 'n', 'r']),
    LanguageSubtag::new(['t', 'n', 's']),
    LanguageSubtag::new(['t', 'n', 't']),
    LanguageSubtag::new(['t', 'n', 'u']),
    LanguageSubtag::new(['t', 'n', 'v']),
    LanguageSubtag::new(['t', 'n', 'w']),
    LanguageSubtag::new(['t', 'n', 'x']),
    LanguageSubtag::new(['t', 'n', 'y']),
    LanguageSubtag::new(['t', 'n', 'z']),
    LanguageSubtag::new(['t', 'o', ' ']),
    LanguageSubtag::new(['t', 'o', 'b']),
    LanguageSubtag::new(['t', 'o', 'c']),
    LanguageSubtag::new(['t', 'o', 'd']),
    LanguageSubtag::new(['t', 'o', 'e']),
    LanguageSubtag::new(['t', 'o', 'f']),
    LanguageSubtag::new(['t', 'o', 'g']),
    LanguageSubtag::new(['t', 'o', 'h']),
    LanguageSubtag::new(['t', 'o', 'i']),
    LanguageSubtag::new(['t', 'o', 'j']),
    LanguageSubtag::new(['t', 'o', 'l']),
    LanguageSubtag::new(['t', 'o', 'm']),
    LanguageSubtag::new(['t', 'o', 'o']),
    LanguageSubtag::new(['t', 'o', 'p']),
    LanguageSubtag::new(['t', 'o', 'q']),
    LanguageSubtag::new(['t', 'o', 'r']),
    LanguageSubtag::new(['t', 'o', 's']),
    LanguageSubtag::new(['t', 'o', 'u']),
    LanguageSubtag::new(['t', 'o', 'v']),
    LanguageSubtag::new(['t', 'o', 'w']),
    LanguageSubtag::new(['t', 'o', 'x']),
    LanguageSubtag::new(['t', 'o', 'y']),
    LanguageSubtag::new(['t', 'o', 'z']),
    LanguageSubtag::new(['t', 'p', 'a']),
    LanguageSubtag::new(['t', 'p', 'c']),
    LanguageSubtag::new(['t', 'p', 'e']),
    LanguageSubtag::new(['t', 'p', 'f']),
    LanguageSubtag::new(['t', 'p', 'g']),
    LanguageSubtag::new(['t', 'p', 'i']),
    LanguageSubtag::new(['t', 'p', 'j']),
    LanguageSubtag::new(['t', 'p', 'k']),
    LanguageSubtag::new(['t', 'p', 'l']),
    LanguageSubtag::new(['t', 'p', 'm']),
    LanguageSubtag::new(['t', 'p', 'n']),
    LanguageSubtag::new(['t', 'p', 'o']),
    LanguageSubtag::new(['t', 'p', 'p']),
    LanguageSubtag::new(['t', 'p', 'q']),
    LanguageSubtag::new(['t', 'p', 'r']),
    LanguageSubtag::new(['t', 'p', 't']),
    LanguageSubtag::new(['t', 'p', 'u']),
    LanguageSubtag::new(['t', 'p', 'v']),
    LanguageSubtag::new(['t', 'p', 'w']),
    LanguageSubtag::new(['t', 'p', 'x']),
    LanguageSubtag::new(['t', 'p', 'y']),
    LanguageSubtag::new(['t', 'p', 'z']),
    LanguageSubtag::new(['t', 'q', 'b']),
    LanguageSubtag::new(['t', 'q', 'l']),
    LanguageSubtag::new(['t', 'q', 'm']),
    LanguageSubtag::new(['t', 'q', 'n']),
    LanguageSubtag::new(['t', 'q', 'o']),
    LanguageSubtag::new(['t', 'q', 'p']),
    LanguageSubtag::new(['t', 'q', 'q']),
    LanguageSubtag::new(['t', 'q', 'r']),
    LanguageSubtag::new(['t', 'q', 't']),
    LanguageSubtag::new(['t', 'q', 'u']),
    LanguageSubtag::new(['t', 'q', 'w']),
    LanguageSubtag::new(['t', 'r', ' ']),
    LanguageSubtag::new(['t', 'r', 'a']),
    LanguageSubtag::new(['t', 'r', 'b']),
    LanguageSubtag::new(['t', 'r', 'c']),
    LanguageSubtag::new(['t', 'r', 'd']),
    LanguageSubtag::new(['t', 'r', 'e']),
    LanguageSubtag::new(['t', 'r', 'f']),
    LanguageSubtag::new(['t', 'r', 'g']),
    LanguageSubtag::new(['t', 'r', 'h']),
    LanguageSubtag::new(['t', 'r', 'i']),
    LanguageSubtag::new(['t', 'r', 'j']),
    LanguageSubtag::new(['t', 'r', 'k']),
    LanguageSubtag::new(['t', 'r', 'l']),
    LanguageSubtag::new(['t', 'r', 'm']),
    LanguageSubtag::new(['t', 'r', 'n']),
    LanguageSubtag::new(['t', 'r', 'o']),
    LanguageSubtag::new(['t', 'r', 'p']),
    LanguageSubtag::new(['t', 'r', 'q']),
    LanguageSubtag::new(['t', 'r', 'r']),
    LanguageSubtag::new(['t', 'r', 's']),
    LanguageSubtag::new(['t', 'r', 't']),
    LanguageSubtag::new(['t', 'r', 'u']),
    LanguageSubtag::new(['t', 'r', 'v']),
    LanguageSubtag::new(['t', 'r', 'w']),
    LanguageSubtag::new(['t', 'r', 'x']),
    LanguageSubtag::new(['t', 'r', 'y']),
    LanguageSubtag::new(['t', 'r', 'z']),
    LanguageSubtag::new(['t', 's', ' ']),
    LanguageSubtag::new(['t', 's', 'a']),
    LanguageSubtag::new(['t', 's', 'b']),
    LanguageSubtag::new(['t', 's', 'c']),
    LanguageSubtag::new(['t', 's', 'd']),
    LanguageSubtag::new(['t', 's', 'e']),
    LanguageSubtag::new(['t', 's', 'f']),
    LanguageSubtag::new(['t', 's', 'g']),
    LanguageSubtag::new(['t', 's', 'h']),
    LanguageSubtag::new(['t', 's', 'i']),
    LanguageSubtag::new(['t', 's', 'j']),
    LanguageSubtag::new(['t', 's', 'k']),
    LanguageSubtag::new(['t', 's', 'l']),
    LanguageSubtag::new(['t', 's', 'm']),
    LanguageSubtag::new(['t', 's', 'p']),
    LanguageSubtag::new(['t', 's', 'q']),
    LanguageSubtag::new(['t', 's', 'r']),
    LanguageSubtag::new(['t', 's', 's']),
    LanguageSubtag::new(['t', 's', 't']),
    LanguageSubtag::new(['t', 's', 'u']),
    LanguageSubtag::new(['t', 's', 'v']),
    LanguageSubtag::new(['t', 's', 'w']),
    LanguageSubtag::new(['t', 's', 'x']),
    LanguageSubtag::new(['t', 's', 'y']),
    LanguageSubtag::new(['t', 's', 'z']),
    LanguageSubtag::new(['t', 't', ' ']),
    LanguageSubtag::new(['t', 't', 'a']),
    LanguageSubtag::new(['t', 't', 'b']),
    LanguageSubtag::new(['t', 't', 'c']),
    LanguageSubtag::new(['t', 't', 'd']),
    LanguageSubtag::new(['t', 't', 'e']),
    LanguageSubtag::new(['t', 't', 'f']),
    LanguageSubtag::new(['t', 't', 'g']),
    LanguageSubtag::new(['t', 't', 'h']),
    LanguageSubtag::new(['t', 't', 'i']),
    LanguageSubtag::new(['t', 't', 'j']),
    LanguageSubtag::new(['t', 't', 'k']),
    LanguageSubtag::new(['t', 't', 'l']),
    LanguageSubtag::new(['t', 't', 'm']),
    LanguageSubtag::new(['t', 't', 'n']),
    LanguageSubtag::new(['t', 't', 'o']),
    LanguageSubtag::new(['t', 't', 'p']),
    LanguageSubtag::new(['t', 't', 'q']),
    LanguageSubtag::new(['t', 't', 'r']),
    LanguageSubtag::new(['t', 't', 's']),
    LanguageSubtag::new(['t', 't', 't']),
    LanguageSubtag::new(['t', 't', 'u']),
    LanguageSubtag::new(['t', 't', 'v']),
    LanguageSubtag::new(['t', 't', 'w']),
    LanguageSubtag::new(['t', 't', 'y']),
    LanguageSubtag::new(['t', 't', 'z']),
    LanguageSubtag::new(['t', 'u', 'a']),
    LanguageSubtag::new(['t', 'u', 'b']),
    LanguageSubtag::new(['t', 'u', 'c']),
    LanguageSubtag::new(['t', 'u', 'd']),
    LanguageSubtag::new(['t', 'u', 'e']),
    LanguageSubtag::new(['t', 'u', 'f']),
    LanguageSubtag::new(['t', 'u', 'g']),
    LanguageSubtag::new(['t', 'u', 'h']),
    LanguageSubtag::new(['t', 'u', 'i']),
    LanguageSubtag::new(['t', 'u', 'j']),
    LanguageSubtag::new(['t', 'u', 'l']),
    LanguageSubtag::new(['t', 'u', 'm']),
    LanguageSubtag::new(['t', 'u', 'n']),
    LanguageSubtag::new(['t', 'u', 'o']),
    LanguageSubtag::new(['t', 'u', 'p']),
    LanguageSubtag::new(['t', 'u', 'q']),
    LanguageSubtag::new(['t', 'u', 's']),
    LanguageSubtag::new(['t', 'u', 't']),
    LanguageSubtag::new(['t', 'u', 'u']),
    LanguageSubtag::new(['t', 'u', 'v']),
    LanguageSubtag::new(['t', 'u', 'w']),
    LanguageSubtag::new(['t', 'u', 'x']),
    LanguageSubtag::new(['t', 'u', 'y']),
    LanguageSubtag::new(['t', 'u', 'z']),
    LanguageSubtag::new(['t', 'v', 'a']),
    LanguageSubtag::new(['t', 'v', 'd']),
    LanguageSubtag::new(['t', 'v', 'e']),
    LanguageSubtag::new(['t', 'v', 'k']),
    LanguageSubtag::new(['t', 'v', 'l']),
    LanguageSubtag::new(['t', 'v', 'm']),
    LanguageSubtag::new(['t', 'v', 'n']),
    LanguageSubtag::new(['t', 'v', 'o']),
    LanguageSubtag::new(['t', 'v', 's']),
    LanguageSubtag::new(['t', 'v', 't']),
    LanguageSubtag::new(['t', 'v', 'u']),
    LanguageSubtag::new(['t', 'v', 'w']),
    LanguageSubtag::new(['t', 'v', 'x']),
    LanguageSubtag::new(['t', 'v', 'y']),
    LanguageSubtag::new(['t', 'w', ' ']),
    LanguageSubtag::new(['t', 'w', 'a']),
    LanguageSubtag::new(['t', 'w', 'b']),
    LanguageSubtag::new(['t', 'w', 'c']),
    LanguageSubtag::new(['t', 'w', 'd']),
    LanguageSubtag::new(['t', 'w', 'e']),
    LanguageSubtag::new(['t', 'w', 'f']),
    LanguageSubtag::new(['t', 'w', 'g']),
    LanguageSubtag::new(['t', 'w', 'h']),
    LanguageSubtag::new(['t', 'w', 'l']),
    LanguageSubtag::new(['t', 'w', 'm']),
    LanguageSubtag::new(['t', 'w', 'n']),
    LanguageSubtag::new(['t', 'w', 'o']),
    LanguageSubtag::new(['t', 'w', 'p']),
    LanguageSubtag::new(['t', 'w', 'q']),
    LanguageSubtag::new(['t', 'w', 'r']),
    LanguageSubtag::new(['t', 'w', 't']),
    LanguageSubtag::new(['t', 'w', 'u']),
    LanguageSubtag::new(['t', 'w', 'w']),
    LanguageSubtag::new(['t', 'w', 'x']),
    LanguageSubtag::new(['t', 'w', 'y']),
    LanguageSubtag::new(['t', 'x', 'a']),
    LanguageSubtag::new(['t', 'x', 'b']),
    LanguageSubtag::new(['t', 'x', 'c']),
    LanguageSubtag::new(['t', 'x', 'e']),
    LanguageSubtag::new(['t', 'x', 'g']),
    LanguageSubtag::new(['t', 'x', 'h']),
    LanguageSubtag::new(['t', 'x', 'i']),
    LanguageSubtag::new(['t', 'x', 'j']),
    LanguageSubtag::new(['t', 'x', 'm']),
    LanguageSubtag::new(['t', 'x', 'n']),
    LanguageSubtag::new(['t', 'x', 'o']),
    LanguageSubtag::new(['t', 'x', 'q']),
    LanguageSubtag::new(['t', 'x', 'r']),
    LanguageSubtag::new(['t', 'x', 's']),
    LanguageSubtag::new(['t', 'x', 't']),
    LanguageSubtag::new(['t', 'x', 'u']),
    LanguageSubtag::new(['t', 'x', 'x']),
    LanguageSubtag::new(['t', 'x', 'y']),
    LanguageSubtag::new(['t', 'y', ' ']),
    LanguageSubtag::new(['t', 'y', 'a']),
    LanguageSubtag::new(['t', 'y', 'e']),
    LanguageSubtag::new(['t', 'y', 'h']),
    LanguageSubtag::new(['t', 'y', 'i']),
    LanguageSubtag::new(['t', 'y', 'j']),
    LanguageSubtag::new(['t', 'y', 'l']),
    LanguageSubtag::new(['t', 'y', 'n']),
    LanguageSubtag::new(['t', 'y', 'p']),
    LanguageSubtag::new(['t', 'y', 'r']),
    LanguageSubtag::new(['t', 'y', 's']),
    LanguageSubtag::new(['t', 'y', 't']),
    LanguageSubtag::new(['t', 'y', 'u']),
    LanguageSubtag::new(['t', 'y', 'v']),
    LanguageSubtag::new(['t', 'y', 'x']),
    LanguageSubtag::new(['t', 'y', 'y']),
    LanguageSubtag::new(['t', 'y', 'z']),
    LanguageSubtag::new(['t', 'z', 'a']),
    LanguageSubtag::new(['t', 'z', 'h']),
    LanguageSubtag::new(['t', 'z', 'j']),
    LanguageSubtag::new(['t', 'z', 'l']),
    LanguageSubtag::new(['t', 'z', 'm']),
    LanguageSubtag::new(['t', 'z', 'n']),
    LanguageSubtag::new(['t', 'z', 'o']),
    LanguageSubtag::new(['t', 'z', 'x']),
    LanguageSubtag::new(['u', 'a', 'm']),
    LanguageSubtag::new(['u', 'a', 'n']),
    LanguageSubtag::new(['u', 'a', 'r']),
    LanguageSubtag::new(['u', 'b', 'a']),
    LanguageSubtag::new(['u', 'b', 'i']),
    LanguageSubtag::new(['u', 'b', 'l']),
    LanguageSubtag::new(['u', 'b', 'r']),
    LanguageSubtag::new(['u', 'b', 'u']),
    LanguageSubtag::new(['u', 'b', 'y']),
    LanguageSubtag::new(['u', 'd', 'a']),
    LanguageSubtag::new(['u', 'd', 'e']),
    LanguageSubtag::new(['u', 'd', 'g']),
    LanguageSubtag::new(['u', 'd', 'i']),
    LanguageSubtag::new(['u', 'd', 'j']),
    LanguageSubtag::new(['u', 'd', 'l']),
    LanguageSubtag::new(['u', 'd', 'm']),
    LanguageSubtag::new(['u', 'd', 'u']),
    LanguageSubtag::new(['u', 'e', 's']),
    LanguageSubtag::new(['u', 'f', 'i']),
    LanguageSubtag::new(['u', 'g', ' ']),
    LanguageSubtag::new(['u', 'g', 'a']),
    LanguageSubtag::new(['u', 'g', 'b']),
    LanguageSubtag::new(['u', 'g', 'e']),
    LanguageSubtag::new(['u', 'g', 'n']),
    LanguageSubtag::new(['u', 'g', 'o']),
    LanguageSubtag::new(['u', 'g', 'y']),
    LanguageSubtag::new(['u', 'h', 'a']),
    LanguageSubtag::new(['u', 'h', 'n']),
    LanguageSubtag::new(['u', 'i', 's']),
    LanguageSubtag::new(['u', 'i', 'v']),
    LanguageSubtag::new(['u', 'j', 'i']),
    LanguageSubtag::new(['u', 'k', ' ']),
    LanguageSubtag::new(['u', 'k', 'a']),
    LanguageSubtag::new(['u', 'k', 'g']),
    LanguageSubtag::new(['u', 'k', 'h']),
    LanguageSubtag::new(['u', 'k', 'i']),
    LanguageSubtag::new(['u', 'k', 'k']),
    LanguageSubtag::new(['u', 'k', 'l']),
    LanguageSubtag::new(['u', 'k', 'p']),
    LanguageSubtag::new(['u', 'k', 'q']),
    LanguageSubtag::new(['u', 'k', 's']),
    LanguageSubtag::new(['u', 'k', 'u']),
    LanguageSubtag::new(['u', 'k', 'v']),
    LanguageSubtag::new(['u', 'k', 'w']),
    LanguageSubtag::new(['u', 'k', 'y']),
    LanguageSubtag::new(['u', 'l', 'a']),
    LanguageSubtag::new(['u', 'l', 'b']),
    LanguageSubtag::new(['u', 'l', 'c']),
    LanguageSubtag::new(['u', 'l', 'e']),
    LanguageSubtag::new(['u', 'l', 'f']),
    LanguageSubtag::new(['u', 'l', 'i']),
    LanguageSubtag::new(['u', 'l', 'k']),
    LanguageSubtag::new(['u', 'l', 'l']),
    LanguageSubtag::new(['u', 'l', 'm']),
    LanguageSubtag::new(['u', 'l', 'n']),
    LanguageSubtag::new(['u', 'l', 'u']),
    LanguageSubtag::new(['u', 'l', 'w']),
    LanguageSubtag::new(['u', 'm', 'a']),
    LanguageSubtag::new(['u', 'm', 'b']),
    LanguageSubtag::new(['u', 'm', 'c']),
    LanguageSubtag::new(['u', 'm', 'd']),
    LanguageSubtag::new(['u', 'm', 'g']),
    LanguageSubtag::new(['u', 'm', 'i']),
    LanguageSubtag::new(['u', 'm', 'm']),
    LanguageSubtag::new(['u', 'm', 'n']),
    LanguageSubtag::new(['u', 'm', 'o']),
    LanguageSubtag::new(['u', 'm', 'p']),
    LanguageSubtag::new(['u', 'm', 'r']),
    LanguageSubtag::new(['u', 'm', 's']),
    LanguageSubtag::new(['u', 'm', 'u']),
    LanguageSubtag::new(['u', 'n', 'a']),
    LanguageSubtag::new(['u', 'n', 'd']),
    LanguageSubtag::new(['u', 'n', 'e']),
    LanguageSubtag::new(['u', 'n', 'g']),
    LanguageSubtag::new(['u', 'n', 'i']),
    LanguageSubtag::new(['u', 'n', 'k']),
    LanguageSubtag::new(['u', 'n', 'm']),
    LanguageSubtag::new(['u', 'n', 'n']),
    LanguageSubtag::new(['u', 'n', 'p']),
    LanguageSubtag::new(['u', 'n', 'r']),
    LanguageSubtag::new(['u', 'n', 'u']),
    LanguageSubtag::new(['u', 'n', 'x']),
    LanguageSubtag::new(['u', 'n', 'z']),
    LanguageSubtag::new(['u', 'o', 'k']),
    LanguageSubtag::new(['u', 'p', 'i']),
    LanguageSubtag::new(['u', 'p', 'v']),
    LanguageSubtag::new(['u', 'r', ' ']),
    LanguageSubtag::new(['u', 'r', 'a']),
    LanguageSubtag::new(['u', 'r', 'b']),
    LanguageSubtag::new(['u', 'r', 'c']),
    LanguageSubtag::new(['u', 'r', 'e']),
    LanguageSubtag::new(['u', 'r', 'f']),
    LanguageSubtag::new(['u', 'r', 'g']),
    LanguageSubtag::new(['u', 'r', 'h']),
    LanguageSubtag::new(['u', 'r', 'i']),
    LanguageSubtag::new(['u', 'r', 'j']),
    LanguageSubtag::new(['u', 'r', 'k']),
    LanguageSubtag::new(['u', 'r', 'l']),
    LanguageSubtag::new(['u', 'r', 'm']),
    LanguageSubtag::new(['u', 'r', 'n']),
    LanguageSubtag::new(['u', 'r', 'o']),
    LanguageSubtag::new(['u', 'r', 'p']),
    LanguageSubtag::new(['u', 'r', 'r']),
    LanguageSubtag::new(['u', 'r', 't']),
    LanguageSubtag::new(['u', 'r', 'u']),
    LanguageSubtag::new(['u', 'r', 'v']),
    LanguageSubtag::new(['u', 'r', 'w']),
    LanguageSubtag::new(['u', 'r', 'x']),
    LanguageSubtag::new(['u', 'r', 'y']),
    LanguageSubtag::new(['u', 'r', 'z']),
    LanguageSubtag::new(['u', 's', 'a']),
    LanguageSubtag::new(['u', 's', 'h']),
    LanguageSubtag::new(['u', 's', 'i']),
    LanguageSubtag::new(['u', 's', 'k']),
    LanguageSubtag::new(['u', 's', 'p']),
    LanguageSubtag::new(['u', 's', 's']),
    LanguageSubtag::new(['u', 's', 'u']),
    LanguageSubtag::new(['u', 't', 'a']),
    LanguageSubtag::new(['u', 't', 'e']),
    LanguageSubtag::new(['u', 't', 'h']),
    LanguageSubtag::new(['u', 't', 'p']),
    LanguageSubtag::new(['u', 't', 'r']),
    LanguageSubtag::new(['u', 't', 'u']),
    LanguageSubtag::new(['u', 'u', 'm']),
    LanguageSubtag::new(['u', 'u', 'n']),
    LanguageSubtag::new(['u', 'u', 'r']),
    LanguageSubtag::new(['u', 'u', 'u']),
    LanguageSubtag::new(['u', 'v', 'e']),
    LanguageSubtag::new(['u', 'v', 'h']),
    LanguageSubtag::new(['u', 'v', 'l']),
    LanguageSubtag::new(['u', 'w', 'a']),
    LanguageSubtag::new(['u', 'y', 'a']),
    LanguageSubtag::new(['u', 'z', ' ']),
    LanguageSubtag::new(['u', 'z', 'n']),
    LanguageSubtag::new(['u', 'z', 's']),
    LanguageSubtag::new(['v', 'a', 'a']),
    LanguageSubtag::new(['v', 'a', 'e']),
    LanguageSubtag::new(['v', 'a', 'f']),
    LanguageSubtag::new(['v', 'a', 'g']),
    LanguageSubtag::new(['v', 'a', 'h']),
    LanguageSubtag::new(['v', 'a', 'i']),
    LanguageSubtag::new(['v', 'a', 'j']),
    LanguageSubtag::new(['v', 'a', 'l']),
    LanguageSubtag::new(['v', 'a', 'm']),
    LanguageSubtag::new(['v', 'a', 'n']),
    LanguageSubtag::new(['v', 'a', 'o']),
    LanguageSubtag::new(['v', 'a', 'p']),
    LanguageSubtag::new(['v', 'a', 'r']),
    LanguageSubtag::new(['v', 'a', 's']),
    LanguageSubtag::new(['v', 'a', 'u']),
    LanguageSubtag::new(['v', 'a', 'v']),
    LanguageSubtag::new(['v', 'a', 'y']),
    LanguageSubtag::new(['v', 'b', 'b']),
    LanguageSubtag::new(['v', 'b', 'k']),
    LanguageSubtag::new(['v', 'e', ' ']),
    LanguageSubtag::new(['v', 'e', 'c']),
    LanguageSubtag::new(['v', 'e', 'd']),
    LanguageSubtag::new(['v', 'e', 'l']),
    LanguageSubtag::new(['v', 'e', 'm']),
    LanguageSubtag::new(['v', 'e', 'o']),
    LanguageSubtag::new(['v', 'e', 'p']),
    LanguageSubtag::new(['v', 'e', 'r']),
    LanguageSubtag::new(['v', 'g', 'r']),
    LanguageSubtag::new(['v', 'g', 't']),
    LanguageSubtag::new(['v', 'i', ' ']),
    LanguageSubtag::new(['v', 'i', 'c']),
    LanguageSubtag::new(['v', 'i', 'd']),
    LanguageSubtag::new(['v', 'i', 'f']),
    LanguageSubtag::new(['v', 'i', 'g']),
    LanguageSubtag::new(['v', 'i', 'l']),
    LanguageSubtag::new(['v', 'i', 'n']),
    LanguageSubtag::new(['v', 'i', 's']),
    LanguageSubtag::new(['v', 'i', 't']),
    LanguageSubtag::new(['v', 'i', 'v']),
    LanguageSubtag::new(['v', 'k', 'a']),
    LanguageSubtag::new(['v', 'k', 'i']),
    LanguageSubtag::new(['v', 'k', 'j']),
    LanguageSubtag::new(['v', 'k', 'k']),
    LanguageSubtag::new(['v', 'k', 'l']),
    LanguageSubtag::new(['v', 'k', 'm']),
    LanguageSubtag::new(['v', 'k', 'n']),
    LanguageSubtag::new(['v', 'k', 'o']),
    LanguageSubtag::new(['v', 'k', 'p']),
    LanguageSubtag::new(['v', 'k', 't']),
    LanguageSubtag::new(['v', 'k', 'u']),
    LanguageSubtag::new(['v', 'k', 'z']),
    LanguageSubtag::new(['v', 'l', 'p']),
    LanguageSubtag::new(['v', 'l', 's']),
    LanguageSubtag::new(['v', 'm', 'a']),
    LanguageSubtag::new(['v', 'm', 'b']),
    LanguageSubtag::new(['v', 'm', 'c']),
    LanguageSubtag::new(['v', 'm', 'd']),
    LanguageSubtag::new(['v', 'm', 'e']),
    LanguageSubtag::new(['v', 'm', 'f']),
    LanguageSubtag::new(['v', 'm', 'g']),
    LanguageSubtag::new(['v', 'm', 'h']),
    LanguageSubtag::new(['v', 'm', 'i']),
    LanguageSubtag::new(['v', 'm', 'j']),
    LanguageSubtag::new(['v', 'm', 'k']),
    LanguageSubtag::new(['v', 'm', 'l']),
    LanguageSubtag::new(['v', 'm', 'm']),
    LanguageSubtag::new(['v', 'm', 'p']),
    LanguageSubtag::new(['v', 'm', 'q']),
    LanguageSubtag::new(['v', 'm', 'r']),
    LanguageSubtag::new(['v', 'm', 's']),
    LanguageSubtag::new(['v', 'm', 'u']),
    LanguageSubtag::new(['v', 'm', 'v']),
    LanguageSubtag::new(['v', 'm', 'w']),
    LanguageSubtag::new(['v', 'm', 'x']),
    LanguageSubtag::new(['v', 'm', 'y']),
    LanguageSubtag::new(['v', 'm', 'z']),
    LanguageSubtag::new(['v', 'n', 'k']),
    LanguageSubtag::new(['v', 'n', 'm']),
    LanguageSubtag::new(['v', 'n', 'p']),
    LanguageSubtag::new(['v', 'o', ' ']),
    LanguageSubtag::new(['v', 'o', 'r']),
    LanguageSubtag::new(['v', 'o', 't']),
    LanguageSubtag::new(['v', 'r', 'a']),
    LanguageSubtag::new(['v', 'r', 'o']),
    LanguageSubtag::new(['v', 'r', 's']),
    LanguageSubtag::new(['v', 'r', 't']),
    LanguageSubtag::new(['v', 's', 'i']),
    LanguageSubtag::new(['v', 's', 'l']),
    LanguageSubtag::new(['v', 's', 'v']),
    LanguageSubtag::new(['v', 't', 'o']),
    LanguageSubtag::new(['v', 'u', 'm']),
    LanguageSubtag::new(['v', 'u', 'n']),
    LanguageSubtag::new(['v', 'u', 't']),
    LanguageSubtag::new(['v', 'w', 'a']),
    LanguageSubtag::new(['w', 'a', ' ']),
    LanguageSubtag::new(['w', 'a', 'a']),
    LanguageSubtag::new(['w', 'a', 'b']),
    LanguageSubtag::new(['w', 'a', 'c']),
    LanguageSubtag::new(['w', 'a', 'd']),
    LanguageSubtag::new(['w', 'a', 'e']),
    LanguageSubtag::new(['w', 'a', 'f']),
    LanguageSubtag::new(['w', 'a', 'g']),
    LanguageSubtag::new(['w', 'a', 'h']),
    LanguageSubtag::new(['w', 'a', 'i']),
    LanguageSubtag::new(['w', 'a', 'j']),
    LanguageSubtag::new(['w', 'a', 'k']),
    LanguageSubtag::new(['w', 'a', 'l']),
    LanguageSubtag::new(['w', 'a', 'm']),
    LanguageSubtag::new(['w', 'a', 'n']),
    LanguageSubtag::new(['w', 'a', 'o']),
    LanguageSubtag::new(['w', 'a', 'p']),
    LanguageSubtag::new(['w', 'a', 'q']),
    LanguageSubtag::new(['w', 'a', 'r']),
    LanguageSubtag::new(['w', 'a', 's']),
    LanguageSubtag::new(['w', 'a', 't']),
    LanguageSubtag::new(['w', 'a', 'u']),
    LanguageSubtag::new(['w', 'a', 'v']),
    LanguageSubtag::new(['w', 'a', 'w']),
    LanguageSubtag::new(['w', 'a', 'x']),
    LanguageSubtag::new(['w', 'a', 'y']),
    LanguageSubtag::new(['w', 'a', 'z']),
    LanguageSubtag::new(['w', 'b', 'a']),
    LanguageSubtag::new(['w', 'b', 'b']),
    LanguageSubtag::new(['w', 'b', 'e']),
    LanguageSubtag::new(['w', 'b', 'f']),
    LanguageSubtag::new(['w', 'b', 'h']),
    LanguageSubtag::new(['w', 'b', 'i']),
    LanguageSubtag::new(['w', 'b', 'j']),
    LanguageSubtag::new(['w', 'b', 'k']),
    LanguageSubtag::new(['w', 'b', 'l']),
    LanguageSubtag::new(['w', 'b', 'm']),
    LanguageSubtag::new(['w', 'b', 'p']),
    LanguageSubtag::new(['w', 'b', 'q']),
    LanguageSubtag::new(['w', 'b', 'r']),
    LanguageSubtag::new(['w', 'b', 's']),
    LanguageSubtag::new(['w', 'b', 't']),
    LanguageSubtag::new(['w', 'b', 'v']),
    LanguageSubtag::new(['w', 'b', 'w']),
    LanguageSubtag::new(['w', 'c', 'a']),
    LanguageSubtag::new(['w', 'c', 'i']),
    LanguageSubtag::new(['w', 'd', 'd']),
    LanguageSubtag::new(['w', 'd', 'g']),
    LanguageSubtag::new(['w', 'd', 'j']),
    LanguageSubtag::new(['w', 'd', 'k']),
    LanguageSubtag::new(['w', 'd', 'u']),
    LanguageSubtag::new(['w', 'd', 'y']),
    LanguageSubtag::new(['w', 'e', 'a']),
    LanguageSubtag::new(['w', 'e', 'c']),
    LanguageSubtag::new(['w', 'e', 'd']),
    LanguageSubtag::new(['w', 'e', 'g']),
    LanguageSubtag::new(['w', 'e', 'h']),
    LanguageSubtag::new(['w', 'e', 'i']),
    LanguageSubtag::new(['w', 'e', 'm']),
    LanguageSubtag::new(['w', 'e', 'n']),
    LanguageSubtag::new(['w', 'e', 'o']),
    LanguageSubtag::new(['w', 'e', 'p']),
    LanguageSubtag::new(['w', 'e', 'r']),
    LanguageSubtag::new(['w', 'e', 's']),
    LanguageSubtag::new(['w', 'e', 't']),
    LanguageSubtag::new(['w', 'e', 'u']),
    LanguageSubtag::new(['w', 'e', 'w']),
    LanguageSubtag::new(['w', 'f', 'g']),
    LanguageSubtag::new(['w', 'g', 'a']),
    LanguageSubtag::new(['w', 'g', 'b']),
    LanguageSubtag::new(['w', 'g', 'g']),
    LanguageSubtag::new(['w', 'g', 'i']),
    LanguageSubtag::new(['w', 'g', 'o']),
    LanguageSubtag::new(['w', 'g', 'u']),
    LanguageSubtag::new(['w', 'g', 'w']),
    LanguageSubtag::new(['w', 'g', 'y']),
    LanguageSubtag::new(['w', 'h', 'a']),
    LanguageSubtag::new(['w', 'h', 'g']),
    LanguageSubtag::new(['w', 'h', 'k']),
    LanguageSubtag::new(['w', 'h', 'u']),
    LanguageSubtag::new(['w', 'i', 'b']),
    LanguageSubtag::new(['w', 'i', 'c']),
    LanguageSubtag::new(['w', 'i', 'e']),
    LanguageSubtag::new(['w', 'i', 'f']),
    LanguageSubtag::new(['w', 'i', 'g']),
    LanguageSubtag::new(['w', 'i', 'h']),
    LanguageSubtag::new(['w', 'i', 'i']),
    LanguageSubtag::new(['w', 'i', 'j']),
    LanguageSubtag::new(['w', 'i', 'k']),
    LanguageSubtag::new(['w', 'i', 'l']),
    LanguageSubtag::new(['w', 'i', 'm']),
    LanguageSubtag::new(['w', 'i', 'n']),
    LanguageSubtag::new(['w', 'i', 'r']),
    LanguageSubtag::new(['w', 'i', 't']),
    LanguageSubtag::new(['w', 'i', 'u']),
    LanguageSubtag::new(['w', 'i', 'v']),
    LanguageSubtag::new(['w', 'i', 'w']),
    LanguageSubtag::new(['w', 'i', 'y']),
    LanguageSubtag::new(['w', 'j', 'a']),
    LanguageSubtag::new(['w', 'j', 'i']),
    LanguageSubtag::new(['w', 'k', 'a']),
    LanguageSubtag::new(['w', 'k', 'b']),
    LanguageSubtag::new(['w', 'k', 'd']),
    LanguageSubtag::new(['w', 'k', 'l']),
    LanguageSubtag::new(['w', 'k', 'r']),
    LanguageSubtag::new(['w', 'k', 'u']),
    LanguageSubtag::new(['w', 'k', 'w']),
    LanguageSubtag::new(['w', 'k', 'y']),
    LanguageSubtag::new(['w', 'l', 'a']),
    LanguageSubtag::new(['w', 'l', 'c']),
    LanguageSubtag::new(['w', 'l', 'e']),
    LanguageSubtag::new(['w', 'l', 'g']),
    LanguageSubtag::new(['w', 'l', 'h']),
    LanguageSubtag::new(['w', 'l', 'i']),
    LanguageSubtag::new(['w', 'l', 'k']),
    LanguageSubtag::new(['w', 'l', 'l']),
    LanguageSubtag::new(['w', 'l', 'm']),
    LanguageSubtag::new(['w', 'l', 'o']),
    LanguageSubtag::new(['w', 'l', 'r']),
    LanguageSubtag::new(['w', 'l', 's']),
    LanguageSubtag::new(['w', 'l', 'u']),
    LanguageSubtag::new(['w', 'l', 'v']),
    LanguageSubtag::new(['w', 'l', 'w']),
    LanguageSubtag::new(['w', 'l', 'x']),
    LanguageSubtag::new(['w', 'l', 'y']),
    LanguageSubtag::new(['w', 'm', 'a']),
    LanguageSubtag::new(['w', 'm', 'b']),
    LanguageSubtag::new(['w', 'm', 'c']),
    LanguageSubtag::new(['w', 'm', 'd']),
    LanguageSubtag::new(['w', 'm', 'e']),
    LanguageSubtag::new(['w', 'm', 'g']),
    LanguageSubtag::new(['w', 'm', 'h']),
    LanguageSubtag::new(['w', 'm', 'i']),
    LanguageSubtag::new(['w', 'm', 'm']),
    LanguageSubtag::new(['w', 'm', 'n']),
    LanguageSubtag::new(['w', 'm', 'o']),
    LanguageSubtag::new(['w', 'm', 's']),
    LanguageSubtag::new(['w', 'm', 't']),
    LanguageSubtag::new(['w', 'm', 'w']),
    LanguageSubtag::new(['w', 'm', 'x']),
    LanguageSubtag::new(['w', 'n', 'b']),
    LanguageSubtag::new(['w', 'n', 'c']),
    LanguageSubtag::new(['w', 'n', 'd']),
    LanguageSubtag::new(['w', 'n', 'e']),
    LanguageSubtag::new(['w', 'n', 'g']),
    LanguageSubtag::new(['w', 'n', 'i']),
    LanguageSubtag::new(['w', 'n', 'k']),
    LanguageSubtag::new(['w', 'n', 'm']),
    LanguageSubtag::new(['w', 'n', 'n']),
    LanguageSubtag::new(['w', 'n', 'o']),
    LanguageSubtag::new(['w', 'n', 'p']),
    LanguageSubtag::new(['w', 'n', 'u']),
    LanguageSubtag::new(['w', 'n', 'w']),
    LanguageSubtag::new(['w', 'n', 'y']),
    LanguageSubtag::new(['w', 'o', ' ']),
    LanguageSubtag::new(['w', 'o', 'a']),
    LanguageSubtag::new(['w', 'o', 'b']),
    LanguageSubtag::new(['w', 'o', 'c']),
    LanguageSubtag::new(['w', 'o', 'd']),
    LanguageSubtag::new(['w', 'o', 'e']),
    LanguageSubtag::new(['w', 'o', 'f']),
    LanguageSubtag::new(['w', 'o', 'g']),
    LanguageSubtag::new(['w', 'o', 'i']),
    LanguageSubtag::new(['w', 'o', 'k']),
    LanguageSubtag::new(['w', 'o', 'm']),
    LanguageSubtag::new(['w', 'o', 'n']),
    LanguageSubtag::new(['w', 'o', 'o']),
    LanguageSubtag::new(['w', 'o', 'r']),
    LanguageSubtag::new(['w', 'o', 's']),
    LanguageSubtag::new(['w', 'o', 'w']),
    LanguageSubtag::new(['w', 'o', 'y']),
    LanguageSubtag::new(['w', 'p', 'c']),
    LanguageSubtag::new(['w', 'r', 'a']),
    LanguageSubtag::new(['w', 'r', 'b']),
    LanguageSubtag::new(['w', 'r', 'd']),
    LanguageSubtag::new(['w', 'r', 'g']),
    LanguageSubtag::new(['w', 'r', 'h']),
    LanguageSubtag::new(['w', 'r', 'i']),
    LanguageSubtag::new(['w', 'r', 'k']),
    LanguageSubtag::new(['w', 'r', 'l']),
    LanguageSubtag::new(['w', 'r', 'm']),
    LanguageSubtag::new(['w', 'r', 'n']),
    LanguageSubtag::new(['w', 'r', 'o']),
    LanguageSubtag::new(['w', 'r', 'p']),
    LanguageSubtag::new(['w', 'r', 'r']),
    LanguageSubtag::new(['w', 'r', 's']),
    LanguageSubtag::new(['w', 'r', 'u']),
    LanguageSubtag::new(['w', 'r', 'v']),
    LanguageSubtag::new(['w', 'r', 'w']),
    LanguageSubtag::new(['w', 'r', 'x']),
    LanguageSubtag::new(['w', 'r', 'y']),
    LanguageSubtag::new(['w', 'r', 'z']),
    LanguageSubtag::new(['w', 's', 'a']),
    LanguageSubtag::new(['w', 's', 'g']),
    LanguageSubtag::new(['w', 's', 'i']),
    LanguageSubtag::new(['w', 's', 'k']),
    LanguageSubtag::new(['w', 's', 'r']),
    LanguageSubtag::new(['w', 's', 's']),
    LanguageSubtag::new(['w', 's', 'u']),
    LanguageSubtag::new(['w', 's', 'v']),
    LanguageSubtag::new(['w', 't', 'f']),
    LanguageSubtag::new(['w', 't', 'h']),
    LanguageSubtag::new(['w', 't', 'i']),
    LanguageSubtag::new(['w', 't', 'k']),
    LanguageSubtag::new(['w', 't', 'm']),
    LanguageSubtag::new(['w', 't', 'w']),
    LanguageSubtag::new(['w', 'u', 'a']),
    LanguageSubtag::new(['w', 'u', 'b']),
    LanguageSubtag::new(['w', 'u', 'd']),
    LanguageSubtag::new(['w', 'u', 'h']),
    LanguageSubtag::new(['w', 'u', 'l']),
    LanguageSubtag::new(['w', 'u', 'm']),
    LanguageSubtag::new(['w', 'u', 'n']),
    LanguageSubtag::new(['w', 'u', 'r']),
    LanguageSubtag::new(['w', 'u', 't']),
    LanguageSubtag::new(['w', 'u', 'u']),
    LanguageSubtag::new(['w', 'u', 'v']),
    LanguageSubtag::new(['w', 'u', 'x']),
    LanguageSubtag::new(['w', 'u', 'y']),
    LanguageSubtag::new(['w', 'w', 'a']),
    LanguageSubtag::new(['w', 'w', 'b']),
    LanguageSubtag::new(['w', 'w', 'o']),
    LanguageSubtag::new(['w', 'w', 'r']),
    LanguageSubtag::new(['w', 'w', 'w']),
    LanguageSubtag::new(['w', 'x', 'a']),
    LanguageSubtag::new(['w', 'x', 'w']),
    LanguageSubtag::new(['w', 'y', 'a']),
    LanguageSubtag::new(['w', 'y', 'b']),
    LanguageSubtag::new(['w', 'y', 'i']),
    LanguageSubtag::new(['w', 'y', 'm']),
    LanguageSubtag::new(['w', 'y', 'r']),
    LanguageSubtag::new(['w', 'y', 'y']),
    LanguageSubtag::new(['x', 'a', 'a']),
    LanguageSubtag::new(['x', 'a', 'b']),
    LanguageSubtag::new(['x', 'a', 'c']),
    LanguageSubtag::new(['x', 'a', 'd']),
    LanguageSubtag::new(['x', 'a', 'e']),
    LanguageSubtag::new(['x', 'a', 'g']),
    LanguageSubtag::new(['x', 'a', 'i']),
    LanguageSubtag::new(['x', 'a', 'j']),
    LanguageSubtag::new(['x', 'a', 'k']),
    LanguageSubtag::new(['x', 'a', 'l']),
    LanguageSubtag::new(['x', 'a', 'm']),
    LanguageSubtag::new(['x', 'a', 'n']),
    LanguageSubtag::new(['x', 'a', 'o']),
    LanguageSubtag::new(['x', 'a', 'p']),
    LanguageSubtag::new(['x', 'a', 'q']),
    LanguageSubtag::new(['x', 'a', 'r']),
    LanguageSubtag::new(['x', 'a', 's']),
    LanguageSubtag::new(['x', 'a', 't']),
    LanguageSubtag::new(['x', 'a', 'u']),
    LanguageSubtag::new(['x', 'a', 'v']),
    LanguageSubtag::new(['x', 'a', 'w']),
    LanguageSubtag::new(['x', 'a', 'y']),
    LanguageSubtag::new(['x', 'b', 'a']),
    LanguageSubtag::new(['x', 'b', 'b']),
    LanguageSubtag::new(['x', 'b', 'c']),
    LanguageSubtag::new(['x', 'b', 'd']),
    LanguageSubtag::new(['x', 'b', 'e']),
    LanguageSubtag::new(['x', 'b', 'g']),
    LanguageSubtag::new(['x', 'b', 'i']),
    LanguageSubtag::new(['x', 'b', 'j']),
    LanguageSubtag::new(['x', 'b', 'm']),
    LanguageSubtag::new(['x', 'b', 'n']),
    LanguageSubtag::new(['x', 'b', 'o']),
    LanguageSubtag::new(['x', 'b', 'p']),
    LanguageSubtag::new(['x', 'b', 'r']),
    LanguageSubtag::new(['x', 'b', 'w']),
    LanguageSubtag::new(['x', 'b', 'x']),
    LanguageSubtag::new(['x', 'b', 'y']),
    LanguageSubtag::new(['x', 'c', 'b']),
    LanguageSubtag::new(['x', 'c', 'c']),
    LanguageSubtag::new(['x', 'c', 'e']),
    LanguageSubtag::new(['x', 'c', 'g']),
    LanguageSubtag::new(['x', 'c', 'h']),
    LanguageSubtag::new(['x', 'c', 'l']),
    LanguageSubtag::new(['x', 'c', 'm']),
    LanguageSubtag::new(['x', 'c', 'n']),
    LanguageSubtag::new(['x', 'c', 'o']),
    LanguageSubtag::new(['x', 'c', 'r']),
    LanguageSubtag::new(['x', 'c', 't']),
    LanguageSubtag::new(['x', 'c', 'u']),
    LanguageSubtag::new(['x', 'c', 'v']),
    LanguageSubtag::new(['x', 'c', 'w']),
    LanguageSubtag::new(['x', 'c', 'y']),
    LanguageSubtag::new(['x', 'd', 'a']),
    LanguageSubtag::new(['x', 'd', 'c']),
    LanguageSubtag::new(['x', 'd', 'k']),
    LanguageSubtag::new(['x', 'd', 'm']),
    LanguageSubtag::new(['x', 'd', 'o']),
    LanguageSubtag::new(['x', 'd', 'y']),
    LanguageSubtag::new(['x', 'e', 'b']),
    LanguageSubtag::new(['x', 'e', 'd']),
    LanguageSubtag::new(['x', 'e', 'g']),
    LanguageSubtag::new(['x', 'e', 'l']),
    LanguageSubtag::new(['x', 'e', 'm']),
    LanguageSubtag::new(['x', 'e', 'p']),
    LanguageSubtag::new(['x', 'e', 'r']),
    LanguageSubtag::new(['x', 'e', 's']),
    LanguageSubtag::new(['x', 'e', 't']),
    LanguageSubtag::new(['x', 'e', 'u']),
    LanguageSubtag::new(['x', 'f', 'a']),
    LanguageSubtag::new(['x', 'g', 'a']),
    LanguageSubtag::new(['x', 'g', 'b']),
    LanguageSubtag::new(['x', 'g', 'd']),
    LanguageSubtag::new(['x', 'g', 'f']),
    LanguageSubtag::new(['x', 'g', 'g']),
    LanguageSubtag::new(['x', 'g', 'i']),
    LanguageSubtag::new(['x', 'g', 'l']),
    LanguageSubtag::new(['x', 'g', 'm']),
    LanguageSubtag::new(['x', 'g', 'n']),
    LanguageSubtag::new(['x', 'g', 'r']),
    LanguageSubtag::new(['x', 'g', 'u']),
    LanguageSubtag::new(['x', 'g', 'w']),
    LanguageSubtag::new(['x', 'h', ' ']),
    LanguageSubtag::new(['x', 'h', 'a']),
    LanguageSubtag::new(['x', 'h', 'c']),
    LanguageSubtag::new(['x', 'h', 'd']),
    LanguageSubtag::new(['x', 'h', 'e']),
    LanguageSubtag::new(['x', 'h', 'r']),
    LanguageSubtag::new(['x', 'h', 't']),
    LanguageSubtag::new(['x', 'h', 'u']),
    LanguageSubtag::new(['x', 'h', 'v']),
    LanguageSubtag::new(['x', 'i', 'a']),
    LanguageSubtag::new(['x', 'i', 'b']),
    LanguageSubtag::new(['x', 'i', 'i']),
    LanguageSubtag::new(['x', 'i', 'l']),
    LanguageSubtag::new(['x', 'i', 'n']),
    LanguageSubtag::new(['x', 'i', 'p']),
    LanguageSubtag::new(['x', 'i', 'r']),
    LanguageSubtag::new(['x', 'i', 's']),
    LanguageSubtag::new(['x', 'i', 'v']),
    LanguageSubtag::new(['x', 'i', 'y']),
    LanguageSubtag::new(['x', 'j', 'b']),
    LanguageSubtag::new(['x', 'j', 't']),
    LanguageSubtag::new(['x', 'k', 'a']),
    LanguageSubtag::new(['x', 'k', 'b']),
    LanguageSubtag::new(['x', 'k', 'c']),
    LanguageSubtag::new(['x', 'k', 'd']),
    LanguageSubtag::new(['x', 'k', 'e']),
    LanguageSubtag::new(['x', 'k', 'f']),
    LanguageSubtag::new(['x', 'k', 'g']),
    LanguageSubtag::new(['x', 'k', 'h']),
    LanguageSubtag::new(['x', 'k', 'i']),
    LanguageSubtag::new(['x', 'k', 'j']),
    LanguageSubtag::new(['x', 'k', 'k']),
    LanguageSubtag::new(['x', 'k', 'l']),
    LanguageSubtag::new(['x', 'k', 'n']),
    LanguageSubtag::new(['x', 'k', 'o']),
    LanguageSubtag::new(['x', 'k', 'p']),
    LanguageSubtag::new(['x', 'k', 'q']),
    LanguageSubtag::new(['x', 'k', 'r']),
    LanguageSubtag::new(['x', 'k', 's']),
    LanguageSubtag::new(['x', 'k', 't']),
    LanguageSubtag::new(['x', 'k', 'u']),
    LanguageSubtag::new(['x', 'k', 'v']),
    LanguageSubtag::new(['x', 'k', 'w']),
    LanguageSubtag::new(['x', 'k', 'x']),
    LanguageSubtag::new(['x', 'k', 'y']),
    LanguageSubtag::new(['x', 'k', 'z']),
    LanguageSubtag::new(['x', 'l', 'a']),
    LanguageSubtag::new(['x', 'l', 'b']),
    LanguageSubtag::new(['x', 'l', 'c']),
    LanguageSubtag::new(['x', 'l', 'd']),
    LanguageSubtag::new(['x', 'l', 'e']),
    LanguageSubtag::new(['x', 'l', 'g']),
    LanguageSubtag::new(['x', 'l', 'i']),
    LanguageSubtag::new(['x', 'l', 'n']),
    LanguageSubtag::new(['x', 'l', 'o']),
    LanguageSubtag::new(['x', 'l', 'p']),
    LanguageSubtag::new(['x', 'l', 's']),
    LanguageSubtag::new(['x', 'l', 'u']),
    LanguageSubtag::new(['x', 'l', 'y']),
    LanguageSubtag::new(['x', 'm', 'a']),
    LanguageSubtag::new(['x', 'm', 'b']),
    LanguageSubtag::new(['x', 'm', 'c']),
    LanguageSubtag::new(['x', 'm', 'd']),
    LanguageSubtag::new(['x', 'm', 'e']),
    LanguageSubtag::new(['x', 'm', 'f']),
    LanguageSubtag::new(['x', 'm', 'g']),
    LanguageSubtag::new(['x', 'm', 'h']),
    LanguageSubtag::new(['x', 'm', 'j']),
    LanguageSubtag::new(['x', 'm', 'k']),
    LanguageSubtag::new(['x', 'm', 'l']),
    LanguageSubtag::new(['x', 'm', 'm']),
    LanguageSubtag::new(['x', 'm', 'n']),
    LanguageSubtag::new(['x', 'm', 'o']),
    LanguageSubtag::new(['x', 'm', 'p']),
    LanguageSubtag::new(['x', 'm', 'q']),
    LanguageSubtag::new(['x', 'm', 'r']),
    LanguageSubtag::new(['x', 'm', 's']),
    LanguageSubtag::new(['x', 'm', 't']),
    LanguageSubtag::new(['x', 'm', 'u']),
    LanguageSubtag::new(['x', 'm', 'v']),
    LanguageSubtag::new(['x', 'm', 'w']),
    LanguageSubtag::new(['x', 'm', 'x']),
    LanguageSubtag::new(['x', 'm', 'y']),
    LanguageSubtag::new(['x', 'm', 'z']),
    LanguageSubtag::new(['x', 'n', 'a']),
    LanguageSubtag::new(['x', 'n', 'b']),
    LanguageSubtag::new(['x', 'n', 'd']),
    LanguageSubtag::new(['x', 'n', 'g']),
    LanguageSubtag::new(['x', 'n', 'h']),
    LanguageSubtag::new(['x', 'n', 'i']),
    LanguageSubtag::new(['x', 'n', 'j']),
    LanguageSubtag::new(['x', 'n', 'k']),
    LanguageSubtag::new(['x', 'n', 'm']),
    LanguageSubtag::new(['x', 'n', 'n']),
    LanguageSubtag::new(['x', 'n', 'o']),
    LanguageSubtag::new(['x', 'n', 'q']),
    LanguageSubtag::new(['x', 'n', 'r']),
    LanguageSubtag::new(['x', 'n', 's']),
    LanguageSubtag::new(['x', 'n', 't']),
    LanguageSubtag::new(['x', 'n', 'u']),
    LanguageSubtag::new(['x', 'n', 'y']),
    LanguageSubtag::new(['x', 'n', 'z']),
    LanguageSubtag::new(['x', 'o', 'c']),
    LanguageSubtag::new(['x', 'o', 'd']),
    LanguageSubtag::new(['x', 'o', 'g']),
    LanguageSubtag::new(['x', 'o', 'i']),
    LanguageSubtag::new(['x', 'o', 'k']),
    LanguageSubtag::new(['x', 'o', 'm']),
    LanguageSubtag::new(['x', 'o', 'n']),
    LanguageSubtag::new(['x', 'o', 'o']),
    LanguageSubtag::new(['x', 'o', 'p']),
    LanguageSubtag::new(['x', 'o', 'r']),
    LanguageSubtag::new(['x', 'o', 'w']),
    LanguageSubtag::new(['x', 'p', 'a']),
    LanguageSubtag::new(['x', 'p', 'b']),
    LanguageSubtag::new(['x', 'p', 'c']),
    LanguageSubtag::new(['x', 'p', 'd']),
    LanguageSubtag::new(['x', 'p', 'e']),
    LanguageSubtag::new(['x', 'p', 'f']),
    LanguageSubtag::new(['x', 'p', 'g']),
    LanguageSubtag::new(['x', 'p', 'h']),
    LanguageSubtag::new(['x', 'p', 'i']),
    LanguageSubtag::new(['x', 'p', 'j']),
    LanguageSubtag::new(['x', 'p', 'k']),
    LanguageSubtag::new(['x', 'p', 'l']),
    LanguageSubtag::new(['x', 'p', 'm']),
    LanguageSubtag::new(['x', 'p', 'n']),
    LanguageSubtag::new(['x', 'p', 'o']),
    LanguageSubtag::new(['x', 'p', 'p']),
    LanguageSubtag::new(['x', 'p', 'q']),
    LanguageSubtag::new(['x', 'p', 'r']),
    LanguageSubtag::new(['x', 'p', 's']),
    LanguageSubtag::new(['x', 'p', 't']),
    LanguageSubtag::new(['x', 'p', 'u']),
    LanguageSubtag::new(['x', 'p', 'v']),
    LanguageSubtag::new(['x', 'p', 'w']),
    LanguageSubtag::new(['x', 'p', 'x']),
    LanguageSubtag::new(['x', 'p', 'y']),
    LanguageSubtag::new(['x', 'p', 'z']),
    LanguageSubtag::new(['x', 'q', 'a']),
    LanguageSubtag::new(['x', 'q', 't']),
    LanguageSubtag::new(['x', 'r', 'a']),
    LanguageSubtag::new(['x', 'r', 'b']),
    LanguageSubtag::new(['x', 'r', 'd']),
    LanguageSubtag::new(['x', 'r', 'e']),
    LanguageSubtag::new(['x', 'r', 'g']),
    LanguageSubtag::new(['x', 'r', 'i']),
    LanguageSubtag::new(['x', 'r', 'm']),
    LanguageSubtag::new(['x', 'r', 'n']),
    LanguageSubtag::new(['x', 'r', 'q']),
    LanguageSubtag::new(['x', 'r', 'r']),
    LanguageSubtag::new(['x', 'r', 't']),
    LanguageSubtag::new(['x', 'r', 'u']),
    LanguageSubtag::new(['x', 'r', 'w']),
    LanguageSubtag::new(['x', 's', 'a']),
    LanguageSubtag::new(['x', 's', 'b']),
    LanguageSubtag::new(['x', 's', 'c']),
    LanguageSubtag::new(['x', 's', 'd']),
    LanguageSubtag::new(['x', 's', 'e']),
    LanguageSubtag::new(['x', 's', 'h']),
    LanguageSubtag::new(['x', 's', 'i']),
    LanguageSubtag::new(['x', 's', 'j']),
    LanguageSubtag::new(['x', 's', 'l']),
    LanguageSubtag::new(['x', 's', 'm']),
    LanguageSubtag::new(['x', 's', 'n']),
    LanguageSubtag::new(['x', 's', 'o']),
    LanguageSubtag::new(['x', 's', 'p']),
    LanguageSubtag::new(['x', 's', 'q']),
    LanguageSubtag::new(['x', 's', 'r']),
    LanguageSubtag::new(['x', 's', 's']),
    LanguageSubtag::new(['x', 's', 'u']),
    LanguageSubtag::new(['x', 's', 'v']),
    LanguageSubtag::new(['x', 's', 'y']),
    LanguageSubtag::new(['x', 't', 'a']),
    LanguageSubtag::new(['x', 't', 'b']),
    LanguageSubtag::new(['x', 't', 'c']),
    LanguageSubtag::new(['x', 't', 'd']),
    LanguageSubtag::new(['x', 't', 'e']),
    LanguageSubtag::new(['x', 't', 'g']),
    LanguageSubtag::new(['x', 't', 'h']),
    LanguageSubtag::new(['x', 't', 'i']),
    LanguageSubtag::new(['x', 't', 'j']),
    LanguageSubtag::new(['x', 't', 'l']),
    LanguageSubtag::new(['x', 't', 'm']),
    LanguageSubtag::new(['x', 't', 'n']),
    LanguageSubtag::new(['x', 't', 'o']),
    LanguageSubtag::new(['x', 't', 'p']),
    LanguageSubtag::new(['x', 't', 'q']),
    LanguageSubtag::new(['x', 't', 'r']),
    LanguageSubtag::new(['x', 't', 's']),
    LanguageSubtag::new(['x', 't', 't']),
    LanguageSubtag::new(['x', 't', 'u']),
    LanguageSubtag::new(['x', 't', 'v']),
    LanguageSubtag::new(['x', 't', 'w']),
    LanguageSubtag::new(['x', 't', 'y']),
    LanguageSubtag::new(['x', 't', 'z']),
    LanguageSubtag::new(['x', 'u', 'a']),
    LanguageSubtag::new(['x', 'u', 'b']),
    LanguageSubtag::new(['x', 'u', 'd']),
    LanguageSubtag::new(['x', 'u', 'g']),
    LanguageSubtag::new(['x', 'u', 'j']),
    LanguageSubtag::new(['x', 'u', 'l']),
    LanguageSubtag::new(['x', 'u', 'm']),
    LanguageSubtag::new(['x', 'u', 'n']),
    LanguageSubtag::new(['x', 'u', 'o']),
    LanguageSubtag::new(['x', 'u', 'p']),
    LanguageSubtag::new(['x', 'u', 'r']),
    LanguageSubtag::new(['x', 'u', 't']),
    LanguageSubtag::new(['x', 'u', 'u']),
    LanguageSubtag::new(['x', 'v', 'e']),
    LanguageSubtag::new(['x', 'v', 'i']),
    LanguageSubtag::new(['x', 'v', 'n']),
    LanguageSubtag::new(['x', 'v', 'o']),
    LanguageSubtag::new(['x', 'v', 's']),
    LanguageSubtag::new(['x', 'w', 'a']),
    LanguageSubtag::new(['x', 'w', 'c']),
    LanguageSubtag::new(['x', 'w', 'd']),
    LanguageSubtag::new(['x', 'w', 'e']),
    LanguageSubtag::new(['x', 'w', 'g']),
    LanguageSubtag::new(['x', 'w', 'j']),
    LanguageSubtag::new(['x', 'w', 'k']),
    LanguageSubtag::new(['x', 'w', 'l']),
    LanguageSubtag::new(['x', 'w', 'o']),
    LanguageSubtag::new(['x', 'w', 'r']),
    LanguageSubtag::new(['x', 'w', 't']),
    LanguageSubtag::new(['x', 'w', 'w']),
    LanguageSubtag::new(['x', 'x', 'b']),
    LanguageSubtag::new(['x', 'x', 'k']),
    LanguageSubtag::new(['x', 'x', 'm']),
    LanguageSubtag::new(['x', 'x', 'r']),
    LanguageSubtag::new(['x', 'x', 't']),
    LanguageSubtag::new(['x', 'y', 'a']),
    LanguageSubtag::new(['x', 'y', 'b']),
    LanguageSubtag::new(['x', 'y', 'j']),
    LanguageSubtag::new(['x', 'y', 'k']),
    LanguageSubtag::new(['x', 'y', 'l']),
    LanguageSubtag::new(['x', 'y', 't']),
    LanguageSubtag::new(['x', 'y', 'y']),
    LanguageSubtag::new(['x', 'z', 'h']),
    LanguageSubtag::new(['x', 'z', 'm']),
    LanguageSubtag::new(['x', 'z', 'p']),
    LanguageSubtag::new(['y', 'a', 'a']),
    LanguageSubtag::new(['y', 'a', 'b']),
    LanguageSubtag::new(['y', 'a', 'c']),
    LanguageSubtag::new(['y', 'a', 'd']),
    LanguageSubtag::new(['y', 'a', 'e']),
    LanguageSubtag::new(['y', 'a', 'f']),
    LanguageSubtag::new(['y', 'a', 'g']),
    LanguageSubtag::new(['y', 'a', 'h']),
    LanguageSubtag::new(['y', 'a', 'i']),
    LanguageSubtag::new(['y', 'a', 'j']),
    LanguageSubtag::new(['y', 'a', 'k']),
    LanguageSubtag::new(['y', 'a', 'l']),
    LanguageSubtag::new(['y', 'a', 'm']),
    LanguageSubtag::new(['y', 'a', 'n']),
    LanguageSubtag::new(['y', 'a', 'o']),
    LanguageSubtag::new(['y', 'a', 'p']),
    LanguageSubtag::new(['y', 'a', 'q']),
    LanguageSubtag::new(['y', 'a', 'r']),
    LanguageSubtag::new(['y', 'a', 's']),
    LanguageSubtag::new(['y', 'a', 't']),
    LanguageSubtag::new(['y', 'a', 'u']),
    LanguageSubtag::new(['y', 'a', 'v']),
    LanguageSubtag::new(['y', 'a', 'w']),
    LanguageSubtag::new(['y', 'a', 'x']),
    LanguageSubtag::new(['y', 'a', 'y']),
    LanguageSubtag::new(['y', 'a', 'z']),
    LanguageSubtag::new(['y', 'b', 'a']),
    LanguageSubtag::new(['y', 'b', 'b']),
    LanguageSubtag::new(['y', 'b', 'd']),
    LanguageSubtag::new(['y', 'b', 'e']),
    LanguageSubtag::new(['y', 'b', 'h']),
    LanguageSubtag::new(['y', 'b', 'i']),
    LanguageSubtag::new(['y', 'b', 'j']),
    LanguageSubtag::new(['y', 'b', 'k']),
    LanguageSubtag::new(['y', 'b', 'l']),
    LanguageSubtag::new(['y', 'b', 'm']),
    LanguageSubtag::new(['y', 'b', 'n']),
    LanguageSubtag::new(['y', 'b', 'o']),
    LanguageSubtag::new(['y', 'b', 'x']),
    LanguageSubtag::new(['y', 'b', 'y']),
    LanguageSubtag::new(['y', 'c', 'h']),
    LanguageSubtag::new(['y', 'c', 'l']),
    LanguageSubtag::new(['y', 'c', 'n']),
    LanguageSubtag::new(['y', 'c', 'p']),
    LanguageSubtag::new(['y', 'd', 'a']),
    LanguageSubtag::new(['y', 'd', 'd']),
    LanguageSubtag::new(['y', 'd', 'e']),
    LanguageSubtag::new(['y', 'd', 'g']),
    LanguageSubtag::new(['y', 'd', 'k']),
    LanguageSubtag::new(['y', 'd', 's']),
    LanguageSubtag::new(['y', 'e', 'a']),
    LanguageSubtag::new(['y', 'e', 'c']),
    LanguageSubtag::new(['y', 'e', 'e']),
    LanguageSubtag::new(['y', 'e', 'i']),
    LanguageSubtag::new(['y', 'e', 'j']),
    LanguageSubtag::new(['y', 'e', 'l']),
    LanguageSubtag::new(['y', 'e', 'n']),
    LanguageSubtag::new(['y', 'e', 'r']),
    LanguageSubtag::new(['y', 'e', 's']),
    LanguageSubtag::new(['y', 'e', 't']),
    LanguageSubtag::new(['y', 'e', 'u']),
    LanguageSubtag::new(['y', 'e', 'v']),
    LanguageSubtag::new(['y', 'e', 'y']),
    LanguageSubtag::new(['y', 'g', 'a']),
    LanguageSubtag::new(['y', 'g', 'i']),
    LanguageSubtag::new(['y', 'g', 'l']),
    LanguageSubtag::new(['y', 'g', 'm']),
    LanguageSubtag::new(['y', 'g', 'p']),
    LanguageSubtag::new(['y', 'g', 'r']),
    LanguageSubtag::new(['y', 'g', 's']),
    LanguageSubtag::new(['y', 'g', 'u']),
    LanguageSubtag::new(['y', 'g', 'w']),
    LanguageSubtag::new(['y', 'h', 'a']),
    LanguageSubtag::new(['y', 'h', 'd']),
    LanguageSubtag::new(['y', 'h', 'l']),
    LanguageSubtag::new(['y', 'h', 's']),
    LanguageSubtag::new(['y', 'i', ' ']),
    LanguageSubtag::new(['y', 'i', 'a']),
    LanguageSubtag::new(['y', 'i', 'f']),
    LanguageSubtag::new(['y', 'i', 'g']),
    LanguageSubtag::new(['y', 'i', 'h']),
    LanguageSubtag::new(['y', 'i', 'i']),
    LanguageSubtag::new(['y', 'i', 'j']),
    LanguageSubtag::new(['y', 'i', 'k']),
    LanguageSubtag::new(['y', 'i', 'l']),
    LanguageSubtag::new(['y', 'i', 'm']),
    LanguageSubtag::new(['y', 'i', 'n']),
    LanguageSubtag::new(['y', 'i', 'p']),
    LanguageSubtag::new(['y', 'i', 'q']),
    LanguageSubtag::new(['y', 'i', 'r']),
    LanguageSubtag::new(['y', 'i', 's']),
    LanguageSubtag::new(['y', 'i', 't']),
    LanguageSubtag::new(['y', 'i', 'u']),
    LanguageSubtag::new(['y', 'i', 'v']),
    LanguageSubtag::new(['y', 'i', 'x']),
    LanguageSubtag::new(['y', 'i', 'y']),
    LanguageSubtag::new(['y', 'i', 'z']),
    LanguageSubtag::new(['y', 'k', 'a']),
    LanguageSubtag::new(['y', 'k', 'g']),
    LanguageSubtag::new(['y', 'k', 'i']),
    LanguageSubtag::new(['y', 'k', 'k']),
    LanguageSubtag::new(['y', 'k', 'l']),
    LanguageSubtag::new(['y', 'k', 'm']),
    LanguageSubtag::new(['y', 'k', 'n']),
    LanguageSubtag::new(['y', 'k', 'o']),
    LanguageSubtag::new(['y', 'k', 'r']),
    LanguageSubtag::new(['y', 'k', 't']),
    LanguageSubtag::new(['y', 'k', 'u']),
    LanguageSubtag::new(['y', 'k', 'y']),
    LanguageSubtag::new(['y', 'l', 'a']),
    LanguageSubtag::new(['y', 'l', 'b']),
    LanguageSubtag::new(['y', 'l', 'e']),
    LanguageSubtag::new(['y', 'l', 'g']),
    LanguageSubtag::new(['y', 'l', 'i']),
    LanguageSubtag::new(['y', 'l', 'l']),
    LanguageSubtag::new(['y', 'l', 'm']),
    LanguageSubtag::new(['y', 'l', 'n']),
    LanguageSubtag::new(['y', 'l', 'o']),
    LanguageSubtag::new(['y', 'l', 'r']),
    LanguageSubtag::new(['y', 'l', 'u']),
    LanguageSubtag::new(['y', 'l', 'y']),
    LanguageSubtag::new(['y', 'm', 'a']),
    LanguageSubtag::new(['y', 'm', 'b']),
    LanguageSubtag::new(['y', 'm', 'c']),
    LanguageSubtag::new(['y', 'm', 'd']),
    LanguageSubtag::new(['y', 'm', 'e']),
    LanguageSubtag::new(['y', 'm', 'g']),
    LanguageSubtag::new(['y', 'm', 'h']),
    LanguageSubtag::new(['y', 'm', 'i']),
    LanguageSubtag::new(['y', 'm', 'k']),
    LanguageSubtag::new(['y', 'm', 'l']),
    LanguageSubtag::new(['y', 'm', 'm']),
    LanguageSubtag::new(['y', 'm', 'n']),
    LanguageSubtag::new(['y', 'm', 'o']),
    LanguageSubtag::new(['y', 'm', 'p']),
    LanguageSubtag::new(['y', 'm', 'q']),
    LanguageSubtag::new(['y', 'm', 'r']),
    LanguageSubtag::new(['y', 'm', 's']),
    LanguageSubtag::new(['y', 'm', 't']),
    LanguageSubtag::new(['y', 'm', 'x']),
    LanguageSubtag::new(['y', 'm', 'z']),
    LanguageSubtag::new(['y', 'n', 'a']),
    LanguageSubtag::new(['y', 'n', 'd']),
    LanguageSubtag::new(['y', 'n', 'e']),
    LanguageSubtag::new(['y', 'n', 'g']),
    LanguageSubtag::new(['y', 'n', 'h']),
    LanguageSubtag::new(['y', 'n', 'k']),
    LanguageSubtag::new(['y', 'n', 'l']),
    LanguageSubtag::new(['y', 'n', 'n']),
    LanguageSubtag::new(['y', 'n', 'o']),
    LanguageSubtag::new(['y', 'n', 'q']),
    LanguageSubtag::new(['y', 'n', 's']),
    LanguageSubtag::new(['y', 'n', 'u']),
    LanguageSubtag::new(['y', 'o', ' ']),
    LanguageSubtag::new(['y', 'o', 'b']),
    LanguageSubtag::new(['y', 'o', 'g']),
    LanguageSubtag::new(['y', 'o', 'i']),
    LanguageSubtag::new(['y', 'o', 'k']),
    LanguageSubtag::new(['y', 'o', 'l']),
    LanguageSubtag::new(['y', 'o', 'm']),
    LanguageSubtag::new(['y', 'o', 'n']),
    LanguageSubtag::new(['y', 'o', 's']),
    LanguageSubtag::new(['y', 'o', 't']),
    LanguageSubtag::new(['y', 'o', 'x']),
    LanguageSubtag::new(['y', 'o', 'y']),
    LanguageSubtag::new(['y', 'p', 'a']),
    LanguageSubtag::new(['y', 'p', 'b']),
    LanguageSubtag::new(['y', 'p', 'g']),
    LanguageSubtag::new(['y', 'p', 'h']),
    LanguageSubtag::new(['y', 'p', 'k']),
    LanguageSubtag::new(['y', 'p', 'm']),
    LanguageSubtag::new(['y', 'p', 'n']),
    LanguageSubtag::new(['y', 'p', 'o']),
    LanguageSubtag::new(['y', 'p', 'p']),
    LanguageSubtag::new(['y', 'p', 'z']),
    LanguageSubtag::new(['y', 'r', 'a']),
    LanguageSubtag::new(['y', 'r', 'b']),
    LanguageSubtag::new(['y', 'r', 'e']),
    LanguageSubtag::new(['y', 'r', 'i']),
    LanguageSubtag::new(['y', 'r', 'k']),
    LanguageSubtag::new(['y', 'r', 'l']),
    LanguageSubtag::new(['y', 'r', 'm']),
    LanguageSubtag::new(['y', 'r', 'n']),
    LanguageSubtag::new(['y', 'r', 'o']),
    LanguageSubtag::new(['y', 'r', 's']),
    LanguageSubtag::new(['y', 'r', 'w']),
    LanguageSubtag::new(['y', 'r', 'y']),
    LanguageSubtag::new(['y', 's', 'c']),
    LanguageSubtag::new(['y', 's', 'd']),
    LanguageSubtag::new(['y', 's', 'g']),
    LanguageSubtag::new(['y', 's', 'l']),
    LanguageSubtag::new(['y', 's', 'm']),
    LanguageSubtag::new(['y', 's', 'n']),
    LanguageSubtag::new(['y', 's', 'o']),
    LanguageSubtag::new(['y', 's', 'p']),
    LanguageSubtag::new(['y', 's', 'r']),
    LanguageSubtag::new(['y', 's', 's']),
    LanguageSubtag::new(['y', 's', 'y']),
    LanguageSubtag::new(['y', 't', 'a']),
    LanguageSubtag::new(['y', 't', 'l']),
    LanguageSubtag::new(['y', 't', 'p']),
    LanguageSubtag::new(['y', 't', 'w']),
    LanguageSubtag::new(['y', 't', 'y']),
    LanguageSubtag::new(['y', 'u', 'a']),
    LanguageSubtag::new(['y', 'u', 'b']),
    LanguageSubtag::new(['y', 'u', 'c']),
    LanguageSubtag::new(['y', 'u', 'd']),
    LanguageSubtag::new(['y', 'u', 'e']),
    LanguageSubtag::new(['y', 'u', 'f']),
    LanguageSubtag::new(['y', 'u', 'g']),
    LanguageSubtag::new(['y', 'u', 'i']),
    LanguageSubtag::new(['y', 'u', 'j']),
    LanguageSubtag::new(['y', 'u', 'k']),
    LanguageSubtag::new(['y', 'u', 'l']),
    LanguageSubtag::new(['y', 'u', 'm']),
    LanguageSubtag::new(['y', 'u', 'n']),
    LanguageSubtag::new(['y', 'u', 'p']),
    LanguageSubtag::new(['y', 'u', 'q']),
    LanguageSubtag::new(['y', 'u', 'r']),
    LanguageSubtag::new(['y', 'u', 't']),
    LanguageSubtag::new(['y', 'u', 'u']),
    LanguageSubtag::new(['y', 'u', 'w']),
    LanguageSubtag::new(['y', 'u', 'x']),
    LanguageSubtag::new(['y', 'u', 'y']),
    LanguageSubtag::new(['y', 'u', 'z']),
    LanguageSubtag::new(['y', 'v', 'a']),
    LanguageSubtag::new(['y', 'v', 't']),
    LanguageSubtag::new(['y', 'w', 'a']),
    LanguageSubtag::new(['y', 'w', 'g']),
    LanguageSubtag::new(['y', 'w', 'l']),
    LanguageSubtag::new(['y', 'w', 'n']),
    LanguageSubtag::new(['y', 'w', 'q']),
    LanguageSubtag::new(['y', 'w', 'r']),
    LanguageSubtag::new(['y', 'w', 't']),
    LanguageSubtag::new(['y', 'w', 'u']),
    LanguageSubtag::new(['y', 'w', 'w']),
    LanguageSubtag::new(['y', 'x', 'a']),
    LanguageSubtag::new(['y', 'x', 'g']),
    LanguageSubtag::new(['y', 'x', 'l']),
    LanguageSubtag::new(['y', 'x', 'm']),
    LanguageSubtag::new(['y', 'x', 'u']),
    LanguageSubtag::new(['y', 'x', 'y']),
    LanguageSubtag::new(['y', 'y', 'r']),
    LanguageSubtag::new(['y', 'y', 'u']),
    LanguageSubtag::new(['y', 'y', 'z']),
    LanguageSubtag::new(['y', 'z', 'g']),
    LanguageSubtag::new(['y', 'z', 'k']),
    LanguageSubtag::new(['z', 'a', ' ']),
    LanguageSubtag::new(['z', 'a', 'a']),
    LanguageSubtag::new(['z', 'a', 'b']),
    LanguageSubtag::new(['z', 'a', 'c']),
    LanguageSubtag::new(['z', 'a', 'd']),
    LanguageSubtag::new(['z', 'a', 'e']),
    LanguageSubtag::new(['z', 'a', 'f']),
    LanguageSubtag::new(['z', 'a', 'g']),
    LanguageSubtag::new(['z', 'a', 'h']),
    LanguageSubtag::new(['z', 'a', 'i']),
    LanguageSubtag::new(['z', 'a', 'j']),
    LanguageSubtag::new(['z', 'a', 'k']),
    LanguageSubtag::new(['z', 'a', 'l']),
    LanguageSubtag::new(['z', 'a', 'm']),
    LanguageSubtag::new(['z', 'a', 'o']),
    LanguageSubtag::new(['z', 'a', 'p']),
    LanguageSubtag::new(['z', 'a', 'q']),
    LanguageSubtag::new(['z', 'a', 'r']),
    LanguageSubtag::new(['z', 'a', 's']),
    LanguageSubtag::new(['z', 'a', 't']),
    LanguageSubtag::new(['z', 'a', 'u']),
    LanguageSubtag::new(['z', 'a', 'v']),
    LanguageSubtag::new(['z', 'a', 'w']),
    LanguageSubtag::new(['z', 'a', 'x']),
    LanguageSubtag::new(['z', 'a', 'y']),
    LanguageSubtag::new(['z', 'a', 'z']),
    LanguageSubtag::new(['z', 'b', 'a']),
    LanguageSubtag::new(['z', 'b', 'c']),
    LanguageSubtag::new(['z', 'b', 'e']),
    LanguageSubtag::new(['z', 'b', 'l']),
    LanguageSubtag::new(['z', 'b', 't']),
    LanguageSubtag::new(['z', 'b', 'u']),
    LanguageSubtag::new(['z', 'b', 'w']),
    LanguageSubtag::new(['z', 'c', 'a']),
    LanguageSubtag::new(['z', 'c', 'h']),
    LanguageSubtag::new(['z', 'd', 'j']),
    LanguageSubtag::new(['z', 'e', 'a']),
    LanguageSubtag::new(['z', 'e', 'g']),
    LanguageSubtag::new(['z', 'e', 'h']),
    LanguageSubtag::new(['z', 'e', 'n']),
    LanguageSubtag::new(['z', 'g', 'a']),
    LanguageSubtag::new(['z', 'g', 'b']),
    LanguageSubtag::new(['z', 'g', 'h']),
    LanguageSubtag::new(['z', 'g', 'm']),
    LanguageSubtag::new(['z', 'g', 'n']),
    LanguageSubtag::new(['z', 'g', 'r']),
    LanguageSubtag::new(['z', 'h', ' ']),
    LanguageSubtag::new(['z', 'h', 'b']),
    LanguageSubtag::new(['z', 'h', 'd']),
    LanguageSubtag::new(['z', 'h', 'i']),
    LanguageSubtag::new(['z', 'h', 'n']),
    LanguageSubtag::new(['z', 'h', 'w']),
    LanguageSubtag::new(['z', 'h', 'x']),
    LanguageSubtag::new(['z', 'i', 'a']),
    LanguageSubtag::new(['z', 'i', 'b']),
    LanguageSubtag::new(['z', 'i', 'k']),
    LanguageSubtag::new(['z', 'i', 'l']),
    LanguageSubtag::new(['z', 'i', 'm']),
    LanguageSubtag::new(['z', 'i', 'n']),
    LanguageSubtag::new(['z', 'i', 'r']),
    LanguageSubtag::new(['z', 'i', 'w']),
    LanguageSubtag::new(['z', 'i', 'z']),
    LanguageSubtag::new(['z', 'k', 'a']),
    LanguageSubtag::new(['z', 'k', 'b']),
    LanguageSubtag::new(['z', 'k', 'd']),
    LanguageSubtag::new(['z', 'k', 'g']),
    LanguageSubtag::new(['z', 'k', 'h']),
    LanguageSubtag::new(['z', 'k', 'k']),
    LanguageSubtag::new(['z', 'k', 'n']),
    LanguageSubtag::new(['z', 'k', 'o']),
    LanguageSubtag::new(['z', 'k', 'p']),
    LanguageSubtag::new(['z', 'k', 'r']),
    LanguageSubtag::new(['z', 'k', 't']),
    LanguageSubtag::new(['z', 'k', 'u']),
    LanguageSubtag::new(['z', 'k', 'v']),
    LanguageSubtag::new(['z', 'k', 'z']),
    LanguageSubtag::new(['z', 'l', 'a']),
    LanguageSubtag::new(['z', 'l', 'e']),
    LanguageSubtag::new(['z', 'l', 'j']),
    LanguageSubtag::new(['z', 'l', 'm']),
    LanguageSubtag::new(['z', 'l', 'n']),
    LanguageSubtag::new(['z', 'l', 'q']),
    LanguageSubtag::new(['z', 'l', 's']),
    LanguageSubtag::new(['z', 'l', 'w']),
    LanguageSubtag::new(['z', 'm', 'a']),
    LanguageSubtag::new(['z', 'm', 'b']),
    LanguageSubtag::new(['z', 'm', 'c']),
    LanguageSubtag::new(['z', 'm', 'd']),
    LanguageSubtag::new(['z', 'm', 'e']),
    LanguageSubtag::new(['z', 'm', 'f']),
    LanguageSubtag::new(['z', 'm', 'g']),
    LanguageSubtag::new(['z', 'm', 'h']),
    LanguageSubtag::new(['z', 'm', 'i']),
    LanguageSubtag::new(['z', 'm', 'j']),
    LanguageSubtag::new(['z', 'm', 'k']),
    LanguageSubtag::new(['z', 'm', 'l']),
    LanguageSubtag::new(['z', 'm', 'm']),
    LanguageSubtag::new(['z', 'm', 'n']),
    LanguageSubtag::new(['z', 'm', 'o']),
    LanguageSubtag::new(['z', 'm', 'p']),
    LanguageSubtag::new(['z', 'm', 'q']),
    LanguageSubtag::new(['z', 'm', 'r']),
    LanguageSubtag::new(['z', 'm', 's']),
    LanguageSubtag::new(['z', 'm', 't']),
    LanguageSubtag::new(['z', 'm', 'u']),
    LanguageSubtag::new(['z', 'm', 'v']),
    LanguageSubtag::new(['z', 'm', 'w']),
    LanguageSubtag::new(['z', 'm', 'x']),
    LanguageSubtag::new(['z', 'm', 'y']),
    LanguageSubtag::new(['z', 'm', 'z']),
    LanguageSubtag::new(['z', 'n', 'a']),
    LanguageSubtag::new(['z', 'n', 'd']),
    LanguageSubtag::new(['z', 'n', 'e']),
    LanguageSubtag::new(['z', 'n', 'g']),
    LanguageSubtag::new(['z', 'n', 'k']),
    LanguageSubtag::new(['z', 'n', 's']),
    LanguageSubtag::new(['z', 'o', 'c']),
    LanguageSubtag::new(['z', 'o', 'h']),
    LanguageSubtag::new(['z', 'o', 'm']),
    LanguageSubtag::new(['z', 'o', 'o']),
    LanguageSubtag::new(['z', 'o', 'q']),
    LanguageSubtag::new(['z', 'o', 'r']),
    LanguageSubtag::new(['z', 'o', 's']),
    LanguageSubtag::new(['z', 'p', 'a']),
    LanguageSubtag::new(['z', 'p', 'b']),
    LanguageSubtag::new(['z', 'p', 'c']),
    LanguageSubtag::new(['z', 'p', 'd']),
    LanguageSubtag::new(['z', 'p', 'e']),
    LanguageSubtag::new(['z', 'p', 'f']),
    LanguageSubtag::new(['z', 'p', 'g']),
    LanguageSubtag::new(['z', 'p', 'h']),
    LanguageSubtag::new(['z', 'p', 'i']),
    LanguageSubtag::new(['z', 'p', 'j']),
    LanguageSubtag::new(['z', 'p', 'k']),
    LanguageSubtag::new(['z', 'p', 'l']),
    LanguageSubtag::new(['z', 'p', 'm']),
    LanguageSubtag::new(['z', 'p', 'n']),
    LanguageSubtag::new(['z', 'p', 'o']),
    LanguageSubtag::new(['z', 'p', 'p']),
    LanguageSubtag::new(['z', 'p', 'q']),
    LanguageSubtag::new(['z', 'p', 'r']),
    LanguageSubtag::new(['z', 'p', 's']),
    LanguageSubtag::new(['z', 'p', 't']),
    LanguageSubtag::new(['z', 'p', 'u']),
    LanguageSubtag::new(['z', 'p', 'v']),
    LanguageSubtag::new(['z', 'p', 'w']),
    LanguageSubtag::new(['z', 'p', 'x']),
    LanguageSubtag::new(['z', 'p', 'y']),
    LanguageSubtag::new(['z', 'p', 'z']),
    LanguageSubtag::new(['z', 'q', 'e']),
    LanguageSubtag::new(['z', 'r', 'a']),
    LanguageSubtag::new(['z', 'r', 'g']),
    LanguageSubtag::new(['z', 'r', 'n']),
    LanguageSubtag::new(['z', 'r', 'o']),
    LanguageSubtag::new(['z', 'r', 'p']),
    LanguageSubtag::new(['z', 'r', 's']),
    LanguageSubtag::new(['z', 's', 'a']),
    LanguageSubtag::new(['z', 's', 'k']),
    LanguageSubtag::new(['z', 's', 'l']),
    LanguageSubtag::new(['z', 's', 'm']),
    LanguageSubtag::new(['z', 's', 'r']),
    LanguageSubtag::new(['z', 's', 'u']),
    LanguageSubtag::new(['z', 't', 'e']),
    LanguageSubtag::new(['z', 't', 'g']),
    LanguageSubtag::new(['z', 't', 'l']),
    LanguageSubtag::new(['z', 't', 'm']),
    LanguageSubtag::new(['z', 't', 'n']),
    LanguageSubtag::new(['z', 't', 'p']),
    LanguageSubtag::new(['z', 't', 'q']),
    LanguageSubtag::new(['z', 't', 's']),
    LanguageSubtag::new(['z', 't', 't']),
    LanguageSubtag::new(['z', 't', 'u']),
    LanguageSubtag::new(['z', 't', 'x']),
    LanguageSubtag::new(['z', 't', 'y']),
    LanguageSubtag::new(['z', 'u', ' ']),
    LanguageSubtag::new(['z', 'u', 'a']),
    LanguageSubtag::new(['z', 'u', 'h']),
    LanguageSubtag::new(['z', 'u', 'm']),
    LanguageSubtag::new(['z', 'u', 'n']),
    LanguageSubtag::new(['z', 'u', 'y']),
    LanguageSubtag::new(['z', 'w', 'a']),
    LanguageSubtag::new(['z', 'x', 'x']),
    LanguageSubtag::new(['z', 'y', 'b']),
    LanguageSubtag::new(['z', 'y', 'g']),
    LanguageSubtag::new(['z', 'y', 'j']),
    LanguageSubtag::new(['z', 'y', 'n']),
    LanguageSubtag::new(['z', 'y', 'p']),
    LanguageSubtag::new(['z', 'z', 'a']),
    LanguageSubtag::new(['z', 'z', 'j']),
];

pub const LANGUAGES_PREFERRED_VALUE: [(LanguageSubtag, LanguageSubtag); 92] = [
    (
        LanguageSubtag::new(['a', 'a', 'm']),
        LanguageSubtag::new(['a', 'a', 's']),
    ),
    (
        LanguageSubtag::new(['a', 'd', 'p']),
        LanguageSubtag::new(['d', 'z', ' ']),
    ),
    (
        LanguageSubtag::new(['a', 's', 'd']),
        LanguageSubtag::new(['s', 'n', 'z']),
    ),
    (
        LanguageSubtag::new(['a', 'u', 'e']),
        LanguageSubtag::new(['k', 't', 'z']),
    ),
    (
        LanguageSubtag::new(['a', 'y', 'x']),
        LanguageSubtag::new(['n', 'u', 'n']),
    ),
    (
        LanguageSubtag::new(['b', 'g', 'm']),
        LanguageSubtag::new(['b', 'c', 'g']),
    ),
    (
        LanguageSubtag::new(['b', 'i', 'c']),
        LanguageSubtag::new(['b', 'i', 'r']),
    ),
    (
        LanguageSubtag::new(['b', 'j', 'd']),
        LanguageSubtag::new(['d', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['b', 'l', 'g']),
        LanguageSubtag::new(['i', 'b', 'a']),
    ),
    (
        LanguageSubtag::new(['c', 'c', 'q']),
        LanguageSubtag::new(['r', 'k', 'i']),
    ),
    (
        LanguageSubtag::new(['c', 'j', 'r']),
        LanguageSubtag::new(['m', 'o', 'm']),
    ),
    (
        LanguageSubtag::new(['c', 'k', 'a']),
        LanguageSubtag::new(['c', 'm', 'r']),
    ),
    (
        LanguageSubtag::new(['c', 'm', 'k']),
        LanguageSubtag::new(['x', 'c', 'h']),
    ),
    (
        LanguageSubtag::new(['c', 'o', 'y']),
        LanguageSubtag::new(['p', 'i', 'j']),
    ),
    (
        LanguageSubtag::new(['c', 'q', 'u']),
        LanguageSubtag::new(['q', 'u', 'h']),
    ),
    (
        LanguageSubtag::new(['d', 'i', 't']),
        LanguageSubtag::new(['d', 'i', 'f']),
    ),
    (
        LanguageSubtag::new(['d', 'r', 'h']),
        LanguageSubtag::new(['k', 'h', 'k']),
    ),
    (
        LanguageSubtag::new(['d', 'r', 'r']),
        LanguageSubtag::new(['k', 'z', 'k']),
    ),
    (
        LanguageSubtag::new(['d', 'r', 'w']),
        LanguageSubtag::new(['p', 'r', 's']),
    ),
    (
        LanguageSubtag::new(['g', 'a', 'v']),
        LanguageSubtag::new(['d', 'e', 'v']),
    ),
    (
        LanguageSubtag::new(['g', 'f', 'x']),
        LanguageSubtag::new(['v', 'a', 'j']),
    ),
    (
        LanguageSubtag::new(['g', 'g', 'n']),
        LanguageSubtag::new(['g', 'v', 'r']),
    ),
    (
        LanguageSubtag::new(['g', 'l', 'i']),
        LanguageSubtag::new(['k', 'z', 'k']),
    ),
    (
        LanguageSubtag::new(['g', 't', 'i']),
        LanguageSubtag::new(['n', 'y', 'c']),
    ),
    (
        LanguageSubtag::new(['g', 'u', 'v']),
        LanguageSubtag::new(['d', 'u', 'z']),
    ),
    (
        LanguageSubtag::new(['h', 'r', 'r']),
        LanguageSubtag::new(['j', 'a', 'l']),
    ),
    (
        LanguageSubtag::new(['i', 'b', 'i']),
        LanguageSubtag::new(['o', 'p', 'a']),
    ),
    (
        LanguageSubtag::new(['i', 'l', 'w']),
        LanguageSubtag::new(['g', 'a', 'l']),
    ),
    (
        LanguageSubtag::new(['i', 'n', ' ']),
        LanguageSubtag::new(['i', 'd', ' ']),
    ),
    (
        LanguageSubtag::new(['i', 'w', ' ']),
        LanguageSubtag::new(['h', 'e', ' ']),
    ),
    (
        LanguageSubtag::new(['j', 'e', 'g']),
        LanguageSubtag::new(['o', 'y', 'b']),
    ),
    (
        LanguageSubtag::new(['j', 'i', ' ']),
        LanguageSubtag::new(['y', 'i', ' ']),
    ),
    (
        LanguageSubtag::new(['j', 'w', ' ']),
        LanguageSubtag::new(['j', 'v', ' ']),
    ),
    (
        LanguageSubtag::new(['k', 'g', 'c']),
        LanguageSubtag::new(['t', 'd', 'f']),
    ),
    (
        LanguageSubtag::new(['k', 'g', 'h']),
        LanguageSubtag::new(['k', 'm', 'l']),
    ),
    (
        LanguageSubtag::new(['k', 'o', 'j']),
        LanguageSubtag::new(['k', 'w', 'v']),
    ),
    (
        LanguageSubtag::new(['k', 'r', 'm']),
        LanguageSubtag::new(['b', 'm', 'f']),
    ),
    (
        LanguageSubtag::new(['k', 't', 'r']),
        LanguageSubtag::new(['d', 't', 'p']),
    ),
    (
        LanguageSubtag::new(['k', 'v', 's']),
        LanguageSubtag::new(['g', 'd', 'j']),
    ),
    (
        LanguageSubtag::new(['k', 'w', 'q']),
        LanguageSubtag::new(['y', 'a', 'm']),
    ),
    (
        LanguageSubtag::new(['k', 'x', 'e']),
        LanguageSubtag::new(['t', 'v', 'd']),
    ),
    (
        LanguageSubtag::new(['k', 'x', 'l']),
        LanguageSubtag::new(['k', 'r', 'u']),
    ),
    (
        LanguageSubtag::new(['k', 'z', 'j']),
        LanguageSubtag::new(['d', 't', 'p']),
    ),
    (
        LanguageSubtag::new(['k', 'z', 't']),
        LanguageSubtag::new(['d', 't', 'p']),
    ),
    (
        LanguageSubtag::new(['l', 'i', 'i']),
        LanguageSubtag::new(['r', 'a', 'q']),
    ),
    (
        LanguageSubtag::new(['l', 'l', 'o']),
        LanguageSubtag::new(['n', 'g', 't']),
    ),
    (
        LanguageSubtag::new(['l', 'm', 'm']),
        LanguageSubtag::new(['r', 'm', 'x']),
    ),
    (
        LanguageSubtag::new(['m', 'e', 'g']),
        LanguageSubtag::new(['c', 'i', 'r']),
    ),
    (
        LanguageSubtag::new(['m', 'o', ' ']),
        LanguageSubtag::new(['r', 'o', ' ']),
    ),
    (
        LanguageSubtag::new(['m', 's', 't']),
        LanguageSubtag::new(['m', 'r', 'y']),
    ),
    (
        LanguageSubtag::new(['m', 'w', 'j']),
        LanguageSubtag::new(['v', 'a', 'j']),
    ),
    (
        LanguageSubtag::new(['m', 'y', 'd']),
        LanguageSubtag::new(['a', 'o', 'g']),
    ),
    (
        LanguageSubtag::new(['m', 'y', 't']),
        LanguageSubtag::new(['m', 'r', 'y']),
    ),
    (
        LanguageSubtag::new(['n', 'a', 'd']),
        LanguageSubtag::new(['x', 'n', 'y']),
    ),
    (
        LanguageSubtag::new(['n', 'c', 'p']),
        LanguageSubtag::new(['k', 'd', 'z']),
    ),
    (
        LanguageSubtag::new(['n', 'n', 's']),
        LanguageSubtag::new(['n', 'b', 'r']),
    ),
    (
        LanguageSubtag::new(['n', 'n', 'x']),
        LanguageSubtag::new(['n', 'g', 'v']),
    ),
    (
        LanguageSubtag::new(['n', 't', 's']),
        LanguageSubtag::new(['p', 'i', 'j']),
    ),
    (
        LanguageSubtag::new(['n', 'x', 'u']),
        LanguageSubtag::new(['b', 'p', 'p']),
    ),
    (
        LanguageSubtag::new(['o', 'u', 'n']),
        LanguageSubtag::new(['v', 'a', 'j']),
    ),
    (
        LanguageSubtag::new(['p', 'a', 't']),
        LanguageSubtag::new(['k', 'x', 'r']),
    ),
    (
        LanguageSubtag::new(['p', 'c', 'r']),
        LanguageSubtag::new(['a', 'd', 'x']),
    ),
    (
        LanguageSubtag::new(['p', 'm', 'c']),
        LanguageSubtag::new(['h', 'u', 'w']),
    ),
    (
        LanguageSubtag::new(['p', 'm', 'u']),
        LanguageSubtag::new(['p', 'h', 'r']),
    ),
    (
        LanguageSubtag::new(['p', 'p', 'a']),
        LanguageSubtag::new(['b', 'f', 'y']),
    ),
    (
        LanguageSubtag::new(['p', 'p', 'r']),
        LanguageSubtag::new(['l', 'c', 'q']),
    ),
    (
        LanguageSubtag::new(['p', 'r', 'y']),
        LanguageSubtag::new(['p', 'r', 't']),
    ),
    (
        LanguageSubtag::new(['p', 'u', 'z']),
        LanguageSubtag::new(['p', 'u', 'b']),
    ),
    (
        LanguageSubtag::new(['s', 'c', 'a']),
        LanguageSubtag::new(['h', 'l', 'e']),
    ),
    (
        LanguageSubtag::new(['s', 'k', 'k']),
        LanguageSubtag::new(['o', 'y', 'b']),
    ),
    (
        LanguageSubtag::new(['t', 'd', 'u']),
        LanguageSubtag::new(['d', 't', 'p']),
    ),
    (
        LanguageSubtag::new(['t', 'h', 'c']),
        LanguageSubtag::new(['t', 'p', 'o']),
    ),
    (
        LanguageSubtag::new(['t', 'h', 'w']),
        LanguageSubtag::new(['o', 'l', 'a']),
    ),
    (
        LanguageSubtag::new(['t', 'h', 'x']),
        LanguageSubtag::new(['o', 'y', 'b']),
    ),
    (
        LanguageSubtag::new(['t', 'i', 'e']),
        LanguageSubtag::new(['r', 'a', 's']),
    ),
    (
        LanguageSubtag::new(['t', 'k', 'k']),
        LanguageSubtag::new(['t', 'w', 'm']),
    ),
    (
        LanguageSubtag::new(['t', 'l', 'w']),
        LanguageSubtag::new(['w', 'e', 'o']),
    ),
    (
        LanguageSubtag::new(['t', 'm', 'p']),
        LanguageSubtag::new(['t', 'y', 'j']),
    ),
    (
        LanguageSubtag::new(['t', 'n', 'e']),
        LanguageSubtag::new(['k', 'a', 'k']),
    ),
    (
        LanguageSubtag::new(['t', 'n', 'f']),
        LanguageSubtag::new(['p', 'r', 's']),
    ),
    (
        LanguageSubtag::new(['t', 's', 'f']),
        LanguageSubtag::new(['t', 'a', 'j']),
    ),
    (
        LanguageSubtag::new(['u', 'o', 'k']),
        LanguageSubtag::new(['e', 'm', 'a']),
    ),
    (
        LanguageSubtag::new(['x', 'b', 'a']),
        LanguageSubtag::new(['c', 'a', 'x']),
    ),
    (
        LanguageSubtag::new(['x', 'i', 'a']),
        LanguageSubtag::new(['a', 'c', 'n']),
    ),
    (
        LanguageSubtag::new(['x', 'k', 'h']),
        LanguageSubtag::new(['w', 'a', 'w']),
    ),
    (
        LanguageSubtag::new(['x', 'r', 'q']),
        LanguageSubtag::new(['d', 'm', 'w']),
    ),
    (
        LanguageSubtag::new(['y', 'b', 'd']),
        LanguageSubtag::new(['r', 'k', 'i']),
    ),
    (
        LanguageSubtag::new(['y', 'm', 'a']),
        LanguageSubtag::new(['l', 'r', 'r']),
    ),
    (
        LanguageSubtag::new(['y', 'm', 't']),
        LanguageSubtag::new(['m', 't', 'm']),
    ),
    (
        LanguageSubtag::new(['y', 'o', 's']),
        LanguageSubtag::new(['z', 'o', 'm']),
    ),
    (
        LanguageSubtag::new(['y', 'u', 'u']),
        LanguageSubtag::new(['y', 'u', 'g']),
    ),
    (
        LanguageSubtag::new(['z', 'i', 'r']),
        LanguageSubtag::new(['s', 'c', 'v']),
    ),
];

pub const LANGUAGES_SUPPRESS_SCRIPT: [(LanguageSubtag, ScriptSubtag); 134] = [
    (
        LanguageSubtag::new(['a', 'b', ' ']),
        ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['a', 'f', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['a', 'm', ' ']),
        ScriptSubtag::new(['E', 't', 'h', 'i']),
    ),
    (
        LanguageSubtag::new(['a', 'r', ' ']),
        ScriptSubtag::new(['A', 'r', 'a', 'b']),
    ),
    (
        LanguageSubtag::new(['a', 's', ' ']),
        ScriptSubtag::new(['B', 'e', 'n', 'g']),
    ),
    (
        LanguageSubtag::new(['a', 'y', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['b', 'e', ' ']),
        ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['b', 'g', ' ']),
        ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['b', 'n', ' ']),
        ScriptSubtag::new(['B', 'e', 'n', 'g']),
    ),
    (
        LanguageSubtag::new(['b', 's', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['c', 'a', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['c', 'h', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['c', 's', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['c', 'y', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['d', 'a', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['d', 'e', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['d', 's', 'b']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['d', 'v', ' ']),
        ScriptSubtag::new(['T', 'h', 'a', 'a']),
    ),
    (
        LanguageSubtag::new(['d', 'z', ' ']),
        ScriptSubtag::new(['T', 'i', 'b', 't']),
    ),
    (
        LanguageSubtag::new(['e', 'l', ' ']),
        ScriptSubtag::new(['G', 'r', 'e', 'k']),
    ),
    (
        LanguageSubtag::new(['e', 'n', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['e', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['e', 's', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['e', 't', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['e', 'u', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['f', 'a', ' ']),
        ScriptSubtag::new(['A', 'r', 'a', 'b']),
    ),
    (
        LanguageSubtag::new(['f', 'i', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['f', 'j', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['f', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['f', 'r', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['f', 'r', 'r']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['f', 'r', 's']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['f', 'y', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['g', 'a', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['g', 'l', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['g', 'n', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['g', 's', 'w']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['g', 'u', ' ']),
        ScriptSubtag::new(['G', 'u', 'j', 'r']),
    ),
    (
        LanguageSubtag::new(['g', 'v', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['h', 'e', ' ']),
        ScriptSubtag::new(['H', 'e', 'b', 'r']),
    ),
    (
        LanguageSubtag::new(['h', 'i', ' ']),
        ScriptSubtag::new(['D', 'e', 'v', 'a']),
    ),
    (
        LanguageSubtag::new(['h', 'r', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['h', 's', 'b']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['h', 't', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['h', 'u', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['h', 'y', ' ']),
        ScriptSubtag::new(['A', 'r', 'm', 'n']),
    ),
    (
        LanguageSubtag::new(['i', 'd', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['i', 'n', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['i', 's', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['i', 't', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['i', 'w', ' ']),
        ScriptSubtag::new(['H', 'e', 'b', 'r']),
    ),
    (
        LanguageSubtag::new(['j', 'a', ' ']),
        ScriptSubtag::new(['J', 'p', 'a', 'n']),
    ),
    (
        LanguageSubtag::new(['k', 'a', ' ']),
        ScriptSubtag::new(['G', 'e', 'o', 'r']),
    ),
    (
        LanguageSubtag::new(['k', 'k', ' ']),
        ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['k', 'l', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['k', 'm', ' ']),
        ScriptSubtag::new(['K', 'h', 'm', 'r']),
    ),
    (
        LanguageSubtag::new(['k', 'n', ' ']),
        ScriptSubtag::new(['K', 'n', 'd', 'a']),
    ),
    (
        LanguageSubtag::new(['k', 'o', ' ']),
        ScriptSubtag::new(['K', 'o', 'r', 'e']),
    ),
    (
        LanguageSubtag::new(['k', 'o', 'k']),
        ScriptSubtag::new(['D', 'e', 'v', 'a']),
    ),
    (
        LanguageSubtag::new(['l', 'a', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['l', 'b', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['l', 'n', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['l', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 'o', 'o']),
    ),
    (
        LanguageSubtag::new(['l', 't', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['l', 'v', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 'a', 'i']),
        ScriptSubtag::new(['D', 'e', 'v', 'a']),
    ),
    (
        LanguageSubtag::new(['m', 'e', 'n']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 'g', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 'h', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 'k', ' ']),
        ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['m', 'l', ' ']),
        ScriptSubtag::new(['M', 'l', 'y', 'm']),
    ),
    (
        LanguageSubtag::new(['m', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 'r', ' ']),
        ScriptSubtag::new(['D', 'e', 'v', 'a']),
    ),
    (
        LanguageSubtag::new(['m', 's', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 't', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 'y', ' ']),
        ScriptSubtag::new(['M', 'y', 'm', 'r']),
    ),
    (
        LanguageSubtag::new(['n', 'a', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'b', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'd', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'd', 's']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'e', ' ']),
        ScriptSubtag::new(['D', 'e', 'v', 'a']),
    ),
    (
        LanguageSubtag::new(['n', 'i', 'u']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'l', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'n', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'q', 'o']),
        ScriptSubtag::new(['N', 'k', 'o', 'o']),
    ),
    (
        LanguageSubtag::new(['n', 'r', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 's', 'o']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'y', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['o', 'm', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['o', 'r', ' ']),
        ScriptSubtag::new(['O', 'r', 'y', 'a']),
    ),
    (
        LanguageSubtag::new(['p', 'a', ' ']),
        ScriptSubtag::new(['G', 'u', 'r', 'u']),
    ),
    (
        LanguageSubtag::new(['p', 'l', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['p', 's', ' ']),
        ScriptSubtag::new(['A', 'r', 'a', 'b']),
    ),
    (
        LanguageSubtag::new(['p', 't', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['q', 'u', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['r', 'm', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['r', 'n', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['r', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['r', 'u', ' ']),
        ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['r', 'w', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'g', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'i', ' ']),
        ScriptSubtag::new(['S', 'i', 'n', 'h']),
    ),
    (
        LanguageSubtag::new(['s', 'k', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'l', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'm', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'q', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 's', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 't', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'v', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['s', 'w', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'a', ' ']),
        ScriptSubtag::new(['T', 'a', 'm', 'l']),
    ),
    (
        LanguageSubtag::new(['t', 'e', ' ']),
        ScriptSubtag::new(['T', 'e', 'l', 'u']),
    ),
    (
        LanguageSubtag::new(['t', 'e', 'm']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'h', ' ']),
        ScriptSubtag::new(['T', 'h', 'a', 'i']),
    ),
    (
        LanguageSubtag::new(['t', 'i', ' ']),
        ScriptSubtag::new(['E', 't', 'h', 'i']),
    ),
    (
        LanguageSubtag::new(['t', 'k', 'l']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'l', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'm', 'h']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'n', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'o', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'p', 'i']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'r', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 's', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['t', 'v', 'l']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['u', 'k', ' ']),
        ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['u', 'r', ' ']),
        ScriptSubtag::new(['A', 'r', 'a', 'b']),
    ),
    (
        LanguageSubtag::new(['v', 'e', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['v', 'i', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['x', 'h', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
    (
        LanguageSubtag::new(['y', 'i', ' ']),
        ScriptSubtag::new(['H', 'e', 'b', 'r']),
    ),
    (
        LanguageSubtag::new(['z', 'b', 'l']),
        ScriptSubtag::new(['B', 'l', 'i', 's']),
    ),
    (
        LanguageSubtag::new(['z', 'u', ' ']),
        ScriptSubtag::new(['L', 'a', 't', 'n']),
    ),
];

pub const EXTLANGS: [(LanguageSubtag, &str); 245] = [
    (LanguageSubtag::new(['a', 'a', 'o']), "ar-"),
    (LanguageSubtag::new(['a', 'b', 'h']), "ar-"),
    (LanguageSubtag::new(['a', 'b', 'v']), "ar-"),
    (LanguageSubtag::new(['a', 'c', 'm']), "ar-"),
    (LanguageSubtag::new(['a', 'c', 'q']), "ar-"),
    (LanguageSubtag::new(['a', 'c', 'w']), "ar-"),
    (LanguageSubtag::new(['a', 'c', 'x']), "ar-"),
    (LanguageSubtag::new(['a', 'c', 'y']), "ar-"),
    (LanguageSubtag::new(['a', 'd', 'f']), "ar-"),
    (LanguageSubtag::new(['a', 'd', 's']), "sgn-"),
    (LanguageSubtag::new(['a', 'e', 'b']), "ar-"),
    (LanguageSubtag::new(['a', 'e', 'c']), "ar-"),
    (LanguageSubtag::new(['a', 'e', 'd']), "sgn-"),
    (LanguageSubtag::new(['a', 'e', 'n']), "sgn-"),
    (LanguageSubtag::new(['a', 'f', 'b']), "ar-"),
    (LanguageSubtag::new(['a', 'f', 'g']), "sgn-"),
    (LanguageSubtag::new(['a', 'j', 'p']), "ar-"),
    (LanguageSubtag::new(['a', 'p', 'c']), "ar-"),
    (LanguageSubtag::new(['a', 'p', 'd']), "ar-"),
    (LanguageSubtag::new(['a', 'r', 'b']), "ar-"),
    (LanguageSubtag::new(['a', 'r', 'q']), "ar-"),
    (LanguageSubtag::new(['a', 'r', 's']), "ar-"),
    (LanguageSubtag::new(['a', 'r', 'y']), "ar-"),
    (LanguageSubtag::new(['a', 'r', 'z']), "ar-"),
    (LanguageSubtag::new(['a', 's', 'e']), "sgn-"),
    (LanguageSubtag::new(['a', 's', 'f']), "sgn-"),
    (LanguageSubtag::new(['a', 's', 'p']), "sgn-"),
    (LanguageSubtag::new(['a', 's', 'q']), "sgn-"),
    (LanguageSubtag::new(['a', 's', 'w']), "sgn-"),
    (LanguageSubtag::new(['a', 'u', 'z']), "ar-"),
    (LanguageSubtag::new(['a', 'v', 'l']), "ar-"),
    (LanguageSubtag::new(['a', 'y', 'h']), "ar-"),
    (LanguageSubtag::new(['a', 'y', 'l']), "ar-"),
    (LanguageSubtag::new(['a', 'y', 'n']), "ar-"),
    (LanguageSubtag::new(['a', 'y', 'p']), "ar-"),
    (LanguageSubtag::new(['b', 'b', 'z']), "ar-"),
    (LanguageSubtag::new(['b', 'f', 'i']), "sgn-"),
    (LanguageSubtag::new(['b', 'f', 'k']), "sgn-"),
    (LanguageSubtag::new(['b', 'j', 'n']), "ms-"),
    (LanguageSubtag::new(['b', 'o', 'g']), "sgn-"),
    (LanguageSubtag::new(['b', 'q', 'n']), "sgn-"),
    (LanguageSubtag::new(['b', 'q', 'y']), "sgn-"),
    (LanguageSubtag::new(['b', 't', 'j']), "ms-"),
    (LanguageSubtag::new(['b', 'v', 'e']), "ms-"),
    (LanguageSubtag::new(['b', 'v', 'l']), "sgn-"),
    (LanguageSubtag::new(['b', 'v', 'u']), "ms-"),
    (LanguageSubtag::new(['b', 'z', 's']), "sgn-"),
    (LanguageSubtag::new(['c', 'd', 'o']), "zh-"),
    (LanguageSubtag::new(['c', 'd', 's']), "sgn-"),
    (LanguageSubtag::new(['c', 'j', 'y']), "zh-"),
    (LanguageSubtag::new(['c', 'm', 'n']), "zh-"),
    (LanguageSubtag::new(['c', 'n', 'p']), "zh-"),
    (LanguageSubtag::new(['c', 'o', 'a']), "ms-"),
    (LanguageSubtag::new(['c', 'p', 'x']), "zh-"),
    (LanguageSubtag::new(['c', 's', 'c']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'd']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'e']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'f']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'g']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'n']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'p']), "zh-"),
    (LanguageSubtag::new(['c', 's', 'q']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'r']), "sgn-"),
    (LanguageSubtag::new(['c', 's', 'x']), "sgn-"),
    (LanguageSubtag::new(['c', 'z', 'h']), "zh-"),
    (LanguageSubtag::new(['c', 'z', 'o']), "zh-"),
    (LanguageSubtag::new(['d', 'o', 'q']), "sgn-"),
    (LanguageSubtag::new(['d', 's', 'e']), "sgn-"),
    (LanguageSubtag::new(['d', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['d', 'u', 'p']), "ms-"),
    (LanguageSubtag::new(['e', 'c', 's']), "sgn-"),
    (LanguageSubtag::new(['e', 'h', 's']), "sgn-"),
    (LanguageSubtag::new(['e', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['e', 's', 'n']), "sgn-"),
    (LanguageSubtag::new(['e', 's', 'o']), "sgn-"),
    (LanguageSubtag::new(['e', 't', 'h']), "sgn-"),
    (LanguageSubtag::new(['f', 'c', 's']), "sgn-"),
    (LanguageSubtag::new(['f', 's', 'e']), "sgn-"),
    (LanguageSubtag::new(['f', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['f', 's', 's']), "sgn-"),
    (LanguageSubtag::new(['g', 'a', 'n']), "zh-"),
    (LanguageSubtag::new(['g', 'd', 's']), "sgn-"),
    (LanguageSubtag::new(['g', 'o', 'm']), "kok-"),
    (LanguageSubtag::new(['g', 's', 'e']), "sgn-"),
    (LanguageSubtag::new(['g', 's', 'g']), "sgn-"),
    (LanguageSubtag::new(['g', 's', 'm']), "sgn-"),
    (LanguageSubtag::new(['g', 's', 's']), "sgn-"),
    (LanguageSubtag::new(['g', 'u', 's']), "sgn-"),
    (LanguageSubtag::new(['h', 'a', 'b']), "sgn-"),
    (LanguageSubtag::new(['h', 'a', 'f']), "sgn-"),
    (LanguageSubtag::new(['h', 'a', 'k']), "zh-"),
    (LanguageSubtag::new(['h', 'd', 's']), "sgn-"),
    (LanguageSubtag::new(['h', 'j', 'i']), "ms-"),
    (LanguageSubtag::new(['h', 'k', 's']), "sgn-"),
    (LanguageSubtag::new(['h', 'o', 's']), "sgn-"),
    (LanguageSubtag::new(['h', 'p', 's']), "sgn-"),
    (LanguageSubtag::new(['h', 's', 'h']), "sgn-"),
    (LanguageSubtag::new(['h', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['h', 's', 'n']), "zh-"),
    (LanguageSubtag::new(['i', 'c', 'l']), "sgn-"),
    (LanguageSubtag::new(['i', 'k', 's']), "sgn-"),
    (LanguageSubtag::new(['i', 'l', 's']), "sgn-"),
    (LanguageSubtag::new(['i', 'n', 'l']), "sgn-"),
    (LanguageSubtag::new(['i', 'n', 's']), "sgn-"),
    (LanguageSubtag::new(['i', 's', 'e']), "sgn-"),
    (LanguageSubtag::new(['i', 's', 'g']), "sgn-"),
    (LanguageSubtag::new(['i', 's', 'r']), "sgn-"),
    (LanguageSubtag::new(['j', 'a', 'k']), "ms-"),
    (LanguageSubtag::new(['j', 'a', 'x']), "ms-"),
    (LanguageSubtag::new(['j', 'c', 's']), "sgn-"),
    (LanguageSubtag::new(['j', 'h', 's']), "sgn-"),
    (LanguageSubtag::new(['j', 'k', 's']), "sgn-"),
    (LanguageSubtag::new(['j', 'l', 's']), "sgn-"),
    (LanguageSubtag::new(['j', 'o', 's']), "sgn-"),
    (LanguageSubtag::new(['j', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['j', 'u', 's']), "sgn-"),
    (LanguageSubtag::new(['k', 'g', 'i']), "sgn-"),
    (LanguageSubtag::new(['k', 'n', 'n']), "kok-"),
    (LanguageSubtag::new(['k', 'v', 'b']), "ms-"),
    (LanguageSubtag::new(['k', 'v', 'k']), "sgn-"),
    (LanguageSubtag::new(['k', 'v', 'r']), "ms-"),
    (LanguageSubtag::new(['k', 'x', 'd']), "ms-"),
    (LanguageSubtag::new(['l', 'b', 's']), "sgn-"),
    (LanguageSubtag::new(['l', 'c', 'e']), "ms-"),
    (LanguageSubtag::new(['l', 'c', 'f']), "ms-"),
    (LanguageSubtag::new(['l', 'i', 'w']), "ms-"),
    (LanguageSubtag::new(['l', 'l', 's']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'b']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'g']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'n']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'o']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'p']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 't']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'v']), "sgn-"),
    (LanguageSubtag::new(['l', 's', 'y']), "sgn-"),
    (LanguageSubtag::new(['l', 't', 'g']), "lv-"),
    (LanguageSubtag::new(['l', 'v', 's']), "lv-"),
    (LanguageSubtag::new(['l', 'w', 's']), "sgn-"),
    (LanguageSubtag::new(['l', 'z', 'h']), "zh-"),
    (LanguageSubtag::new(['m', 'a', 'x']), "ms-"),
    (LanguageSubtag::new(['m', 'd', 'l']), "sgn-"),
    (LanguageSubtag::new(['m', 'e', 'o']), "ms-"),
    (LanguageSubtag::new(['m', 'f', 'a']), "ms-"),
    (LanguageSubtag::new(['m', 'f', 'b']), "ms-"),
    (LanguageSubtag::new(['m', 'f', 's']), "sgn-"),
    (LanguageSubtag::new(['m', 'i', 'n']), "ms-"),
    (LanguageSubtag::new(['m', 'n', 'p']), "zh-"),
    (LanguageSubtag::new(['m', 'q', 'g']), "ms-"),
    (LanguageSubtag::new(['m', 'r', 'e']), "sgn-"),
    (LanguageSubtag::new(['m', 's', 'd']), "sgn-"),
    (LanguageSubtag::new(['m', 's', 'i']), "ms-"),
    (LanguageSubtag::new(['m', 's', 'r']), "sgn-"),
    (LanguageSubtag::new(['m', 'u', 'i']), "ms-"),
    (LanguageSubtag::new(['m', 'z', 'c']), "sgn-"),
    (LanguageSubtag::new(['m', 'z', 'g']), "sgn-"),
    (LanguageSubtag::new(['m', 'z', 'y']), "sgn-"),
    (LanguageSubtag::new(['n', 'a', 'n']), "zh-"),
    (LanguageSubtag::new(['n', 'b', 's']), "sgn-"),
    (LanguageSubtag::new(['n', 'c', 's']), "sgn-"),
    (LanguageSubtag::new(['n', 's', 'i']), "sgn-"),
    (LanguageSubtag::new(['n', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['n', 's', 'p']), "sgn-"),
    (LanguageSubtag::new(['n', 's', 'r']), "sgn-"),
    (LanguageSubtag::new(['n', 'z', 's']), "sgn-"),
    (LanguageSubtag::new(['o', 'k', 'l']), "sgn-"),
    (LanguageSubtag::new(['o', 'r', 'n']), "ms-"),
    (LanguageSubtag::new(['o', 'r', 's']), "ms-"),
    (LanguageSubtag::new(['p', 'e', 'l']), "ms-"),
    (LanguageSubtag::new(['p', 'g', 'a']), "ar-"),
    (LanguageSubtag::new(['p', 'g', 'z']), "sgn-"),
    (LanguageSubtag::new(['p', 'k', 's']), "sgn-"),
    (LanguageSubtag::new(['p', 'r', 'l']), "sgn-"),
    (LanguageSubtag::new(['p', 'r', 'z']), "sgn-"),
    (LanguageSubtag::new(['p', 's', 'c']), "sgn-"),
    (LanguageSubtag::new(['p', 's', 'd']), "sgn-"),
    (LanguageSubtag::new(['p', 's', 'e']), "ms-"),
    (LanguageSubtag::new(['p', 's', 'g']), "sgn-"),
    (LanguageSubtag::new(['p', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['p', 's', 'o']), "sgn-"),
    (LanguageSubtag::new(['p', 's', 'p']), "sgn-"),
    (LanguageSubtag::new(['p', 's', 'r']), "sgn-"),
    (LanguageSubtag::new(['p', 'y', 's']), "sgn-"),
    (LanguageSubtag::new(['r', 'm', 's']), "sgn-"),
    (LanguageSubtag::new(['r', 's', 'i']), "sgn-"),
    (LanguageSubtag::new(['r', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['r', 's', 'm']), "sgn-"),
    (LanguageSubtag::new(['s', 'd', 'l']), "sgn-"),
    (LanguageSubtag::new(['s', 'f', 'b']), "sgn-"),
    (LanguageSubtag::new(['s', 'f', 's']), "sgn-"),
    (LanguageSubtag::new(['s', 'g', 'g']), "sgn-"),
    (LanguageSubtag::new(['s', 'g', 'x']), "sgn-"),
    (LanguageSubtag::new(['s', 'h', 'u']), "ar-"),
    (LanguageSubtag::new(['s', 'l', 'f']), "sgn-"),
    (LanguageSubtag::new(['s', 'l', 's']), "sgn-"),
    (LanguageSubtag::new(['s', 'q', 'k']), "sgn-"),
    (LanguageSubtag::new(['s', 'q', 's']), "sgn-"),
    (LanguageSubtag::new(['s', 'q', 'x']), "sgn-"),
    (LanguageSubtag::new(['s', 's', 'h']), "ar-"),
    (LanguageSubtag::new(['s', 's', 'p']), "sgn-"),
    (LanguageSubtag::new(['s', 's', 'r']), "sgn-"),
    (LanguageSubtag::new(['s', 'v', 'k']), "sgn-"),
    (LanguageSubtag::new(['s', 'w', 'c']), "sw-"),
    (LanguageSubtag::new(['s', 'w', 'h']), "sw-"),
    (LanguageSubtag::new(['s', 'w', 'l']), "sgn-"),
    (LanguageSubtag::new(['s', 'y', 'y']), "sgn-"),
    (LanguageSubtag::new(['s', 'z', 's']), "sgn-"),
    (LanguageSubtag::new(['t', 'm', 'w']), "ms-"),
    (LanguageSubtag::new(['t', 's', 'e']), "sgn-"),
    (LanguageSubtag::new(['t', 's', 'm']), "sgn-"),
    (LanguageSubtag::new(['t', 's', 'q']), "sgn-"),
    (LanguageSubtag::new(['t', 's', 's']), "sgn-"),
    (LanguageSubtag::new(['t', 's', 'y']), "sgn-"),
    (LanguageSubtag::new(['t', 'z', 'a']), "sgn-"),
    (LanguageSubtag::new(['u', 'g', 'n']), "sgn-"),
    (LanguageSubtag::new(['u', 'g', 'y']), "sgn-"),
    (LanguageSubtag::new(['u', 'k', 'l']), "sgn-"),
    (LanguageSubtag::new(['u', 'k', 's']), "sgn-"),
    (LanguageSubtag::new(['u', 'r', 'k']), "ms-"),
    (LanguageSubtag::new(['u', 'z', 'n']), "uz-"),
    (LanguageSubtag::new(['u', 'z', 's']), "uz-"),
    (LanguageSubtag::new(['v', 'g', 't']), "sgn-"),
    (LanguageSubtag::new(['v', 'k', 'k']), "ms-"),
    (LanguageSubtag::new(['v', 'k', 't']), "ms-"),
    (LanguageSubtag::new(['v', 's', 'i']), "sgn-"),
    (LanguageSubtag::new(['v', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['v', 's', 'v']), "sgn-"),
    (LanguageSubtag::new(['w', 'b', 's']), "sgn-"),
    (LanguageSubtag::new(['w', 'u', 'u']), "zh-"),
    (LanguageSubtag::new(['x', 'k', 'i']), "sgn-"),
    (LanguageSubtag::new(['x', 'm', 'l']), "sgn-"),
    (LanguageSubtag::new(['x', 'm', 'm']), "ms-"),
    (LanguageSubtag::new(['x', 'm', 's']), "sgn-"),
    (LanguageSubtag::new(['y', 'd', 's']), "sgn-"),
    (LanguageSubtag::new(['y', 'g', 's']), "sgn-"),
    (LanguageSubtag::new(['y', 'h', 's']), "sgn-"),
    (LanguageSubtag::new(['y', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['y', 's', 'm']), "sgn-"),
    (LanguageSubtag::new(['y', 'u', 'e']), "zh-"),
    (LanguageSubtag::new(['z', 'i', 'b']), "sgn-"),
    (LanguageSubtag::new(['z', 'l', 'm']), "ms-"),
    (LanguageSubtag::new(['z', 'm', 'i']), "ms-"),
    (LanguageSubtag::new(['z', 's', 'l']), "sgn-"),
    (LanguageSubtag::new(['z', 's', 'm']), "ms-"),
];

pub const EXTLANGS_PREFERRED_VALUE: [(LanguageSubtag, LanguageSubtag); 245] = [
    (
        LanguageSubtag::new(['a', 'a', 'o']),
        LanguageSubtag::new(['a', 'a', 'o']),
    ),
    (
        LanguageSubtag::new(['a', 'b', 'h']),
        LanguageSubtag::new(['a', 'b', 'h']),
    ),
    (
        LanguageSubtag::new(['a', 'b', 'v']),
        LanguageSubtag::new(['a', 'b', 'v']),
    ),
    (
        LanguageSubtag::new(['a', 'c', 'm']),
        LanguageSubtag::new(['a', 'c', 'm']),
    ),
    (
        LanguageSubtag::new(['a', 'c', 'q']),
        LanguageSubtag::new(['a', 'c', 'q']),
    ),
    (
        LanguageSubtag::new(['a', 'c', 'w']),
        LanguageSubtag::new(['a', 'c', 'w']),
    ),
    (
        LanguageSubtag::new(['a', 'c', 'x']),
        LanguageSubtag::new(['a', 'c', 'x']),
    ),
    (
        LanguageSubtag::new(['a', 'c', 'y']),
        LanguageSubtag::new(['a', 'c', 'y']),
    ),
    (
        LanguageSubtag::new(['a', 'd', 'f']),
        LanguageSubtag::new(['a', 'd', 'f']),
    ),
    (
        LanguageSubtag::new(['a', 'd', 's']),
        LanguageSubtag::new(['a', 'd', 's']),
    ),
    (
        LanguageSubtag::new(['a', 'e', 'b']),
        LanguageSubtag::new(['a', 'e', 'b']),
    ),
    (
        LanguageSubtag::new(['a', 'e', 'c']),
        LanguageSubtag::new(['a', 'e', 'c']),
    ),
    (
        LanguageSubtag::new(['a', 'e', 'd']),
        LanguageSubtag::new(['a', 'e', 'd']),
    ),
    (
        LanguageSubtag::new(['a', 'e', 'n']),
        LanguageSubtag::new(['a', 'e', 'n']),
    ),
    (
        LanguageSubtag::new(['a', 'f', 'b']),
        LanguageSubtag::new(['a', 'f', 'b']),
    ),
    (
        LanguageSubtag::new(['a', 'f', 'g']),
        LanguageSubtag::new(['a', 'f', 'g']),
    ),
    (
        LanguageSubtag::new(['a', 'j', 'p']),
        LanguageSubtag::new(['a', 'j', 'p']),
    ),
    (
        LanguageSubtag::new(['a', 'p', 'c']),
        LanguageSubtag::new(['a', 'p', 'c']),
    ),
    (
        LanguageSubtag::new(['a', 'p', 'd']),
        LanguageSubtag::new(['a', 'p', 'd']),
    ),
    (
        LanguageSubtag::new(['a', 'r', 'b']),
        LanguageSubtag::new(['a', 'r', 'b']),
    ),
    (
        LanguageSubtag::new(['a', 'r', 'q']),
        LanguageSubtag::new(['a', 'r', 'q']),
    ),
    (
        LanguageSubtag::new(['a', 'r', 's']),
        LanguageSubtag::new(['a', 'r', 's']),
    ),
    (
        LanguageSubtag::new(['a', 'r', 'y']),
        LanguageSubtag::new(['a', 'r', 'y']),
    ),
    (
        LanguageSubtag::new(['a', 'r', 'z']),
        LanguageSubtag::new(['a', 'r', 'z']),
    ),
    (
        LanguageSubtag::new(['a', 's', 'e']),
        LanguageSubtag::new(['a', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['a', 's', 'f']),
        LanguageSubtag::new(['a', 's', 'f']),
    ),
    (
        LanguageSubtag::new(['a', 's', 'p']),
        LanguageSubtag::new(['a', 's', 'p']),
    ),
    (
        LanguageSubtag::new(['a', 's', 'q']),
        LanguageSubtag::new(['a', 's', 'q']),
    ),
    (
        LanguageSubtag::new(['a', 's', 'w']),
        LanguageSubtag::new(['a', 's', 'w']),
    ),
    (
        LanguageSubtag::new(['a', 'u', 'z']),
        LanguageSubtag::new(['a', 'u', 'z']),
    ),
    (
        LanguageSubtag::new(['a', 'v', 'l']),
        LanguageSubtag::new(['a', 'v', 'l']),
    ),
    (
        LanguageSubtag::new(['a', 'y', 'h']),
        LanguageSubtag::new(['a', 'y', 'h']),
    ),
    (
        LanguageSubtag::new(['a', 'y', 'l']),
        LanguageSubtag::new(['a', 'y', 'l']),
    ),
    (
        LanguageSubtag::new(['a', 'y', 'n']),
        LanguageSubtag::new(['a', 'y', 'n']),
    ),
    (
        LanguageSubtag::new(['a', 'y', 'p']),
        LanguageSubtag::new(['a', 'y', 'p']),
    ),
    (
        LanguageSubtag::new(['b', 'b', 'z']),
        LanguageSubtag::new(['b', 'b', 'z']),
    ),
    (
        LanguageSubtag::new(['b', 'f', 'i']),
        LanguageSubtag::new(['b', 'f', 'i']),
    ),
    (
        LanguageSubtag::new(['b', 'f', 'k']),
        LanguageSubtag::new(['b', 'f', 'k']),
    ),
    (
        LanguageSubtag::new(['b', 'j', 'n']),
        LanguageSubtag::new(['b', 'j', 'n']),
    ),
    (
        LanguageSubtag::new(['b', 'o', 'g']),
        LanguageSubtag::new(['b', 'o', 'g']),
    ),
    (
        LanguageSubtag::new(['b', 'q', 'n']),
        LanguageSubtag::new(['b', 'q', 'n']),
    ),
    (
        LanguageSubtag::new(['b', 'q', 'y']),
        LanguageSubtag::new(['b', 'q', 'y']),
    ),
    (
        LanguageSubtag::new(['b', 't', 'j']),
        LanguageSubtag::new(['b', 't', 'j']),
    ),
    (
        LanguageSubtag::new(['b', 'v', 'e']),
        LanguageSubtag::new(['b', 'v', 'e']),
    ),
    (
        LanguageSubtag::new(['b', 'v', 'l']),
        LanguageSubtag::new(['b', 'v', 'l']),
    ),
    (
        LanguageSubtag::new(['b', 'v', 'u']),
        LanguageSubtag::new(['b', 'v', 'u']),
    ),
    (
        LanguageSubtag::new(['b', 'z', 's']),
        LanguageSubtag::new(['b', 'z', 's']),
    ),
    (
        LanguageSubtag::new(['c', 'd', 'o']),
        LanguageSubtag::new(['c', 'd', 'o']),
    ),
    (
        LanguageSubtag::new(['c', 'd', 's']),
        LanguageSubtag::new(['c', 'd', 's']),
    ),
    (
        LanguageSubtag::new(['c', 'j', 'y']),
        LanguageSubtag::new(['c', 'j', 'y']),
    ),
    (
        LanguageSubtag::new(['c', 'm', 'n']),
        LanguageSubtag::new(['c', 'm', 'n']),
    ),
    (
        LanguageSubtag::new(['c', 'n', 'p']),
        LanguageSubtag::new(['c', 'n', 'p']),
    ),
    (
        LanguageSubtag::new(['c', 'o', 'a']),
        LanguageSubtag::new(['c', 'o', 'a']),
    ),
    (
        LanguageSubtag::new(['c', 'p', 'x']),
        LanguageSubtag::new(['c', 'p', 'x']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'c']),
        LanguageSubtag::new(['c', 's', 'c']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'd']),
        LanguageSubtag::new(['c', 's', 'd']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'e']),
        LanguageSubtag::new(['c', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'f']),
        LanguageSubtag::new(['c', 's', 'f']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'g']),
        LanguageSubtag::new(['c', 's', 'g']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'l']),
        LanguageSubtag::new(['c', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'n']),
        LanguageSubtag::new(['c', 's', 'n']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'p']),
        LanguageSubtag::new(['c', 's', 'p']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'q']),
        LanguageSubtag::new(['c', 's', 'q']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'r']),
        LanguageSubtag::new(['c', 's', 'r']),
    ),
    (
        LanguageSubtag::new(['c', 's', 'x']),
        LanguageSubtag::new(['c', 's', 'x']),
    ),
    (
        LanguageSubtag::new(['c', 'z', 'h']),
        LanguageSubtag::new(['c', 'z', 'h']),
    ),
    (
        LanguageSubtag::new(['c', 'z', 'o']),
        LanguageSubtag::new(['c', 'z', 'o']),
    ),
    (
        LanguageSubtag::new(['d', 'o', 'q']),
        LanguageSubtag::new(['d', 'o', 'q']),
    ),
    (
        LanguageSubtag::new(['d', 's', 'e']),
        LanguageSubtag::new(['d', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['d', 's', 'l']),
        LanguageSubtag::new(['d', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['d', 'u', 'p']),
        LanguageSubtag::new(['d', 'u', 'p']),
    ),
    (
        LanguageSubtag::new(['e', 'c', 's']),
        LanguageSubtag::new(['e', 'c', 's']),
    ),
    (
        LanguageSubtag::new(['e', 'h', 's']),
        LanguageSubtag::new(['e', 'h', 's']),
    ),
    (
        LanguageSubtag::new(['e', 's', 'l']),
        LanguageSubtag::new(['e', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['e', 's', 'n']),
        LanguageSubtag::new(['e', 's', 'n']),
    ),
    (
        LanguageSubtag::new(['e', 's', 'o']),
        LanguageSubtag::new(['e', 's', 'o']),
    ),
    (
        LanguageSubtag::new(['e', 't', 'h']),
        LanguageSubtag::new(['e', 't', 'h']),
    ),
    (
        LanguageSubtag::new(['f', 'c', 's']),
        LanguageSubtag::new(['f', 'c', 's']),
    ),
    (
        LanguageSubtag::new(['f', 's', 'e']),
        LanguageSubtag::new(['f', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['f', 's', 'l']),
        LanguageSubtag::new(['f', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['f', 's', 's']),
        LanguageSubtag::new(['f', 's', 's']),
    ),
    (
        LanguageSubtag::new(['g', 'a', 'n']),
        LanguageSubtag::new(['g', 'a', 'n']),
    ),
    (
        LanguageSubtag::new(['g', 'd', 's']),
        LanguageSubtag::new(['g', 'd', 's']),
    ),
    (
        LanguageSubtag::new(['g', 'o', 'm']),
        LanguageSubtag::new(['g', 'o', 'm']),
    ),
    (
        LanguageSubtag::new(['g', 's', 'e']),
        LanguageSubtag::new(['g', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['g', 's', 'g']),
        LanguageSubtag::new(['g', 's', 'g']),
    ),
    (
        LanguageSubtag::new(['g', 's', 'm']),
        LanguageSubtag::new(['g', 's', 'm']),
    ),
    (
        LanguageSubtag::new(['g', 's', 's']),
        LanguageSubtag::new(['g', 's', 's']),
    ),
    (
        LanguageSubtag::new(['g', 'u', 's']),
        LanguageSubtag::new(['g', 'u', 's']),
    ),
    (
        LanguageSubtag::new(['h', 'a', 'b']),
        LanguageSubtag::new(['h', 'a', 'b']),
    ),
    (
        LanguageSubtag::new(['h', 'a', 'f']),
        LanguageSubtag::new(['h', 'a', 'f']),
    ),
    (
        LanguageSubtag::new(['h', 'a', 'k']),
        LanguageSubtag::new(['h', 'a', 'k']),
    ),
    (
        LanguageSubtag::new(['h', 'd', 's']),
        LanguageSubtag::new(['h', 'd', 's']),
    ),
    (
        LanguageSubtag::new(['h', 'j', 'i']),
        LanguageSubtag::new(['h', 'j', 'i']),
    ),
    (
        LanguageSubtag::new(['h', 'k', 's']),
        LanguageSubtag::new(['h', 'k', 's']),
    ),
    (
        LanguageSubtag::new(['h', 'o', 's']),
        LanguageSubtag::new(['h', 'o', 's']),
    ),
    (
        LanguageSubtag::new(['h', 'p', 's']),
        LanguageSubtag::new(['h', 'p', 's']),
    ),
    (
        LanguageSubtag::new(['h', 's', 'h']),
        LanguageSubtag::new(['h', 's', 'h']),
    ),
    (
        LanguageSubtag::new(['h', 's', 'l']),
        LanguageSubtag::new(['h', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['h', 's', 'n']),
        LanguageSubtag::new(['h', 's', 'n']),
    ),
    (
        LanguageSubtag::new(['i', 'c', 'l']),
        LanguageSubtag::new(['i', 'c', 'l']),
    ),
    (
        LanguageSubtag::new(['i', 'k', 's']),
        LanguageSubtag::new(['i', 'k', 's']),
    ),
    (
        LanguageSubtag::new(['i', 'l', 's']),
        LanguageSubtag::new(['i', 'l', 's']),
    ),
    (
        LanguageSubtag::new(['i', 'n', 'l']),
        LanguageSubtag::new(['i', 'n', 'l']),
    ),
    (
        LanguageSubtag::new(['i', 'n', 's']),
        LanguageSubtag::new(['i', 'n', 's']),
    ),
    (
        LanguageSubtag::new(['i', 's', 'e']),
        LanguageSubtag::new(['i', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['i', 's', 'g']),
        LanguageSubtag::new(['i', 's', 'g']),
    ),
    (
        LanguageSubtag::new(['i', 's', 'r']),
        LanguageSubtag::new(['i', 's', 'r']),
    ),
    (
        LanguageSubtag::new(['j', 'a', 'k']),
        LanguageSubtag::new(['j', 'a', 'k']),
    ),
    (
        LanguageSubtag::new(['j', 'a', 'x']),
        LanguageSubtag::new(['j', 'a', 'x']),
    ),
    (
        LanguageSubtag::new(['j', 'c', 's']),
        LanguageSubtag::new(['j', 'c', 's']),
    ),
    (
        LanguageSubtag::new(['j', 'h', 's']),
        LanguageSubtag::new(['j', 'h', 's']),
    ),
    (
        LanguageSubtag::new(['j', 'k', 's']),
        LanguageSubtag::new(['j', 'k', 's']),
    ),
    (
        LanguageSubtag::new(['j', 'l', 's']),
        LanguageSubtag::new(['j', 'l', 's']),
    ),
    (
        LanguageSubtag::new(['j', 'o', 's']),
        LanguageSubtag::new(['j', 'o', 's']),
    ),
    (
        LanguageSubtag::new(['j', 's', 'l']),
        LanguageSubtag::new(['j', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['j', 'u', 's']),
        LanguageSubtag::new(['j', 'u', 's']),
    ),
    (
        LanguageSubtag::new(['k', 'g', 'i']),
        LanguageSubtag::new(['k', 'g', 'i']),
    ),
    (
        LanguageSubtag::new(['k', 'n', 'n']),
        LanguageSubtag::new(['k', 'n', 'n']),
    ),
    (
        LanguageSubtag::new(['k', 'v', 'b']),
        LanguageSubtag::new(['k', 'v', 'b']),
    ),
    (
        LanguageSubtag::new(['k', 'v', 'k']),
        LanguageSubtag::new(['k', 'v', 'k']),
    ),
    (
        LanguageSubtag::new(['k', 'v', 'r']),
        LanguageSubtag::new(['k', 'v', 'r']),
    ),
    (
        LanguageSubtag::new(['k', 'x', 'd']),
        LanguageSubtag::new(['k', 'x', 'd']),
    ),
    (
        LanguageSubtag::new(['l', 'b', 's']),
        LanguageSubtag::new(['l', 'b', 's']),
    ),
    (
        LanguageSubtag::new(['l', 'c', 'e']),
        LanguageSubtag::new(['l', 'c', 'e']),
    ),
    (
        LanguageSubtag::new(['l', 'c', 'f']),
        LanguageSubtag::new(['l', 'c', 'f']),
    ),
    (
        LanguageSubtag::new(['l', 'i', 'w']),
        LanguageSubtag::new(['l', 'i', 'w']),
    ),
    (
        LanguageSubtag::new(['l', 'l', 's']),
        LanguageSubtag::new(['l', 'l', 's']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'b']),
        LanguageSubtag::new(['l', 's', 'b']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'g']),
        LanguageSubtag::new(['l', 's', 'g']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'l']),
        LanguageSubtag::new(['l', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'n']),
        LanguageSubtag::new(['l', 's', 'n']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'o']),
        LanguageSubtag::new(['l', 's', 'o']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'p']),
        LanguageSubtag::new(['l', 's', 'p']),
    ),
    (
        LanguageSubtag::new(['l', 's', 't']),
        LanguageSubtag::new(['l', 's', 't']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'v']),
        LanguageSubtag::new(['l', 's', 'v']),
    ),
    (
        LanguageSubtag::new(['l', 's', 'y']),
        LanguageSubtag::new(['l', 's', 'y']),
    ),
    (
        LanguageSubtag::new(['l', 't', 'g']),
        LanguageSubtag::new(['l', 't', 'g']),
    ),
    (
        LanguageSubtag::new(['l', 'v', 's']),
        LanguageSubtag::new(['l', 'v', 's']),
    ),
    (
        LanguageSubtag::new(['l', 'w', 's']),
        LanguageSubtag::new(['l', 'w', 's']),
    ),
    (
        LanguageSubtag::new(['l', 'z', 'h']),
        LanguageSubtag::new(['l', 'z', 'h']),
    ),
    (
        LanguageSubtag::new(['m', 'a', 'x']),
        LanguageSubtag::new(['m', 'a', 'x']),
    ),
    (
        LanguageSubtag::new(['m', 'd', 'l']),
        LanguageSubtag::new(['m', 'd', 'l']),
    ),
    (
        LanguageSubtag::new(['m', 'e', 'o']),
        LanguageSubtag::new(['m', 'e', 'o']),
    ),
    (
        LanguageSubtag::new(['m', 'f', 'a']),
        LanguageSubtag::new(['m', 'f', 'a']),
    ),
    (
        LanguageSubtag::new(['m', 'f', 'b']),
        LanguageSubtag::new(['m', 'f', 'b']),
    ),
    (
        LanguageSubtag::new(['m', 'f', 's']),
        LanguageSubtag::new(['m', 'f', 's']),
    ),
    (
        LanguageSubtag::new(['m', 'i', 'n']),
        LanguageSubtag::new(['m', 'i', 'n']),
    ),
    (
        LanguageSubtag::new(['m', 'n', 'p']),
        LanguageSubtag::new(['m', 'n', 'p']),
    ),
    (
        LanguageSubtag::new(['m', 'q', 'g']),
        LanguageSubtag::new(['m', 'q', 'g']),
    ),
    (
        LanguageSubtag::new(['m', 'r', 'e']),
        LanguageSubtag::new(['m', 'r', 'e']),
    ),
    (
        LanguageSubtag::new(['m', 's', 'd']),
        LanguageSubtag::new(['m', 's', 'd']),
    ),
    (
        LanguageSubtag::new(['m', 's', 'i']),
        LanguageSubtag::new(['m', 's', 'i']),
    ),
    (
        LanguageSubtag::new(['m', 's', 'r']),
        LanguageSubtag::new(['m', 's', 'r']),
    ),
    (
        LanguageSubtag::new(['m', 'u', 'i']),
        LanguageSubtag::new(['m', 'u', 'i']),
    ),
    (
        LanguageSubtag::new(['m', 'z', 'c']),
        LanguageSubtag::new(['m', 'z', 'c']),
    ),
    (
        LanguageSubtag::new(['m', 'z', 'g']),
        LanguageSubtag::new(['m', 'z', 'g']),
    ),
    (
        LanguageSubtag::new(['m', 'z', 'y']),
        LanguageSubtag::new(['m', 'z', 'y']),
    ),
    (
        LanguageSubtag::new(['n', 'a', 'n']),
        LanguageSubtag::new(['n', 'a', 'n']),
    ),
    (
        LanguageSubtag::new(['n', 'b', 's']),
        LanguageSubtag::new(['n', 'b', 's']),
    ),
    (
        LanguageSubtag::new(['n', 'c', 's']),
        LanguageSubtag::new(['n', 'c', 's']),
    ),
    (
        LanguageSubtag::new(['n', 's', 'i']),
        LanguageSubtag::new(['n', 's', 'i']),
    ),
    (
        LanguageSubtag::new(['n', 's', 'l']),
        LanguageSubtag::new(['n', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['n', 's', 'p']),
        LanguageSubtag::new(['n', 's', 'p']),
    ),
    (
        LanguageSubtag::new(['n', 's', 'r']),
        LanguageSubtag::new(['n', 's', 'r']),
    ),
    (
        LanguageSubtag::new(['n', 'z', 's']),
        LanguageSubtag::new(['n', 'z', 's']),
    ),
    (
        LanguageSubtag::new(['o', 'k', 'l']),
        LanguageSubtag::new(['o', 'k', 'l']),
    ),
    (
        LanguageSubtag::new(['o', 'r', 'n']),
        LanguageSubtag::new(['o', 'r', 'n']),
    ),
    (
        LanguageSubtag::new(['o', 'r', 's']),
        LanguageSubtag::new(['o', 'r', 's']),
    ),
    (
        LanguageSubtag::new(['p', 'e', 'l']),
        LanguageSubtag::new(['p', 'e', 'l']),
    ),
    (
        LanguageSubtag::new(['p', 'g', 'a']),
        LanguageSubtag::new(['p', 'g', 'a']),
    ),
    (
        LanguageSubtag::new(['p', 'g', 'z']),
        LanguageSubtag::new(['p', 'g', 'z']),
    ),
    (
        LanguageSubtag::new(['p', 'k', 's']),
        LanguageSubtag::new(['p', 'k', 's']),
    ),
    (
        LanguageSubtag::new(['p', 'r', 'l']),
        LanguageSubtag::new(['p', 'r', 'l']),
    ),
    (
        LanguageSubtag::new(['p', 'r', 'z']),
        LanguageSubtag::new(['p', 'r', 'z']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'c']),
        LanguageSubtag::new(['p', 's', 'c']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'd']),
        LanguageSubtag::new(['p', 's', 'd']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'e']),
        LanguageSubtag::new(['p', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'g']),
        LanguageSubtag::new(['p', 's', 'g']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'l']),
        LanguageSubtag::new(['p', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'o']),
        LanguageSubtag::new(['p', 's', 'o']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'p']),
        LanguageSubtag::new(['p', 's', 'p']),
    ),
    (
        LanguageSubtag::new(['p', 's', 'r']),
        LanguageSubtag::new(['p', 's', 'r']),
    ),
    (
        LanguageSubtag::new(['p', 'y', 's']),
        LanguageSubtag::new(['p', 'y', 's']),
    ),
    (
        LanguageSubtag::new(['r', 'm', 's']),
        LanguageSubtag::new(['r', 'm', 's']),
    ),
    (
        LanguageSubtag::new(['r', 's', 'i']),
        LanguageSubtag::new(['r', 's', 'i']),
    ),
    (
        LanguageSubtag::new(['r', 's', 'l']),
        LanguageSubtag::new(['r', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['r', 's', 'm']),
        LanguageSubtag::new(['r', 's', 'm']),
    ),
    (
        LanguageSubtag::new(['s', 'd', 'l']),
        LanguageSubtag::new(['s', 'd', 'l']),
    ),
    (
        LanguageSubtag::new(['s', 'f', 'b']),
        LanguageSubtag::new(['s', 'f', 'b']),
    ),
    (
        LanguageSubtag::new(['s', 'f', 's']),
        LanguageSubtag::new(['s', 'f', 's']),
    ),
    (
        LanguageSubtag::new(['s', 'g', 'g']),
        LanguageSubtag::new(['s', 'g', 'g']),
    ),
    (
        LanguageSubtag::new(['s', 'g', 'x']),
        LanguageSubtag::new(['s', 'g', 'x']),
    ),
    (
        LanguageSubtag::new(['s', 'h', 'u']),
        LanguageSubtag::new(['s', 'h', 'u']),
    ),
    (
        LanguageSubtag::new(['s', 'l', 'f']),
        LanguageSubtag::new(['s', 'l', 'f']),
    ),
    (
        LanguageSubtag::new(['s', 'l', 's']),
        LanguageSubtag::new(['s', 'l', 's']),
    ),
    (
        LanguageSubtag::new(['s', 'q', 'k']),
        LanguageSubtag::new(['s', 'q', 'k']),
    ),
    (
        LanguageSubtag::new(['s', 'q', 's']),
        LanguageSubtag::new(['s', 'q', 's']),
    ),
    (
        LanguageSubtag::new(['s', 'q', 'x']),
        LanguageSubtag::new(['s', 'q', 'x']),
    ),
    (
        LanguageSubtag::new(['s', 's', 'h']),
        LanguageSubtag::new(['s', 's', 'h']),
    ),
    (
        LanguageSubtag::new(['s', 's', 'p']),
        LanguageSubtag::new(['s', 's', 'p']),
    ),
    (
        LanguageSubtag::new(['s', 's', 'r']),
        LanguageSubtag::new(['s', 's', 'r']),
    ),
    (
        LanguageSubtag::new(['s', 'v', 'k']),
        LanguageSubtag::new(['s', 'v', 'k']),
    ),
    (
        LanguageSubtag::new(['s', 'w', 'c']),
        LanguageSubtag::new(['s', 'w', 'c']),
    ),
    (
        LanguageSubtag::new(['s', 'w', 'h']),
        LanguageSubtag::new(['s', 'w', 'h']),
    ),
    (
        LanguageSubtag::new(['s', 'w', 'l']),
        LanguageSubtag::new(['s', 'w', 'l']),
    ),
    (
        LanguageSubtag::new(['s', 'y', 'y']),
        LanguageSubtag::new(['s', 'y', 'y']),
    ),
    (
        LanguageSubtag::new(['s', 'z', 's']),
        LanguageSubtag::new(['s', 'z', 's']),
    ),
    (
        LanguageSubtag::new(['t', 'm', 'w']),
        LanguageSubtag::new(['t', 'm', 'w']),
    ),
    (
        LanguageSubtag::new(['t', 's', 'e']),
        LanguageSubtag::new(['t', 's', 'e']),
    ),
    (
        LanguageSubtag::new(['t', 's', 'm']),
        LanguageSubtag::new(['t', 's', 'm']),
    ),
    (
        LanguageSubtag::new(['t', 's', 'q']),
        LanguageSubtag::new(['t', 's', 'q']),
    ),
    (
        LanguageSubtag::new(['t', 's', 's']),
        LanguageSubtag::new(['t', 's', 's']),
    ),
    (
        LanguageSubtag::new(['t', 's', 'y']),
        LanguageSubtag::new(['t', 's', 'y']),
    ),
    (
        LanguageSubtag::new(['t', 'z', 'a']),
        LanguageSubtag::new(['t', 'z', 'a']),
    ),
    (
        LanguageSubtag::new(['u', 'g', 'n']),
        LanguageSubtag::new(['u', 'g', 'n']),
    ),
    (
        LanguageSubtag::new(['u', 'g', 'y']),
        LanguageSubtag::new(['u', 'g', 'y']),
    ),
    (
        LanguageSubtag::new(['u', 'k', 'l']),
        LanguageSubtag::new(['u', 'k', 'l']),
    ),
    (
        LanguageSubtag::new(['u', 'k', 's']),
        LanguageSubtag::new(['u', 'k', 's']),
    ),
    (
        LanguageSubtag::new(['u', 'r', 'k']),
        LanguageSubtag::new(['u', 'r', 'k']),
    ),
    (
        LanguageSubtag::new(['u', 'z', 'n']),
        LanguageSubtag::new(['u', 'z', 'n']),
    ),
    (
        LanguageSubtag::new(['u', 'z', 's']),
        LanguageSubtag::new(['u', 'z', 's']),
    ),
    (
        LanguageSubtag::new(['v', 'g', 't']),
        LanguageSubtag::new(['v', 'g', 't']),
    ),
    (
        LanguageSubtag::new(['v', 'k', 'k']),
        LanguageSubtag::new(['v', 'k', 'k']),
    ),
    (
        LanguageSubtag::new(['v', 'k', 't']),
        LanguageSubtag::new(['v', 'k', 't']),
    ),
    (
        LanguageSubtag::new(['v', 's', 'i']),
        LanguageSubtag::new(['v', 's', 'i']),
    ),
    (
        LanguageSubtag::new(['v', 's', 'l']),
        LanguageSubtag::new(['v', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['v', 's', 'v']),
        LanguageSubtag::new(['v', 's', 'v']),
    ),
    (
        LanguageSubtag::new(['w', 'b', 's']),
        LanguageSubtag::new(['w', 'b', 's']),
    ),
    (
        LanguageSubtag::new(['w', 'u', 'u']),
        LanguageSubtag::new(['w', 'u', 'u']),
    ),
    (
        LanguageSubtag::new(['x', 'k', 'i']),
        LanguageSubtag::new(['x', 'k', 'i']),
    ),
    (
        LanguageSubtag::new(['x', 'm', 'l']),
        LanguageSubtag::new(['x', 'm', 'l']),
    ),
    (
        LanguageSubtag::new(['x', 'm', 'm']),
        LanguageSubtag::new(['x', 'm', 'm']),
    ),
    (
        LanguageSubtag::new(['x', 'm', 's']),
        LanguageSubtag::new(['x', 'm', 's']),
    ),
    (
        LanguageSubtag::new(['y', 'd', 's']),
        LanguageSubtag::new(['y', 'd', 's']),
    ),
    (
        LanguageSubtag::new(['y', 'g', 's']),
        LanguageSubtag::new(['y', 'g', 's']),
    ),
    (
        LanguageSubtag::new(['y', 'h', 's']),
        LanguageSubtag::new(['y', 'h', 's']),
    ),
    (
        LanguageSubtag::new(['y', 's', 'l']),
        LanguageSubtag::new(['y', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['y', 's', 'm']),
        LanguageSubtag::new(['y', 's', 'm']),
    ),
    (
        LanguageSubtag::new(['y', 'u', 'e']),
        LanguageSubtag::new(['y', 'u', 'e']),
    ),
    (
        LanguageSubtag::new(['z', 'i', 'b']),
        LanguageSubtag::new(['z', 'i', 'b']),
    ),
    (
        LanguageSubtag::new(['z', 'l', 'm']),
        LanguageSubtag::new(['z', 'l', 'm']),
    ),
    (
        LanguageSubtag::new(['z', 'm', 'i']),
        LanguageSubtag::new(['z', 'm', 'i']),
    ),
    (
        LanguageSubtag::new(['z', 's', 'l']),
        LanguageSubtag::new(['z', 's', 'l']),
    ),
    (
        LanguageSubtag::new(['z', 's', 'm']),
        LanguageSubtag::new(['z', 's', 'm']),
    ),
];

pub const SCRIPTS: [ScriptSubtag; 208] = [
    ScriptSubtag::new(['A', 'd', 'l', 'm']),
    ScriptSubtag::new(['A', 'f', 'a', 'k']),
    ScriptSubtag::new(['A', 'g', 'h', 'b']),
    ScriptSubtag::new(['A', 'h', 'o', 'm']),
    ScriptSubtag::new(['A', 'r', 'a', 'b']),
    ScriptSubtag::new(['A', 'r', 'a', 'n']),
    ScriptSubtag::new(['A', 'r', 'm', 'i']),
    ScriptSubtag::new(['A', 'r', 'm', 'n']),
    ScriptSubtag::new(['A', 'v', 's', 't']),
    ScriptSubtag::new(['B', 'a', 'l', 'i']),
    ScriptSubtag::new(['B', 'a', 'm', 'u']),
    ScriptSubtag::new(['B', 'a', 's', 's']),
    ScriptSubtag::new(['B', 'a', 't', 'k']),
    ScriptSubtag::new(['B', 'e', 'n', 'g']),
    ScriptSubtag::new(['B', 'h', 'k', 's']),
    ScriptSubtag::new(['B', 'l', 'i', 's']),
    ScriptSubtag::new(['B', 'o', 'p', 'o']),
    ScriptSubtag::new(['B', 'r', 'a', 'h']),
    ScriptSubtag::new(['B', 'r', 'a', 'i']),
    ScriptSubtag::new(['B', 'u', 'g', 'i']),
    ScriptSubtag::new(['B', 'u', 'h', 'd']),
    ScriptSubtag::new(['C', 'a', 'k', 'm']),
    ScriptSubtag::new(['C', 'a', 'n', 's']),
    ScriptSubtag::new(['C', 'a', 'r', 'i']),
    ScriptSubtag::new(['C', 'h', 'a', 'm']),
    ScriptSubtag::new(['C', 'h', 'e', 'r']),
    ScriptSubtag::new(['C', 'h', 'r', 's']),
    ScriptSubtag::new(['C', 'i', 'r', 't']),
    ScriptSubtag::new(['C', 'o', 'p', 't']),
    ScriptSubtag::new(['C', 'p', 'm', 'n']),
    ScriptSubtag::new(['C', 'p', 'r', 't']),
    ScriptSubtag::new(['C', 'y', 'r', 'l']),
    ScriptSubtag::new(['C', 'y', 'r', 's']),
    ScriptSubtag::new(['D', 'e', 'v', 'a']),
    ScriptSubtag::new(['D', 'i', 'a', 'k']),
    ScriptSubtag::new(['D', 'o', 'g', 'r']),
    ScriptSubtag::new(['D', 's', 'r', 't']),
    ScriptSubtag::new(['D', 'u', 'p', 'l']),
    ScriptSubtag::new(['E', 'g', 'y', 'd']),
    ScriptSubtag::new(['E', 'g', 'y', 'h']),
    ScriptSubtag::new(['E', 'g', 'y', 'p']),
    ScriptSubtag::new(['E', 'l', 'b', 'a']),
    ScriptSubtag::new(['E', 'l', 'y', 'm']),
    ScriptSubtag::new(['E', 't', 'h', 'i']),
    ScriptSubtag::new(['G', 'e', 'o', 'k']),
    ScriptSubtag::new(['G', 'e', 'o', 'r']),
    ScriptSubtag::new(['G', 'l', 'a', 'g']),
    ScriptSubtag::new(['G', 'o', 'n', 'g']),
    ScriptSubtag::new(['G', 'o', 'n', 'm']),
    ScriptSubtag::new(['G', 'o', 't', 'h']),
    ScriptSubtag::new(['G', 'r', 'a', 'n']),
    ScriptSubtag::new(['G', 'r', 'e', 'k']),
    ScriptSubtag::new(['G', 'u', 'j', 'r']),
    ScriptSubtag::new(['G', 'u', 'r', 'u']),
    ScriptSubtag::new(['H', 'a', 'n', 'b']),
    ScriptSubtag::new(['H', 'a', 'n', 'g']),
    ScriptSubtag::new(['H', 'a', 'n', 'i']),
    ScriptSubtag::new(['H', 'a', 'n', 'o']),
    ScriptSubtag::new(['H', 'a', 'n', 's']),
    ScriptSubtag::new(['H', 'a', 'n', 't']),
    ScriptSubtag::new(['H', 'a', 't', 'r']),
    ScriptSubtag::new(['H', 'e', 'b', 'r']),
    ScriptSubtag::new(['H', 'i', 'r', 'a']),
    ScriptSubtag::new(['H', 'l', 'u', 'w']),
    ScriptSubtag::new(['H', 'm', 'n', 'g']),
    ScriptSubtag::new(['H', 'm', 'n', 'p']),
    ScriptSubtag::new(['H', 'r', 'k', 't']),
    ScriptSubtag::new(['H', 'u', 'n', 'g']),
    ScriptSubtag::new(['I', 'n', 'd', 's']),
    ScriptSubtag::new(['I', 't', 'a', 'l']),
    ScriptSubtag::new(['J', 'a', 'm', 'o']),
    ScriptSubtag::new(['J', 'a', 'v', 'a']),
    ScriptSubtag::new(['J', 'p', 'a', 'n']),
    ScriptSubtag::new(['J', 'u', 'r', 'c']),
    ScriptSubtag::new(['K', 'a', 'l', 'i']),
    ScriptSubtag::new(['K', 'a', 'n', 'a']),
    ScriptSubtag::new(['K', 'h', 'a', 'r']),
    ScriptSubtag::new(['K', 'h', 'm', 'r']),
    ScriptSubtag::new(['K', 'h', 'o', 'j']),
    ScriptSubtag::new(['K', 'i', 't', 'l']),
    ScriptSubtag::new(['K', 'i', 't', 's']),
    ScriptSubtag::new(['K', 'n', 'd', 'a']),
    ScriptSubtag::new(['K', 'o', 'r', 'e']),
    ScriptSubtag::new(['K', 'p', 'e', 'l']),
    ScriptSubtag::new(['K', 't', 'h', 'i']),
    ScriptSubtag::new(['L', 'a', 'n', 'a']),
    ScriptSubtag::new(['L', 'a', 'o', 'o']),
    ScriptSubtag::new(['L', 'a', 't', 'f']),
    ScriptSubtag::new(['L', 'a', 't', 'g']),
    ScriptSubtag::new(['L', 'a', 't', 'n']),
    ScriptSubtag::new(['L', 'e', 'k', 'e']),
    ScriptSubtag::new(['L', 'e', 'p', 'c']),
    ScriptSubtag::new(['L', 'i', 'm', 'b']),
    ScriptSubtag::new(['L', 'i', 'n', 'a']),
    ScriptSubtag::new(['L', 'i', 'n', 'b']),
    ScriptSubtag::new(['L', 'i', 's', 'u']),
    ScriptSubtag::new(['L', 'o', 'm', 'a']),
    ScriptSubtag::new(['L', 'y', 'c', 'i']),
    ScriptSubtag::new(['L', 'y', 'd', 'i']),
    ScriptSubtag::new(['M', 'a', 'h', 'j']),
    ScriptSubtag::new(['M', 'a', 'k', 'a']),
    ScriptSubtag::new(['M', 'a', 'n', 'd']),
    ScriptSubtag::new(['M', 'a', 'n', 'i']),
    ScriptSubtag::new(['M', 'a', 'r', 'c']),
    ScriptSubtag::new(['M', 'a', 'y', 'a']),
    ScriptSubtag::new(['M', 'e', 'd', 'f']),
    ScriptSubtag::new(['M', 'e', 'n', 'd']),
    ScriptSubtag::new(['M', 'e', 'r', 'c']),
    ScriptSubtag::new(['M', 'e', 'r', 'o']),
    ScriptSubtag::new(['M', 'l', 'y', 'm']),
    ScriptSubtag::new(['M', 'o', 'd', 'i']),
    ScriptSubtag::new(['M', 'o', 'n', 'g']),
    ScriptSubtag::new(['M', 'o', 'o', 'n']),
    ScriptSubtag::new(['M', 'r', 'o', 'o']),
    ScriptSubtag::new(['M', 't', 'e', 'i']),
    ScriptSubtag::new(['M', 'u', 'l', 't']),
    ScriptSubtag::new(['M', 'y', 'm', 'r']),
    ScriptSubtag::new(['N', 'a', 'n', 'd']),
    ScriptSubtag::new(['N', 'a', 'r', 'b']),
    ScriptSubtag::new(['N', 'b', 'a', 't']),
    ScriptSubtag::new(['N', 'e', 'w', 'a']),
    ScriptSubtag::new(['N', 'k', 'd', 'b']),
    ScriptSubtag::new(['N', 'k', 'g', 'b']),
    ScriptSubtag::new(['N', 'k', 'o', 'o']),
    ScriptSubtag::new(['N', 's', 'h', 'u']),
    ScriptSubtag::new(['O', 'g', 'a', 'm']),
    ScriptSubtag::new(['O', 'l', 'c', 'k']),
    ScriptSubtag::new(['O', 'r', 'k', 'h']),
    ScriptSubtag::new(['O', 'r', 'y', 'a']),
    ScriptSubtag::new(['O', 's', 'g', 'e']),
    ScriptSubtag::new(['O', 's', 'm', 'a']),
    ScriptSubtag::new(['O', 'u', 'g', 'r']),
    ScriptSubtag::new(['P', 'a', 'l', 'm']),
    ScriptSubtag::new(['P', 'a', 'u', 'c']),
    ScriptSubtag::new(['P', 'c', 'u', 'n']),
    ScriptSubtag::new(['P', 'e', 'l', 'm']),
    ScriptSubtag::new(['P', 'e', 'r', 'm']),
    ScriptSubtag::new(['P', 'h', 'a', 'g']),
    ScriptSubtag::new(['P', 'h', 'l', 'i']),
    ScriptSubtag::new(['P', 'h', 'l', 'p']),
    ScriptSubtag::new(['P', 'h', 'l', 'v']),
    ScriptSubtag::new(['P', 'h', 'n', 'x']),
    ScriptSubtag::new(['P', 'i', 'q', 'd']),
    ScriptSubtag::new(['P', 'l', 'r', 'd']),
    ScriptSubtag::new(['P', 'r', 't', 'i']),
    ScriptSubtag::new(['P', 's', 'i', 'n']),
    ScriptSubtag::new(['R', 'a', 'n', 'j']),
    ScriptSubtag::new(['R', 'j', 'n', 'g']),
    ScriptSubtag::new(['R', 'o', 'h', 'g']),
    ScriptSubtag::new(['R', 'o', 'r', 'o']),
    ScriptSubtag::new(['R', 'u', 'n', 'r']),
    ScriptSubtag::new(['S', 'a', 'm', 'r']),
    ScriptSubtag::new(['S', 'a', 'r', 'a']),
    ScriptSubtag::new(['S', 'a', 'r', 'b']),
    ScriptSubtag::new(['S', 'a', 'u', 'r']),
    ScriptSubtag::new(['S', 'g', 'n', 'w']),
    ScriptSubtag::new(['S', 'h', 'a', 'w']),
    ScriptSubtag::new(['S', 'h', 'r', 'd']),
    ScriptSubtag::new(['S', 'h', 'u', 'i']),
    ScriptSubtag::new(['S', 'i', 'd', 'd']),
    ScriptSubtag::new(['S', 'i', 'n', 'd']),
    ScriptSubtag::new(['S', 'i', 'n', 'h']),
    ScriptSubtag::new(['S', 'o', 'g', 'd']),
    ScriptSubtag::new(['S', 'o', 'g', 'o']),
    ScriptSubtag::new(['S', 'o', 'r', 'a']),
    ScriptSubtag::new(['S', 'o', 'y', 'o']),
    ScriptSubtag::new(['S', 'u', 'n', 'd']),
    ScriptSubtag::new(['S', 'y', 'l', 'o']),
    ScriptSubtag::new(['S', 'y', 'r', 'c']),
    ScriptSubtag::new(['S', 'y', 'r', 'e']),
    ScriptSubtag::new(['S', 'y', 'r', 'j']),
    ScriptSubtag::new(['S', 'y', 'r', 'n']),
    ScriptSubtag::new(['T', 'a', 'g', 'b']),
    ScriptSubtag::new(['T', 'a', 'k', 'r']),
    ScriptSubtag::new(['T', 'a', 'l', 'e']),
    ScriptSubtag::new(['T', 'a', 'l', 'u']),
    ScriptSubtag::new(['T', 'a', 'm', 'l']),
    ScriptSubtag::new(['T', 'a', 'n', 'g']),
    ScriptSubtag::new(['T', 'a', 'v', 't']),
    ScriptSubtag::new(['T', 'e', 'l', 'u']),
    ScriptSubtag::new(['T', 'e', 'n', 'g']),
    ScriptSubtag::new(['T', 'f', 'n', 'g']),
    ScriptSubtag::new(['T', 'g', 'l', 'g']),
    ScriptSubtag::new(['T', 'h', 'a', 'a']),
    ScriptSubtag::new(['T', 'h', 'a', 'i']),
    ScriptSubtag::new(['T', 'i', 'b', 't']),
    ScriptSubtag::new(['T', 'i', 'r', 'h']),
    ScriptSubtag::new(['T', 'n', 's', 'a']),
    ScriptSubtag::new(['T', 'o', 't', 'o']),
    ScriptSubtag::new(['U', 'g', 'a', 'r']),
    ScriptSubtag::new(['V', 'a', 'i', 'i']),
    ScriptSubtag::new(['V', 'i', 's', 'p']),
    ScriptSubtag::new(['V', 'i', 't', 'h']),
    ScriptSubtag::new(['W', 'a', 'r', 'a']),
    ScriptSubtag::new(['W', 'c', 'h', 'o']),
    ScriptSubtag::new(['W', 'o', 'l', 'e']),
    ScriptSubtag::new(['X', 'p', 'e', 'o']),
    ScriptSubtag::new(['X', 's', 'u', 'x']),
    ScriptSubtag::new(['Y', 'e', 'z', 'i']),
    ScriptSubtag::new(['Y', 'i', 'i', 'i']),
    ScriptSubtag::new(['Z', 'a', 'n', 'b']),
    ScriptSubtag::new(['Z', 'i', 'n', 'h']),
    ScriptSubtag::new(['Z', 'm', 't', 'h']),
    ScriptSubtag::new(['Z', 's', 'y', 'e']),
    ScriptSubtag::new(['Z', 's', 'y', 'm']),
    ScriptSubtag::new(['Z', 'x', 'x', 'x']),
    ScriptSubtag::new(['Z', 'y', 'y', 'y']),
    ScriptSubtag::new(['Z', 'z', 'z', 'z']),
];

pub const SCRIPTS_PREFERRED_VALUE: [(ScriptSubtag, ScriptSubtag); 0] = [];

pub const REGIONS: [RegionSubtag; 302] = [
    RegionSubtag::new(['0', '0', '1']),
    RegionSubtag::new(['0', '0', '2']),
    RegionSubtag::new(['0', '0', '3']),
    RegionSubtag::new(['0', '0', '5']),
    RegionSubtag::new(['0', '0', '9']),
    RegionSubtag::new(['0', '1', '1']),
    RegionSubtag::new(['0', '1', '3']),
    RegionSubtag::new(['0', '1', '4']),
    RegionSubtag::new(['0', '1', '5']),
    RegionSubtag::new(['0', '1', '7']),
    RegionSubtag::new(['0', '1', '8']),
    RegionSubtag::new(['0', '1', '9']),
    RegionSubtag::new(['0', '2', '1']),
    RegionSubtag::new(['0', '2', '9']),
    RegionSubtag::new(['0', '3', '0']),
    RegionSubtag::new(['0', '3', '4']),
    RegionSubtag::new(['0', '3', '5']),
    RegionSubtag::new(['0', '3', '9']),
    RegionSubtag::new(['0', '5', '3']),
    RegionSubtag::new(['0', '5', '4']),
    RegionSubtag::new(['0', '5', '7']),
    RegionSubtag::new(['0', '6', '1']),
    RegionSubtag::new(['1', '4', '2']),
    RegionSubtag::new(['1', '4', '3']),
    RegionSubtag::new(['1', '4', '5']),
    RegionSubtag::new(['1', '5', '0']),
    RegionSubtag::new(['1', '5', '1']),
    RegionSubtag::new(['1', '5', '4']),
    RegionSubtag::new(['1', '5', '5']),
    RegionSubtag::new(['2', '0', '2']),
    RegionSubtag::new(['4', '1', '9']),
    RegionSubtag::new(['A', 'A', ' ']),
    RegionSubtag::new(['A', 'C', ' ']),
    RegionSubtag::new(['A', 'D', ' ']),
    RegionSubtag::new(['A', 'E', ' ']),
    RegionSubtag::new(['A', 'F', ' ']),
    RegionSubtag::new(['A', 'G', ' ']),
    RegionSubtag::new(['A', 'I', ' ']),
    RegionSubtag::new(['A', 'L', ' ']),
    RegionSubtag::new(['A', 'M', ' ']),
    RegionSubtag::new(['A', 'N', ' ']),
    RegionSubtag::new(['A', 'O', ' ']),
    RegionSubtag::new(['A', 'Q', ' ']),
    RegionSubtag::new(['A', 'R', ' ']),
    RegionSubtag::new(['A', 'S', ' ']),
    RegionSubtag::new(['A', 'T', ' ']),
    RegionSubtag::new(['A', 'U', ' ']),
    RegionSubtag::new(['A', 'W', ' ']),
    RegionSubtag::new(['A', 'X', ' ']),
    RegionSubtag::new(['A', 'Z', ' ']),
    RegionSubtag::new(['B', 'A', ' ']),
    RegionSubtag::new(['B', 'B', ' ']),
    RegionSubtag::new(['B', 'D', ' ']),
    RegionSubtag::new(['B', 'E', ' ']),
    RegionSubtag::new(['B', 'F', ' ']),
    RegionSubtag::new(['B', 'G', ' ']),
    RegionSubtag::new(['B', 'H', ' ']),
    RegionSubtag::new(['B', 'I', ' ']),
    RegionSubtag::new(['B', 'J', ' ']),
    RegionSubtag::new(['B', 'L', ' ']),
    RegionSubtag::new(['B', 'M', ' ']),
    RegionSubtag::new(['B', 'N', ' ']),
    RegionSubtag::new(['B', 'O', ' ']),
    RegionSubtag::new(['B', 'Q', ' ']),
    RegionSubtag::new(['B', 'R', ' ']),
    RegionSubtag::new(['B', 'S', ' ']),
    RegionSubtag::new(['B', 'T', ' ']),
    RegionSubtag::new(['B', 'U', ' ']),
    RegionSubtag::new(['B', 'V', ' ']),
    RegionSubtag::new(['B', 'W', ' ']),
    RegionSubtag::new(['B', 'Y', ' ']),
    RegionSubtag::new(['B', 'Z', ' ']),
    RegionSubtag::new(['C', 'A', ' ']),
    RegionSubtag::new(['C', 'C', ' ']),
    RegionSubtag::new(['C', 'D', ' ']),
    RegionSubtag::new(['C', 'F', ' ']),
    RegionSubtag::new(['C', 'G', ' ']),
    RegionSubtag::new(['C', 'H', ' ']),
    RegionSubtag::new(['C', 'I', ' ']),
    RegionSubtag::new(['C', 'K', ' ']),
    RegionSubtag::new(['C', 'L', ' ']),
    RegionSubtag::new(['C', 'M', ' ']),
    RegionSubtag::new(['C', 'N', ' ']),
    RegionSubtag::new(['C', 'O', ' ']),
    RegionSubtag::new(['C', 'P', ' ']),
    RegionSubtag::new(['C', 'R', ' ']),
    RegionSubtag::new(['C', 'S', ' ']),
    RegionSubtag::new(['C', 'U', ' ']),
    RegionSubtag::new(['C', 'V', ' ']),
    RegionSubtag::new(['C', 'W', ' ']),
    RegionSubtag::new(['C', 'X', ' ']),
    RegionSubtag::new(['C', 'Y', ' ']),
    RegionSubtag::new(['C', 'Z', ' ']),
    RegionSubtag::new(['D', 'D', ' ']),
    RegionSubtag::new(['D', 'E', ' ']),
    RegionSubtag::new(['D', 'G', ' ']),
    RegionSubtag::new(['D', 'J', ' ']),
    RegionSubtag::new(['D', 'K', ' ']),
    RegionSubtag::new(['D', 'M', ' ']),
    RegionSubtag::new(['D', 'O', ' ']),
    RegionSubtag::new(['D', 'Z', ' ']),
    RegionSubtag::new(['E', 'A', ' ']),
    RegionSubtag::new(['E', 'C', ' ']),
    RegionSubtag::new(['E', 'E', ' ']),
    RegionSubtag::new(['E', 'G', ' ']),
    RegionSubtag::new(['E', 'H', ' ']),
    RegionSubtag::new(['E', 'R', ' ']),
    RegionSubtag::new(['E', 'S', ' ']),
    RegionSubtag::new(['E', 'T', ' ']),
    RegionSubtag::new(['E', 'U', ' ']),
    RegionSubtag::new(['E', 'Z', ' ']),
    RegionSubtag::new(['F', 'I', ' ']),
    RegionSubtag::new(['F', 'J', ' ']),
    RegionSubtag::new(['F', 'K', ' ']),
    RegionSubtag::new(['F', 'M', ' ']),
    RegionSubtag::new(['F', 'O', ' ']),
    RegionSubtag::new(['F', 'R', ' ']),
    RegionSubtag::new(['F', 'X', ' ']),
    RegionSubtag::new(['G', 'A', ' ']),
    RegionSubtag::new(['G', 'B', ' ']),
    RegionSubtag::new(['G', 'D', ' ']),
    RegionSubtag::new(['G', 'E', ' ']),
    RegionSubtag::new(['G', 'F', ' ']),
    RegionSubtag::new(['G', 'G', ' ']),
    RegionSubtag::new(['G', 'H', ' ']),
    RegionSubtag::new(['G', 'I', ' ']),
    RegionSubtag::new(['G', 'L', ' ']),
    RegionSubtag::new(['G', 'M', ' ']),
    RegionSubtag::new(['G', 'N', ' ']),
    RegionSubtag::new(['G', 'P', ' ']),
    RegionSubtag::new(['G', 'Q', ' ']),
    RegionSubtag::new(['G', 'R', ' ']),
    RegionSubtag::new(['G', 'S', ' ']),
    RegionSubtag::new(['G', 'T', ' ']),
    RegionSubtag::new(['G', 'U', ' ']),
    RegionSubtag::new(['G', 'W', ' ']),
    RegionSubtag::new(['G', 'Y', ' ']),
    RegionSubtag::new(['H', 'K', ' ']),
    RegionSubtag::new(['H', 'M', ' ']),
    RegionSubtag::new(['H', 'N', ' ']),
    RegionSubtag::new(['H', 'R', ' ']),
    RegionSubtag::new(['H', 'T', ' ']),
    RegionSubtag::new(['H', 'U', ' ']),
    RegionSubtag::new(['I', 'C', ' ']),
    RegionSubtag::new(['I', 'D', ' ']),
    RegionSubtag::new(['I', 'E', ' ']),
    RegionSubtag::new(['I', 'L', ' ']),
    RegionSubtag::new(['I', 'M', ' ']),
    RegionSubtag::new(['I', 'N', ' ']),
    RegionSubtag::new(['I', 'O', ' ']),
    RegionSubtag::new(['I', 'Q', ' ']),
    RegionSubtag::new(['I', 'R', ' ']),
    RegionSubtag::new(['I', 'S', ' ']),
    RegionSubtag::new(['I', 'T', ' ']),
    RegionSubtag::new(['J', 'E', ' ']),
    RegionSubtag::new(['J', 'M', ' ']),
    RegionSubtag::new(['J', 'O', ' ']),
    RegionSubtag::new(['J', 'P', ' ']),
    RegionSubtag::new(['K', 'E', ' ']),
    RegionSubtag::new(['K', 'G', ' ']),
    RegionSubtag::new(['K', 'H', ' ']),
    RegionSubtag::new(['K', 'I', ' ']),
    RegionSubtag::new(['K', 'M', ' ']),
    RegionSubtag::new(['K', 'N', ' ']),
    RegionSubtag::new(['K', 'P', ' ']),
    RegionSubtag::new(['K', 'R', ' ']),
    RegionSubtag::new(['K', 'W', ' ']),
    RegionSubtag::new(['K', 'Y', ' ']),
    RegionSubtag::new(['K', 'Z', ' ']),
    RegionSubtag::new(['L', 'A', ' ']),
    RegionSubtag::new(['L', 'B', ' ']),
    RegionSubtag::new(['L', 'C', ' ']),
    RegionSubtag::new(['L', 'I', ' ']),
    RegionSubtag::new(['L', 'K', ' ']),
    RegionSubtag::new(['L', 'R', ' ']),
    RegionSubtag::new(['L', 'S', ' ']),
    RegionSubtag::new(['L', 'T', ' ']),
    RegionSubtag::new(['L', 'U', ' ']),
    RegionSubtag::new(['L', 'V', ' ']),
    RegionSubtag::new(['L', 'Y', ' ']),
    RegionSubtag::new(['M', 'A', ' ']),
    RegionSubtag::new(['M', 'C', ' ']),
    RegionSubtag::new(['M', 'D', ' ']),
    RegionSubtag::new(['M', 'E', ' ']),
    RegionSubtag::new(['M', 'F', ' ']),
    RegionSubtag::new(['M', 'G', ' ']),
    RegionSubtag::new(['M', 'H', ' ']),
    RegionSubtag::new(['M', 'K', ' ']),
    RegionSubtag::new(['M', 'L', ' ']),
    RegionSubtag::new(['M', 'M', ' ']),
    RegionSubtag::new(['M', 'N', ' ']),
    RegionSubtag::new(['M', 'O', ' ']),
    RegionSubtag::new(['M', 'P', ' ']),
    RegionSubtag::new(['M', 'Q', ' ']),
    RegionSubtag::new(['M', 'R', ' ']),
    RegionSubtag::new(['M', 'S', ' ']),
    RegionSubtag::new(['M', 'T', ' ']),
    RegionSubtag::new(['M', 'U', ' ']),
    RegionSubtag::new(['M', 'V', ' ']),
    RegionSubtag::new(['M', 'W', ' ']),
    RegionSubtag::new(['M', 'X', ' ']),
    RegionSubtag::new(['M', 'Y', ' ']),
    RegionSubtag::new(['M', 'Z', ' ']),
    RegionSubtag::new(['N', 'A', ' ']),
    RegionSubtag::new(['N', 'C', ' ']),
    RegionSubtag::new(['N', 'E', ' ']),
    RegionSubtag::new(['N', 'F', ' ']),
    RegionSubtag::new(['N', 'G', ' ']),
    RegionSubtag::new(['N', 'I', ' ']),
    RegionSubtag::new(['N', 'L', ' ']),
    RegionSubtag::new(['N', 'O', ' ']),
    RegionSubtag::new(['N', 'P', ' ']),
    RegionSubtag::new(['N', 'R', ' ']),
    RegionSubtag::new(['N', 'T', ' ']),
    RegionSubtag::new(['N', 'U', ' ']),
    RegionSubtag::new(['N', 'Z', ' ']),
    RegionSubtag::new(['O', 'M', ' ']),
    RegionSubtag::new(['P', 'A', ' ']),
    RegionSubtag::new(['P', 'E', ' ']),
    RegionSubtag::new(['P', 'F', ' ']),
    RegionSubtag::new(['P', 'G', ' ']),
    RegionSubtag::new(['P', 'H', ' ']),
    RegionSubtag::new(['P', 'K', ' ']),
    RegionSubtag::new(['P', 'L', ' ']),
    RegionSubtag::new(['P', 'M', ' ']),
    RegionSubtag::new(['P', 'N', ' ']),
    RegionSubtag::new(['P', 'R', ' ']),
    RegionSubtag::new(['P', 'S', ' ']),
    RegionSubtag::new(['P', 'T', ' ']),
    RegionSubtag::new(['P', 'W', ' ']),
    RegionSubtag::new(['P', 'Y', ' ']),
    RegionSubtag::new(['Q', 'A', ' ']),
    RegionSubtag::new(['R', 'E', ' ']),
    RegionSubtag::new(['R', 'O', ' ']),
    RegionSubtag::new(['R', 'S', ' ']),
    RegionSubtag::new(['R', 'U', ' ']),
    RegionSubtag::new(['R', 'W', ' ']),
    RegionSubtag::new(['S', 'A', ' ']),
    RegionSubtag::new(['S', 'B', ' ']),
    RegionSubtag::new(['S', 'C', ' ']),
    RegionSubtag::new(['S', 'D', ' ']),
    RegionSubtag::new(['S', 'E', ' ']),
    RegionSubtag::new(['S', 'G', ' ']),
    RegionSubtag::new(['S', 'H', ' ']),
    RegionSubtag::new(['S', 'I', ' ']),
    RegionSubtag::new(['S', 'J', ' ']),
    RegionSubtag::new(['S', 'K', ' ']),
    RegionSubtag::new(['S', 'L', ' ']),
    RegionSubtag::new(['S', 'M', ' ']),
    RegionSubtag::new(['S', 'N', ' ']),
    RegionSubtag::new(['S', 'O', ' ']),
    RegionSubtag::new(['S', 'R', ' ']),
    RegionSubtag::new(['S', 'S', ' ']),
    RegionSubtag::new(['S', 'T', ' ']),
    RegionSubtag::new(['S', 'U', ' ']),
    RegionSubtag::new(['S', 'V', ' ']),
    RegionSubtag::new(['S', 'X', ' ']),
    RegionSubtag::new(['S', 'Y', ' ']),
    RegionSubtag::new(['S', 'Z', ' ']),
    RegionSubtag::new(['T', 'A', ' ']),
    RegionSubtag::new(['T', 'C', ' ']),
    RegionSubtag::new(['T', 'D', ' ']),
    RegionSubtag::new(['T', 'F', ' ']),
    RegionSubtag::new(['T', 'G', ' ']),
    RegionSubtag::new(['T', 'H', ' ']),
    RegionSubtag::new(['T', 'J', ' ']),
    RegionSubtag::new(['T', 'K', ' ']),
    RegionSubtag::new(['T', 'L', ' ']),
    RegionSubtag::new(['T', 'M', ' ']),
    RegionSubtag::new(['T', 'N', ' ']),
    RegionSubtag::new(['T', 'O', ' ']),
    RegionSubtag::new(['T', 'P', ' ']),
    RegionSubtag::new(['T', 'R', ' ']),
    RegionSubtag::new(['T', 'T', ' ']),
    RegionSubtag::new(['T', 'V', ' ']),
    RegionSubtag::new(['T', 'W', ' ']),
    RegionSubtag::new(['T', 'Z', ' ']),
    RegionSubtag::new(['U', 'A', ' ']),
    RegionSubtag::new(['U', 'G', ' ']),
    RegionSubtag::new(['U', 'M', ' ']),
    RegionSubtag::new(['U', 'N', ' ']),
    RegionSubtag::new(['U', 'S', ' ']),
    RegionSubtag::new(['U', 'Y', ' ']),
    RegionSubtag::new(['U', 'Z', ' ']),
    RegionSubtag::new(['V', 'A', ' ']),
    RegionSubtag::new(['V', 'C', ' ']),
    RegionSubtag::new(['V', 'E', ' ']),
    RegionSubtag::new(['V', 'G', ' ']),
    RegionSubtag::new(['V', 'I', ' ']),
    RegionSubtag::new(['V', 'N', ' ']),
    RegionSubtag::new(['V', 'U', ' ']),
    RegionSubtag::new(['W', 'F', ' ']),
    RegionSubtag::new(['W', 'S', ' ']),
    RegionSubtag::new(['Y', 'D', ' ']),
    RegionSubtag::new(['Y', 'E', ' ']),
    RegionSubtag::new(['Y', 'T', ' ']),
    RegionSubtag::new(['Y', 'U', ' ']),
    RegionSubtag::new(['Z', 'A', ' ']),
    RegionSubtag::new(['Z', 'M', ' ']),
    RegionSubtag::new(['Z', 'R', ' ']),
    RegionSubtag::new(['Z', 'W', ' ']),
    RegionSubtag::new(['Z', 'Z', ' ']),
];

pub const REGIONS_PREFERRED_VALUE: [(RegionSubtag, RegionSubtag); 6] = [
    (
        RegionSubtag::new(['B', 'U', ' ']),
        RegionSubtag::new(['M', 'M', ' ']),
    ),
    (
        RegionSubtag::new(['D', 'D', ' ']),
        RegionSubtag::new(['D', 'E', ' ']),
    ),
    (
        RegionSubtag::new(['F', 'X', ' ']),
        RegionSubtag::new(['F', 'R', ' ']),
    ),
    (
        RegionSubtag::new(['T', 'P', ' ']),
        RegionSubtag::new(['T', 'L', ' ']),
    ),
    (
        RegionSubtag::new(['Y', 'D', ' ']),
        RegionSubtag::new(['Y', 'E', ' ']),
    ),
    (
        RegionSubtag::new(['Z', 'R', ' ']),
        RegionSubtag::new(['C', 'D', ' ']),
    ),
];

pub const VARIANTS: [(&str, &str); 106] = [
    ("1606nict", "frm-"),
    ("1694acad", "fr-"),
    ("1901", "de-"),
    ("1959acad", "be-"),
    (
        "1994",
        "sl-rozaj- sl-rozaj-biske- sl-rozaj-njiva- sl-rozaj-osojs- sl-rozaj-solba-",
    ),
    ("1996", "de-"),
    ("abl1943", "pt-BR-"),
    ("akuapem", "tw-"),
    ("alalc97", ""),
    ("aluku", "djk-"),
    ("ao1990", "pt- gl-"),
    ("aranes", "oc-"),
    ("arevela", "hy-"),
    ("arevmda", "hy-"),
    ("arkaika", "eo-"),
    ("asante", "tw-"),
    ("auvern", "oc-"),
    ("baku1926", "az- ba- crh- kk- krc- ky- sah- tk- tt- uz-"),
    ("balanka", "blo-"),
    ("barla", "kea-"),
    ("basiceng", "en-"),
    ("bauddha", "sa-"),
    ("biscayan", "eu-"),
    ("biske", "sl-rozaj-"),
    ("bohoric", "sl-"),
    ("boont", "en-"),
    ("bornholm", "da-"),
    ("cisaup", "oc-"),
    ("colb1945", "pt-"),
    ("cornu", "en-"),
    ("creiss", "oc-"),
    ("dajnko", "sl-"),
    ("ekavsk", "sr- sr-Latn- sr-Cyrl-"),
    ("emodeng", "en-"),
    ("fonipa", ""),
    ("fonkirsh", ""),
    ("fonnapa", ""),
    ("fonupa", ""),
    ("fonxsamp", ""),
    ("gascon", "oc-"),
    ("grclass", "oc-"),
    ("grital", "oc-"),
    ("grmistr", "oc-"),
    ("hepburn", "ja-Latn-"),
    ("heploc", "ja-Latn-hepburn-"),
    ("hognorsk", "nn-"),
    ("hsistemo", "eo-"),
    ("ijekavsk", "sr- sr-Latn- sr-Cyrl-"),
    ("itihasa", "sa-"),
    ("ivanchov", "bg-"),
    ("jauer", "rm-"),
    ("jyutping", "yue-"),
    ("kkcor", "kw-"),
    ("kociewie", "pl-"),
    ("kscor", "kw-"),
    ("laukika", "sa-"),
    ("lemosin", "oc-"),
    ("lengadoc", "oc-"),
    ("lipaw", "sl-rozaj-"),
    ("luna1918", "ru-"),
    ("metelko", "sl-"),
    ("monoton", "el-"),
    ("ndyuka", "djk-"),
    ("nedis", "sl-"),
    ("newfound", "en-CA-"),
    ("nicard", "oc-"),
    ("njiva", "sl-rozaj-"),
    ("nulik", "vo-"),
    ("osojs", "sl-rozaj-"),
    ("oxendict", "en-"),
    ("pahawh2", "mww- hnj-"),
    ("pahawh3", "mww- hnj-"),
    ("pahawh4", "mww- hnj-"),
    ("pamaka", "djk-"),
    ("peano", "la-"),
    ("petr1708", "ru-"),
    ("pinyin", "zh-Latn- bo-Latn-"),
    ("polyton", "el-"),
    ("provenc", "oc-"),
    ("puter", "rm-"),
    ("rigik", "vo-"),
    ("rozaj", "sl-"),
    ("rumgr", "rm-"),
    ("scotland", "en-"),
    ("scouse", "en-"),
    ("simple", ""),
    ("solba", "sl-rozaj-"),
    ("sotav", "kea-"),
    ("spanglis", "en- es-"),
    ("surmiran", "rm-"),
    ("sursilv", "rm-"),
    ("sutsilv", "rm-"),
    ("tarask", "be-"),
    ("tongyong", "zh-Latn-"),
    ("tunumiit", "kl-"),
    ("uccor", "kw-"),
    ("ucrcor", "kw-"),
    ("ulster", "sco-"),
    ("unifon", "en- hup- kyh- tol- yur-"),
    ("vaidika", "sa-"),
    ("valencia", "ca-"),
    ("vallader", "rm-"),
    ("vecdruka", "lv-"),
    ("vivaraup", "oc-"),
    ("wadegile", "zh-Latn-"),
    ("xsistemo", "eo-"),
];

pub const VARIANTS_PREFERRED_VALUE: [(&str, &str); 1] = [("heploc", "alalc97")];

pub const GRANDFATHEREDS: [&str; 26] = [
    "art-lojban",
    "cel-gaulish",
    "en-GB-oed",
    "i-ami",
    "i-bnn",
    "i-default",
    "i-enochian",
    "i-hak",
    "i-klingon",
    "i-lux",
    "i-mingo",
    "i-navajo",
    "i-pwn",
    "i-tao",
    "i-tay",
    "i-tsu",
    "no-bok",
    "no-nyn",
    "sgn-BE-FR",
    "sgn-BE-NL",
    "sgn-CH-DE",
    "zh-guoyu",
    "zh-hakka",
    "zh-min",
    "zh-min-nan",
    "zh-xiang",
];

pub const GRANDFATHEREDS_PREFERRED_VALUE: [(&str, &str); 21] = [
    ("art-lojban", "jbo"),
    ("en-GB-oed", "en-GB-oxendict"),
    ("i-ami", "ami"),
    ("i-bnn", "bnn"),
    ("i-hak", "hak"),
    ("i-klingon", "tlh"),
    ("i-lux", "lb"),
    ("i-navajo", "nv"),
    ("i-pwn", "pwn"),
    ("i-tao", "tao"),
    ("i-tay", "tay"),
    ("i-tsu", "tsu"),
    ("no-bok", "nb"),
    ("no-nyn", "nn"),
    ("sgn-BE-FR", "sfb"),
    ("sgn-BE-NL", "vgt"),
    ("sgn-CH-DE", "sgg"),
    ("zh-guoyu", "cmn"),
    ("zh-hakka", "hak"),
    ("zh-min-nan", "nan"),
    ("zh-xiang", "hsn"),
];

pub const REDUNDANTS_PREFERRED_VALUE: [(&str, &str); 25] = [
    ("sgn-BR", "bzs"),
    ("sgn-CO", "csn"),
    ("sgn-DE", "gsg"),
    ("sgn-DK", "dsl"),
    ("sgn-ES", "ssp"),
    ("sgn-FR", "fsl"),
    ("sgn-GB", "bfi"),
    ("sgn-GR", "gss"),
    ("sgn-IE", "isg"),
    ("sgn-IT", "ise"),
    ("sgn-JP", "jsl"),
    ("sgn-MX", "mfs"),
    ("sgn-NI", "ncs"),
    ("sgn-NL", "dse"),
    ("sgn-NO", "nsl"),
    ("sgn-PT", "psr"),
    ("sgn-SE", "swl"),
    ("sgn-US", "ase"),
    ("sgn-ZA", "sfs"),
    ("zh-cmn", "cmn"),
    ("zh-cmn-Hans", "cmn-Hans"),
    ("zh-cmn-Hant", "cmn-Hant"),
    ("zh-gan", "gan"),
    ("zh-wuu", "wuu"),
    ("zh-yue", "yue"),
];
