/*
 * Copyright (C) 2021 Andreas Steffen, strongSec GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include "test_suite.h"

#include <plugins/plugin_feature.h>

typedef struct rsa_pkcs1_test_t rsa_pkcs1_test_t;

struct rsa_pkcs1_test_t {
	chunk_t msg;
	chunk_t ct;
};

/**
 * Private RSA key to load
 */
static chunk_t rsa_2048_key = chunk_from_chars( /* PKCS#8 format */
		0x30,0x82,0x04,0xbd,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x04,0xa7,0x30,0x82,0x04,0xa3,0x02,0x01,
		0x00,0x02,0x82,0x01,0x01,0x00,0xb3,0x51,0x0a,0x2b,0xcd,0x4c,0xe6,0x44,0xc5,0xb5,
		0x94,0xae,0x50,0x59,0xe1,0x2b,0x2f,0x05,0x4b,0x65,0x8d,0x5d,0xa5,0x95,0x9a,0x2f,
		0xdf,0x18,0x71,0xb8,0x08,0xbc,0x3d,0xf3,0xe6,0x28,0xd2,0x79,0x2e,0x51,0xaa,0xd5,
		0xc1,0x24,0xb4,0x3b,0xda,0x45,0x3d,0xca,0x5c,0xde,0x4b,0xcf,0x28,0xe7,0xbd,0x4e,
		0xff,0xba,0x0c,0xb4,0xb7,0x42,0xbb,0xb6,0xd5,0xa0,0x13,0xcb,0x63,0xd1,0xaa,0x3a,
		0x89,0xe0,0x26,0x27,0xef,0x53,0x98,0xb5,0x2c,0x0c,0xfd,0x97,0xd2,0x08,0xab,0xeb,
		0x8d,0x7c,0x9b,0xce,0x0b,0xbe,0xb0,0x19,0xa8,0x6d,0xdb,0x58,0x9b,0xeb,0x29,0xa5,
		0xb7,0x4b,0xf8,0x61,0x07,0x5c,0x67,0x7c,0x81,0xd4,0x30,0xf0,0x30,0xc2,0x65,0x24,
		0x7a,0xf9,0xd3,0xc9,0x14,0x0c,0xcb,0x65,0x30,0x9d,0x07,0xe0,0xad,0xc1,0xef,0xd1,
		0x5c,0xf1,0x7e,0x7b,0x05,0x5d,0x7d,0xa3,0x86,0x8e,0x46,0x48,0xcc,0x3a,0x18,0x0f,
		0x0e,0xe7,0xf8,0xe1,0xe7,0xb1,0x80,0x98,0xa3,0x39,0x1b,0x4c,0xe7,0x16,0x1e,0x98,
		0xd5,0x7a,0xf8,0xa9,0x47,0xe2,0x01,0xa4,0x63,0xe2,0xd6,0xbb,0xca,0x80,0x59,0xe5,
		0x70,0x6e,0x9d,0xfe,0xd8,0xf4,0x85,0x64,0x65,0xff,0xa7,0x12,0xed,0x1a,0xa1,0x8e,
		0x88,0x8d,0x12,0xdc,0x6a,0xa0,0x9c,0xe9,0x5e,0xcf,0xca,0x83,0xcc,0x5b,0x0b,0x15,
		0xdb,0x09,0xc8,0x64,0x7f,0x5d,0x52,0x4c,0x0f,0x2e,0x76,0x20,0xa3,0x41,0x6b,0x96,
		0x23,0xca,0xdc,0x0f,0x09,0x7a,0xf5,0x73,0x26,0x1c,0x98,0xc8,0x40,0x0a,0xa1,0x2a,
		0xf3,0x8e,0x43,0xca,0xd8,0x4d,0x02,0x03,0x01,0x00,0x01,0x02,0x82,0x01,0x00,0x1a,
		0x50,0x2d,0x0e,0xea,0x6c,0x7b,0x69,0xe2,0x1d,0x58,0x39,0x10,0x1f,0x70,0x54,0x56,
		0xed,0x0e,0xf8,0x52,0xfb,0x47,0xfe,0x21,0x07,0x1f,0x54,0xc5,0xf3,0x3c,0x8c,0xeb,
		0x06,0x6c,0x62,0xd7,0x27,0xe3,0x2d,0x26,0xc5,0x81,0x37,0x32,0x9f,0x89,0xd3,0x19,
		0x53,0x25,0xb7,0x95,0x26,0x4c,0x19,0x5d,0x85,0x47,0x2f,0x75,0x07,0xdb,0xd0,0x96,
		0x1d,0x29,0x51,0xf9,0x35,0xa2,0x6b,0x34,0xf0,0xac,0x24,0xd1,0x54,0x90,0xe1,0x12,
		0x8a,0x9b,0x71,0x38,0x91,0x5b,0xc7,0xdb,0xfa,0x8f,0xe3,0x96,0x35,0x71,0x31,0xc5,
		0x43,0xae,0x9c,0x98,0x50,0x73,0x68,0xd9,0xce,0xb0,0x8c,0x1c,0x61,0x98,0xa3,0xed,
		0xa7,0xae,0xa1,0x85,0xa0,0xe9,0x76,0xcd,0x42,0xc2,0x2d,0x00,0xf0,0x03,0xd9,0xf1,
		0x9d,0x96,0xea,0x4c,0x9a,0xfc,0xbf,0xe1,0x44,0x1c,0xcc,0x80,0x2c,0xfb,0x06,0x89,
		0xf5,0x9d,0x80,0x4c,0x6a,0x4e,0x4f,0x40,0x4c,0x15,0x17,0x47,0x45,0xed,0x6c,0xb8,
		0xbc,0x88,0xef,0x0b,0x33,0xba,0x0d,0x2a,0x80,0xe3,0x5e,0x43,0xbc,0x90,0xf3,0x50,
		0x05,0x2e,0x72,0x01,0x6e,0x75,0xb0,0x0d,0x35,0x7a,0x38,0x1c,0x9c,0x0d,0x46,0x70,
		0x69,0xca,0x66,0x08,0x87,0xc9,0x87,0x76,0x63,0x49,0xfc,0xc4,0x34,0x60,0xb4,0xaa,
		0x51,0x6b,0xce,0x07,0x9e,0xdd,0x87,0xba,0x16,0x43,0x07,0xb7,0x52,0xc2,0x77,0xed,
		0x95,0x28,0xad,0x3b,0xa0,0xbf,0x18,0x77,0x34,0x9e,0xd3,0xb7,0x96,0x6a,0x6c,0x24,
		0x01,0x10,0x40,0x9b,0xf4,0xd0,0xfa,0xde,0x0c,0x68,0xfd,0xad,0xd8,0x47,0xfd,0x02,
		0x81,0x81,0x00,0xec,0x12,0x5c,0xf3,0x7e,0x31,0x0a,0x2f,0xf4,0x62,0x63,0xb9,0xb2,
		0xe0,0x62,0x9d,0x63,0x90,0x00,0x5e,0xc8,0x89,0x13,0xd4,0xfb,0x71,0xbd,0x4d,0xd8,
		0x56,0x12,0x44,0x98,0xaa,0xeb,0xa9,0x83,0xd7,0xba,0x2b,0xd9,0x42,0xe6,0x4d,0x22,
		0x3f,0xeb,0x7a,0x23,0xaf,0x4d,0x60,0x5e,0xfe,0xea,0x6b,0xd7,0x0d,0x39,0xaf,0xe9,
		0x9d,0x35,0xa3,0xaa,0x15,0xe7,0x4a,0x17,0x68,0x77,0x80,0x93,0xbe,0x0e,0xdd,0x4a,
		0x8d,0x09,0xb2,0xde,0xf6,0xdc,0x9b,0x67,0xff,0x85,0x76,0x46,0x25,0xc2,0xe1,0x92,
		0x36,0xdb,0x4c,0x40,0x1c,0xe3,0x0a,0x25,0x72,0xd3,0xec,0xb4,0xf9,0x69,0xb7,0xad,
		0x19,0xc5,0x22,0xc0,0x2d,0x77,0x44,0x65,0x67,0x6e,0x1a,0x37,0x76,0xc5,0x4d,0x62,
		0x48,0x34,0x8b,0x02,0x81,0x81,0x00,0xc2,0x74,0x2a,0xbc,0xd9,0x89,0x7b,0xd4,0xb0,
		0xb6,0x71,0xf9,0x73,0xfc,0x82,0xa8,0xf8,0x4a,0xbf,0x57,0x05,0xff,0x88,0xdd,0x41,
		0x94,0x86,0x23,0xaf,0xe9,0xdc,0xa6,0x0d,0xc6,0x54,0x33,0x90,0x76,0x7f,0xea,0xeb,
		0xeb,0x53,0x95,0x76,0xee,0x8b,0xfa,0x61,0xb5,0xfc,0xbc,0xa9,0x4a,0x7c,0xef,0x75,
		0xa0,0x91,0x50,0xc5,0x40,0xfa,0x96,0x94,0xdd,0x80,0x04,0xad,0x23,0x71,0x8c,0x88,
		0x90,0x49,0x21,0x93,0x69,0xc9,0x9f,0x44,0x58,0xd4,0xaf,0xc1,0x48,0xf6,0xf0,0x7d,
		0xf8,0x73,0x24,0xa9,0x6d,0x9c,0xf7,0xb3,0x85,0xdd,0x86,0x22,0x41,0x4a,0x18,0x32,
		0xf9,0xf2,0x94,0x46,0xf0,0x50,0xc2,0xd5,0xa6,0x40,0x76,0x49,0xdc,0x41,0xab,0x70,
		0xe2,0x3b,0x3d,0xcc,0x22,0xc9,0x87,0x02,0x81,0x81,0x00,0x96,0xa9,0x79,0x8d,0x25,
		0x02,0x63,0x40,0x0b,0xb6,0x27,0x73,0x42,0x88,0x16,0x27,0xe0,0x7c,0xec,0xdf,0x91,
		0x18,0x7b,0x01,0xb8,0x9f,0xf4,0x73,0x14,0x18,0x8a,0x7c,0x20,0xfb,0x24,0x80,0x01,
		0x56,0xd2,0xc8,0x5d,0x56,0x66,0xe8,0xdf,0x6c,0xef,0xf9,0xf9,0x80,0x4d,0xdf,0xad,
		0x80,0xff,0x57,0x67,0xde,0x56,0xec,0xc0,0x29,0xc7,0x2b,0xf6,0xc7,0x17,0xdf,0x9f,
		0x64,0xda,0xaf,0xc2,0x9a,0xcf,0x9d,0xc7,0x90,0x8f,0x9a,0x0a,0xd6,0x7e,0x20,0xe8,
		0x94,0x99,0x36,0xcc,0xba,0x18,0xd0,0x21,0xa2,0xc4,0xfe,0xbb,0x04,0x34,0x9a,0x2b,
		0x20,0x47,0xc4,0x90,0x13,0x85,0xb6,0xe5,0xd0,0xc6,0x91,0xd1,0x18,0xb3,0x3f,0x81,
		0x80,0x2b,0x32,0xac,0x27,0x2e,0xf0,0x9e,0x42,0xfa,0xd5,0x02,0x81,0x80,0x05,0x54,
		0xf4,0x1b,0x0b,0x87,0xf6,0x8a,0x45,0x72,0x2b,0x3b,0xe0,0xcf,0x4a,0xb1,0xe1,0x65,
		0x03,0x4c,0x1a,0x91,0x00,0x2a,0xb8,0xf2,0x9e,0x9e,0xf9,0xe2,0xda,0xb6,0xfe,0xe7,
		0xb2,0x45,0x5b,0xaf,0xb4,0x20,0x37,0xe9,0xd2,0xf7,0xe5,0x33,0xf3,0x48,0xa1,0x47,
		0x41,0x2f,0xd7,0x20,0x80,0xbe,0x7c,0x26,0x33,0xf5,0xd8,0x02,0xc9,0x1c,0x39,0xe6,
		0xbc,0xec,0xe3,0xe6,0x75,0xe5,0x99,0x95,0x03,0x3c,0x55,0x73,0x70,0x20,0xda,0xd9,
		0xe8,0xb3,0x0d,0x04,0xb8,0x28,0xad,0xfb,0x93,0x04,0xad,0x54,0xa1,0x1a,0x35,0xa4,
		0xf5,0x07,0x09,0x87,0x6a,0xc5,0xb1,0x18,0x23,0x6b,0xa7,0x6a,0x4d,0x7c,0x9a,0x29,
		0x1d,0xd9,0x60,0x7b,0x16,0x9d,0xe1,0xd1,0x82,0x38,0x56,0x91,0x99,0x9f,0x02,0x81,
		0x80,0x1c,0x64,0x01,0x89,0xd9,0xbf,0xe8,0xc6,0x23,0x83,0x32,0x10,0xa7,0x6c,0x42,
		0x0c,0x6f,0x44,0xe5,0xd7,0x60,0xe2,0x59,0x91,0x6c,0xec,0x2a,0xe2,0xb1,0x56,0x45,
		0x69,0x60,0xfd,0x95,0xe2,0x74,0x76,0x60,0xc3,0x89,0x56,0x22,0x50,0xf0,0x55,0x04,
		0x9c,0xfa,0xb7,0xe5,0xc3,0x03,0x95,0x49,0x38,0x4a,0x7a,0x2a,0xae,0xb1,0xc8,0x24,
		0xd3,0xaf,0x70,0x94,0x82,0xa8,0xcf,0x9b,0x58,0x70,0x22,0xa0,0x0b,0x1f,0x07,0x22,
		0xdb,0x50,0xf3,0x3c,0xb2,0x6d,0xc2,0x0d,0xd2,0x24,0x5d,0x52,0x65,0xdf,0x61,0xee,
		0x29,0x83,0xc9,0x38,0xc2,0x16,0x7d,0xce,0xe1,0x21,0xfc,0x4b,0x44,0x79,0xc2,0x37,
		0xe7,0x28,0xcf,0x63,0x3a,0xb6,0x0a,0x8c,0x0e,0xcd,0x04,0xfc,0xe7,0xe3,0xba,0xa5,
		0x59);

/**
 * RSA PKCS1 decryption test vectors from https://boringssl.googlesource.com/
 */
static rsa_pkcs1_test_t rsa_2048_pkcs1_tests[] = {
	{ chunk_from_chars(                                                          /* 2 */
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00),
	  chunk_from_chars(
		0xa9,0xac,0xec,0x7e,0x58,0x76,0x1d,0x91,0x91,0x24,0x9f,0xf7,0xea,0x5d,0xb4,0x99,
		0xca,0xdc,0xcc,0x51,0xd2,0x9f,0x8e,0x7f,0xd0,0xaa,0x2c,0xb9,0x96,0x20,0x95,0x62,
		0x6f,0x1c,0xad,0xae,0x29,0x66,0x6f,0x04,0xce,0x2a,0xfd,0x4b,0x65,0x0b,0xe5,0x9d,
		0x07,0x1d,0x06,0x44,0x6d,0x59,0x10,0x7e,0xb5,0x08,0xcc,0x60,0x54,0x57,0x27,0xb0,
		0x56,0x7d,0xfb,0x4f,0x2f,0x94,0xca,0x60,0xb9,0x39,0xc6,0x0b,0xe1,0x11,0x17,0x2f,
		0x36,0x7d,0xfd,0x23,0x55,0x16,0xe4,0xa6,0x00,0x61,0x64,0x8c,0x67,0xf5,0x53,0x66,
		0x50,0x82,0x1a,0xc2,0xa6,0x07,0x44,0xbe,0x3c,0xf6,0xbe,0xfa,0x8f,0x66,0xe7,0x6a,
		0x3e,0x7c,0x5f,0xbc,0x6d,0xfa,0x4d,0xda,0x55,0xec,0xbd,0xbf,0xfd,0xc9,0x8d,0x61,
		0x0d,0xe5,0x66,0x7a,0x4f,0x48,0x5f,0x61,0x68,0xb5,0x2b,0xbe,0x47,0x0e,0x60,0x14,
		0x25,0x38,0x74,0xce,0x7b,0x78,0xe5,0x09,0x93,0x7e,0x0b,0xc5,0xf0,0x28,0x57,0xe1,
		0xad,0x3c,0xf5,0x51,0x39,0xbb,0xe6,0xdc,0x7a,0xc4,0xb1,0xed,0x50,0x97,0xbf,0x78,
		0x1b,0x76,0x71,0xca,0x9b,0xb5,0x81,0x87,0xaa,0x6c,0x71,0xc5,0x8a,0xc0,0x56,0x1c,
		0x5a,0xac,0xf9,0x6c,0x35,0xde,0xb2,0x4e,0x39,0x5b,0x68,0x23,0xde,0x7f,0xc9,0x6b,
		0x80,0x31,0xb5,0x90,0x6a,0x34,0xc4,0xdc,0x57,0xe4,0xf1,0x22,0x61,0x57,0xb9,0xab,
		0xd8,0x49,0xe1,0x36,0x7d,0xda,0x01,0x4f,0xbf,0x9e,0xd4,0xca,0x51,0x5a,0x7a,0x04,
		0xcf,0x87,0x78,0x79,0x45,0x00,0x7e,0x4f,0x63,0xc0,0x36,0x6a,0x5b,0xbc,0x34,0x89)
	},
	{ chunk_from_chars(                                                          /* 3 */
		0x54,0x65,0x73,0x74),
	  chunk_from_chars(
		0x45,0x01,0xb4,0xd6,0x69,0xe0,0x1b,0x9e,0xf2,0xdc,0x80,0x0a,0xa1,0xb0,0x6d,0x49,
		0x19,0x6f,0x5a,0x09,0xfe,0x8f,0xbc,0xd0,0x37,0x32,0x3c,0x60,0xea,0xf0,0x27,0xbf,
		0xb9,0x84,0x32,0xbe,0x4e,0x4a,0x26,0xc5,0x67,0xff,0xec,0x71,0x8b,0xcb,0xea,0x97,
		0x7d,0xd2,0x68,0x12,0xfa,0x07,0x1c,0x33,0x80,0x8b,0x4d,0x5e,0xbb,0x74,0x2d,0x98,
		0x79,0x80,0x60,0x94,0xb6,0xfb,0xee,0xa6,0x3d,0x25,0xea,0x31,0x41,0x73,0x3b,0x60,
		0xe3,0x1c,0x69,0x12,0x10,0x6e,0x1b,0x75,0x8a,0x7f,0xe0,0x01,0x4f,0x07,0x51,0x93,
		0xfa,0xa8,0xb4,0x62,0x2b,0xfd,0x5d,0x30,0x13,0xf0,0xa3,0x21,0x90,0xa9,0x5d,0xe6,
		0x1a,0x36,0x04,0x71,0x1b,0xc6,0x29,0x45,0xf9,0x5a,0x65,0x22,0xbd,0x4d,0xfe,0xd0,
		0xa9,0x94,0xef,0x18,0x5b,0x28,0xc2,0x81,0xf7,0xb5,0xe4,0xc8,0xed,0x41,0x17,0x6d,
		0x12,0xd9,0xfc,0x1b,0x83,0x7e,0x6a,0x01,0x11,0xd0,0x13,0x2d,0x08,0xa6,0xd6,0xf0,
		0x58,0x0d,0xe0,0xc9,0xee,0xd8,0xed,0x10,0x55,0x31,0x79,0x94,0x82,0xd1,0xe4,0x66,
		0xc6,0x8c,0x23,0xb0,0xc2,0x22,0xaf,0x7f,0xc1,0x2a,0xc2,0x79,0xbc,0x4f,0xf5,0x7e,
		0x7b,0x45,0x86,0xd2,0x09,0x37,0x1b,0x38,0xc4,0xc1,0x03,0x5e,0xdd,0x41,0x8d,0xc5,
		0xf9,0x60,0x44,0x1c,0xb2,0x1e,0xa2,0xbe,0xdb,0xfe,0xa8,0x6d,0xe0,0xd7,0x86,0x1e,
		0x81,0x02,0x1b,0x65,0x0a,0x1d,0xe5,0x10,0x02,0xc3,0x15,0xf1,0xe7,0xc1,0x2d,0xeb,
		0xe4,0xdc,0xeb,0xf7,0x90,0xca,0xaa,0x54,0xa2,0xf2,0x6b,0x14,0x9c,0xf9,0xe7,0x7d)
	},
	{ chunk_from_chars(                                                          /* 4 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x45,0x5f,0xe8,0xc7,0xc5,0x9d,0x08,0xc0,0x68,0xb5,0xff,0x73,0x9d,0x8d,0xab,0x91,
		0x2b,0x63,0x9c,0x8e,0x9e,0xad,0xe5,0xd0,0x51,0x9d,0x58,0xf4,0xea,0xd7,0x20,0x8d,
		0x5a,0x75,0x3b,0x4a,0x88,0xfe,0x77,0x14,0x75,0xad,0xc8,0x2d,0x10,0xab,0x29,0xde,
		0xd2,0x8c,0xaf,0x03,0xf9,0x03,0x4d,0x3a,0x11,0x1b,0x52,0x04,0x40,0xc0,0x22,0x76,
		0xe1,0xb6,0x41,0x7c,0x42,0xee,0xc0,0x25,0x7f,0x1f,0x05,0x48,0x28,0x68,0x98,0x7f,
		0x2f,0x75,0xbd,0x33,0xd1,0xec,0x3d,0xbc,0x79,0x9d,0x7b,0x5b,0xf2,0x5c,0x4a,0x05,
		0x43,0x79,0x3a,0x4d,0x3c,0xe3,0x05,0xcc,0x43,0x64,0x6b,0xc4,0x50,0x34,0x4e,0x62,
		0x4f,0xd3,0x81,0xe2,0x4d,0x8e,0x57,0xef,0x28,0x40,0xdd,0x9d,0x57,0x6d,0xa5,0x54,
		0xba,0x40,0x8e,0xe6,0x58,0x01,0x59,0xe6,0xd8,0x84,0x38,0xa2,0x8d,0x66,0x25,0x0b,
		0x3b,0x3f,0xe3,0xbc,0x66,0x24,0x40,0x60,0x22,0xa9,0xe4,0xee,0x27,0x78,0xc3,0x82,
		0x30,0x67,0x4f,0x63,0x5f,0x56,0xb9,0xd6,0xad,0xcf,0x2b,0xe6,0xbf,0xab,0x34,0xa8,
		0xa4,0x31,0x16,0x9d,0x76,0x98,0x76,0x42,0x2f,0x70,0x77,0xde,0xd3,0x1f,0xa6,0xf2,
		0x99,0x93,0xdd,0x19,0x72,0xb2,0xd2,0xd2,0x4b,0x05,0x13,0xa7,0xa1,0x93,0xf6,0xa8,
		0x8d,0x53,0xc4,0x9c,0xde,0x2c,0x03,0x0f,0x85,0xe3,0xdd,0xfb,0xc9,0xf9,0x9b,0x4a,
		0x66,0x7f,0xd9,0xc6,0x52,0x38,0x22,0x38,0x16,0x6f,0x3d,0x39,0xeb,0x2b,0x78,0xde,
		0x53,0xad,0x24,0xc9,0x76,0x99,0xfe,0x57,0x38,0xa7,0xa7,0x05,0xa2,0xab,0x14,0x1b)
	},
	{ chunk_from_chars(                                                          /* 8 */
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78),
	  chunk_from_chars(
	  	0x7e,0x15,0x1b,0x7b,0x34,0xe3,0xb4,0xab,0xe0,0x45,0xcf,0x70,0x86,0x40,0xe6,0x15,
	  	0x01,0xc5,0x0f,0xdc,0xa6,0x29,0xae,0xca,0x10,0x25,0x9d,0x45,0xd1,0x5e,0xee,0xc6,
	  	0xa2,0x00,0x8b,0x63,0x36,0xf5,0x70,0x21,0xac,0x6f,0xda,0xd9,0xa6,0xb2,0x9d,0x65,
	  	0xf0,0x98,0xab,0xff,0x76,0xf9,0x37,0x22,0xa8,0xe2,0x3f,0xb5,0xe1,0x53,0xdb,0x07,
	  	0x50,0x05,0x57,0x5d,0xad,0x6a,0xdc,0xcb,0x7c,0x02,0x0c,0xd7,0x41,0xc3,0x41,0x99,
	  	0x46,0xb8,0x2d,0x36,0x9a,0x07,0xfa,0xd5,0xb0,0xd5,0x5d,0x51,0x77,0x4f,0x89,0x91,
	  	0xbd,0x65,0xe9,0xe8,0x28,0xd8,0xf5,0xa9,0x89,0xc8,0x66,0xa0,0x24,0xa4,0xa7,0x84,
	  	0x34,0xe9,0xaf,0xfd,0x0a,0xf2,0xc7,0x2f,0x91,0x85,0xd4,0x50,0xb6,0x27,0x00,0x8a,
	  	0x8a,0x09,0x68,0xfc,0x63,0x73,0xca,0x34,0x04,0x10,0x30,0x6a,0x58,0x92,0x1c,0xce,
	  	0x12,0x07,0xbb,0x6f,0x6c,0x14,0xe3,0xd1,0xf2,0x14,0x30,0x4f,0x9f,0x6b,0xb9,0x19,
	  	0x99,0x09,0xe1,0x61,0x03,0x22,0xe8,0x34,0xb0,0xce,0x9f,0x55,0xb1,0x83,0x5d,0x76,
	  	0x23,0xb8,0x2e,0xf5,0x48,0x54,0x5f,0x98,0x4e,0xa5,0x14,0x66,0x25,0x01,0x59,0x34,
	  	0x4d,0xde,0x90,0x2a,0x0f,0x02,0x1b,0xa4,0xba,0xf2,0x6b,0x16,0xd8,0xc6,0xa4,0x20,
	  	0x03,0xf4,0xd5,0xdc,0xae,0x53,0x11,0x87,0xdc,0x7e,0x3f,0x87,0xc9,0xe0,0x44,0x70,
	  	0x59,0x9e,0xb6,0x23,0xe0,0x4f,0xca,0x26,0x6e,0x86,0xf9,0x8c,0xab,0xb6,0x86,0x60,
	  	0x04,0xe7,0xfc,0x80,0xb3,0x6c,0x39,0x77,0x45,0x6e,0x51,0xeb,0x64,0xf4,0xb6,0x5f)
	}
};

static void test_rsa_pkcs1(chunk_t key, encryption_scheme_t encryption_scheme,
						  rsa_pkcs1_test_t *test)
{
	private_key_t *privkey;
	public_key_t *pubkey;
	chunk_t msg = chunk_empty, ct = chunk_empty;

	privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_ANY,
								 BUILD_BLOB_ASN1_DER, key, BUILD_END);
	ck_assert(privkey != NULL);
	ck_assert(privkey->get_type(privkey) == KEY_RSA);

	pubkey = privkey->get_public_key(privkey);
	ck_assert(pubkey != NULL);

	/* decrypt */
	ck_assert(privkey->decrypt(privkey, encryption_scheme, NULL, test->ct, &msg));
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&msg);

	/* encrypt and decrypt again */
	ck_assert(pubkey->encrypt(pubkey, encryption_scheme, NULL, test->msg, &ct));
	ck_assert(privkey->decrypt(privkey, encryption_scheme, NULL, ct, &msg));
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&ct);
	chunk_free(&msg);

	/* cleanup */
	pubkey->destroy(pubkey);
	privkey->destroy(privkey);
}

START_TEST(test_rsa_2048_pkcs1)
{
	test_rsa_pkcs1(rsa_2048_key, ENCRYPT_RSA_PKCS1,
				  &rsa_2048_pkcs1_tests[_i]);
}
END_TEST

Suite *rsa_pkcs1_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("rsa_pkcs1");

	tc = tcase_create("rsa_2048_pkcs1");
	tcase_add_loop_test(tc, test_rsa_2048_pkcs1, 0, countof(rsa_2048_pkcs1_tests));
	suite_add_tcase(s, tc);

	return s;
}
