/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "VtkMeshComponent.h"
#include "VtkMeshComponentProperties.h"
#include "VtkMeshComponentPopup.h"

// -- CamiTK stuff
#include <Application.h>
#include <MeshComponent.h>
#include <Geometry.h>
#include <InteractiveViewer.h>

// -- vtk stuff
#include <vtkPolyData.h>
#include <vtkCleanPolyData.h>
#include <vtkPointData.h>
#include <vtkCell.h>
#include <vtkTriangleFilter.h>
#include <vtkGeometryFilter.h>
#include <vtkPolyDataNormals.h>
#include <vtkDoubleArray.h>

using namespace camitk;


// -------------------- default constructor  --------------------
VtkMeshComponent::VtkMeshComponent (const QString& fileName)  throw(AbortException) : MeshComponent ( fileName ) {

    whatIsIt = VtkMeshUtil::typeOf(myFileName.toStdString());

    if (whatIsIt == VtkMeshUtil::UNKNOWN) {
        throw AbortException("Unable to recognize this vtk file\n(it is neither an unstructured grid nor a poly data).");
    }

    setName(VtkMeshUtil::getVtkPointSetHeaderString(myFileName.toStdString()).c_str());

    myPopupMenu = NULL;
    myProperties = NULL;
    initialPointData = NULL;
    demoPointData = NULL;

    // build the vtk data structure from the file
    vtkSmartPointer<vtkPointSet> data = VtkMeshUtil::buildVtkPointSet(myFileName.toStdString().c_str(), whatIsIt);

    // instanciate the Geometry
    initRepresentation(data);

    // initialize point data value (for demo or display)
    initPointData();

}

// -------------------- destructor --------------------
VtkMeshComponent::~VtkMeshComponent() {
    setPointData ( NULL );
    InteractiveViewer::get3DViewer()->setColorScale ( false );
}

// -------------------- initPointData --------------------
void VtkMeshComponent::initPointData() {
    // create a new vtkDataArray to store the demo value
    demoPointData = vtkSmartPointer<vtkDoubleArray>::New();

    int numberOfPoints = getPointSet()->GetNumberOfPoints();
    demoPointData->SetNumberOfValues ( numberOfPoints );
    demoPointData->SetName ( "the point data" );
    for ( vtkIdType i = 0; i < numberOfPoints;++i ) {
        demoPointData->SetValue ( i, 0.0 );
    }
    // backup the read point data
    initialPointData = vtkSmartPointer<vtkPointData>::New();
    initialPointData->ShallowCopy(myGeometry->getPointSet()->GetPointData());

    // remove the point data
    setPointData(NULL);
}

// -------------------- pointPicked --------------------
void VtkMeshComponent::pointPicked ( vtkIdType pointId, bool sel/* picking does not change the selection state, do not bother with the 2nd parameter*/ ) {
    MeshComponent::pointPicked(pointId, sel);
    // picking show demo point data
    demoPointData->SetValue ( pointId, 1.0 - double ( pointId ) / double ( getPointSet()->GetNumberOfPoints() ) );
    demoPointData->Modified(); // force an update on the vtk pipeline
    refresh(); // refresh display
}

// -------------------- cellPicked --------------------
void VtkMeshComponent::cellPicked ( vtkIdType cellId, bool  sel /* picking does not change the selection state, do not bother with the 2nd parameter*/ ) {
    MeshComponent::cellPicked(cellId, sel);
    // change data for all cell points
    vtkSmartPointer<vtkCell> c = getPointSet()->GetCell ( cellId );
    for ( vtkIdType i = 0; i < c->GetNumberOfPoints(); i++ ) {
        vtkIdType pointId = c->GetPointId ( i );
        demoPointData->SetValue ( pointId, 1.0 - double ( pointId ) / double ( getPointSet()->GetNumberOfPoints() ) );
    }
    demoPointData->Modified(); // force an update on the vtk pipeline
    refresh(); // refresh display
}

// -------------------- showPointData --------------------
void VtkMeshComponent::showPointData(VTK_COMPONENT_POINT_DATA_TYPE type) {
    switch (type) {
    case NONE:
        setPointData ( NULL );
        InteractiveViewer::get3DViewer()->setColorScale ( false );
        break;
    case DEMO:
        setPointData ( demoPointData );
        InteractiveViewer::get3DViewer()->setColorScaleMinMax ( 0.0, getPointSet()->GetNumberOfPoints() );
        InteractiveViewer::get3DViewer()->setColorScaleTitle("Demo Point Data");
        InteractiveViewer::get3DViewer()->setColorScale ( true );
        break;
    case INITIAL:
        myGeometry->getPointSet()->GetPointData()->ShallowCopy(initialPointData);
        if (initialPointData->GetNumberOfArrays() > 0) {
            double range[2];
            myGeometry->getPointSet()->Update(); // force vtk to compute the range
            myGeometry->getPointSet()->GetPointData()->GetScalars()->GetRange ( range );
            InteractiveViewer::get3DViewer()->setColorScaleMinMax ( range[0], range[1] );
            InteractiveViewer::get3DViewer()->setColorScaleTitle("VTK Point Data");
            InteractiveViewer::get3DViewer()->setColorScale ( true );
        }
        else {
            InteractiveViewer::get3DViewer()->setColorScale ( false );
        }
        break;
    }
    refresh();
}

// -------------------- exportMDL --------------------
bool VtkMeshComponent::exportMDL ( std::string filename ) {
    if ( myGeometry ) {
        int i;
        // MDL format supports only triangles. Hence, this dataset triangularized before
        // being exported (this operation is then NOT reversible).
        // if the dataset is a volumetric mesh (e.g. hexahedrons), only the external surface is exported.

        //extract external surface
        vtkSmartPointer<vtkGeometryFilter> geomF = vtkSmartPointer<vtkGeometryFilter>::New();
        geomF->SetInput(getPointSet());


        // triangles
        vtkSmartPointer<vtkTriangleFilter> triangleF = vtkSmartPointer<vtkTriangleFilter>::New();
        triangleF->SetInput(geomF->GetOutput());

        // clean unused
        vtkSmartPointer<vtkCleanPolyData> cleaner = vtkSmartPointer<vtkCleanPolyData>::New();
        cleaner->SetInput(triangleF->GetOutput());
        cleaner->Update();

        vtkSmartPointer<vtkPolyData> ds = cleaner->GetOutput();



        //--- write as .mdl file (Registration format)
        std::ofstream o ( filename.c_str() );

        //--- name
        o << "[Name, STRING]" << std::endl;
        o << getName().toStdString() << std::endl;
        o << std::endl;


        //--- vertices
        o << "[Vertices, ARRAY1<POINT3D>]" << std::endl;
        o << ds->GetPoints()->GetNumberOfPoints() << std::endl;

        double pt[3];

        for ( i = 0; i < ds->GetPoints()->GetNumberOfPoints(); i++ ) {
            ds->GetPoints()->GetPoint ( i, pt );
            o << pt[0] << " " << pt[1] << " " << pt[2] << std::endl;
        }

        o << std::endl;


        //--- normals
        o << "[Normals, ARRAY1<VECTOR3D>]" << std::endl;

        if ( ds->GetPointData() && ds->GetPointData()->GetNormals() ) {
            // save existing normals
            o << ds->GetPointData()->GetNormals()->GetNumberOfTuples() << std::endl;

            for ( i = 0; i < ds->GetPointData()->GetNormals()->GetNumberOfTuples(); i++ ) {
                ds->GetPointData()->GetNormals()->GetTuple ( i, pt );
                o << pt[0] << " " << pt[1] << " " << pt[2] << std::endl;
            }
        }
        else {
            // compute the normals
            vtkPolyDataNormals * pNormals = vtkPolyDataNormals::New();
            pNormals->SetInput ( ds );
            pNormals->Update();

            o << pNormals->GetOutput()->GetPointData()->GetNormals()->GetNumberOfTuples() << std::endl;

            for ( i = 0; i < pNormals->GetOutput()->GetPointData()->GetNormals()->GetNumberOfTuples(); i++ ) {
                pNormals->GetOutput()->GetPointData()->GetNormals()->GetTuple ( i, pt );
                o << pt[0] << " " << pt[1] << " " << pt[2] << std::endl;
            }

        }

        o << std::endl;


        //--- triangles
        int j;
        o << "[Triangles, ARRAY1<STRING>]" << std::endl;
        o << ds->GetNumberOfCells() << std::endl;

        for ( i = 0; i < ds->GetNumberOfCells(); i++ ) {
            // write it twice
            for ( j = 0;j < ds->GetCell ( i )->GetNumberOfPoints();j++ )
                o << ds->GetCell ( i )->GetPointId ( j ) << " ";

            for ( j = 0;j < ds->GetCell ( i )->GetNumberOfPoints();j++ )
                o << ds->GetCell ( i )->GetPointId ( j ) << " ";

            o << std::endl;
        }
        return true;

    }
    else
        return false;
}


// -------------------- getPopupMenu --------------------
QMenu * VtkMeshComponent::getPopupMenu ( QWidget* parent ) {
    if ( !myPopupMenu ) {
        myPopupMenu = new VtkMeshComponentPopup ( this, parent );
    }

    return myPopupMenu; // Do NOT forget to delete this object in the destructor
}

// --------------------   getPropertyObject   ---------------------------------
QObject * VtkMeshComponent::getPropertyObject() {
    if (!myProperties) {
        myProperties = new VtkMeshComponentProperties(this);
    }

    return myProperties; // Do NOT forget to delete this object in the destructor
}

//------------------------ getPixmap ---------------------
#include "vtklogo_20x20.xpm"
QPixmap * VtkMeshComponent::myPixmap = NULL;
QPixmap VtkMeshComponent::getIcon() {
    if ( !myPixmap ) {
        myPixmap = new QPixmap ( vtklogo_20x20 );
    }
    return ( *myPixmap );
}


