#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <algorithm>
#include <unistd.h>
#include <cstring>

#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/err.h>

#include "opensslgenericutil.h"

namespace AuthN {
namespace OpenSSL {

  void trim_blank(std::string& str) {
    // Trim the blank in the begining and end of a string
    std::string::size_type pos1, pos2;
    std::string blanks(" \t");
    pos1 = str.find_first_not_of(blanks);
    pos2 = str.find_last_not_of(blanks);
    str = str.substr(pos1, pos2-pos1+1);
  }

  std::string upper_string(const std::string& str) {
    std::string upper;
    std::transform(str.begin(), str.end(), std::back_inserter(upper), toupper);
    return upper;
  }

  std::string lower_string(const std::string& str) {
    std::string lower;
    std::transform(str.begin(), str.end(), std::back_inserter(lower), tolower);
    return lower;
  }

  std::string get_hash_from_x509str(const std::string& x509_str) {
    X509* cert = NULL;
    X509_NAME* name;
  
    BIO* bio = BIO_new(BIO_s_mem());
    if(bio != NULL) {
      BIO_write(bio, x509_str.c_str(), x509_str.size());
      cert = PEM_read_bio_X509(bio, NULL, NULL, NULL);
    }
    else return std::string();

    std::string hash_str;
    if(cert != NULL) {
      char hash[32];
      memset(hash, 0, 32);
      snprintf(hash, 32, "%08lx", X509_NAME_hash(cert->cert_info->subject));
      hash_str = hash;
      X509_free(cert);
    }

    BIO_set_close(bio, BIO_CLOSE); 
    BIO_free_all(bio);
    return hash_str;
  }

  std::string GetOpenSSLError(int err) {
    std::string errstr;
    unsigned long errors;
    char errors_buf[256] = { 0 };
    if(err == 0) {
      errors = ERR_get_error();
    } else {
      errors = err;
    };
    while(errors != 0){
      ERR_error_string_n(errors,errors_buf,sizeof(errors_buf));
      errors_buf[sizeof(errors_buf)-1] = 0;
      errstr += errors_buf;
      errstr += "\n";
      errors = ERR_get_error();
    };
    return errstr;
  }

}
}


