#ifndef H_CDW_EXTERNAL_TOOLS
#define H_CDW_EXTERNAL_TOOLS

#include <stdbool.h>
#include <stddef.h> /* size_t */

#include "main.h"
#include "cdw_widgets.h"
#include "cdw_cdio.h"
#include "cdw_disc.h"
#include "cdw_ncurses.h"


/* don't change order and values of labels from MKISOFS to SHA512SUM,
   they are used to index a table in external_tools.c */
enum cdw_tool {
	CDW_TOOL_NONE = -1,

	CDW_TOOL_MKISOFS = 0, /* both: original mkisofs and genisoimage */
	CDW_TOOL_CDRECORD,    /* both: original cdrecord and wodim */

	CDW_TOOL_GROWISOFS,
	CDW_TOOL_DVD_RW_FORMAT,
	CDW_TOOL_DVD_RW_MEDIAINFO,

	CDW_TOOL_XORRISO,

	CDW_TOOL_MD5SUM,
	CDW_TOOL_SHA1SUM,
	CDW_TOOL_SHA224SUM,
	CDW_TOOL_SHA256SUM,
	CDW_TOOL_SHA384SUM,
	CDW_TOOL_SHA512SUM,

	CDW_TOOL_LIBCDIO
}; /** \brief Tool used to perform given task */

#define CDW_EXT_TOOLS_N_DIGEST_TOOLS 6 /* md5sum, sha1sum, sha224sum, sha256sum, sha384, sha512sum */


/* twelve external tools available:
   cdrecord, mkisofs,
   growisofs, dvd-rw+mediainfo, dvd-rw+format, xorriso,
   md5sum, sha1sum, sha224sum, sha256sum, sha384, sha512sum;

   I'm not including "libcdio", because this is a library (although
   I should check for it somehow)  */
#define CDW_EXT_TOOLS_N_TOOLS 12


/* maximal number of instances of a tool (locations in filesystem), that can
   be recognized and handled by cdw; 10 instances seems to be reasonable limit */
#define CDW_EXT_TOOLS_N_INSTANCES_MAX 10


cdw_rv_t cdw_ext_tools_init(void);
void cdw_ext_tools_clean(void);

/* basic getters */
const char *cdw_ext_tools_get_tool_fullpath(cdw_id_t tool_id);
const char *cdw_ext_tools_get_tool_name(cdw_id_t tool_id);
bool        cdw_ext_tools_is_tool_available(cdw_id_t tool_id);
bool        cdw_ext_tools_is_digest_tool(cdw_id_t tool_id);

/* additional getters */
cdw_id_t cdw_ext_tools_config_get_cd_handler_id(void);
cdw_id_t cdw_ext_tools_config_get_dvd_handler_id(void);
cdw_id_t cdw_ext_tools_config_get_iso9660_id(void);
bool cdw_ext_tools_config_is_manual(void);
bool cdw_ext_tools_is_cdrecord_wodim(void);
bool cdw_ext_tools_digest_tool_available(void);

typedef struct {
	bool manual_selection;
	CDW_DROPDOWN *iso9660;
	CDW_DROPDOWN *cd_handlers;
	CDW_DROPDOWN *dvd_handlers;
	CDW_DROPDOWN *mkisofs;
	CDW_DROPDOWN *cdrecord;
	CDW_DROPDOWN *digest_handlers;
} cdw_ext_tools_configuration_t;



bool cdw_ext_tools_set_burn_tool(cdw_disc_t *disc, cdw_id_label_t *burn_tool, cdw_id_label_t *iso9660_tool);
bool cdw_ext_tools_set_digest_tool(cdw_id_label_t *tool);
bool cdw_ext_tools_set_iso9660_otf_tool(cdw_disc_t *disc, cdw_id_label_t *tool);
bool cdw_ext_tools_set_iso9660_sa_tool(cdw_id_label_t *tool);
bool cdw_ext_tools_set_erase_tool(cdw_disc_t *disc, cdw_id_label_t *tool);
bool cdw_ext_tools_set_media_info_tool(cdw_disc_t *disc, cdw_id_label_t *tool);


/* functions used in cdw configuration window */
CDW_DROPDOWN *cdw_ext_tools_make_iso9660_dropdown(WINDOW *window, int row, int col, int width);
CDW_DROPDOWN *cdw_ext_tools_make_cd_handlers_dropdown(WINDOW *window, int row, int col, int width);
CDW_DROPDOWN *cdw_ext_tools_make_dvd_handlers_dropdown(WINDOW *window, int row, int col, int width);
CDW_DROPDOWN *cdw_ext_tools_make_mkisofs_dropdown(WINDOW *window, int row, int col, int width);
CDW_DROPDOWN *cdw_ext_tools_make_cdrecord_dropdown(WINDOW *window, int row, int col, int width);
CDW_DROPDOWN *cdw_ext_tools_make_digest_handlers_dropdown(WINDOW *window, int row, int col, int width);

void cdw_ext_tools_save_configuration(cdw_ext_tools_configuration_t *etc);


/* debug and error message functions */
void cdw_ext_tools_debug_print_config(void);
cdw_rv_t cdw_ext_tools_error_dialog(cdw_id_t tool_id);


#endif /* H_CDW_EXTERNAL_TOOLS */
