# Copyright 2015 Cloudbase Solutions Srl
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""
Test class for MSFT OCS PowerInterface
"""

import mock

from ironic.common import exception
from ironic.common import states
from ironic.conductor import task_manager
from ironic.drivers.modules.msftocs import common as msftocs_common
from ironic.drivers.modules.msftocs import msftocsclient
from ironic.tests.unit.conductor import mgr_utils
from ironic.tests.unit.db import base as db_base
from ironic.tests.unit.db import utils as db_utils
from ironic.tests.unit.objects import utils as obj_utils

INFO_DICT = db_utils.get_test_msftocs_info()


class MSFTOCSPowerTestCase(db_base.DbTestCase):
    def setUp(self):
        super(MSFTOCSPowerTestCase, self).setUp()
        mgr_utils.mock_the_extension_manager(driver='fake_msftocs')
        self.info = INFO_DICT
        self.node = obj_utils.create_test_node(self.context,
                                               driver='fake_msftocs',
                                               driver_info=self.info)

    def test_get_properties(self):
        expected = msftocs_common.REQUIRED_PROPERTIES
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=True) as task:
            self.assertEqual(expected, task.driver.get_properties())

    @mock.patch.object(msftocs_common, 'parse_driver_info', autospec=True)
    def test_validate(self, mock_drvinfo):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=True) as task:
            task.driver.power.validate(task)
            mock_drvinfo.assert_called_once_with(task.node)

    @mock.patch.object(msftocs_common, 'parse_driver_info', autospec=True)
    def test_validate_fail(self, mock_drvinfo):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=True) as task:
            mock_drvinfo.side_effect = iter(
                [exception.InvalidParameterValue('x')])
            self.assertRaises(exception.InvalidParameterValue,
                              task.driver.power.validate,
                              task)

    @mock.patch.object(msftocs_common, 'get_client_info', autospec=True)
    def test_get_power_state(self, mock_gci):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=True) as task:
            mock_c = mock.MagicMock(spec=msftocsclient.MSFTOCSClientApi)
            blade_id = task.node.driver_info['msftocs_blade_id']
            mock_gci.return_value = (mock_c, blade_id)
            mock_c.get_blade_state.return_value = msftocsclient.POWER_STATUS_ON

            self.assertEqual(states.POWER_ON,
                             task.driver.power.get_power_state(task))
            mock_gci.assert_called_once_with(task.node.driver_info)
            mock_c.get_blade_state.assert_called_once_with(blade_id)

    @mock.patch.object(msftocs_common, 'get_client_info', autospec=True)
    def test_set_power_state_on(self, mock_gci):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=False) as task:
            mock_c = mock.MagicMock(spec=msftocsclient.MSFTOCSClientApi)
            blade_id = task.node.driver_info['msftocs_blade_id']
            mock_gci.return_value = (mock_c, blade_id)

            task.driver.power.set_power_state(task, states.POWER_ON)
            mock_gci.assert_called_once_with(task.node.driver_info)
            mock_c.set_blade_on.assert_called_once_with(blade_id)

    @mock.patch.object(msftocs_common, 'get_client_info', autospec=True)
    def test_set_power_state_off(self, mock_gci):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=False) as task:
            mock_c = mock.MagicMock(spec=msftocsclient.MSFTOCSClientApi)
            blade_id = task.node.driver_info['msftocs_blade_id']
            mock_gci.return_value = (mock_c, blade_id)

            task.driver.power.set_power_state(task, states.POWER_OFF)
            mock_gci.assert_called_once_with(task.node.driver_info)
            mock_c.set_blade_off.assert_called_once_with(blade_id)

    @mock.patch.object(msftocs_common, 'get_client_info', autospec=True)
    def test_set_power_state_blade_on_fail(self, mock_gci):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=False) as task:
            mock_c = mock.MagicMock(spec=msftocsclient.MSFTOCSClientApi)
            blade_id = task.node.driver_info['msftocs_blade_id']
            mock_gci.return_value = (mock_c, blade_id)

            ex = exception.MSFTOCSClientApiException('x')
            mock_c.set_blade_on.side_effect = ex

            pstate = states.POWER_ON
            self.assertRaises(exception.PowerStateFailure,
                              task.driver.power.set_power_state,
                              task, pstate)
            mock_gci.assert_called_once_with(task.node.driver_info)
            mock_c.set_blade_on.assert_called_once_with(blade_id)

    @mock.patch.object(msftocs_common, 'get_client_info', autospec=True)
    def test_set_power_state_invalid_parameter_fail(self, mock_gci):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=False) as task:
            mock_c = mock.MagicMock(spec=msftocsclient.MSFTOCSClientApi)
            blade_id = task.node.driver_info['msftocs_blade_id']
            mock_gci.return_value = (mock_c, blade_id)

            pstate = states.ERROR
            self.assertRaises(exception.InvalidParameterValue,
                              task.driver.power.set_power_state,
                              task, pstate)
            mock_gci.assert_called_once_with(task.node.driver_info)

    @mock.patch.object(msftocs_common, 'get_client_info', autospec=True)
    def test_reboot(self, mock_gci):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=False) as task:
            mock_c = mock.MagicMock(spec=msftocsclient.MSFTOCSClientApi)
            blade_id = task.node.driver_info['msftocs_blade_id']
            mock_gci.return_value = (mock_c, blade_id)

            task.driver.power.reboot(task)
            mock_gci.assert_called_once_with(task.node.driver_info)
            mock_c.set_blade_power_cycle.assert_called_once_with(blade_id)

    @mock.patch.object(msftocs_common, 'get_client_info', autospec=True)
    def test_reboot_fail(self, mock_gci):
        with task_manager.acquire(self.context, self.node.uuid,
                                  shared=False) as task:
            mock_c = mock.MagicMock(spec=msftocsclient.MSFTOCSClientApi)
            blade_id = task.node.driver_info['msftocs_blade_id']
            mock_gci.return_value = (mock_c, blade_id)

            ex = exception.MSFTOCSClientApiException('x')
            mock_c.set_blade_power_cycle.side_effect = ex

            self.assertRaises(exception.PowerStateFailure,
                              task.driver.power.reboot,
                              task)
            mock_gci.assert_called_once_with(task.node.driver_info)
            mock_c.set_blade_power_cycle.assert_called_once_with(blade_id)
