package latexDraw.parsers.svg.elements;

import java.awt.Color;
import java.text.ParseException;

import latexDraw.parsers.svg.*;
import latexDraw.parsers.svg.parsers.CSSStyleParser;
import latexDraw.parsers.svg.parsers.SVGLengthParser;

import org.w3c.dom.*;

/**
 * Defines an SVG element.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public abstract class SVGElement implements Element, Cloneable
{
	/** The attributes of the element. @since 0.1 */
	protected SVGNamedNodeMap attributes;
	
	/** The children of this element. @since 0.1 */
	protected SVGNodeList children;
	
	/** The parent SVGElement of this element. @since 0.1 */
	protected SVGElement parent;
	
	/** The name of the tag. @since 0.1 */
	protected String name;
	
	/** The list of transformations which concern the element. @since 0.1 */
	protected SVGTransformList transform;
	
	/** The list of the CSS styles of the SVG attribute style. @since 0.1 */
	protected CSSStyleList stylesCSS;
	
	/** The document containing the element. @since 0.1 */
	protected SVGDocument ownerDocument;
	
	
	
	/**
	 * The constructor by default.
	 * @since 0.1
	 */
	public SVGElement()
	{
		
		children 	= new SVGNodeList();
		attributes 	= null;
		transform	= null;
		stylesCSS	= null;
		parent 		= null;
		name 		= null;
		ownerDocument = null;
	}
	
	
	
	/**
	 * Creates an simple SVGElement with the owner document.
	 * @param owner The owner document.
	 * @since 0.1
	 */
	public SVGElement(SVGDocument owner)
	{
		this();
		
		if(owner==null)
			throw new IllegalArgumentException();
		
		ownerDocument = owner;
	}
	
	
	
	/**
	 * The constructor using a node in order to be initialised.
	 * @param n The node.
	 * @throws MalformedSVGDocument If the element is not well formed.
	 * @since 0.1
	 */
	public SVGElement(Node n) throws MalformedSVGDocument
	{
		this(n, null);
	}

	

	/**
	 * The constructor using a node to create the SVG element and an SVG element to be its parent.
	 * @param n The node.
	 * @param p The parent SVG element.
	 * @throws MalformedSVGDocument If the element is not well formed.
	 * @since 0.1
	 */
	public SVGElement(Node n, SVGElement p) throws MalformedSVGDocument
	{
		this();
		
		if(n==null)
			throw new IllegalArgumentException();
		
		if(p!=null)
		{
			ownerDocument = p.getOwnerDocument();
			setParent(p);
		}
		
		setAttributes(n);
		setNodeValue(n.getNodeValue());
		setNodeName(n.getNodeName());
		
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_TRANSFORM);
		
		if(v!=null)
		{
			transform = new SVGTransformList();
			transform.addTransformations(getAttribute(getUsablePrefix()+SVGAttributes.SVG_TRANSFORM));
		}
		
		v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_STYLE);
		
		if(v!=null)
		{
			stylesCSS = new CSSStyleList();
			try { new CSSStyleParser(v, stylesCSS).parse(); }
			catch(ParseException e) {e.printStackTrace(); }
		}
		
		if(!checkAttributes())
			throw new MalformedSVGDocument();
		
		NodeList nl = n.getChildNodes();
		int i, size = nl.getLength();
		
		if(size==1)
			if(nl.item(0).getNodeName().equals("#text") || nl.item(0).getNodeName().equals("#cdata-section"))//$NON-NLS-1$//$NON-NLS-2$
				setTextContent(n.getTextContent());
		
		for(i=0; i<size; i++)
			SVGElementsFactory.createSVGElement(nl.item(i), this);
	}
	
	
	
	protected void setAttributes(Node n)
	{
		if(n==null || n.getAttributes()==null)
			return;
		
		NamedNodeMap nnm = n.getAttributes();
		int i, size = nnm.getLength();
		
		if(size>0)
		{
			if(attributes==null)
				attributes = new SVGNamedNodeMap();
			
			for(i=0; i<size; i++)
				attributes.getAttributes().add(new SVGAttr(nnm.item(i).getNodeName(), nnm.item(i).getNodeValue(), this));
		}
	}
	
	
	
	/**
	 * Allows to get the root of the SVG document.
	 * @return The root.
	 * @since 0.1
	 */
	public SVGElement getRootElement()
	{
		if(parent==null)
			return this;
		
		return parent.getRootElement();
	}

	

	/**
	 * @return the parent.
	 * @since 0.1
	 */
	public SVGElement getParent()
	{
		return parent;
	}



	/**
	 * @param parent the parent to set.
	 * @since 0.1
	 */
	public void setParent(SVGElement parent)
	{
		if(this.parent!=parent)
		{
			if(this.parent!=null)
				this.parent.children.getNodes().remove(this);
				
			this.parent = parent;
			
			if(this.parent!=null && !this.parent.children.getNodes().contains(this))
				this.parent.children.getNodes().add(this);
		}
	}



	public NamedNodeMap getAttributes()
	{
		return attributes;
	}


	
	@Override
	public String toString()
	{
		StringBuffer str = new StringBuffer().append('[');
		int i, size = children==null ? 0 : children.getLength();
		
		str.append("name=").append(name).append(",");//$NON-NLS-1$//$NON-NLS-2$
		
		if(!hasChildNodes())
			str.append("textContent=").append(getTextContent()).append(",");//$NON-NLS-1$//$NON-NLS-2$
		
		str.append("attributes=");//$NON-NLS-1$
		
		if(attributes!=null)
			str.append(attributes.toString());
		
		str.append(", children={");//$NON-NLS-1$
		
		for(i=0; i<size-1; i++)
			str.append(children.item(i).toString()).append(',');
		
		if(size>0)
			str.append(children.getNodes().lastElement().toString());
		
		str.append('}');
		
		return str.append(']').toString();
	}


	
	public String getNodeName()
	{
		return name;
	}



	public void setNodeName(String name)
	{
		this.name = name;
	}



	public String getAttribute(String nameAttr)
	{
		if(attributes==null || nameAttr==null)
			return null;
		
		Node n = attributes.getNamedItem(nameAttr);
		
		if(n!=null)
			return n.getNodeValue();
		
		return null;
	}



	public Attr getAttributeNode(String nameAttr)
	{
		if(attributes==null)
			return null;
		
		return (Attr)attributes.getNamedItem(nameAttr);
	}


	public String getTagName()
	{
		return getNodeName();
	}



	public Node appendChild(Node newChild) throws DOMException
	{
		if(!(newChild instanceof SVGElement))
			throw new DOMException(DOMException.TYPE_MISMATCH_ERR, "SVGElement excepted here.");//$NON-NLS-1$
		
		if(children.getNodes().contains(newChild))
			children.getNodes().remove(newChild);
			
		children.getNodes().add((SVGElement)newChild);
		((SVGElement)newChild).setParent(this);
		
		return newChild;
	}



	public NodeList getChildNodes()
	{
		return children;
	}



	public Node getFirstChild()
	{
		return children==null || children.getNodes()==null || children.getNodes().isEmpty() ? null : children.getNodes().firstElement();
	}



	public Node getLastChild()
	{
		return children==null || children.getNodes()==null || children.getNodes().isEmpty() ? null : children.getNodes().lastElement();
	}



	public short getNodeType()
	{
		return Node.ELEMENT_NODE;
	}



	public SVGDocument getOwnerDocument()
	{
		return ownerDocument;
	}



	public Node getParentNode()
	{
		return parent;
	}



	public boolean hasAttributes()
	{
		return attributes!=null && attributes.getAttributes()!=null && !attributes.getAttributes().isEmpty();
	}



	public boolean hasChildNodes()
	{
		return children!=null && children.getNodes()!=null && !children.getNodes().isEmpty();
	}



	public Node insertBefore(Node newChild, Node refChild) throws DOMException
	{
		boolean ok = false;
		
		if(children!=null && newChild!=null && refChild!=null)
		{
			int pos = children.getNodes().indexOf(refChild);
			
			if(pos!=-1 && newChild instanceof SVGElement)
			{
				children.getNodes().add(pos, (SVGElement)newChild);
				ok = true;
			}
		}
		
		return ok ? newChild : null;
	}




	public boolean isEqualNode(Node node)
	{
		if(node==null || !(node instanceof SVGElement))
			return false;
		
		SVGElement elt = (SVGElement)node;
		String uri = lookupNamespaceURI(null);
		String val = getNodeValue();
		boolean valEq = val==null ? elt.getNodeValue()==null : val.equals(elt.getNodeValue());
		boolean uriEq = uri==null ? elt.lookupNamespaceURI(null)==null : uri.equals(elt.lookupNamespaceURI(null));
		boolean attrEq = attributes==null ? elt.attributes==null : attributes.equals(elt.attributes);
		
		return name.equals(elt.name) && getUsablePrefix().equals(elt.getUsablePrefix()) && uriEq && valEq && attrEq;
	}



	public boolean isSameNode(Node other)
	{
		return other!=null && other==this;
	}



	public Node removeChild(Node oldChild) throws DOMException
	{
		boolean ok = false;
		
		if(children!=null && oldChild!=null)
			ok = children.getNodes().remove(oldChild);
		
		return ok ? oldChild : null;
	}



	public void setTextContent(String textContent) throws DOMException
	{
		if(textContent==null)
			throw new DOMException(DOMException.DOMSTRING_SIZE_ERR, "textContent is null.");//$NON-NLS-1$
		
		appendChild(new SVGText(textContent, getOwnerDocument()));
	}


	public String lookupPrefix(String namespaceURI)
	{
		if(namespaceURI == null)
			return null;
        
		String pref = null, xmlns = "xmlns";//$NON-NLS-1$
		
		if(attributes!=null)
		{
			int i=0, size = attributes.getLength();
			boolean again = true;
			
			while(i<size && again)
			{
				SVGAttr attr = attributes.getAttributes().elementAt(i);
				String attrName = attr.getName();
				
				if(attrName!=null && attrName.startsWith(xmlns) && namespaceURI.equals(attr.getValue()))
				{
					int index = attrName.indexOf(':');
					
					pref = index==-1 ? "" : attrName.substring(index+1); //$NON-NLS-1$
					again = false;
				}
				else
					i++;
			}
		}
		
		if(pref!=null)
			return pref;
		
		if(getParentNode()==null)
			return null;
		
		return getParentNode().lookupPrefix(namespaceURI);
	}

	
	public boolean hasAttribute(String nameAttr)
	{ 
		return attributes!=null && attributes.getLength()>0;
	}
	
	
	
	
	public String getNamespaceURI()
	{ 
		return lookupNamespaceURI(getPrefix());
	}
	
	
	
	
	public String lookupNamespaceURI(String pref)
	{ 
		String uri = null;
		
		if(attributes!=null)
			if(pref==null)
			{
				int i=0, size = attributes.getLength();
				boolean again = true;
				String xmlns = "xmlns";//$NON-NLS-1$
				
				while(i<size && again)
				{
					SVGAttr attr = attributes.getAttributes().elementAt(i);
					String attrName = attr.getName();
					
					if(attrName!=null && attrName.equals(xmlns))
					{
						uri = attr.getNodeValue();
						again = false;
					}
					else
						i++;
				}
			}
			else
			{
				int i=0, size = attributes.getLength();
				boolean again = true;
				String xmlns = "xmlns:";//$NON-NLS-1$
				
				while(i<size && again)
				{
					SVGAttr attr = attributes.getAttributes().elementAt(i);
					String attrName = attr.getName();
					
					if(attrName!=null && attrName.startsWith(xmlns) && pref.equals(attrName.substring(xmlns.length())))
					{
						uri = attr.getNodeValue();
						again = false;
					}
					else
						i++;
				}
			}
		
		if(uri!=null)
			return uri;
		
		if(getParentNode()==null)
			return null;
		
		return parent.lookupNamespaceURI(pref);
	}
	
	
	
	public String getPrefix()
	{ 
		if(getNodeName()==null)
			return null;
		
		int index = getNodeName().indexOf(':');
		
		if(index!=-1)
			return getNodeName().substring(0, index);
		
		return null;
	}
	
	
	public String getUsablePrefix()
	{ 
		if(getNodeName()==null)
			return "";
		
		int index = getNodeName().indexOf(':');
		
		if(index!=-1)
			return getNodeName().substring(0, index+1);
		
		return "";//$NON-NLS-1$
	}
	
	
	public void setAttribute(String name, String value) throws DOMException
	{ 
		if(value==null || name==null)
			throw new DOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid name or/and value");//$NON-NLS-1$
		
		if(attributes==null)
			attributes = new SVGNamedNodeMap();

		attributes.setNamedItem(new SVGAttr(name, value, this));
	}

	
	public Node getNextSibling()
	{
		if(parent==null)
			return null;
		
		SVGNodeList nl = (SVGNodeList)parent.getChildNodes();
		int index = nl.getNodes().indexOf(this);
		
		if(index==-1)
			return null;
		
		return index+1>=nl.getLength() ? null : nl.getNodes().elementAt(index+1);
		
	}

	
	
	public Node getPreviousSibling()
	{
		if(parent==null)
			return null;
		
		SVGNodeList nl = (SVGNodeList)parent.getChildNodes();
		int index = nl.getNodes().indexOf(this);
		
		if(index==-1)
			return null;
		
		return index-1<0 ? null : nl.getNodes().elementAt(index+1);
	}
	
	
	
	
	public NodeList getElementsByTagName(String nameElt)
	{ 
		if("*".equals(nameElt)) //$NON-NLS-1$
			return getChildNodes();
		
		SVGNodeList nl = new SVGNodeList();
		
		if(nameElt!=null)
		{
			NodeList nl2 = getChildNodes();
			int i, size = nl2.getLength();
			Node n;
			
			for(i=0; i<size; i++)
			{
				n = nl2.item(i);
				
				if(n!=null && nameElt.equals(n.getNodeName()) && n instanceof SVGElement)
					nl.getNodes().add((SVGElement)n);
			}
		}
		
		return nl;
	}
		
	

	
	public NodeList getElementsByTagNameNS(String namespaceURI, String localName) throws DOMException
	{
		String all = "*"; //$NON-NLS-1$
		
		if(all.equals(namespaceURI))
			return getElementsByTagName(localName);
		
		SVGNodeList nl = new SVGNodeList();
		
		if(namespaceURI!=null && localName!=null)
		{
			boolean getAll = all.equals(localName);
			NodeList nl2 = getChildNodes();
			int i, size = nl2.getLength();
			Node n;
			
			for(i=0; i<size; i++)
			{
				n = nl2.item(i);
				
				if(n!=null && n instanceof SVGElement && namespaceURI.equals(n.getNamespaceURI()) && (getAll || n.getNodeName().endsWith(localName)))
						nl.getNodes().add((SVGElement)n);
			}
		}
		
		return nl;
	}
	
	
	
	public String getTextContent() throws DOMException
	{
		String content = ""; //$NON-NLS-1$
		NodeList nl = getElementsByTagName(SVGText.TEXT_NODE_NAME);
		
		for(int i=0, size=nl.getLength(); i<size; i++)
			content += ((SVGText)nl.item(i)).getData();
		
		return content;
	}
	
	
	public String getLocalName()
	{ 
		return name; 
	}
	
	
	public void removeAttribute(String nameAttr) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void removeAttributeNS(String namespaceURI, String localName) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Attr removeAttributeNode(Attr oldAttr) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void setAttributeNS(String namespaceURI, String qualifiedName, String value) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Attr setAttributeNode(Attr newAttr) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Attr setAttributeNodeNS(Attr newAttr) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void setIdAttribute(String name, boolean isId) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void setIdAttributeNS(String namespaceURI, String localName, boolean isId) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void setIdAttributeNode(Attr idAttr, boolean isId) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Node cloneNode(boolean deep)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public short compareDocumentPosition(Node other) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public String getBaseURI()
	{ return null; }

	public TypeInfo getSchemaTypeInfo()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	public Object getFeature(String feature, String version)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	public Object getUserData(String key)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	public boolean isDefaultNamespace(String namespaceURI)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	public boolean isSupported(String feature, String version)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void normalize()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	public Node replaceChild(Node newChild, Node oldChild) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	public Object setUserData(String key, Object data, UserDataHandler handler)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void setPrefix(String prefix) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	
	public String getNodeValue() throws DOMException
	{ return null; }

	public void setNodeValue(String nodeValue) throws DOMException
	{ /* No value. */ }
	
	
	/**
	 * Returns the prefix of the given namespace URI with the ':' character after or an empty string
	 * if no prefix is found.
	 * @param namespaceURI The URI to look for.
	 * @return the prefix followed by ':' or an empty string.
	 * @since 0.1
	 */
	public String lookupPrefixUsable(String namespaceURI)
	{
		String pref = lookupPrefix(namespaceURI);
		
		if(pref==null)
			pref = "";//$NON-NLS-1$
		else
			pref = pref + ":";//$NON-NLS-1$
		
		return pref;
	}
	
	
	/**
	 * Check if the SVG element is valid according to the SVG specification.
	 * @return True if valid.
	 * @since 0.1
	 */
	public abstract boolean checkAttributes();
	
	
	/**
	 * According to the SVG specification, some attributes may lead to disables rendering of the element (e.g. width=0, 
	 * height=0,...). This method checks these conditions depending of the SVG element.
	 * @return True if the element can be rendered.
	 * @since 0.1
	 */
	public abstract boolean enableRendering();
	
	
	
	public Attr getAttributeNodeNS(String namespaceURI, String localName)
	{
		if(localName==null)
			return null;
		
		return getAttributeNode(lookupPrefixUsable(namespaceURI) + localName);
	}
	
	
	public String getAttributeNS(String namespaceURI, String localName) throws DOMException
	{
		if(localName==null)
			return null;
		
		return getAttribute(lookupPrefixUsable(namespaceURI) + localName);
	}
	
	
	
	public boolean hasAttributeNS(String namespaceURI, String localName) throws DOMException
	{
		return  getAttributeNodeNS(namespaceURI, localName)!=null;
	}
	
	
	/**
	 * @return The root element of the SVG document: an SVGSVGElement.
	 * @since 0.1
	 */
	public SVGSVGElement getSVGRoot()
	{
		SVGElement e = getRootElement();
		
		if(e instanceof SVGSVGElement)
			return (SVGSVGElement)e;
		
		return null;
	}
	
	
	
	/**
	 * Allows to get a definition (a tag in the 'def' part) with the identifier 'id'.
	 * @param id The identifier of the wanted definition.
	 * @return The definition or null.
	 * @since 0.1
	 */
	public SVGElement getDef(String id)
	{
		SVGSVGElement root = getSVGRoot();
		
		if(root==null)
			return null;
		
		return root.getDefs().getDef(id);
	}
	
	
	
	/**
	 * @return The identifier of the SVGElement.
	 * @since 0.1
	 */
	public String getId()
	{
		return getAttribute(getUsablePrefix()+SVGAttributes.SVG_ID);
	}
	
	
	
	/**
	 * Allow to get a set of children having the name 'nodeName'
	 * @param nodeName The name of the wanted nodes.
	 * @return The set of nodes (may be empty but not null).
	 * @since 0.1
	 */
	public SVGNodeList getChildren(String nodeName)
	{
		SVGNodeList nl 	= new SVGNodeList();
		NodeList nl2 	=  getChildNodes();
		
		for(int i=0, size = nl2.getLength(); i<size; i++)
			if(nl2.item(i).getNodeName().equals(nodeName))
				nl.getNodes().add((SVGElement)nl2.item(i));
		
		return nl;
	}
	
	
	
	/**
	 * Sets the stroke width of the SVG shape.
	 * @param strokeW The new stroke width (must be greater than 0).
	 * @since 0.1
	 */
	public void setStrokeWidth(double strokeW)
	{
		if(strokeW>0)
			setAttribute(getUsablePrefix()+SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(strokeW));
	}
	
	
	
	/**
	 * @return The stroke width of the element (if it is possible) or 1.
	 * @since 0.1
	 */
	public double getStrokeWidth()
	{
		String swStr = getSVGAttribute(SVGAttributes.SVG_STROKE_WIDTH, getUsablePrefix());
		double sw;
		
		try { 
			sw = swStr==null ? parent==null ? 1 : parent.getStrokeWidth() : 
				new SVGLengthParser(swStr).parseLength().getValue();
		} catch(ParseException e){ sw = 1; }
		
		return sw;
	}
	
	
	
	/**
	 * @return The dash array of the element (if it is possible) or null.
	 * @since 0.1
	 */
	public String getStrokeDasharray()
	{
		String da = getSVGAttribute(SVGAttributes.SVG_STROKE_DASHARRAY, getUsablePrefix());
		
		return da==null ? parent==null ? SVGAttributes.SVG_VALUE_NONE : parent.getStrokeDasharray() : da;
	}
	
	
	
	/**
	 * @return The linecap of the element (if it is possible) or null.
	 * @since 0.1
	 */
	public String getLinecap()
	{
		String linecap = getSVGAttribute(SVGAttributes.SVG_STROKE_LINECAP, getUsablePrefix());

		return linecap==null ? parent==null ? SVGAttributes.SVG_LINECAP_VALUE_BUTT : parent.getLinecap() : linecap;
	}
	
	
	
	/**
	 * Sets the colour of the filling.
	 * @param c The new filling colour.
	 * @since 0.1
	 */
	public void setFill(Color c)
	{
		if(c!=null)
			setAttribute(getUsablePrefix()+SVGAttributes.SVG_FILL, CSSColors.getColorName(c, true));
	}
	
	
	
	/**
	 * @return The fill content of the element (if it is possible) or null.
	 * @since 0.1
	 */
	public String getFill()
	{
		String fill = getSVGAttribute(SVGAttributes.SVG_FILL, getUsablePrefix());
		
		return fill==null ? parent==null ? CSSColors.CSS_BLACK_NAME : parent.getFill() : fill;
	}
	
	
	/**
	 * @return The font-size value in point of the element, or from one of its parents.
	 */
	public float getFontSize() {
		String fs = getSVGAttribute(SVGAttributes.SVG_FONT_SIZE, getUsablePrefix());
		
		return fs==null ? parent==null ? SVGLengthParser.FontSize.MEDIUM.getPointValue() : 
					parent.getFontSize() : SVGLengthParser.fontSizetoPoint(fs);
	}
	
	
	/**
	 * @return The defined or inherited font family. Otherwise, an empty string.
	 */
	public String getFontFamily() {
		String fam = getSVGAttribute(SVGAttributes.SVG_FONT_FAMILY, getUsablePrefix());
		
		return fam==null ? parent==null ? "" : parent.getFontFamily() : fam;
	}
	
	
	/**
	 * @return The defined or inherited font style. Otherwise, the default value "normal" is returned.
	 */
	public String getFontStyle() {
		String style = getSVGAttribute(SVGAttributes.SVG_FONT_STYLE, getUsablePrefix());
		
		return style==null ? parent==null ? SVGAttributes.SVG_FONT_STYLE_NORMAL : parent.getFontStyle() : style;
	}
	
	
	/**
	 * @return The defined or inherited font weight. Otherwise, the default value "normal" is returned.
	 */
	public String getFontWeight() {
		String weight = getSVGAttribute(SVGAttributes.SVG_FONT_WEIGHT, getUsablePrefix());
		
		return weight==null ? parent==null ? SVGAttributes.SVG_FONT_WEIGHT_NORMAL : parent.getFontWeight() : weight;
	}
	
	
	
	/**
	 * Sets The colour of the stroke.
	 * @param c The new colour of the stroke (must not be null).
	 * @since 0.1
	 */
	public void setStroke(Color c)
	{
		if(c!=null)
			setAttribute(getUsablePrefix()+SVGAttributes.SVG_STROKE, CSSColors.getColorName(c, true));
	}
	
	
	
	/**
	 * @return The fill content of the element (if it is possible) or null.
	 * @since 0.1
	 */
	public Color getStroke()
	{
		String stroke = getSVGAttribute(SVGAttributes.SVG_STROKE, getUsablePrefix());
		
		return stroke==null ? parent==null ? null : parent.getStroke() : CSSColors.getRGBColour(stroke);
	}
	
	
	
	/**
	 * @return The prefix followed by ':' if there is a prefix. An empty string is returned in the other case.
	 * @since 0.1
	 */
	public String getUsablePrefix(String uri)
	{
		String pref = lookupPrefix(uri);
		
		return pref==null || pref.length()==0 ? "" : pref+':';//$NON-NLS-1$
	}



	/**
	 * @return The list of transformations of the current SVG element (may be null).
	 * @since 0.1
	 */
	public SVGTransformList getTransform()
	{
		return transform;
	}
	
	
	
	
	/**
	 * @return The list of all the transformations of the node's parents followed by the node's transformations. 
	 * The first transformations will be the transformations of the oldest parent and the last ones, the 
	 * transformations of the current node. If no transformation are defined, an empty list is returned.
	 * @since 0.1
	 */
	public SVGTransformList getWholeTransform()
	{
		SVGTransformList tl = new SVGTransformList(); 	// The list that will be returned.
		SVGElement p = getParent();						// A parent element.
		
		if(getTransform()!=null)
			tl.addAll(getTransform());
		
		while(p!=null)
		{
			if(p.getTransform()!=null)
				tl.addAll(0, p.getTransform());
			
			p = p.getParent();
		}
		
		return tl;
	}
	
	
	
	/**
	 * Sets the owner document of the node.
	 * @param doc The document to set.
	 * @since 2.0.0
	 */
	public void setOwnerDocument(SVGDocument doc)
	{
		if(doc!=null)
		{
			ownerDocument = doc;
			
			for(int i=0, size=children.getLength(); i<size; i++)
				children.item(i).setOwnerDocument(doc);
		}
	}
	
	
	
	public int getNbChildren(int deep)
	{
		if(children==null || deep<1)
			return 0;
		
		int cpt = children.getLength();
		
		for(SVGElement e : children.getNodes())
			cpt += e.getNbChildren(deep-1);
		
		return cpt;
	}


	/**
	 * @return the stylesCSS
	 * @since 0.1
	 */
	public CSSStyleList getStylesCSS()
	{
		return stylesCSS;
	}
	
	
	
	
	/**
	 * An SVG attribute can be defined in: its corresponding attribute (e.g. fill="...");
	 * the attribute style (e.g. style="fill:...;..."); a CSS stylesheet. This function
	 * returns the value of an SVG attribute by testing: 1. Its corresponding attribute;
	 * 2. The attribute style is 1. fails. TODO CSS stylesheet are not managed yet.
	 * @param attrName The name of the researched attribute.
	 * @param prefix The usable prefix.
	 * @return The found value or null.
	 * @since 2.0.6
	 */
	public String getSVGAttribute(String attrName, String prefix) {
		if(attrName==null) return null;
		
		String value = getAttribute((prefix==null?"":prefix) + attrName);
		
		if(value==null && stylesCSS!=null)
			value = stylesCSS.get(attrName);
		
		return value;
	}
}
