/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.transaction;

import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.util.TestingUtil;
import org.jboss.cache.interceptors.OrderedSynchronizationHandler;
import org.jboss.cache.transaction.NotifyingTransactionManager.Notification;
import org.jgroups.JChannel;
import static org.testng.AssertJUnit.*;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.transaction.RollbackException;
import javax.transaction.Synchronization;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

/**
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 */
@Test(groups = {"functional"})
public class AbortionTest
{
   private CacheSPI cache1, cache2, cache3;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      cache1 = initCache(false);
      TestingUtil.sleepThread(1500); // to ensure cache1 is the coordinator
      cache2 = initCache(false);
      cache3 = initCache(true);
   }

   @AfterMethod(alwaysRun = true)
   public void tearDown() throws Exception
   {
      TestingUtil.killCaches(cache3, cache2, cache1);
   }

   private CacheSPI initCache(boolean notifying)
   {
      CacheSPI c = (CacheSPI) new DefaultCacheFactory().createCache(false);
      c.getConfiguration().setCacheMode("REPL_SYNC");
      c.getConfiguration().setClusterConfig(getJGroupsStack());
      c.getConfiguration().setFetchInMemoryState(false);
      if (!notifying)
      {
         c.getConfiguration().setTransactionManagerLookupClass("org.jboss.cache.transaction.DummyTransactionManagerLookup");
      }
      else
      {
         c.getConfiguration().setTransactionManagerLookupClass("org.jboss.cache.transaction.NotifyingTransactionManager");
      }
      c.start();
      return c;
   }

   // we need a 'special' stack that does not attempt redelivery since we kill a channel midway during a tx in this test.
   private String getJGroupsStack()
   {
      return JChannel.DEFAULT_PROTOCOL_STACK;
   }

   private void destroyCache(CacheSPI c)
   {
      if (c != null)
      {
         c.stop();
         c.destroy();
      }
   }

   public void testSyncCaches() throws Exception
   {
      performTest(false, false);
   }

   public void testSyncCachesSyncCommitRollback() throws Exception
   {
      performTest(true, false);
   }

   /**
    * Note that this tests a *remote* beforeCompletion abort - which is a part of the calling instance's afterCompletion.
    *
    * @throws Exception
    */
   public void testAbortBeforeCompletion() throws Exception
   {
      performTest(true, true);
   }

   private void performTest(boolean syncCommitRollback, boolean abortBeforeCompletion) throws Exception
   {
      cache1.getConfiguration().setSyncCommitPhase(syncCommitRollback);
      cache1.getConfiguration().setSyncRollbackPhase(syncCommitRollback);
      cache2.getConfiguration().setSyncCommitPhase(syncCommitRollback);
      cache2.getConfiguration().setSyncRollbackPhase(syncCommitRollback);
      cache3.getConfiguration().setSyncCommitPhase(syncCommitRollback);
      cache3.getConfiguration().setSyncRollbackPhase(syncCommitRollback);

      TransactionManager mgr1 = cache1.getTransactionManager();
      TransactionManager mgr2 = cache2.getTransactionManager();
      assertTrue(cache3.getTransactionManager() instanceof NotifyingTransactionManager);
      NotifyingTransactionManager mgr3 = (NotifyingTransactionManager) cache3.getTransactionManager();
      mgr3.setCache(cache3);

      assertSame(mgr1, mgr2);
      assertNotSame(mgr1, mgr3);
      assertNotSame(mgr2, mgr3);

      assertTrue(mgr1 instanceof DummyTransactionManager);
      assertTrue(mgr2 instanceof DummyTransactionManager);

      cache1.put("/test", "key", "value");

      assertEquals("value", cache1.get("/test", "key"));
      assertEquals("value", cache2.get("/test", "key"));
      assertEquals("value", cache3.get("/test", "key"));

      mgr3.setNotification(new TestNotification(abortBeforeCompletion));

      mgr1.begin();
      cache1.put("/test", "key", "value2");
      mgr1.commit();

      TestingUtil.sleepThread(5000);

      // only test cache1 and cache2.  Assume cache3 has crashed out.
      assertEquals(0, cache1.getNumberOfLocksHeld());
      assertEquals(0, cache2.getNumberOfLocksHeld());
      assertEquals("put in transaction should NOT have been rolled back", "value2", cache1.get("/test", "key"));
      assertEquals("put in transaction should NOT have been rolled back", "value2", cache2.get("/test", "key"));

   }

   class TestNotification implements Notification
   {
      boolean abortBeforeCompletion;

      public TestNotification(boolean abortBeforeCompletion)
      {
         this.abortBeforeCompletion = abortBeforeCompletion;
      }

      public void notify(Transaction tx, TransactionEntry entry) throws SystemException, RollbackException
      {
         OrderedSynchronizationHandler osh = entry.getOrderedSynchronizationHandler();

         final Transaction finalTx = tx;
         System.out.println("Notify called.");
         // add an aborting sync handler.
         Synchronization abort = new Synchronization()
         {

            public void beforeCompletion()
            {
               if (abortBeforeCompletion)
               {
                  cache3.getConfiguration().getRuntimeConfig().getChannel().close();
                  System.out.println("Returning from abort.beforeCompletion");
                  try
                  {
                     finalTx.setRollbackOnly();
                  }
                  catch (SystemException e)
                  {
                     throw new RuntimeException("Unable to set rollback", e);
                  }
                  throw new RuntimeException("Dummy exception");
               }
            }

            public void afterCompletion(int i)
            {
               if (!abortBeforeCompletion)
               {
                  cache3.getConfiguration().getRuntimeConfig().getChannel().close();
                  System.out.println("Returning from abort.afterCompletion");
                  throw new RuntimeException("Dummy exception");
               }
            }
         };

         osh.registerAtHead(abort);
         System.out.println("Added sync handler.");
      }

   }
}
