/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_DBUSSERVICE_H
#define MEDIASCANNER_DBUSSERVICE_H

// C++ Standard Library
#include <set>
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/dbusutils.h"
#include "mediascanner/mediautils.h"

namespace mediascanner {
namespace dbus {

using std::set;
using std::string;
using std::vector;

enum MediaChangeType {
    MEDIA_INFO_CREATED,
    MEDIA_INFO_UPDATED,
    MEDIA_INFO_REMOVED
};

template<typename InterfaceType>
class MediaScannerInterface : public InterfaceType {
private:
    typedef typename MethodTrait
            < InterfaceType,
              ArgumentList<string>,
              ArgumentList<bool> >::
              type MediaInfoExistsMethodType;
    typedef typename MethodTrait
            < InterfaceType,
              ArgumentList<string, vector<string> >,
              ArgumentList<MediaInfo> >::
              type LookupMediaInfoMethodType;
    typedef typename MethodTrait
            < InterfaceType,
              ArgumentList<string, vector<string>, int32_t, int32_t> >::
              type QueryMediaInfoMethodType;
    typedef typename MethodTrait
            < InterfaceType,
              ArgumentList< MediaInfo >,
              ArgumentList< set<string> > >::
              type StoreMediaInfoMethodType;
    typedef typename MethodTrait
            < InterfaceType, ArgumentList<string> >::
              type RemoveMediaInfoMethodType;

    typedef typename PropertyTrait
            < InterfaceType, ReadOnly, string >::
              type IndexPathPropertyType;
    typedef typename PropertyTrait
            < InterfaceType, ReadOnly, vector<string> >::
              type MediaRootsPropertyType;

    typedef typename SignalTrait
            < InterfaceType, ArgumentList<uint32_t, vector<MediaInfo> > >::
              type MediaInfoAvailableSignalType;

    typedef typename SignalTrait
            < InterfaceType, ArgumentList<MediaChangeType, vector<string> > >::
              type MediaInfoChangedSignalType;

public:
    class MediaInfoExistsMethod : public MediaInfoExistsMethodType {
    public:
        typedef MediaInfoExistsMethodType inherited;
        typedef typename inherited::input_args_type input_args_type;
        typedef typename inherited::output_args_type output_args_type;

        explicit MediaInfoExistsMethod(InterfaceType *parent = nullptr)
            : inherited(parent, "MediaInfoExists",
                        input_args_type(Argument<string>("url")),
                        output_args_type(Argument<bool>("exists"))) {
        }
    };

    class LookupMediaInfoMethod : public LookupMediaInfoMethodType {
    public:
        typedef LookupMediaInfoMethodType inherited;
        typedef typename inherited::input_args_type input_args_type;
        typedef typename inherited::output_args_type output_args_type;

        explicit LookupMediaInfoMethod(InterfaceType *parent = nullptr)
            : inherited(parent, "LookupMediaInfo",
                        input_args_type(Argument<string>("url"),
                                        Argument<vector<string> >("fields")),
                        output_args_type(Argument<MediaInfo>("item"))) {
        }
    };

    class QueryMediaInfoMethod : public QueryMediaInfoMethodType {
    public:
        typedef QueryMediaInfoMethodType inherited;
        typedef typename inherited::input_args_type input_args_type;
        typedef typename inherited::output_args_type output_args_type;

        explicit QueryMediaInfoMethod(InterfaceType *parent = nullptr)
            : inherited(parent, "QueryMediaInfo",
                        input_args_type(Argument<string>("query"),
                                        Argument<vector<string> >("fields"),
                                        Argument<int32_t>("offset"),
                                        Argument<int32_t>("limit"))) {
        }
    };

    /**
     * @brief This class implements the @c StoreMediaInfo method of the media
     * scanner's D-Bus service. It inserts new media information into the index.
     * The actual URL of the media information is retreived from @p item.
     * @param[in] item A key-value map of the information to store.
     * @see WritableMediaIndex::Insert()
     * @ingroup dbus
     */
    class StoreMediaInfoMethod : public StoreMediaInfoMethodType {
    public:
        typedef StoreMediaInfoMethodType inherited;
        typedef typename inherited::input_args_type input_args_type;
        typedef typename inherited::output_args_type output_args_type;

        explicit StoreMediaInfoMethod(InterfaceType *parent = nullptr)
            : inherited(parent, "StoreMediaInfo",
                        input_args_type(Argument<MediaInfo>("item")),
                        output_args_type(Argument<set<string> >("bad_keys"))) {
        }
    };

    /**
     * @brief This class implements the @c RemoveMediaInfo method of the media
     * scanner's D-Bus service. It removes all information about the media
     * referenced by @p url.
     * @param[in] url The URL of the media to remove.
     * @see WritableMediaIndex::Delete()
     * @ingroup dbus
     */
    class RemoveMediaInfoMethod : public RemoveMediaInfoMethodType {
    public:
        typedef RemoveMediaInfoMethodType inherited;
        typedef typename inherited::input_args_type input_args_type;
        typedef typename inherited::output_args_type output_args_type;

        explicit RemoveMediaInfoMethod(InterfaceType *parent = nullptr)
            : inherited(parent, "RemoveMediaInfo",
                        input_args_type(Argument<string>("url"))) {
        }
    };

    /**
     * @brief This class implements the @c IndexPath property of the media
     * scanner's D-Bus service. It contains the local file system path of
     * the media index.
     * @see MediaIndex::path()
     * @ingroup dbus
     */
    class IndexPathProperty : public IndexPathPropertyType {
    public:
        typedef IndexPathPropertyType inherited;

        IndexPathProperty()
            : inherited("IndexPath") {
        }
    };

    /**
     * @brief This class implements the @c MediaRoots property of the media
     * scanner's D-Bus service. It contains the paths of all the directories the
     * file system scanner is watching.
     * the media index.
     * @see FileSystemScanner::directories()
     * @ingroup dbus
     */
    class MediaRootsProperty : public MediaRootsPropertyType {
    public:
        typedef MediaRootsPropertyType inherited;

        MediaRootsProperty()
            : inherited("MediaRoots") {
        }
    };

    class MediaInfoAvailableSignal : public MediaInfoAvailableSignalType {
    public:
        typedef MediaInfoAvailableSignalType inherited;
        typedef typename inherited::args_type args_type;

        MediaInfoAvailableSignal()
            : inherited("MediaInfoAvailable",
                        args_type(Argument<uint32_t>("serial"),
                                  Argument<vector<MediaInfo> >("items"))) {
        }
    };

    /**
     * @brief This class describes the @c MediaInfoChanged signal of the media
     * scanner's D-Bus service. This signal is emitted when the media index
     * updates media information. It is emitted in batches to reduce load.
     * @param items A list of key-value maps of the stored information.
     * @ingroup dbus
     */
    class MediaInfoChangedSignal : public MediaInfoChangedSignalType {
    public:
        typedef MediaInfoChangedSignalType inherited;
        typedef typename inherited::args_type args_type;

        MediaInfoChangedSignal()
            : inherited("MediaInfoChanged",
                        args_type(Argument<MediaChangeType>("type"),
                                  Argument<vector<string> >("urls"))) {
        }
    };

    MediaScannerInterface()
        : InterfaceType("com.canonical.MediaScanner") {
        // FIXME(M5): Initialize members here
    }
};

class MediaScannerProxy : public MediaScannerInterface<InterfaceProxy> {
public:
    MediaScannerProxy()
        : MediaInfoExists(this)
        , LookupMediaInfo(this)
        , QueryMediaInfo(this)
        , StoreMediaInfo(this)
        , RemoveMediaInfo(this) {
    }

    void connect(Wrapper<GDBusConnection> connection,
                 Wrapper<GCancellable> cancellable = Wrapper<GCancellable>());

    void connect(Wrapper<GCancellable> cancellable = Wrapper<GCancellable>());

    bool ConnectAndWait(Wrapper<GDBusConnection> connection,
                        Wrapper<GCancellable> cancellable, GError **error);

    bool ConnectAndWait(Wrapper<GCancellable> cancellable, GError **error);

    const MediaInfoExistsMethod MediaInfoExists;
    const LookupMediaInfoMethod LookupMediaInfo;
    const QueryMediaInfoMethod QueryMediaInfo;
    const StoreMediaInfoMethod StoreMediaInfo;
    const RemoveMediaInfoMethod RemoveMediaInfo;

    const IndexPathProperty index_path;
    const MediaRootsProperty media_roots;

    const MediaInfoAvailableSignal media_info_available;
    const MediaInfoChangedSignal media_info_changed;
};

class MediaScannerSkeleton : public MediaScannerInterface<InterfaceSkeleton> {
public:
    static const string& object_path();
    static const string& service_name();
};

string to_string(MediaChangeType change_type);

} // namespace dbus
} // namespace mediascanner

#endif // MEDIASCANNER_DBUSSERVICE_H
